/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

/**
 * TableFilter_CLM_LIKE は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、DBのLIKE に似た動きで、データを絞ります。
 *
 * vals で指定した値の"%" 記号を、前、後ろ、前後 なし 、で判断します。
 * DBの like と異なるのは、文字列の間の "%" と、1文字あいまいの"_" は判定しません。
 *
 * 新しくテーブルを作成するのではなく、既存のテーブルのデータを物理削除しますので、ご注意ください。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 *
 * @og.formSample
 * ●形式：
 *      ① &lt;og:tableFilter classId="CLM_LIKE" keys="CLM1,CLM2..." vals="判定値1,判定値2..." /&gt;
 *
 *      ② &lt;og:tableFilter classId="CLM_LIKE" &gt;
 *               {
 *                   CLM1   : 判定値1   ;
 *                   CLM2   : 判定値2   ;
 *                   ･･･    : ･･･   ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 8.2.1.1 (2022/07/19) 新規追加
 *
 * @version  8.2 (2022/07/19)
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.8,
 */
public class TableFilter_CLM_LIKE extends AbstractTableFilter {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "8.2.1.1 (2022/07/19)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 8.2.1.1 (2022/07/19) 新規追加
	 */
	public TableFilter_CLM_LIKE() {
		super();
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 8.2.1.1 (2022/07/19) 新規追加
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		final DBTableModel table = getDBTableModel();

		final String[] keys = getKeys();			// 判定対象のカラム名の配列
		final int      len  = keys.length;
		final int[]    clmNo = new int[len];		// カラムの番号

		final String[] vals = new String[len];		// 判定値
		final int[]    like = new int[len];			// 判定方法(-1:判定しない、0:そのまま、1:先頭一致、2:後方一致、3:前後曖昧

		for( int i=0; i<len; i++ ) {
			clmNo[i] = table.getColumnNo( keys[i],false );	// カラムが存在しなければ、-1
			final String val =  StringUtil.nval( getValue( keys[i] ) , null );	// なければ、null
			if( clmNo[i] < 0 || val == null )	{ like[i] = -1; }	// -1:判定しない
			else {
				final char chS = val.charAt(0);						// 先頭文字
				final char chE = val.charAt(val.length()-1);		// 終端文字
				if( '%' == chS && '%' == chE )  { like[i] = 3; vals[i] = val.substring(1,val.length()-1); }	// 3:前後曖昧
				else if( '%' == chS )			{ like[i] = 2; vals[i] = val.substring(1); }				// 2:後方一致
				else if( '%' == chE )			{ like[i] = 1; vals[i] = val.substring(0,val.length()-1); }	// 1:先頭一致
				else							{ like[i] = 0; vals[i] = val; }								// 0:そのまま
			}
		}

		final int rowCnt = table.getRowCount();

		// 不一致の場合は、レコードを削除します。よって、逆順にチェックします。
		for( int row=rowCnt-1; row>=0; row-- ) {
			final String[] data = table.getValues( row );

			boolean okFlag = true ;
			for( int j=0; j<len; j++ ) {
				switch( like[j] ) {
					case 0 : okFlag = okFlag && data[clmNo[j]].equals( vals[j] ); break;		// 0:そのまま
					case 1 : okFlag = okFlag && data[clmNo[j]].startsWith( vals[j] ); break;	// 1:先頭一致
					case 2 : okFlag = okFlag && data[clmNo[j]].endsWith( vals[j] ); break;		// 2:後方一致
					case 3 : okFlag = okFlag && data[clmNo[j]].contains( vals[j] ); break;		// 3:前後曖昧
					default : break;				// 判定しない
				}

				if( !okFlag ) {
					table.removeValue( row );
					break;							// 不一致があれば、抜ける
				}
			}
		}

		return table;
	}
}
