/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */
package com.ibm.trl.tcg.pts.integrity;

import java.io.File;
import java.io.FileInputStream;
import java.io.StringReader;
import java.util.Enumeration;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.DOMException;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.ibm.trl.tcg.pts.engine.Event;
import com.ibm.trl.tcg.pts.engine.FiniteStateMachine;
import com.ibm.trl.tcg.pts.engine.PlatformProperties;
import com.ibm.trl.tcg.pts.engine.PlatformProperty;
import com.ibm.trl.tcg.pts.eventlog.EventSet;
import com.ibm.trl.tcg.pts.tools.Base64Tool;
import com.ibm.trl.tcg.pts.tools.HexTool;
import com.ibm.trl.tcg.pts.tools.PcrComposite;
import com.ibm.trl.tcg.pts.tools.SignatureTool;
import com.ibm.trl.tcg.pts.tools.XmlTool;

/**
 * 
 * Container of TCG IWG VerificationResult
 * 
 * 
 * @author Seiji Munetoh
 * 
 */
public class VerificationResult {
	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	// private String _schemasPath = "../../tcg/schemas/";

	// private String[] _schemas = {
	// _schemasPath + "xmldsig-core-schema.xsd",
	// _schemasPath + "SimpleObject_v8_ibm.xsd",
	// _schemasPath + "Core_Integrity_Manifest_v16_ibm.xsd",
	// _schemasPath + "Integrity_Report_Manifest_v19_ibm.xsd",
	// _schemasPath + "Verification_Result_v1_0_ibm.xsd"
	// //_schemasPath + "VerificationResult_v5_ibm.xsd"
	// };

	private String _filename;

	private String _namespace = "http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/verification_result#";

	private Document _doc;

	private Element _result;

	private boolean _indent = false; // 20071120 SM true->false

	//private TCGSchemas _tcgSchemas;

	private PcrComposite _pcrComposite;

	private boolean _verbose = false;

	private int _iidbIndex = -1;

	private Properties _policyProp = null;

	static int VALID = 0;

	static int INVALID = 1;

	static int UNVERIFIED = 2;

	public VerificationResult() {

		// TODO _tcgSchemas = new TCGSchemas();
		init("UNVERIFIED");
		_iidbIndex = -1;
	}

	public VerificationResult(String filename) {
		this.loadFile(filename);
	}

	/**
	 * Create new VerificationResult
	 */
	public void init(String resultStr) {
		// http://java.sun.com/j2se/1.5.0/ja/docs/ja/api/java/util/UUID.html
		//UUID uuid = UUID.randomUUID();

		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			DOMImplementation domImpl = builder.getDOMImplementation();
			_doc = domImpl.createDocument(_namespace, "VerifyResult", null);

			_result = _doc.getDocumentElement();
			_result
					.setAttribute("xmlns:core",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0_1/core_integrity#");
			_result
					.setAttribute("xmlns:ir",
							"http://www.trustedcomputinggroup.org/XML/SCHEMA/1_0/integrity_report#");
			_result.setAttribute("xmlns:xsi",
					"http://www.w3.org/2001/XMLSchema-instance");

			Element ruuid = _doc.createElement("ResultUUID");
			ruuid.setAttribute("DependentResults", "DependentResults");
			ruuid.appendChild(_doc.createTextNode("Demo"));
			_result.appendChild(ruuid);

			Element rs = _doc.createElement("Results");
			rs.setAttribute("RuleUUID", "RuleUUID");
			rs.setAttribute("Result", resultStr);
			rs.setAttribute("ReportUUID", "ReportUUID");
			rs.setAttribute("ReasonStirings", "TEST");

			Element pid = _doc.createElement("PolicyID");
			pid.setAttribute("UriValue", "CSADSAD");
			rs.appendChild(pid);

			_result.appendChild(rs);

			// ruuid.appendChild(_doc.createTextNode("test"));
			// _result.appendChild(ruuid);

		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		}

	}

	public void save(String filename) throws Exception {
		_filename = filename;
		XmlTool.saveToFile(filename, _doc, _indent);
	}

	/**
	 * 
	 * 
	 * @param filename
	 */
	public void loadFile(String filename) {
		_filename = filename;
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			_doc = builder.parse(new File(_filename));
			_result = _doc.getDocumentElement();
			// NodeList nl = _result.getElementsByTagName("core:ComponentID");
			// _cid = (Element) nl.item(0); // TODO check size

			// log.debug(_root.getFirstChild().getNodeValue());
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void loadXacmlResponse(String filename) throws Exception {
		/* Load */
		// _filename = filename;
		String result = null;

		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			Document doc = builder.parse(new File(filename));

			Element response = doc.getDocumentElement();
			NodeList nl = response.getElementsByTagName("Decision");
			result = nl.item(0).getTextContent();

		} catch (Exception e) {
			e.printStackTrace();
		}

		if (result != null){
			if (result.equals("Permit")) {
				// OK
				init("VALID");

			} else if (result.equals("NotApplicable")) {
				init("INVALID");
				throw new Exception("NotApplicable");
			}
			
		} else {
			throw new Exception(result);
		}
	}

	// public void setSchemaPath(String path) throws Exception {
	// _tcgSchemas = new TCGSchemas(path);
	// //_tcgSchemas.setSchemaDir(path);
	// }

	/**
	 * 
	 * @param result
	 *            0 VALID 1 INVALID 2 UNVERIFIED
	 * 
	 */
	public void setResult(int result) {
		if (result == VALID)
			init("VALID");
		else if (result == UNVERIFIED)
			init("UNVERIFIED");
		else if (result == INVALID)
			init("INVALID");
		else {
			log.error("Wrong result " + result);
			init("INVALID"); //
		}
	}

	/**
	 * Get result string
	 * 
	 * @return
	 */
	public String getResult() {
		NodeList nl = _result.getElementsByTagName("Results");
		Element r = (Element) nl.item(0);
		return r.getAttribute("Result");
	}

	/**
	 * Set reason string (internal format is base64)
	 * 
	 * @param message
	 */
	public void setReason(String message) {
		// 20071121 SM fixed
		String b64 = Base64Tool.encode(message.getBytes());
		
		NodeList nl = _result.getElementsByTagName("Results");
		Element r = (Element) nl.item(0);
		r.setAttribute("ReasonStirings", b64);
	}

	/**
	 * Get reason string
	 * 
	 * @return
	 * @throws Exception 
	 */
	public String getReason() throws Exception {
		NodeList nl = _result.getElementsByTagName("Results");
		Element r = (Element) nl.item(0);
		String b64 = r.getAttribute("ReasonStirings");
		byte[] b = Base64Tool.decode(b64);
		String  str = new String(b);
		return str;
	}

	public void parseString(String message) throws Exception {
		_filename = null;
		DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = dbfactory.newDocumentBuilder();
		_doc = builder.parse(new InputSource(new StringReader(message)));
		_result = _doc.getDocumentElement();
	}

	/**
	 * base64 XML string to VR
	 * 
	 * @param base64
	 * @throws Exception
	 */
	public void parseBase64String(String base64) throws Exception {
		if (base64 == null) {
			throw new Exception("Base64String is null. PTS Server Error");
		}
		parseString(new String(Base64Tool.decode(base64)));
	}

	/**
	 * IR validation - From file Herem we wet Reason and Reqult into VR
	 * 
	 * @param irFilename
	 * @param rmFiles
	 * @param option
	 *            1:zipped file
	 * @return result
	 * @throws Exception 
	 * @throws Exception
	 */
	public VerificationResult validateIRfromFile(
			String irFilename,
			byte[] nonce,
			String[] rmFiles,
			int option) throws Exception  {
		if (log.isTraceEnabled()) {
			log.trace("validateIRfromFile");
			log.trace("  irFilename : " + irFilename);
			if (nonce != null) {
				log.trace("  nonce      : " + HexTool.getHexString(nonce));
			}
			log.trace("  rmFiles[0] : " + rmFiles[0]);
		}

		int result = UNVERIFIED;
		String reason = "";

		/* Validate Quote -> Expected PCR Value */
		IntegrityReport ir;
		EventSet[] events;
		try {
			ir = new IntegrityReport(irFilename, option);
			result = this.validateQuoteData(ir);
			if (result == 0) {
				// Good
				if (log.isDebugEnabled()) {
					log.debug("validateQuoteData ...done ");
				}
			} else {
				/* Wrong Quote/Signature */
				log.error("Wrong Quote/Signature");
			}
			_pcrComposite = ir.getPcrComposite();
			events = ir.getEventSet();
		} catch (Exception e) {
			// Internal Error
			log.error("Exception Error at Validate Quote ", e);
			e.printStackTrace();

			this.setReason("Internal Error");
			this.setResult(UNVERIFIED);
			return this;
		}

		/* init */
		if (log.isTraceEnabled()) {
			log.trace("init PlatformProperties");
		}
		PlatformProperties props = new PlatformProperties();
		props.init();

		/* Validate nonce */
		// try {
		if (nonce != null) {
			if (_pcrComposite == null) {

			} else {
				// TODO
				byte[] buf1 = _pcrComposite.getExternalData();

				if (log.isDebugEnabled()) {
					log.debug("nonce res:" + HexTool.getHexString(buf1));
					log.debug("nonce exp:" + HexTool.getHexString(nonce));
				}
				props.setProperty("quote.nonce", Base64Tool.encode(nonce));
			}
		} else {
			if (log.isDebugEnabled()) {
				log.debug("Validate IR without Nonce");
			}
			props.setProperty("quote.nonce", "na");
		}
		// } catch (Exception e) {
		// // Internal Error
		// log.error("Exception Error at Validate Nonce ",e);
		// e.printStackTrace();
		//	
		// this.setReason("Internal Error");
		// this.setResult(UNVERIFIED);
		// return this;
		// }

		/* Validate IML */
		if (log.isTraceEnabled()) {
			log.trace("Validate IML");
		}
		int countValid = 0;
		int countInvalid = 0;
		int countUnverified = 0;
		for (int rmindex = 0; rmindex < rmFiles.length; rmindex++) {
			if (log.isDebugEnabled())
				log.debug("RM " + rmindex);

			/* Setup FSM[] from RM */
			ReferenceManifest rm = new ReferenceManifest(rmFiles[rmindex],
					option);
			// rm.loadFile(rmFiles[rmindex]);

			FiniteStateMachine[] fsms = null;
			try {
				fsms = rm.getValidationModel();
				if (log.isDebugEnabled()) {
					log.debug("FSM # " + fsms.length);
					// for (int i=0;i<fsms.length;i++) {
					// log.debug("FSM[" + i + "]" + fsms[i].getName());
					// }
				}
			} catch (Exception e) {
				log.error("RM does not conains any FSM", e);
				return null;
			}

			/* Validate events by PCR */
			for (int pcrindex = 0; pcrindex < 24; pcrindex++) {
				if (log.isDebugEnabled()) {
					log.debug("RM " + rmindex + " PcrIndex " + pcrindex);
				}

				boolean validate = true;
				boolean noevent = false;
				Event[] event = null;
				FiniteStateMachine fsm = null;

				if (events[pcrindex].pcrIndex != pcrindex) {
					log.error("Wrong Event...");
					return null;
				}

				if (events[pcrindex].size == 0) {
					/* no event */
					if (pcrindex == 17) {
						/* PCR17 VMM */
						if (log.isDebugEnabled()) {
							log.debug("VMM Validate without Event in PCR "
									+ pcrindex);
						}
						noevent = true;
						// TODO revise later
						fsm = fsms[pcrindex];
						if (fsm == null) {
							if (log.isDebugEnabled()) {
								log.debug("Missing FSM for pcrindex="
										+ pcrindex + " SKIP");
							}
							validate = false;
						}
					} else {
						/* else */
						if (log.isDebugEnabled()) {
							log.debug("no event in PCR " + pcrindex + " SKIP");
						}
						validate = false;
					}
				} else {
					/* setup event and validation model(fsm) */
					event = events[pcrindex].getEventList();
					fsm = fsms[pcrindex];
					if (fsm == null) {
						if (log.isDebugEnabled()) {
							log.debug("Missing FSM for pcrindex=" + pcrindex
									+ " SKIP");
						}
						validate = false;
					}
				}

				/* OK, Validate events now */
				if (validate) {
					if (_verbose) {
						System.out.println("--- PCR[" + pcrindex + "] ---");
					}
					if (log.isDebugEnabled()) {
						log.debug("Validate PCR " + pcrindex);
						// for (int i=0;i<event.length;i++) {
						// log.debug(event[i].toString());
						// }
					}

					/* Validate */
					boolean rc = false;
					fsm.reset(props);
					if (_verbose) {
						fsm.setVerbose();
					}

					try {
						/* if validation was faild it issue Exception */
						if (noevent) {
							/* validate without event */
							rc = fsm.validate();
						} else if (_iidbIndex > 0) {
							/* validate with IIDB */
							rc = fsm.validateWithIIDB(event, _iidbIndex);
						} else {
							/* validate without IIDB */
							rc = fsm.validate(event);
						}
						reason += fsm.getReason();
						// System.err.println("REASON : " + reason);

					} catch (Exception e) {
						// TODO For IMA event, Keep validation until last event
						log.error("Internal Error?", e);
						// Fail
						// e.printStackTrace();
						rc = true; // Keep
						reason += "Inertnal error at PCR[" + pcrindex
								+ "] validation,";
						result = INVALID;
						// return null;
					}

					/* Final */
					if (rc == false) {
						if (log.isDebugEnabled()) {
							log.debug("Validation was failed");
						}
						reason += "PCR[" + pcrindex
								+ "] measurement is invalid,";
						// result = INVALID;
						countInvalid++;
					} else {
						// VALID
						if (log.isDebugEnabled()) {
							log.debug("VALID!!!");
						}

						props = fsm.getTveProperties();
						event = fsm.getEvents();

						if (event.length > 0) {
							/* Update Event List for next RM */
							if (log.isDebugEnabled())
								log.debug("Event remains for PCR " + pcrindex);
							events[pcrindex].setEventList(event);
						}
						// reason += "Measurements sounds pretty neat.:-),";
						// result = VALID;
						countValid++;
					}
					if (log.isDebugEnabled())
						log.debug("Validation of PCR " + pcrindex
								+ " events are done...");
				} // validate
			} // pcrindex
		} // rmindex

		if (log.isTraceEnabled()) {
			log.trace("Validate IML - done");
		}

		if (countInvalid > 0) {
			result = INVALID;
		} else if (countUnverified > 0) {
			//
		} else {
			/* Seems no bad measurement */
			reason += "Measurements sounds pretty neat.:-),";
			result = VALID;
		}

		/* Validate Properties by given Policy, PDP */
		if (log.isTraceEnabled()) {
			log.trace("Validate Properties by given Policy,  PDP");
		}

		if (_policyProp != null) {
			String name = null;
			String valueRef;
			String valueActual;
			try {
				int count = 0;
				for (Enumeration e = _policyProp.propertyNames(); e
						.hasMoreElements();) {
					name = (String) e.nextElement();
					valueRef = _policyProp.getProperty(name);
					valueActual = props.getPropertyValueByName(name);

					if (valueRef.equals(valueActual)) { // props.verifyProperty(name,
														// value)) {
						// HIT
						if (log.isDebugEnabled()) {
							log.debug("Policy) " + name + " == " + valueRef
									+ " HIT");
						}

					} else {
						// MISS
						if (log.isDebugEnabled()) {
							log.debug("Policy) " + name + " == " + valueActual
									+ " !=" + valueRef + "(ref) MISS");
						}

						reason += "Policy mismatch (" + name + "!=" + valueRef
								+ " but " + valueActual + "),";
						count++;

					}
				} // for
				if (count > 0) {
					reason += "some policy violation exist,";
					result = INVALID;
				} else {
					reason += "no policy violation,";
					// result = INVALID;
				}

			} catch (Exception e1) {
				reason += "Sorry, PTS Internal Error,";
				// Validation Fail
				// e1.printStackTrace();
				result = INVALID;
			}
		}

		if (result == VALID) {
			/* Validate PcrComposite */
			if (log.isTraceEnabled()) {
				log.trace("Validate PcrComposite");
			}
			if (_pcrComposite != null) {
				int selectedPcrNums = _pcrComposite.getSelectedPcrNums();

				for (int i = 0; i < selectedPcrNums; i++) {
					int pcrindex = _pcrComposite.getPcrIndex(i);
					byte[] pcr = _pcrComposite.getPcrValue(i);

					PlatformProperty p;
					byte[] pcr2 = null;
					try {
						p = props.getPropertyByName("tpm.pcr." + pcrindex);
						pcr2 = Base64Tool.decode(p.getValue());
					} catch (Exception e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}

					int rc = VALID;
					for (int k = 0; k < 20; k++) {
						if (pcr[k] != pcr2[k]) {
							rc = INVALID;
						}
					}

					if (rc == VALID) {
						// OK
						if (log.isDebugEnabled())
							log.debug("good PCR pcrindex " + pcrindex);
						// } else if (rc == 2) {
						// // IML < PCR Selection/Quote
						// if (log.isDebugEnabled()) log.debug("unverified PCR
						// pcrindex " + pcrindex);
						// reason += "PCR[" + pcrindex +"] is unverified,";
					} else {
						// NG
						log.error("wrong PCR pcrindex " + pcrindex);
						log.error("PCR(Quote) " + HexTool.getHexString(pcr));
						log.error("PCR(Calc)  " + HexTool.getHexString(pcr2));
						reason += "PCR[" + pcrindex
								+ "] value does not match with signature,";
					}
				}
			} else { // no QuoteInfo
				props.setProperty("quote", "unverified");
			}

		}
		if (log.isDebugEnabled())
			log.debug("SM DEBUG Validate PcrComposite ...done ");

		if (_verbose) {
			System.out.println("--- properties ---");
			props.printProperties();
			System.out.println("------------------");
		}
		props.log();

		this.setResult(result);
		this.setReason(reason);
		if (log.isDebugEnabled()) {
			log.debug("Result : " + this.getResult());
			log.debug("Reason : " + this.getReason());
		}
		return this;
	}

	/**
	 * 
	 * TODO Just moved from IR...
	 * 
	 * @return 0 valid 1 invalid 2 unverified
	 * @throws Exception
	 * @throws DOMException
	 */
	public int validateQuoteData(IntegrityReport ir) throws Exception {
		/* Validate Signature */

		NodeList nlKV = ir.getElementsByTagName("KeyValue");

		/* IR without Quote */
		if (nlKV.item(0) == null) {
			if (log.isDebugEnabled()) {
				log.debug("No KeyValue in IR, return 2");
			}
			return 2;
		}

		byte[] pubKey = Base64Tool.decode(nlKV.item(0).getTextContent());

		byte[] message = new byte[48];
		NodeList nlQI = ir.getElementsByTagName("QuoteInfo");
		Element qi = (Element) nlQI.item(0);

		message[0] = 1; // TODO get from IR
		message[1] = 1;// TODO
		message[2] = 0;// TODO
		message[3] = 0;// TODO
		message[4] = 'Q';// TODO
		message[5] = 'U';// TODO
		message[6] = 'O';// TODO
		message[7] = 'T';// TODO
		byte[] dv = Base64Tool.decode(qi.getAttribute("DigestValue"));
		for (int i = 0; i < 20; i++)
			message[8 + i] = dv[i];
		byte[] ed = Base64Tool.decode(qi.getAttribute("ExternalData"));
		for (int i = 0; i < 20; i++)
			message[28 + i] = ed[i];
		// TODO Valdate ED if nonce from server

		NodeList nlSV = ir.getElementsByTagName("SignatureValue");
		byte[] signature = Base64Tool.decode(nlSV.item(0).getTextContent());

		if (log.isTraceEnabled()) {
			log.trace("validate signature ");
			log.trace("  pubKey    : " + HexTool.getHexString(pubKey));
			log.trace("  message   : " + HexTool.getHexString(message));
			log.trace("  signature : " + HexTool.getHexString(signature));
		}

		if (SignatureTool.validate(pubKey, message, signature) == false) {
			if (log.isDebugEnabled()) {
				log.debug("Bad Signature, return 1");
			}
			// TODO 20071119 SM ignoe for a while
			log.error("Bad Signature, BUT CONTINUE");
			// return 1; // INVALID
		}

		/* Validate PCR Composite <-> message */
		/* PcrSelection */

		NodeList nlPS = ir.getElementsByTagName("PcrSelection");
		Element ps = (Element) nlPS.item(0);
		int sizeOfSelect = new Integer(ps.getAttribute("SizeOfSelect"));

		_pcrComposite = new PcrComposite();
		_pcrComposite.setPcrNumber(sizeOfSelect * 8); // TODO

		NodeList nlPV = ir.getElementsByTagName("PcrValue");
		for (int i = 0; i < nlPV.getLength(); i++) {
			Element pv = (Element) nlPV.item(i);
			int pcrindex = new Integer(pv.getAttribute("PcrNumber"));
			_pcrComposite.selectPcr(pcrindex, Base64Tool.decode(pv
					.getTextContent()));
		}

		_pcrComposite.setExtraData(ed);

		if (_pcrComposite.validate(message) == false) {
			if (log.isDebugEnabled()) {
				log.debug("Bad pcrComposite, return 1");
			}
			return 1; // INVALID
		}

		if (log.isDebugEnabled()) {
			log.debug("Good QuoteInfo, return 0");
		}
		return 0; // VALID
		// return 2; // UNVERIFYED
	}

	public void setIIDBIndex(int iidbIndex) {
		_iidbIndex = iidbIndex;
	}

	private void setValidationPolicy(String policyFilename) throws Exception {
		File file = new File(policyFilename);
		if (file.exists() == false) {
			throw new Exception("Policy file (properties file) is not found "
					+ policyFilename);
		}

		/* Load Properties */
		_policyProp = new Properties();
		_policyProp.load(new FileInputStream(policyFilename));

	}

	/**
	 * 
	 * Print Verification Detail to stdout
	 * 
	 */
	private void setVerbose() {
		_verbose = true;

	}

	private static void usage() {
		System.out.println("Usage: validate [OPTIONS]");
		System.out.println(" OPTIONS");
		System.out
				.println(" --create --prop propfile --in irfile --out vrfile");
		System.out.println("     create Verification Result from IML");
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		int opType = 0; // 1:create
		// int opMode = 0; // 0:platform 1:runtime
		// int fileCount = 0;
		String inputFilename = null;
		String outputFilename = null;
		String propFilename = null;
		String policyFilename = null;
		boolean verbose = false;
		int iidbIndex = -1;

		for (int i = 0; i < args.length; ++i) {
			if ("--create".equals(args[i])) {
				opType = 1;
			} else if ("--verbose".equals(args[i])) {
				verbose = true;
			} else if ("--prop".equals(args[i])) {
				propFilename = args[++i];
			} else if ("--policy".equals(args[i])) {
				policyFilename = args[++i];
			} else if ("--in".equals(args[i])) {
				inputFilename = args[++i];
			} else if ("--out".equals(args[i])) {
				outputFilename = args[++i];
			} else if ("--iidb".equals(args[i])) {
				iidbIndex = Integer.valueOf(args[++i]).intValue();// args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}

		try {
			switch (opType) {
			case 1: // create VR from IR
				/* Check Properties */
				String propDir = "";
				File file = new File(propFilename);
				if (file.exists() == false) {
					throw new Exception("IR gen properties file is not found "
							+ propFilename);
				} else {
					propDir = file.getParent();
				}

				/* Load Properties */
				Properties prop = new Properties();
				prop.load(new FileInputStream(propFilename));

				String[] rmFiles = {
						propDir + "/" + prop.getProperty("platform.manifest"),// platformRmFilename,
						propDir + "/" + prop.getProperty("runtime.manifest"),// runtimeRmFilename
				};

				// IntegrityReport ir = new IntegrityReport(inputFilename);
				/* Validation */
				VerificationResult vr = new VerificationResult();
				if (verbose)
					vr.setVerbose();
				if (policyFilename != null)
					vr.setValidationPolicy(policyFilename);
				if (iidbIndex > 0)
					vr.setIIDBIndex(iidbIndex);
				vr.validateIRfromFile(inputFilename, null, rmFiles, 0); // TODO
																		// set
																		// nonce

				System.out.println("Result : " + vr.getResult());
				System.out.println("Reason : " + vr.getReason());
				vr.save(outputFilename);

				break;
			default:
				usage();
				break;

			}
		} catch (Exception e) {
			System.err.println("Internal Error");
			e.printStackTrace();
		}
	}

	public boolean validateXML() throws Exception {
		TCGSchemas schema = new TCGSchemas();
		return schema.validate(_filename);
	}

}
