/*
 * 
 * Licensed Materials - Property of IBM
 *
 * Open Platform Trust Services - An open source TCG PTS
 *
 * (C) Copyright International Business Machines Corp. 2007
 *
 */

package com.ibm.trl.tcg.pts.vulnerability.oval;

import java.io.File;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.helpers.DefaultHandler;

import com.ibatis.sqlmap.client.SqlMapClient;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigIidbCreater;
import com.ibm.trl.tcg.pts.ibatis.SqlConfigVul;
import com.ibm.trl.tcg.pts.ibatis.dto.CveDefinitions;
import com.ibm.trl.tcg.pts.ibatis.dto.Measurements;
import com.ibm.trl.tcg.pts.ibatis.dto.OvalDefinitions;
import com.ibm.trl.tcg.pts.ibatis.dto.OvalObjects;
import com.ibm.trl.tcg.pts.ibatis.dto.OvalTests;
import com.ibm.trl.tcg.pts.ibatis.dto.Packages;
import com.ibm.trl.tcg.pts.vulnerability.rpm.RpmVersionTool;
import com.ibm.trl.tcg.pts.vulnerability.tool.GetFileFromInternet;

/**
 * Scan OVAL definition and pick up vul on packages.
 * 
 * @author Megumi Nakamura, Seiji Munetoh
 */
public class ReadOvalRedhat {

	/* Logger */
	private Log log = LogFactory.getLog(this.getClass());

	private SqlMapClient sqlMapVul = SqlConfigVul.getSqlMapInstance();

	private SqlMapClient sqlMapIidb = null;

	/* (String testId) for rpminfo_tests */
	private ArrayList aryTestId = new ArrayList();

	/* (String testId, String objectId) for rpminfo_tests */
	private TreeMap hmTestObjectRef = new TreeMap();

	/* (String testId, String stateId) for rpminfo_tests */
	private TreeMap hmTestStateRef = new TreeMap();

	/* (String objId, String name) for rpminfo_object */
	private TreeMap hmObjName = new TreeMap();

	/* (String stateId, String{version,evr,sig}) for rpminfo_states */
	private TreeMap hmStates = new TreeMap();

	/* (String testId, String defId) for definition */
	private TreeMap hmTestDef = new TreeMap();

	/* (String defId, String testId) for definition */
	private TreeMap hmDefTest = new TreeMap();

	/* (String defId, String cveId) for definition */
	private TreeMap hmDefCve = new TreeMap();

	/* (Integer package_id, String testId) for each status */
	private TreeMap[] mapPackage = null;

	/* (digest_id) for each status */
	private Vector[] vectorDigest = null;

	private static final int STATE_NUM = 5;

	// status of comparing (temporary)
	private static final int DISTRO_MATCH = 1;

	private static final int DISTRO_NOT_MATCH = 2;

	// status of vulnerability (temporary)
	private static final int NO_VUL = 0;

	private static final int VUL = 2;

	private static final int CHECK_AGAIN_VUL = 5;

	// sum of DISTRO + VUL
	public static final int CHECK_AGAIN = 10;

	public static final int NO_VUL_DISTRO_MATCH = NO_VUL + DISTRO_MATCH; // 1

	public static final int NO_VUL_DISTRO_NOT_MATCH = NO_VUL + DISTRO_NOT_MATCH; // 2

	public static final int VUL_DISTRO_MATCH = VUL + DISTRO_MATCH; // 3

	public static final int VUL_DISTRO_NOT_MATCH = VUL + DISTRO_NOT_MATCH; // 4

	private String xmlFile = null;

	private static final int CVEID_LIMIT = 500;

	private static final int TITLE_LIMIT = 150;

	/* RHEL VERSION */
	private boolean version = false;

	private String rhelVer = null;

	/**
	 * Main method to start to load the xmlfile, parse them, and store into the
	 * database.
	 * 
	 * @param args --dbindex, --xmlfile, --distribution, --output
	 */
	public static void main(String[] args) {
		/* default value */
		int dbIndex = 0;
		String xmlFile = "http://people.redhat.com/mjc/oval/com.redhat.rhsa-all.xml";
		String distribution = "rhel5";
		String directory = null;

		for (int i = 0; i < args.length; ++i) {
			if ("--dbindex".equals(args[i])) {
				dbIndex = Integer.valueOf(args[++i]);
			} else if ("--xmlfile".equals(args[i]) || "-x".equals(args[i])) {
				xmlFile = args[++i];
			} else if ("--distribution".equals(args[i]) || "-d".equals(args[i])) {
				distribution = args[++i];
			} else if ("--output".equals(args[i])) {
				directory = args[++i];
			} else {
				System.err.println("Unknown option " + args[i]);
				usage();
				return;
			}
		}
		try {
			ReadOvalRedhat oval = new ReadOvalRedhat(dbIndex, xmlFile, directory);
			oval.run(distribution);
		} catch (Exception e) {
			e.printStackTrace();
		} // directory is null

	}

	/**
	 * Print the usage for main method.
	 */
	private static void usage() {
		System.out.println("usage: oval [option]");
		System.out.println("--dbindex: (required)");
		System.out.println("--xmlfile, -x:");
		System.out.println("	XML file path or url of OVAL file");
		System.out.println("--distribution, -d:");
		System.out.println("	Database name. ex. rhel5");
		System.out.println("--output:");
		System.out.println("	Directory to store xml file.");
		System.exit(0);
	}

	public ReadOvalRedhat() {
	}

	/**
	 * @param dbIndex Index of Integrity Info DB
	 * @param file File location of xml file
	 * @throws Exception
	 */
	public ReadOvalRedhat(int dbIndex, String file) throws Exception {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
		// Load OVAL XML data
		if (file.startsWith("http:") || file.startsWith("ftp:") || file.startsWith("https")) {
			this.xmlFile = new GetFileFromInternet().getFile(file, null, true);
		} else {
			this.xmlFile = file;
		}
		if (log.isDebugEnabled()) {
			log.debug("XML File: " + this.xmlFile);
		}
	}

	/**
	 * 
	 * @param dbIndex Index of Integrity Info DB
	 * @param file File location of xml file
	 * @param directory Directory for downloaded file
	 * @throws Exception
	 */
	public ReadOvalRedhat(int dbIndex, String file, String directory)
			throws Exception {
		sqlMapIidb = SqlConfigIidbCreater.getSqlMapInstance(dbIndex);
		// Load OVAL XML data
		if (file.startsWith("http:") || file.startsWith("ftp:") || file.startsWith("https")) {
			this.xmlFile = new GetFileFromInternet().getFile(file, directory,
					true);
		} else {
			this.xmlFile = file;
		}
		if (log.isDebugEnabled()) {
			log.debug("XML File: " + this.xmlFile);
		}
	}

	/**
	 * Setup the database connections for the vulnerability database and
	 * integrity info database.
	 */
	private void setupDatabase() {
		try {
			sqlMapVul.startTransaction();
			sqlMapIidb.startTransaction();

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
		setup();
	}

	/**
	 * Set the version number.
	 * 
	 * @param distribution
	 *            The distribution name which is expected to contain the figure
	 *            of version.
	 */
	private void setVersion(String distribution) {
		if (distribution == null) {
			rhelVer = "other"; // anything else is ok for dummy
		} else if (distribution.indexOf("5") > 0) {
			rhelVer = "5";
		} else if (distribution.indexOf("4") > 0) {
			rhelVer = "4";
		} else if (distribution.indexOf("3") > 0) {
			rhelVer = "3";
		} else {
			rhelVer = "other"; // anything else is ok for dummy
		}
		if (log.isDebugEnabled()) {
			log.debug("rhel version: " + distribution + " => " + rhelVer);
		}
	}

	/**
	 * Setup the database statements and Initialize the map and vector array.
	 */
	protected void setup() {
		mapPackage = new TreeMap[STATE_NUM];
		vectorDigest = new Vector[STATE_NUM];
		for (int i = 0; i < STATE_NUM; i++) { // initialize for each status
			mapPackage[i] = new TreeMap();
			vectorDigest[i] = new Vector();
		}
	}

	/**
	 * Running the all process of preparing the file, parsing the xml, inserting
	 * into the database.
	 */
	public void run() {
		run(null);
	}

	/**
	 * Running the all process of preparing the file, parsing the xml, inserting
	 * into the database.
	 * @param distribution Distribution name to be checked
	 */
	public void run(String distribution) {
		setVersion(distribution);
		setupDatabase();

		Element root = loadXmlFile();
		parse(root);
		if (log.isDebugEnabled()) {
			log.debug("# PARSE END");
		}

		// definition
		parseInsertDef(root);

		// Insert OVAL data into database (test, object)
		insertTree();
		if (log.isDebugEnabled()) {
			log.debug("# INSERT OVAL DATA END");
		}

		// Check the vulnerability and insert into the database
		makeSqlQuery();
		makeSqlQuery2UpdateRemainder();

		finish();
	}

	/**
	 * Close the database connections.
	 */
	private void finish() {
		try {
			sqlMapVul.commitTransaction();
			sqlMapIidb.commitTransaction();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

		if (log.isDebugEnabled()) {
			log.debug("# END");
		}
	}

	/**
	 * Load the XML file into DOM tree.
	 * 
	 * @return The root element of the xml file
	 */
	protected Element loadXmlFile() {
		try {
			// DOM
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
//			 dbfactory.setValidating(false);
//			 dbfactory.setNamespaceAware(false);
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			builder.setErrorHandler(new DefaultHandler());

			Document doc = builder.parse(new File(xmlFile));
			if (log.isDebugEnabled()) {
				log.debug("XML File: " + xmlFile);
			}

			Element root = doc.getDocumentElement();
			if (log.isDebugEnabled()) {
				log.debug("root: " + root.getTagName()); // oval
			}
			return root;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Parse and check the test element, the object element and the state
	 * element.
	 * 
	 * @param root
	 *            The root element of the xml file
	 */
	protected void parse(Element root) {
		parseRpmTest(root);
		parseRpmObject(root);
		parseRpmState(root);
	}

	/**
	 * Parse the rpminfo_tests, and store them into TreeMap in external
	 * valiable.
	 * 
	 * @param root
	 *            The root element of the xml file
	 */
	private void parseRpmTest(Element root) {
		NodeList testList = root.getElementsByTagName("rpminfo_test");
		for (int i = 0; i < testList.getLength(); i++) {
			Element testElement = (Element) testList.item(i);
			String testId = testElement.getAttribute("id");
			aryTestId.add(testId);
			Element e2 = (Element) testElement.getElementsByTagName("object")
					.item(0);
			hmTestObjectRef.put(testId, e2.getAttribute("object_ref"));
			Element e3 = (Element) testElement.getElementsByTagName("state")
					.item(0);
			hmTestStateRef.put(testId, e3.getAttribute("state_ref"));
		}
	}

	/**
	 * Parse the rpminfo_object, and store them into TreeMap.
	 * 
	 * @param root
	 *            The root element of the xml file
	 */
	private void parseRpmObject(Element root) {
		NodeList objList = root.getElementsByTagName("rpminfo_object");
		for (int i = 0; i < objList.getLength(); i++) {
			Element e1 = (Element) objList.item(i);
			Element e2 = (Element) e1.getElementsByTagName("name").item(0);
			hmObjName.put(e1.getAttribute("id"), e2.getTextContent());
		}
	}

	/**
	 * Parse the rpminfo_states, and store them into TreeMap.
	 * 
	 * @param root
	 *            the root element of the xml file
	 */
	private void parseRpmState(Element root) {
		NodeList statesList = root.getElementsByTagName("rpminfo_state");
		String statesId = null;
		for (int i = 0; i < statesList.getLength(); i++) {
			Element e1 = (Element) statesList.item(i);
			statesId = e1.getAttribute("id");

			Element eVer = (Element) e1.getElementsByTagName("version").item(0);
			if (eVer != null) {
				hmStates.put(statesId, eVer);
			}
			Element eEvr = (Element) e1.getElementsByTagName("evr").item(0);
			if (eEvr != null) {
				hmStates.put(statesId, eEvr);
			}
			Element eSig = (Element) e1.getElementsByTagName("signature_keyid")
					.item(0);
			if (eSig != null) {
				hmStates.put(statesId, eSig);
			}
		}
	}

	/**
	 * Parse the definition, and store them into the database queue.
	 * 
	 * @param root
	 *            the root element of the xml file
	 */
	protected void parseInsertDef(Element root) {
		NodeList defList = root.getElementsByTagName("definition");
		String cveIds = null;
		try {
			for (int i = 0; i < defList.getLength(); i++) {
				Element e1 = (Element) defList.item(i);
				String defId = e1.getAttribute("id");
				NodeList d2 = e1.getElementsByTagName("metadata");
				Element eMetadata = (Element) d2.item(0);
				Element eTitle = (Element) eMetadata.getElementsByTagName(
						"title").item(0);
				Element eDescription = (Element) eMetadata
						.getElementsByTagName("description").item(0);
				Element eReference = (Element) eMetadata.getElementsByTagName(
						"reference").item(0);
				NodeList d3 = e1.getElementsByTagName("criteria");
				parseCriteria(defId, (Element) d3.item(0));
				// CVE
				if (eReference.getAttribute("source").endsWith("CVE")) {
					String cveId = eReference.getAttribute("ref_id");
					String cveUrl = eReference.getAttribute("ref_url");
					if (cveIds == null) {
						cveIds = cveId;
					} else {
						cveIds = cveIds + " " + cveId;
					}
					CveDefinitions cveDef = new CveDefinitions();
					cveDef.setCveId(cveId);
					cveDef.setCveUrl(cveUrl);
					// check the same cve_id  
					CveDefinitions cveDefCheck = (CveDefinitions)sqlMapVul.queryForObject("getCveDefinitionByCveId", cveDef);
					if(cveDefCheck == null) {
						sqlMapVul.insert("insertCveDefinitionUrl", cveDef);
						sqlMapVul.commitTransaction();
					} else {
						sqlMapVul.update("updateCveDefinitionUrl", cveDef);
						sqlMapVul.commitTransaction();
					}
					
				} else {
					Element eAdvisory = (Element) eMetadata
							.getElementsByTagName("advisory").item(0);
					NodeList nCve = eAdvisory.getElementsByTagName("cve");
					if (nCve.getLength() > 0) {
						for (int j = 0; j < nCve.getLength(); j++) {
							String cveUrl = ((Element) nCve.item(j))
									.getAttribute("href");
							String cveId = ((Element) nCve.item(j))
									.getTextContent();
							if (cveIds == null) {
								cveIds = cveId;
							} else {
								cveIds = cveIds + " " + cveId;
							}
							CveDefinitions cveDef = new CveDefinitions();
							cveDef.setCveId(cveId);
							cveDef.setCveUrl(cveUrl);
							// check the same cve_id  
							CveDefinitions cveDefCheck = (CveDefinitions)sqlMapVul.queryForObject("getCveDefinitionByCveId", cveDef);
							if(cveDefCheck == null) {
								sqlMapVul.insert("insertCveDefinitionUrl", cveDef);
								sqlMapVul.commitTransaction();
							} else {
								sqlMapVul.update("updateCveDefinitionUrl", cveDef);
								sqlMapVul.commitTransaction();
							}
						}
					}
				}
				hmDefCve.put(defId, cveIds);
				// Insert into database (definition)
				if (defId != null && eTitle != null && eDescription != null) {
					String title = eTitle.getTextContent();
					if (cveIds.length() > CVEID_LIMIT) { // TODO
						log
								.error("CVE ID is getting too long > 500, "
										+ cveIds);
						log.error("  CVE ID size :" + cveIds.length());
						log.error("  definition_id :  " + defId);
						log.error("  title              :  " + title);
						// } else {
						// _logger.debug("CVE ID size :" + cveIds.length());
					}
					if (title.length() > TITLE_LIMIT) { // TODO
						log.error("title  is too long > 150, " + title);
						log.error("  title  size :" + title.length());
						log.error("  definition_id :  " + defId);
					}
					OvalDefinitions ovalDef = new OvalDefinitions();
					ovalDef.setOvalDefinitionId(defId);
					ovalDef.setOvalTitle(title);
					ovalDef.setOvalDescription(eReference
							.getAttribute("ref_url"));
					ovalDef.setOvalCve(cveIds);
					// check the same def_id  
					OvalDefinitions ovalDefCheck = (OvalDefinitions)sqlMapVul.queryForObject("getOvalDefinition", ovalDef);
					if(ovalDefCheck == null) {
						sqlMapVul.insert("insertOvalDefinition", ovalDef);
						sqlMapVul.commitTransaction();
					} else {
						sqlMapVul.update("updateOvalDefinition", ovalDef);
						sqlMapVul.commitTransaction();
					}
				}

				cveIds = null;

			}
		} catch (DOMException e) {
			e.printStackTrace();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Parse the criteria element, and store the criterion element into TreeMap.
	 * 
	 * @param defId
	 *            The definition id
	 * @param element
	 *            The criteria element
	 */
	private void parseCriteria(String defId, Element element) {
		if (element != null) {
			// String ope = element.getAttribute("operator");
			// _logger.debug("<OPE> " + ope);
			NodeList childrenNode = element.getChildNodes();
			for (int i = 0; i < childrenNode.getLength(); i++) {
				Node childNode = childrenNode.item(i);
				String childName = childNode.getNodeName();
				if (childName.equalsIgnoreCase("criteria")) {
					parseCriteria(defId, (Element) childNode);
				} else if (childName.equalsIgnoreCase("criterion")) {
					parseCriterion(defId, (Element) childNode);
				}
			}
			// _logger.debug("</OPE> " + ope);
		}
	}

	/**
	 * Get the test_ref from the criterion element, and store them into TreeMap.
	 * 
	 * @param defId
	 *            The definition id
	 * @param element
	 *            The criterion element
	 */
	private void parseCriterion(String defId, Element element) {
		String testId = element.getAttribute("test_ref");
		String comment = element.getAttribute("comment");
		if (comment.startsWith("Red Hat Enterprise Linux")
				&& comment.endsWith("is installed")) {
			if (comment.indexOf(rhelVer) > 0) {
				version = true;
			} else {
				version = false;
			}
		}
		if (version || rhelVer.equalsIgnoreCase("other")) {
			if (log.isDebugEnabled()) {
				log.debug("testId> " + testId + " [" + rhelVer + "]");
			}
			hmTestDef.put(testId, defId);
			hmDefTest.put(defId, testId);
			executeTest(testId);
		}
	}

	/**
	 * Check the parameter of the test case.
	 * 
	 * @param testId
	 *            The test id
	 */
	private void executeTest(String testId) {
		String objName = (String) hmObjName.get((String) hmTestObjectRef
				.get(testId));
		Element stateElement = (Element) hmStates.get((String) hmTestStateRef
				.get(testId));
		if (stateElement.getNodeName() == "version") {
			checkVersionValue(objName, stateElement, testId);
		} else if (stateElement.getNodeName() == "evr") {
			checkEvrValue(objName, stateElement, testId);
		}
		// else if(stateElement.getNodeName() == "signature_keyid") {
		// // TODO
		// }
	}

	/**
	 * Make the sql query from the TreeMap(for oval_test, oval_object), and
	 * execute the sql queries.
	 */
	protected void insertTree() {
		try {
			String objId = null;
			for (int i = 0; i < aryTestId.size(); i++) {
				String testId = (String) aryTestId.get(i);
				objId = (String) hmTestObjectRef.get(testId);
				String statesId = (String) hmTestStateRef.get(testId);
				String defId = (String) hmTestDef.get(testId);

				if (defId != null) {
					OvalTests ovalTst = new OvalTests();
					ovalTst.setOvalTestId(testId);
					ovalTst.setOvalTestDefinitionId(defId);
					ovalTst.setOvalTestObjectId(objId);
					ovalTst.setOvalTestStateId(statesId);
					
					// check the same tst_id  
					OvalTests tstCheck = (OvalTests)sqlMapVul.queryForObject("getOvalTest", ovalTst);
					if(tstCheck == null) {
						sqlMapVul.insert("insertOvalTest", ovalTst);
						sqlMapVul.commitTransaction();
					} else {
						sqlMapVul.update("updateOvalTest", ovalTst);
						sqlMapVul.commitTransaction();
					}
					ovalTst = null;
				}
			}

			List<OvalTests> ovalTst = (List<OvalTests>) sqlMapVul
					.queryForList("getOvalTestAll");
			OvalObjects ovalObj = new OvalObjects();
			for (OvalTests o : ovalTst) {
				objId = o.getOvalTestObjectId();
				String objName = (String) hmObjName.get(objId);
				ovalObj.setOvalObjectId(objId);
				ovalObj.setOvalObjectName(objName);
				if (objId != null && objName != null) {
					// check the same obj_id  
					OvalObjects objCheck = (OvalObjects)sqlMapVul.queryForObject("getOvalObject", ovalObj);
					if(objCheck == null) {
						sqlMapVul.insert("insertOvalObject", ovalObj);
						sqlMapVul.commitTransaction();
					} else {
						sqlMapVul.update("updateOvalObject", ovalObj);
						sqlMapVul.commitTransaction();
					}
				}
			}

		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}

	}

	/**
	 * Store the result in TreeMap and Vector.
	 * 
	 * @param packageId
	 *            The package id
	 * @param status
	 *            The result of checking the vulnerability and the difference of
	 *            distributions
	 * @param testId
	 *            The test id
	 */
	private void storeResult(int packageId, int status, String testId) {
		if (status == NO_VUL_DISTRO_MATCH) { // 1
			mapPackage[NO_VUL_DISTRO_MATCH].put(Integer.valueOf(packageId),
					testId);
			vectorDigest[NO_VUL_DISTRO_MATCH].add(Integer.valueOf(packageId));
		} else if (status == NO_VUL_DISTRO_NOT_MATCH) { // 2
			mapPackage[NO_VUL_DISTRO_NOT_MATCH].put(Integer.valueOf(packageId),
					testId);
			vectorDigest[NO_VUL_DISTRO_NOT_MATCH].add(Integer
					.valueOf(packageId));
		} else if (status == VUL_DISTRO_MATCH) { // 3
			mapPackage[VUL_DISTRO_MATCH]
					.put(Integer.valueOf(packageId), testId);
			vectorDigest[VUL_DISTRO_MATCH].add(Integer.valueOf(packageId));
		} else if (status == VUL_DISTRO_NOT_MATCH) { // 4
			mapPackage[VUL_DISTRO_NOT_MATCH].put(Integer.valueOf(packageId),
					testId);
			vectorDigest[VUL_DISTRO_NOT_MATCH].add(Integer.valueOf(packageId));
		} else if (status == CHECK_AGAIN || status >= CHECK_AGAIN_VUL) {
			// status = CHECK_AGAIN;
			mapPackage[0].put(Integer.valueOf(packageId), testId);
			vectorDigest[0].add(Integer.valueOf(packageId));
		}
	}

	/**
	 * Check the version value.
	 * 
	 * @param objName
	 *            The name of object
	 * @param stateElement
	 *            The state element
	 * @param testId
	 *            The test id
	 */
	private void checkVersionValue(String objName, Element stateElement,
			String testId) {
		try {
			List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByName", objName);
			for (Packages p : pkgs) {
				int packageId = p.getPackageId();
				String version = p.getPackageVersion();
				// oval
				String ovalPattern = stateElement.getTextContent();
				// alphabet check
				int distroStatus = DISTRO_MATCH;
				// pattern match
				int vulStatus = compareVersionPattern(version, ovalPattern);
				// TODO
				storeResult(packageId, distroStatus + vulStatus, testId);
			}

		} catch (DOMException e) {
			e.printStackTrace();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Check the pattern matching of the value.
	 * 
	 * @param dbValue
	 *            The value from the database
	 * @param patternValue
	 *            The pattern compared with the value from the database
	 * @return VUL or NO_VUL
	 */
	private int compareVersionPattern(String dbValue, String patternValue) {
		// TODO
		Pattern p = Pattern.compile(patternValue + "([-.].*)?");
		Matcher m = p.matcher(dbValue);
		if (m.matches()) {
			return VUL;
		} else {
			return NO_VUL;
		}
	}

	/**
	 * Compare the distribution name in the version number.
	 * 
	 * @param s1
	 *            The version which is compared with s2
	 * @param s2
	 *            The version which is compared with s1
	 * @return If the name is the same, DISTRO_MATCH is returned. If not,
	 *         DISTRO_NOT_MATCH is returned.
	 */
	private int checkDistribution(String s1, String s2) {
		String dist1 = RpmVersionTool.checkDistributionVersion(s1);
		String dist2 = RpmVersionTool.checkDistributionVersion(s2);
		if (dist1 != null && dist2 != null) {
			if (dist1.equalsIgnoreCase(dist2)) {
				return DISTRO_MATCH;
			} else {
				return DISTRO_NOT_MATCH;
			}
		} else if (dist1 != null || dist2 != null) {
			return DISTRO_NOT_MATCH;
		} else {
			return DISTRO_MATCH;
		}
	}

	/**
	 * Compare the alphabet in the version number.
	 * 
	 * @param objName
	 *            The name of object
	 * @param stateElement
	 *            The state element
	 * @param testId
	 *            The test id
	 */
	private void checkEvrValue(String objName, Element stateElement,
			String testId) {
		try {
			List<Packages> pkgs = (List<Packages>) sqlMapIidb.queryForList(
					"getPackageByName", objName);
			for (Packages p : pkgs) {
				int packageId = p.getPackageId();
				String version = p.getPackageVersion();
				// oval
				String ovalOperation = stateElement.getAttribute("operation");
				String ovalVersion = stateElement.getTextContent();
				log.debug("check> obj: " + objName + ", ope: " + ovalOperation
						+ ", " + ovalVersion);

				// alphabet check
				int distroStatus = checkDistribution(version, ovalVersion);
				// version check
				int vulStatus = compareEvrVersion(ovalOperation, version,
						ovalVersion);
				// TODO
				storeResult(packageId, distroStatus + vulStatus, testId);
			}
		} catch (DOMException e) {
			e.printStackTrace();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Compare the version number.
	 * 
	 * @param ope
	 *            The operand
	 * @param dbValue
	 *            The version value from the database
	 * @param ovalValue
	 *            The version value from the OVAL file
	 * @return VUL or NO_VUL or CHECK_AGAIN_VUL
	 */
	private int compareEvrVersion(String ope, String dbValue, String ovalValue) {
		int rtn = RpmVersionTool.compareRpmOval(dbValue, ovalValue);

		// true or false
		if (rtn > RpmVersionTool.GREATER) {
			return CHECK_AGAIN_VUL;
		} else if ((rtn == RpmVersionTool.GREATER)
				&& ((ope.equalsIgnoreCase("greater than"))
						|| (ope.equalsIgnoreCase("greater than or equal")) || (ope
						.equalsIgnoreCase("not equals")))) {
			return VUL;
		} else if ((rtn == RpmVersionTool.LESS)
				&& ((ope.equalsIgnoreCase("less than"))
						|| (ope.equalsIgnoreCase("less than or equal")) || (ope
						.equalsIgnoreCase("not equals")))) {
			return VUL;
		} else if ((rtn == RpmVersionTool.EQUAL)
				&& ((ope.equalsIgnoreCase("equals"))
						|| (ope.equalsIgnoreCase("greater than or equal")) || (ope
						.equalsIgnoreCase("less than or equal")))) {
			return VUL;
		} else {
			return NO_VUL;
		}
	}

	/**
	 * Make the sql query from the result.
	 */
	protected void makeSqlQuery() {
		try {
			String[] cvssMinMax = null;
			// Package
			for (int j = 0; j < STATE_NUM; j++) {
				Set packageSet = mapPackage[j].keySet(); // map_package =
															// (package_id,
															// testId)
				Iterator it = packageSet.iterator(); // package_id
				while (it.hasNext()) {
					Integer i = (Integer) it.next();
					int packageId = i;
					String testId = (String) (mapPackage[j].get(i));
					String defId = (String) hmTestDef.get(testId);
					String cveId = (String) hmDefCve.get(defId);
					Packages pkgs = new Packages();
					pkgs.setPackageOval(defId);
					pkgs.setPackageCve(cveId);
					pkgs.setPackageId(packageId);

					cvssMinMax = changeCve2CvssMinMax(cveId);
					if (j == 0) {
						pkgs.setPackageVulnerability(CHECK_AGAIN);
						pkgs.setPackageCvssMin(cvssMinMax[0]);
						pkgs.setPackageCvssMax(cvssMinMax[1]);
					} else if (j == NO_VUL_DISTRO_MATCH) {
						pkgs.setPackageVulnerability(NO_VUL_DISTRO_MATCH);
					} else if (j == NO_VUL_DISTRO_NOT_MATCH) {
						pkgs.setPackageVulnerability(NO_VUL_DISTRO_NOT_MATCH);
					} else if (j == VUL_DISTRO_MATCH) {
						pkgs.setPackageVulnerability(VUL_DISTRO_MATCH);
						pkgs.setPackageCvssMin(cvssMinMax[0]);
						pkgs.setPackageCvssMax(cvssMinMax[1]);
					} else if (j == VUL_DISTRO_NOT_MATCH) {
						pkgs.setPackageVulnerability(VUL_DISTRO_NOT_MATCH);
						pkgs.setPackageCvssMin(cvssMinMax[0]);
						pkgs.setPackageCvssMax(cvssMinMax[1]);
					}
					sqlMapIidb.update("updatePackageVulnerabilities", pkgs);
					sqlMapIidb.commitTransaction();
				}
			}
			// Digest
			// TODO
			for (int j = 0; j < STATE_NUM; j++) {
				Iterator it = vectorDigest[j].iterator();
				Measurements mgms = new Measurements();
				if (j == 0) {
					mgms.setDigestVulnerability(CHECK_AGAIN);
				} else {
					mgms.setDigestVulnerability(j);
				}
				if (it.hasNext()) {
					Integer i = (Integer) it.next();
					int packageId = i;
					mgms.setPackageId(packageId);
					// sql = sql + packageId;
					// TODO
					// while (it.hasNext()) {
					// i = (Integer) it.next();
					// packageId = i.toString();
					// sql = sql + " or package_id = " + packageId;
					// }
					sqlMapIidb.update("updateDigestVulnerabilityByPackageId",
							mgms);
					sqlMapIidb.commitTransaction();
				}
			}
		} catch (DOMException e) {
			e.printStackTrace();
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Make the sql query to update the status of remanding packages because
	 * they have no vulnerability information.
	 */
	private void makeSqlQuery2UpdateRemainder() {
		try {
			Measurements mgms = new Measurements();
			mgms.setDigestVulnerability(NO_VUL_DISTRO_MATCH);
			sqlMapIidb.update("updateDigestVulnerabilitybyVulnerability", mgms);
			sqlMapIidb.commitTransaction();

			Packages pkgs = new Packages();
			pkgs.setPackageVulnerability(NO_VUL_DISTRO_MATCH);
			sqlMapIidb.update("updatePackageVulnerabilityByVulnerability", pkgs);
			sqlMapIidb.commitTransaction();
			
		} catch (SQLException ex) {
			do {
				log.error("SQLSTATE: " + ex.getSQLState());
				log.error("ERR-CODE: " + ex.getErrorCode());
				log.error("ERR-MSEG: " + ex.getMessage());
				ex = ex.getNextException();
			} while (null != ex);
		}
	}

	/**
	 * Cve to Cvss{min,max}.
	 * 
	 * @param cveId
	 * @return String[] String[0]=cvss_min, String[1]=cvss_max
	 */
	private String[] changeCve2CvssMinMax(String cveId) {
		String[] cvssMinMax = new String[2];
		try {
			String cvss = "";
			double cvssScore = 0;
			double cvssScoreMin = 0;
			double cvssScoreMax = 0;
			if (cveId != null) {
				String[] cves = cveId.split(" ");
				for (int i = 0; i < cves.length; i++) {
					String cveIdString = cves[i].trim();
					CveDefinitions cveDef = (CveDefinitions) sqlMapVul
							.queryForObject("getCveDefinitionByCveId",
									cveIdString);
					if (cveDef != null) {
						cvss = cveDef.getNvdCvssScore();
						if (cvss != null) {
							cvssScore = new Double(cvss).doubleValue();

							if (cvssScoreMin == 0 && cvssScoreMax == 0) {
								cvssScoreMin = cvssScore;
								cvssScoreMax = cvssScore;
							} else if (cvssScore < cvssScoreMin) {
								cvssScoreMin = cvssScore;
							} else if (cvssScore > cvssScoreMax) {
								cvssScoreMax = cvssScore;
							}
						}
					}
				}
			}
			cvssMinMax[0] = new Double(cvssScoreMin).toString();
			cvssMinMax[1] = new Double(cvssScoreMax).toString();

		} catch (NumberFormatException e) {
			e.printStackTrace();
		} catch (SQLException e) {
			e.printStackTrace();
		}
		return cvssMinMax;
	}

	/**
	 * @return the xmlFile
	 */
	protected String getXmlFile() {
		return xmlFile;
	}

	/**
	 * @param file
	 *            The xmlFile to set
	 */
	protected void setXmlFile(String file) {
		this.xmlFile = file;
	}
}
