/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/ifm.c
 * \brief TCG IF-M protocol
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-04-01
 * cleanup 2011-04-26 SM
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <unistd.h>
#include <fcntl.h>

#ifdef  HAVE_SENDFILE
#include <sys/sendfile.h>
#endif

#include <openpts.h>

// TODO
#define MAX_TLV_MESSAGE_LENGTH 5120000


// DEBUG
// 2011-02-24 SM make check => pass
// 2011-04-01 SM sendfile not work to new ptsc, too fast? <= wrap read/write
// 2011-04-07 SM sendfile not work aggain, RIMM_SET
#undef HAVE_SENDFILE

#ifndef HAVE_SENDFILE
#define SENDFILE_BUF_SIZE 4096

// http://linux.die.net/man/2/sendfile
// sendfile - transfer data between file descriptors
ssize_t my_sendfile(int out_fd, int in_fd, off_t *offset, size_t count) {
    char buf[SENDFILE_BUF_SIZE];
    ssize_t read_size;
    ssize_t write_size;
    ssize_t sum = 0;

    DEBUG_IFM("my_sendfile(), size=%d ############################\n", count);

    // offset == NULL
    do {
        /* set read size */
        if ((count - sum) > SENDFILE_BUF_SIZE) {
            read_size = SENDFILE_BUF_SIZE;
        } else {
            read_size = count - sum;
        }

        /* read */
        read_size = wrapRead(in_fd, buf, read_size);
        if (read_size < 0) {
            // sum = -1;
            break;
        }

        /* write */
        write_size = wrapWrite(out_fd, buf, read_size);

        if (write_size < 0) {
            ERROR("\n");
            sum = -1;
            break;
        }
        if (write_size != read_size) {
            ERROR("\n");
            sum = -1;
            break;
        }

        sum += write_size;
    } while (sum < (ssize_t) count);

    return sum;
}
#define sendfile my_sendfile
#endif  // !HAVE_SENDFILE

/**
 * read IF-M PTS message (standalone)
 *
 * This just fill the PTS_IF_M_Attribute structure.
 * The received packet is parsed by in ptscd.c
 *
 * TODO 2011-04-04 socket -> STDIN
 */
PTS_IF_M_Attribute *readPtsTlv(int fdin) {
    int rc;
    int len;
    BYTE head[12];
    int ptr;
    int rest;
    PTS_Byte * read_msg = NULL;  // TODO(munetoh)
    // Host Byte Order
    PTS_IF_M_Attribute *read_tlv;


    DEBUG_CAL("readPtsTlvFromSock - start\n");

    memset(head, 0, 12);

    /* malloc TLV for read */
    read_tlv = (PTS_IF_M_Attribute *)malloc(sizeof(PTS_IF_M_Attribute));
    if (read_tlv == NULL) {
        ERROR("no memory");
        return NULL;
    }
    memset(read_tlv, 0, sizeof(PTS_IF_M_Attribute));

    /* read IF-M header */
    rc = wrapRead(fdin, head, 12);
    if (rc == 0) {
        ERROR("sock read fail. probably end of the handshake\n");
        free(read_tlv);
        return NULL;
    }

    // copy buf to PTS_IF_M_Attribute (NBO)
    memcpy(read_tlv, head, 12);
    // Convert NBO to Host byte order
    read_tlv->type = ntohl(read_tlv->type);
    read_tlv->length = ntohl(read_tlv->length);

#if 0
    TODO("IF-M type  : 0x%02x%02x%02x%02x (NBO)",
        head[4], head[5], head[6], head[7]);
    TODO("IF-M length: 0x%02x%02x%02x%02x (NBO) %d",
        head[8], head[9], head[10], head[11], read_tlv->length);
#endif

    /* check the length */
    if (read_tlv->length > MAX_TLV_MESSAGE_LENGTH) {
        ERROR("read_tlv->length = %d (0x%X)> %d\n",
            read_tlv->length, read_tlv->length, MAX_TLV_MESSAGE_LENGTH);
        return NULL;
    }

    /* read msg body */
    if (read_tlv->length > 0) {
        read_msg = (PTS_Byte *)malloc(read_tlv->length +1);
        if (read_msg == NULL) {
            ERROR("ERROR malloc %d\n", read_tlv->length +1);
            return NULL;
        } else {
            ptr = 0;
            rest = read_tlv->length;
            while (1) {
                len = wrapRead(fdin, &read_msg[ptr], rest);
                if (len == 0) {
                    break;
                }
                ptr += len;
                rest -= len;

                if (rest < 0) {
                    break;
                }
                // TODO check timeout
            }
        }
        read_msg[read_tlv->length] = 0;
        read_tlv->value = read_msg;
    } else {
        read_tlv->value = NULL;
    }

    /* done */

    DEBUG_IFM("IF-M read,  type=0x%08x, length=%d\n",
        read_tlv->type, read_tlv->length);
    DEBUG_CAL("readPtsTlvFromSock - done\n");

    // NOTE read_tlv->value may contains MBO structure.
    return read_tlv;
}

/**
 * free PTS_IF_M_Attribute
 */
void freePtsTlv(PTS_IF_M_Attribute *tlv) {
    if (tlv == NULL) {
        return;
    }

    /* free*/
    if (tlv->value != NULL) {
        free(tlv->value);
    }
    free(tlv);
}

/**
 * write IF-M PTS message ()
 *
 * we are using sendfile() here and send the data steb by step. 
 * but IF-M of IMC/IMV version need to create whole blob to send.
 *
 * v0.2.4 - sendfile() not work with ptsc. use my_sendfile()
 *
 * Retrun
 *  length of write data
 *  -1 ERROR
 */
int writePtsTlv(OPENPTS_CONTEXT *ctx, int fdout, int type) {
    // Network Byte Order (BigEndian)
    PTS_IF_M_Attribute *write_tlv;
    UINT32 nbou32;
    UINT16 nbou16;
    // Host byte order
    int rc;
    UINT32 length = 0;
    BYTE buf[64];
    int i;
    int fsize[MAX_RM_NUM];
    int fd[MAX_RM_NUM];
    int count[MAX_RM_NUM];
    struct stat st[MAX_RM_NUM];

    OPENPTS_CONFIG *conf;

    /* check */
    if (ctx == NULL) {
        ERROR("ctx is NULL\n");
        return -1;
    }
    conf = ctx->conf;
    if (conf == NULL) {
        ERROR("conf is NULL\n");
        return -1;
    }
    if (conf->uuid == NULL) {
        ERROR("writePtsTlvToSock() uuid is NULL\n");
        return -1;
    }

    /* init */
    for (i = 0; i < MAX_RM_NUM; i++) {
        fd[i] = -1;
    }

    DEBUG_CAL("writePtsTlvToSock - start\n");

    /* malloc TLV */
    write_tlv = (PTS_IF_M_Attribute *)malloc(sizeof(PTS_IF_M_Attribute));
    if (write_tlv == NULL) {
        ERROR("no memory");
        return -1;
    }

    /* setup TLV header (1/2)*/
    write_tlv->flags  = 0;
    write_tlv->vid[0] = 0;
    write_tlv->vid[1] = 0;
    write_tlv->vid[2] = 0;
    write_tlv->type   = htonl(type);
    write_tlv->length = 0;

    /* check the TLV type */
    switch (type) {
    /* Collector <-> Verifier */
    case OPENPTS_CAPABILITIES:
        /* setup TLV header  (2/2) */
        length = sizeof(OPENPTS_IF_M_Capability);
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        // Send values, not use OPENPTS_IF_M_Capability
        /* Send versions */
        rc = wrapWrite(fdout, &ctx->conf->pts_flag, 4);
        rc = wrapWrite(fdout, &ctx->conf->tpm_version, 4);
        rc = wrapWrite(fdout, &ctx->conf->tss_version, 4);
        rc = wrapWrite(fdout, &ctx->conf->pts_version, 4);
        /* Send Platform UUID, ctx->uuid */
        rc = wrapWrite(fdout, ctx->conf->uuid->uuid, 16);
        /* Send RM UUID */
        if (ctx->conf->rm_uuid == NULL) {
            // TODO  verifier does not have Rm UUID. just send Verifier's UUID
            DEBUG("writePtsTlvToSock() RM uuid is NULL, => send platform UUID\n");
            rc = wrapWrite(fdout, ctx->conf->uuid->uuid, 16);
        } else if (ctx->conf->rm_uuid->uuid == NULL) {
            // TODO verifier?
            DEBUG("writePtsTlvToSock() RM uuid is NULL, => send platform UUID, file = %s\n",
                ctx->conf->rm_uuid->filename);
            rc = wrapWrite(fdout, ctx->conf->uuid->uuid, 16);
        } else {
            rc = wrapWrite(fdout, ctx->conf->rm_uuid->uuid, 16);
        }

        DEBUG_IFM("writePtsTlv - OPENPTS_CAPABILITIES, length = %d", length);
        break;

    case DH_NONCE_PARAMETERS_REQUEST:  /* DH: Initiator -> Respondor */
        /* setup TLV header  (2/2) */
        length = 4;
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        /* Send DH Nonce */
        buf[0] = ctx->nonce->req->reserved;
        buf[1] = ctx->nonce->req->min_nonce_len;
        rc = wrapWrite(fdout, buf, 2);
        nbou16 = htons(ctx->nonce->req->dh_group_set);
        rc = wrapWrite(fdout, (BYTE *)&nbou16, 2);

        DEBUG_IFM("writePtsTlv - DH_NONCE_PARAMETERS_REQUEST, length = %d", length);
        break;

    case DH_NONCE_PARAMETORS_RESPONSE:  /* DH: IRespondor -> Initiator */
        /* setup TLV header  (2/2) */
        length =
            4 + 4 +
            ctx->nonce->respondor_nonce_length +
            ctx->nonce->pubkey_length;
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        /* Send DH param  */
        buf[0] = ctx->nonce->res->reserved[0];
        buf[1] = ctx->nonce->res->reserved[1];
        buf[2] = ctx->nonce->res->reserved[2];
        buf[3] = ctx->nonce->res->nonce_length;
        rc = wrapWrite(fdout, buf, 4);
        nbou16 = htons(ctx->nonce->res->selected_dh_group);
        rc = wrapWrite(fdout, (BYTE *)&nbou16, 2);
        nbou16 = htons(ctx->nonce->res->hash_alg_set);
        rc = wrapWrite(fdout, (BYTE *)&nbou16, 2);
        /* nonce */
        rc = wrapWrite(fdout, ctx->nonce->respondor_nonce, ctx->nonce->respondor_nonce_length);
        /* send dh_respondor_public */
        rc = write(fdout, ctx->nonce->pubkey, ctx->nonce->pubkey_length);

        DEBUG_IFM("writePtsTlv - DH_NONCE_PARAMETORS_RESPONSE, length = %d", length);
        break;

    case DH_NONCE_FINISH: /* DH: Initiator -> Respondor */
        /* setup TLV header  (2/2) */
        length =
            4 +
            ctx->nonce->initiator_nonce_length +
            ctx->nonce->pubkey_length;
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        /* Send */
        buf[0] = ctx->nonce->fin->reserved = 0;
        buf[1] = ctx->nonce->fin->nonce_length = ctx->nonce->initiator_nonce_length;
        rc = wrapWrite(fdout, buf, 2);
        nbou16 = htons(ctx->nonce->fin->selected_hash_alg);
        rc = wrapWrite(fdout, (BYTE *)&nbou16, 2);
        /* send dh_initiator_pubkey */
        rc = wrapWrite(fdout, ctx->nonce->pubkey, ctx->nonce->pubkey_length);
        /* send dh_initiator_nonce */
        rc = wrapWrite(fdout, ctx->nonce->initiator_nonce, ctx->nonce->initiator_nonce_length);

        DEBUG_IFM("writePtsTlv - DH_NONCE_FINISH, length = %d", length);
        break;

    case REQUEST_RIMM_SET:  /* RIMM: Verifier -> Collector */
        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
        DEBUG_IFM("writePtsTlv - REQUEST_RIMM_SET, length = 0");
        break;

    case RIMM_SET:  /* RIMM: Collector -> Verifier */
        /* setup TLV header  (2/2) */
        /* open RMs */
        length = 4;  // for RM num
        for (i = 0; i < conf->rm_num; i++) {
            /* open */
            fd[i] = open(ctx->conf->rm_filename[i], O_RDONLY);
            if (fd[i] < 0) {
                // 20101124 SM must be a fullpath for Daemon
                ERROR("Error RM file, %s not found\n", ctx->conf->rm_filename[i]);
                /* send Error massage */
                ctx->ifm_errno = PTS_FATAL;
                ctx->ifm_strerror = smalloc("Manifest not found, initialize the collector");
                goto error;
            }
            stat(ctx->conf->rm_filename[i], &st[i]);
            fsize[i] = st[i].st_size;
            length += 4 + fsize[i];
            /* close */
            close(fd[i]);
            fd[i] = -1;
        }
        write_tlv->length = htonl(length);
        DEBUG_IFM("writePtsTlv - RIMM_SET, length = %d", length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        // NUM of RM
        nbou32 = htonl(conf->rm_num);
        rc = wrapWrite(fdout, (BYTE *)&nbou32, 4);

        for (i = 0; i< conf->rm_num; i++) {
            // length of RM[i]
            nbou32 = htonl(fsize[i]);
            rc = wrapWrite(fdout, (BYTE *)&nbou32, 4);
            // RM[i] body
            fd[i] = open(ctx->conf->rm_filename[i], O_RDONLY);
            // TODO check? must exist
            count[i] = sendfile(fdout, fd[i], NULL, st[i].st_size);
            /* close */
            close(fd[i]);
            fd[i] = -1;
            DEBUG_IFM("RM[%d] len = %d\n", i, count[i]);
        }

        break;

    case REQUEST_NEW_RIMM_SET:  /* NEW RIMM: V -> C */
        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
        DEBUG_IFM("writePtsTlv - REQUEST_NEW_RIMM_SET, length = 0");
        break;

    /* NEW RIMM */
    case NEW_RIMM_SET:
        /* check */
        if (conf->newrm_num == 0) {
            /* New RM is missing => send Error massage */
            ctx->ifm_errno = PTS_FATAL;
            ctx->ifm_strerror = smalloc("New Manifest not found, check the collector");
            goto error;
        }

        /* setup TLV header  (2/2) */
        length = 16 + 4;  // UUID + num
        for (i = 0; i < conf->newrm_num; i++) {
            fd[i] = open(ctx->conf->newrm_filename[i], O_RDONLY);
            if (fd[i] < 0) {
                // 20101124 SM must be a fullpath for Daemon
                ERROR("Error RM file, %s not found\n", ctx->conf->newrm_filename[i]);
                /* send Error massage */
                ctx->ifm_errno = PTS_FATAL;
                ctx->ifm_strerror = smalloc("New Manifest file not found, check the collector");
                goto error;
            }
            stat(ctx->conf->newrm_filename[i], &st[i]);
            fsize[i] = st[i].st_size;
            length += 4 + fsize[i];
            /* close */
            close(fd[i]);
            fd[i] = -1;
        }
        write_tlv->length = htonl(length);

        DEBUG_IFM("writePtsTlv - NEW_RIMM_SET, length = %d", length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        // UUID
        rc = wrapWrite(fdout, ctx->conf->newrm_uuid->uuid, 16);
        // NUM of RM
        nbou32 = htonl(conf->newrm_num);
        rc = wrapWrite(fdout, (BYTE *)&nbou32, 4);
        for (i = 0; i< conf->newrm_num; i++) {
            // length of RM[i]
            nbou32 = htonl(fsize[i]);
            rc = wrapWrite(fdout, (BYTE *)&nbou32, 4);
            // RM[i] body
            fd[i] = open(ctx->conf->newrm_filename[i], O_RDONLY);
            count[i] = sendfile(fdout, fd[i], NULL, st[i].st_size);
            /* close */
            close(fd[i]);
            fd[i] = -1;
            DEBUG_IFM("RM[%d] len = %d\n", i, count[i]);
        }
        break;

    case REQUEST_INTEGRITY_REPORT:  /* IR: V -> C */
        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
        DEBUG_IFM("writePtsTlv - REQUEST_INTEGRITY_REPORT, length = 0");
        break;

    case INTEGRITY_REPORT:  /* IR: C -> V */
        /* setup TLV header  (2/2) */

        /* generate new IR */
        rc = genIr(ctx);
        if (rc != PTS_SUCCESS) {
            ERROR("writePtsTlvToSock - gen IR failed\n");
            // TODO send ERROR?
            /* Send IF-M TLV header with length = 0 */
            // rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
            // break;
            /* send Error massage */
            ctx->ifm_errno = PTS_FATAL;
            ctx->ifm_strerror = smalloc("Generation of IR failed");
            goto error;
        }

        /* check the IR size */
        fd[0] = open(ctx->conf->ir_filename, O_RDONLY);  // TODO(munetoh)
        if (fd[0] < 0) {
            ERROR("Error %s not found\n", ctx->conf->ir_filename);
            /* send Error massage */
            ctx->ifm_errno = PTS_FATAL;
            ctx->ifm_strerror = smalloc("IR file is missing");
            goto error;
        }
        stat(ctx->conf->ir_filename, &st[0]);
        fsize[0] = st[0].st_size;
        length = fsize[0];
        write_tlv->length = htonl(length);
        /* close */
        close(fd[0]);
        fd[0] = -1;

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        // BODY1
        fd[0] = open(ctx->conf->ir_filename, O_RDONLY);
        count[0] = sendfile(fdout, fd[0], NULL, st[0].st_size);
        /* close */
        close(fd[0]);
        fd[0] = -1;

        // DEBUG_IFM("IR len = %d\n", count[0]);
        // TODO check the size?
        DEBUG_IFM("writePtsTlv - INTEGRITY_REPORT, length = %d", length);
        break;

    case VERIFICATION_RESULT:  /* VR: V -> C */
        // TODO N/A
        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
        DEBUG_IFM("writePtsTlv - VERIFICATION_RESULT, length = 0");
        break;

#ifdef CONFIG_AIDE
    case REQUEST_AIDE_DATABASE:  /* AIDE DATABASE: V -> C */
        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
        DEBUG_IFM("writePtsTlv - REQUEST_AIDE_DATABASE, length = 0");
        break;

    case AIDE_DATABASE:  /* AIDE DATABASE: C -> V */
        /* setup TLV header  (2/2) */
        /* body */
        if (ctx->conf->aide_database_filename == NULL) {
            // Test
            DEBUG("writePtsTlvToSock - Error AIDE DB file is not configured\n");
            ctx->ifm_errno = PTS_FATAL;
            ctx->ifm_strerror = smalloc("AIDE DB file is not configured");
            goto error;
        } else {
            fd[0] = open(ctx->conf->aide_database_filename, O_RDONLY);
            if (fd[0] < 0) {
                /* AIDE file is missing, erorr */
                ERROR("writePtsTlvToSock - Error AIDE DB file, %s not found\n",
                    ctx->conf->aide_database_filename);
                /* send Error massage */
                ctx->ifm_errno = PTS_FATAL;
                ctx->ifm_strerror = smalloc("AIDE file not found");
                goto error;
            } else {
                /* OK */
                stat(ctx->conf->aide_database_filename, &st[0]);
                fsize[0] = st[0].st_size;
                length = fsize[0];
                /* close */
                close(fd[0]);
                fd[0] = -1;
            }
        }
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        if (length > 0) {
            // BODY1
            fd[0] = open(ctx->conf->aide_database_filename, O_RDONLY);
            count[0] = sendfile(fdout, fd[0], NULL, st[0].st_size);
            /* close */
            close(fd[0]);
            fd[0] = -1;

            DEBUG_IFM("writePtsTlv - AIDE_DATABASE, file =  %s\n", ctx->conf->aide_database_filename);
            // DEBUG_IFM("AIDE DATABASE len = %d\n", count[0]);
        }
        DEBUG_IFM("writePtsTlv - AIDE_DATABASE, length = %d", length);
        break;
#endif
    /* send TPM_PUBKEY */
    case REQUEST_TPM_PUBKEY:
        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);
        DEBUG_IFM("writePtsTlv - REQUEST_RIMM_SET, length = 0");
        break;

    case TPM_PUBKEY:
        /* setup TLV header  (2/2) */
        write_tlv->type = htonl(type);
        if (ctx->conf->pubkey_length > 0) {
            // DEBUG("TPM_PUBKEY len %d\n", ctx->conf->pubkey_length);
            if (ctx->conf->pubkey != NULL) {
                length = ctx->conf->pubkey_length;
            } else {
                ERROR("writePtsTlvToSock - PUBKEY blob is missing\n");
                ctx->ifm_errno = PTS_FATAL;
                ctx->ifm_strerror = smalloc("Piblic key is missing");
                goto error;
            }
        } else {
            /* missing */
            ERROR("writePtsTlvToSock - PUBKEY blob is missing\n");
            // length = 0;
            ctx->ifm_errno = PTS_FATAL;
            ctx->ifm_strerror = smalloc("Piblic key is missing, initialize the collector");
            goto error;
        }
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        rc = wrapWrite(
                fdout,
                ctx->conf->pubkey,
                ctx->conf->pubkey_length);

        DEBUG_IFM("writePtsTlv - TPM_PUBKEY, length = %d", length);
        break;

    case OPENPTS_ERROR:
        /* setup TLV header  (2/2) */
        // TODO
        if (ctx->ifm_strerror != NULL) {
            length = 4 + 4 + strlen(ctx->ifm_strerror);
        } else {
            length = 4 + 4 + 0;
        }
        write_tlv->length = htonl(length);

        /* Send IF-M TLV header */
        rc = wrapWrite(fdout, (BYTE *)write_tlv, 12);

        {
            UINT32 ifm_errno;
            UINT32 size = 0;
            UINT32 len = 0;
            /* send error code */
            ifm_errno = htonl(ctx->ifm_errno);
            rc = wrapWrite(fdout, (BYTE *)&ifm_errno, 4);
            /* send msg num */

            if (ctx->ifm_strerror != NULL) {
                len = strlen(ctx->ifm_strerror);
                size = htonl(len);
                rc = wrapWrite(fdout, (BYTE *)&size, 4);
                rc = wrapWrite(fdout, (BYTE *)ctx->ifm_strerror, len);
                /* free */
                free(ctx->ifm_strerror);
            } else {
                size = 0;
                rc = wrapWrite(fdout, (BYTE *)&size, 4);
            }
        }

        DEBUG_IFM("writePtsTlv - OPENPTS_ERROR, length = %d", length);
        break;
    default:
        // BAT type
        length = 0;
        ERROR("BAD IFM message TYPE 0x%08x\n", type);
        goto error;  // return -1;
    }


    // DEBUG_IFM("IF-M write, type=0x%08x, length=%d\n",
    //    type, length);
    DEBUG_CAL("writePtsTlvToSock - done\n");

    /* done */
    rc = length;

    /* close */
    for (i = 0; i < MAX_RM_NUM; i++) {
        if (fd[i] >= 0) close(fd[i]);
    }
    free(write_tlv);
    return rc;


    // TODO send ERROR messages
  error:
    ERROR("writePtsTlvToSock()\n");
    /* close files*/
    for (i = 0; i < MAX_RM_NUM; i++) {
        if (fd[i] >= 0) close(fd[i]);
    }
    /* send ERROR */
    rc = writePtsTlv(ctx, fdout, OPENPTS_ERROR);

    free(write_tlv);

    // TODO rc or -1, which?
    return rc;
}


/* TNC, libtnc */

/**
 * get IF-M PTS message (TNC)
 * return *msg (Network Byte Order)
 * TODO use RC core
 */
char* getPtsTlvMessage(OPENPTS_CONTEXT *ctx, int type, int *len) {
    // Network Byte Order (Big Endian)
    PTS_IF_M_Attribute *write_tlv;
    // Host
    int i;
    OPENPTS_CONFIG *conf;
    // TODO 2011-04-11 SM
    UINT32 length = 0;  // endian of host
    // BYTE*  value;

    DEBUG("writePtsTlvToSock - start\n");

    /* check */
    if (ctx == NULL) {
        ERROR("ctx is NULL\n");
        return NULL;
    }
    conf = ctx->conf;
    if (conf == NULL) {
        ERROR("conf is NULL\n");
        return NULL;
    }

    /* TLV */
    *len = 0;

    switch (type) {
    /* Verifier -> Collector */
    case DH_NONCE_PARAMETERS_REQUEST:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;
    case DH_NONCE_FINISH:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;
    case REQUEST_RIMM_SET:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;
    case REQUEST_INTEGRITY_REPORT:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;
    case VERIFICATION_RESULT:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;

    /* Collector -> Verifier */
    case OPENPTS_CAPABILITIES:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;
    case DH_NONCE_PARAMETORS_RESPONSE:
        if ((write_tlv = (PTS_IF_M_Attribute *)
                malloc(sizeof(PTS_IF_M_Attribute))) == NULL) {
            ERROR("no memory");
            return NULL;
        }
        write_tlv->flags  = 0;
        write_tlv->vid[0] = 0;
        write_tlv->vid[1] = 0;
        write_tlv->vid[2] = 0;
        write_tlv->type   = htonl(type);
        write_tlv->length = htonl(length);  // 0
        break;
    case RIMM_SET:
        {
            int rc;
            int num = 2;  // TODO set RM num from conf?
            BYTE * buf;
            int ptr;

            int fsize[MAX_RM_NUM];
            int fd[MAX_RM_NUM];
            struct stat st[MAX_RM_NUM];


            DEBUG_IFM("create TEMPLATE_RIMM_SET_METADATA\n");

            /* body - RMs  */
            length = 4;
            for (i = 0; i< conf->rm_num; i++) {
                fd[i] = open(ctx->conf->rm_filename[i], O_RDONLY);
                if (fd[i] < 0) {
                    printf("Error RM file, '%s' not found\n", ctx->conf->rm_filename[i]);
                    return NULL;
                }
                stat(ctx->conf->rm_filename[i], &st[i]);
                fsize[i] = st[i].st_size;
                length += 4 + fsize[i];
            }

            DEBUG_IFM("size of TEMPLATE_RIMM_SET_METADATA is %d\n", length);

            /* malloc msg buffer */
            if ((buf = malloc(12 + length)) == NULL) {
                ERROR("no memory size=%d\n", 12 + length);
                return NULL;
            }

            write_tlv = (PTS_IF_M_Attribute *)buf;
            write_tlv->flags  = 0;
            write_tlv->vid[0] = 0;
            write_tlv->vid[1] = 0;
            write_tlv->vid[2] = 0;
            write_tlv->type   = htonl(type);
            write_tlv->length = htonl(length);

            /* payload */

            DEBUG_IFM("RIMMs %d\n", num);

            ptr = 12;
            buf[ptr + 0] = (num & 0xFF000000) >> 24;
            buf[ptr + 1] = (num & 0xFF0000) >> 16;
            buf[ptr + 2] = (num & 0xFF00) >> 8;
            buf[ptr + 3] = (num & 0xFF);
            ptr += 4;

            for (i = 0; i < conf->rm_num; i++) {
                DEBUG_IFM("RIMM %d size %d\n", i, fsize[i]);

                buf[ptr + 0] = (fsize[i] & 0xFF000000) >> 24;
                buf[ptr + 1] = (fsize[i] & 0xFF0000) >> 16;
                buf[ptr + 2] = (fsize[i] & 0xFF00) >> 8;
                buf[ptr + 3] = (fsize[i] & 0xFF);
                ptr += 4;

                // BODY1
                while (1) {
                    rc = read(fd[i], &buf[ptr], fsize[i]);
                    if (rc == 0) {
                         break;
                    }
                    ptr += rc;
                    fsize[i] -= rc;
                }
                close(fd[i]);
                fd[i] = -1;
            }
            DEBUG_IFM("create TEMPLATE_RIMM_SET_METADATA ... done\n");
        }
        break;
    case INTEGRITY_REPORT:
        {
            int rc;
            BYTE * buf;
            int ptr;
            int fsize1 = 0;
            int fd1 = 0;
            struct stat st1;

            DEBUG_IFM("create INTEGRITY_REPORT\n");

            if (ctx->conf->iml_mode == 1) {
                /* get IML via securityfs */
                /* reset FSM */
                /* read BIOS IML */
                rc = getBiosImlFile(ctx, ctx->conf->bios_iml_filename, 1);  // TODO endian?
                if (rc != PTS_SUCCESS) {
                    ERROR("load IML %s was failed\n", ctx->conf->bios_iml_filename);
                }
                /* read Runtime IML */
                rc = getImaImlFile(
                        ctx,
                        ctx->conf->runtime_iml_filename,
                        ctx->conf->runtime_iml_type, 1);  // TODO endian?
                /* read PCRS */
                rc = getPcrBySysfsFile(ctx, ctx->conf->pcrs_filename);
                /* TPM Quote */
                TODO("add TPM_quote\n");
                /* save IR */
                rc = writeIr(ctx, ctx->conf->ir_filename);
            } else {
                TODO("get IML/PCR via TSS is not ready\n");
            }

            DEBUG_IFM("send INTEGRITY_REPORT\n");

            /* body - IR  */
            fd1 = open(ctx->conf->ir_filename, O_RDONLY);
            if (fd1 < 0) {
                printf("Error platform RM file, '%s' not found\n", ctx->conf->ir_filename);
                return NULL;
            }
            stat(ctx->conf->ir_filename, &st1);
            fsize1 = st1.st_size;

            length = 4 + fsize1;

            DEBUG_IFM("size of INTEGRITY_REPORT is %d\n", length);

            /* malloc msg buffer */
            if ((buf = malloc(12 + length)) == NULL) {
                ERROR("no memory size=%d\n", 12 + length);
                return NULL;
            }

            write_tlv = (PTS_IF_M_Attribute *)buf;
            write_tlv->flags  = 0;
            write_tlv->vid[0] = 0;
            write_tlv->vid[1] = 0;
            write_tlv->vid[2] = 0;
            write_tlv->type   = htonl(type);
            write_tlv->length = htonl(length);

            /* payload */
            ptr = 12;

            // LEN1
            DEBUG_IFM("IR size %d\n", fsize1);

            buf[ptr + 0] = (fsize1 & 0xFF000000) >> 24;
            buf[ptr + 1] = (fsize1 & 0xFF0000) >> 16;
            buf[ptr + 2] = (fsize1 & 0xFF00) >> 8;
            buf[ptr + 3] = (fsize1 & 0xFF);
            ptr += 4;

            // BODY1
            while (1) {
                rc = read(fd1, &buf[ptr], fsize1);
                if (rc == 0) {
                     break;
                }
                ptr += rc;
                fsize1 -= rc;
            }
            close(fd1);

            DEBUG_IFM("create INTEGRITY_REPORT ... done\n");
        }
        break;

    default:
        // BAT type
        ERROR("BAD TYPE %x\n", type);
        return NULL;
    }

    DEBUG_IFM("IF-M message, type=0x%x, length=%d\n",
        type, length);
    DEBUG("writePtsTlvToSock - done\n");

    if (length == 0) {
        /* just send TLV headder */
        *len = 12;
        return (char*)  write_tlv;
    } else if (length > 0) {
        *len = 12 + length;
        return (char*)  write_tlv;
    } else {
        ERROR("internal error\n");
        return NULL;
    }
}
