/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/imc.c
 * \brief TCG TNC IF-IMC v1.2 R8
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-05-07
 * cleanup 2011-01-22 SM
 *
 * http://www.trustedcomputinggroup.org/resources/tnc_ifimc_specification
 * http://www.trustedcomputinggroup.org/files/resource_files/8CB977E1-1D09-3519-AD48484530EF6639/TNC_IFIMC_v1_2_r8.pdf
 *
 *
 * this library is not a thread safe.
 *  just one IMC<-(IFM)->IMV conenction.
 * 
 *  handshake
 *    0 IMC -> IMV hello
 *    1 IMC <- IMV capability 
 *    2 IMC -> IMV capability
 *
 *    3 IMC <- IMV DH-nonce param req
 *    4 IMC -> IMV DH-nonce param res
 *    5 IMC <- IMV DH-nonce done
 *    6 IMC -> IMV ack
 *
 *    7 IMC <- IMV template RIMM req
 *    8 IMC -> IMV RIMM
 *    9 IMC <- IMV template IR req
 *   10 IMC -> IMV IR
 */

#include <stdio.h>
#include <string.h>

#include <openpts.h>

#include <tncifimc.h>
#include <libtnc.h>  // TODO(munetoh) remove this

// TODO(munetoh) Fix Makefile.am
#include "ifm.c"
#include "iml.c"
#include "ir.c"
#include "base64.c"
#include "tpm.c"
#include "ctx.c"
#include "uml.c"
#include "fsm.c"
#include "rm.c"
#include "log.c"

// TODO(munetoh) for TEST
#define VENDORID 9999  // TODO(munetoh)

/* global variables */
static int initialized = 0;
static TNC_IMCID id = -1;
static TNC_ConnectionID cid = -1;

static TNC_TNCC_ReportMessageTypesPointer    reportMessageTypesPtr;
static TNC_TNCC_RequestHandshakeRetryPointer requestHandshakeRetryPtr;
static TNC_TNCC_SendMessagePointer           sendMessagePtr;

static OPENPTS_CONFIG *conf = NULL;
static OPENPTS_CONTEXT *ctx = NULL;

// static char *config_filename = NULL;

// int verbose = 0;
// int verbose = DEBUG_IFM_FLAG;
int verbose = DEBUG_FLAG | DEBUG_IFM_FLAG;

static TNC_Result sendMessage(
    /*in*/ TNC_IMCID imcID,
    /*in*/ TNC_ConnectionID connectionID,
    /*in*/ TNC_BufferReference message,
    /*in*/ TNC_UInt32 messageLength,
    /*in*/ TNC_MessageType messageType);

/* List of receive message types */
static TNC_MessageType messageTypes[] = {
    TNCMESSAGENUM(TNC_VENDORID_TCG, TNC_SUBTYPE_ANY),  // generic
    TNCMESSAGENUM(VENDORID,  1),
    TNCMESSAGENUM(VENDORID,  3),
    TNCMESSAGENUM(VENDORID,  4),
    TNCMESSAGENUM(VENDORID,  5),
    TNCMESSAGENUM(VENDORID,  7),
    TNCMESSAGENUM(VENDORID,  9),
};

/* IMC Functions */

/**
 * TNC_IMC_Initialize (MANDATORY) 
 * 
 */
TNC_IMC_API TNC_Result TNC_IMC_Initialize(
/*in*/  TNC_IMCID imcID,
/*in*/  TNC_Version minVersion,
/*in*/  TNC_Version maxVersion,
/*out*/ TNC_Version *pOutActualVersion) {
    DEBUG("TNC_IMC_Initialize\n");

    if (initialized) return TNC_RESULT_ALREADY_INITIALIZED;

    /* check version */
    /* Only support version 1 */
    if ((minVersion < TNC_IFIMC_VERSION_1) ||
        (maxVersion > TNC_IFIMC_VERSION_1)) {
        return TNC_RESULT_NO_COMMON_VERSION;
    }

    /* OK */

    *pOutActualVersion = TNC_IFIMC_VERSION_1;
    id = imcID;

    /* initialize PTS */
    conf = newPtsConfig();
    if (conf == NULL) {
        ERROR("no memory\n");
        return TNC_RESULT_FATAL;
    }
    ctx =  newPtsContext(conf);
    if (ctx == NULL) {
        ERROR("no memory\n");
        return TNC_RESULT_FATAL;
    }

    /* setup */
    ctx->conf->ir_filename = "./data/ThinkpadX200_Fedora12/ir_file.xml";

    initialized++;
    return TNC_RESULT_SUCCESS;
}


/**
 * TNC_IMC_NotifyConnectionChange (OPTIONAL)
 * TODO(munetoh) dummy 
 */
TNC_IMC_API TNC_Result TNC_IMC_NotifyConnectionChange(
/*in*/  TNC_IMCID imcID,
/*in*/  TNC_ConnectionID connectionID,
/*in*/  TNC_ConnectionState newState) {
    DEBUG("TNC_IMC_Initialize\n");

    /* check internal status */
    if (!initialized)
        return TNC_RESULT_NOT_INITIALIZED;

    /* check ID */
    if (imcID != id)
        return TNC_RESULT_INVALID_PARAMETER;

    /*  ID */
    cid = connectionID;


    return TNC_RESULT_SUCCESS;
}

/**
 * TNC_IMC_BeginHandshake (MANDATORY)
 */
TNC_IMC_API TNC_Result TNC_IMC_BeginHandshake(
/*in*/  TNC_IMCID imcID,
/*in*/  TNC_ConnectionID connectionID) {
    int rc = 0;
    char* msg = "Hello";  // TBD

    DEBUG("TNC_IMC_BeginHandshake %d %d\n",
            (int)imcID, (int)connectionID);

    /* check internal status */
    if (!initialized)
        return TNC_RESULT_NOT_INITIALIZED;

    /* check ID */
    if (imcID != id)
        return TNC_RESULT_INVALID_PARAMETER;

    /* connection ID */
    cid = connectionID;

    /* just send hello to verifier */

    DEBUG_IFM("C    imcID=%d, connectionID=%d - TNC_IMC_BeginHandshake\n", (int)imcID, (int)connectionID);

    rc = sendMessage(
                imcID,
                connectionID,
                (TNC_BufferReference) msg,
                strlen(msg),
                TNCMESSAGENUM(VENDORID, 0));

    return rc;
}

/**
 * TNC_IMC_ReceiveMessage (OPTIONAL)
 */
TNC_IMC_API TNC_Result TNC_IMC_ReceiveMessage(
/*in*/  TNC_IMCID imcID,
/*in*/  TNC_ConnectionID connectionID,
/*in*/  TNC_BufferReference messageBuffer,
/*in*/  TNC_UInt32 messageLength,
/*in*/  TNC_MessageType messageType) {
    int len = 0;
    PTS_IF_M_Attribute *read_tlv;
    int rc;
    // char* msg = "this is a test message";

    DEBUG("TNC_IMC_ReceiveMessage msg=%s\n", messageBuffer);

    /* check internal status */
    if (!initialized)
        return TNC_RESULT_NOT_INITIALIZED;

    /* check ID */
    if (imcID != id)
        return TNC_RESULT_INVALID_PARAMETER;

    /* connection ID */
    if (connectionID != cid)
        return TNC_RESULT_INVALID_PARAMETER;

    /* */
    DEBUG_IFM("V->C imcID=%d, connectionID=%d, type=0x%x, msg[%d]\n",
        (int)imcID, (int)connectionID, (int)messageType, (int)messageLength);

    /* handshake */

    if (messageType == TNCMESSAGENUM(VENDORID, 1)) {
        /* capability from client  */
        read_tlv = (PTS_IF_M_Attribute *) messageBuffer;
        if (read_tlv->type != OPENPTS_CAPABILITIES) {
            ERROR("bad msg\n");
            return TNC_RESULT_FATAL;
        }

        /* send PTS_CAPABILITIES  */
        char* msg = getPtsTlvMessage(ctx, OPENPTS_CAPABILITIES, &len);
        rc = sendMessage(imcID,
                            connectionID,
                            (TNC_BufferReference)msg,
                            len,
                            TNCMESSAGENUM(VENDORID, 2));
        free(msg);
        DEBUG_IFM("Collector send PTS_CAPABILITIES len=%d\n", len);
        return rc;
    } else if (messageType == TNCMESSAGENUM(VENDORID, 3)) {
        /* DH_NONCE_PARAMETERS_REQUEST from client  */
        read_tlv = (PTS_IF_M_Attribute *) messageBuffer;
        if (read_tlv->type != DH_NONCE_PARAMETERS_REQUEST) {
            ERROR("bad msg\n");
            return TNC_RESULT_FATAL;
        }

        /* send DH_NONCE_PARAMETORS_RESPONSE  */
        char* msg = getPtsTlvMessage(ctx, DH_NONCE_PARAMETORS_RESPONSE, &len);
        rc = sendMessage(imcID,
                            connectionID,
                            (TNC_BufferReference)msg,
                            len,
                            TNCMESSAGENUM(VENDORID, 4));
        free(msg);
        DEBUG_IFM("Collector send PTS_CAPABILITIES len=%d\n", len);
        return rc;
    } else if (messageType == TNCMESSAGENUM(VENDORID, 5)) {
        /* DH_NONCE_FINISH from client  */
        read_tlv = (PTS_IF_M_Attribute *) messageBuffer;
        if (read_tlv->type != DH_NONCE_FINISH) {
            ERROR("bad msg\n");
            return TNC_RESULT_FATAL;
        }
        /* ack to keep TNC handshake  */
        // TODO(munetoh) otherwise TNC HS was terminated.
        char* msg = "ack";
        rc = sendMessage(imcID,
                            connectionID,
                            (TNC_BufferReference)msg,
                            strlen(msg),
                            TNCMESSAGENUM(VENDORID, 6));
        DEBUG_IFM("Collector send PTS_CAPABILITIES len=%d\n", len);
        return rc;
    } else if (messageType == TNCMESSAGENUM(VENDORID, 7)) {
        /* REQUEST_TEMPLATE_RIMM_SET_METADATA from client  */
        read_tlv = (PTS_IF_M_Attribute *) messageBuffer;
        if (read_tlv->type != REQUEST_RIMM_SET) {
            ERROR("bad msg\n");
            return TNC_RESULT_FATAL;
        }

        /* send DH_NONCE_PARAMETORS_RESPONSE  */
        char* msg = getPtsTlvMessage(ctx, RIMM_SET, &len);
        rc = sendMessage(imcID,
                            connectionID,
                            (TNC_BufferReference)msg,
                            len,
                            TNCMESSAGENUM(VENDORID, 8));
        free(msg);
        DEBUG_IFM("Collector send TEMPLATE_RIMM_SET_METADATA len=%d\n", len);
        return rc;
    } else if (messageType == TNCMESSAGENUM(VENDORID, 9)) {
        /* REQUEST_TEMPLATE_RIMM_SET_METADATA from client  */
        read_tlv = (PTS_IF_M_Attribute *) messageBuffer;
        if (read_tlv->type != REQUEST_INTEGRITY_REPORT) {
            ERROR("bad msg\n");
            return TNC_RESULT_FATAL;
        }

        /* send DH_NONCE_PARAMETORS_RESPONSE  */
        char* msg = getPtsTlvMessage(ctx, INTEGRITY_REPORT, &len);
        rc = sendMessage(imcID,
                            connectionID,
                            (TNC_BufferReference)msg,
                            len,
                            TNCMESSAGENUM(VENDORID, 10));
        // free(msg);
        DEBUG_IFM("Collector send INTEGRITY_REPORT len=%d\n", len);
        return rc;
    } else {
        ERROR("bad msg from verifier\n");
        return TNC_RESULT_FATAL;
    }

    return TNC_RESULT_SUCCESS;
}

/**
 * TNC_IMC_BatchEnding (OPTIONAL)
 */
TNC_IMC_API TNC_Result TNC_IMC_BatchEnding(
/*in*/  TNC_IMCID imcID,
/*in*/  TNC_ConnectionID connectionID) {
    DEBUG("TNC_IMC_BatchEnding\n");

    /* check internal status */
    if (!initialized)
        return TNC_RESULT_NOT_INITIALIZED;

    /* check ID */
    if (imcID != id)
        return TNC_RESULT_INVALID_PARAMETER;

    /* connection ID */
    if (connectionID != cid)
        return TNC_RESULT_INVALID_PARAMETER;

    DEBUG_IFM("C    imcID=%d, connectionID=%d - TNC_IMC_BatchEnding\n", (int)imcID, (int)connectionID);

    return TNC_RESULT_SUCCESS;
}

/**
 * TNC_IMC_Terminate (OPTIONAL)
 */
TNC_IMC_API TNC_Result TNC_IMC_Terminate(
/*in*/  TNC_IMCID imcID) {
    DEBUG("TNC_IMC_Terminate\n");

    /* check internal status */
    if (!initialized)
        return TNC_RESULT_NOT_INITIALIZED;

    /* check ID */
    if (imcID != id)
        return TNC_RESULT_INVALID_PARAMETER;

    /* connection ID */
    // TODO(munetoh)

    /* PTS */
    freePtsContext(ctx);
    freePtsConfig(conf);

    DEBUG_IFM("C    imcID=%d - TNC_IMC_Terminate\n", (int)imcID);

    return TNC_RESULT_SUCCESS;
}


/* TNC Client Functions */

/**
 * Call TNC_TNCC_ReportMessageTypes (MANDATORY) in the TNCC
 */
static TNC_Result reportMessageTypes(
    /*in*/ TNC_IMCID imcID,
    /*in*/ TNC_MessageTypeList supportedTypes,
    /*in*/ TNC_UInt32 typeCount) {
    DEBUG("TNC_TNCC_ReportMessageTypes\n");

    if (!reportMessageTypesPtr)
        return TNC_RESULT_FATAL;

    return (*reportMessageTypesPtr)(imcID,
                                     supportedTypes,
                                     typeCount);
}

/**
 * Call TNC_TNCC_SendMessage (MANDATORY) in the TNCC
 */
static TNC_Result sendMessage(
    /*in*/ TNC_IMCID imcID,
    /*in*/ TNC_ConnectionID connectionID,
    /*in*/ TNC_BufferReference message,
    /*in*/ TNC_UInt32 messageLength,
    /*in*/ TNC_MessageType messageType) {
    DEBUG("TNC_TNCC_SendMessage msg='%s' type=%d(0x%x)\n",
            message, (int)messageType, (int)messageType);

    if (!sendMessagePtr)
        return TNC_RESULT_FATAL;

    DEBUG_IFM("C->V imcID=%d, connectionID=%d, type=0x%x, msg[%d]\n",
        (int)imcID, (int)connectionID, (int)messageType, (int)messageLength);

    return (*sendMessagePtr)(imcID,
                              connectionID,
                              message,
                              messageLength,
                              messageType);
}

#if 0
// F12 imc.c:277: error: ‘requestHandshakeRetry’ defined but not used
/**
 * Call TNC_TNCC_RequestHandshakeRetry (MANDATORY) in the TNCC
 */
static TNC_Result requestHandshakeRetry(
    /*in*/ TNC_IMCID imcID,
    /*in*/ TNC_ConnectionID connectionID,
    /*in*/ TNC_RetryReason reason) {
    DEBUG("TNC_TNCC_RequestHandshakeRetry\n");

    if (!requestHandshakeRetryPtr)
        return TNC_RESULT_FATAL;

    return (*requestHandshakeRetryPtr)(imcID, connectionID, reason);
}
#endif

/* Platform-Specific IMC Functions */

/**
 * TNC_IMC_ProvideBindFunction (MANDATORY)
 * 
 * IMCs implementing the UNIX/Linux Dynamic Linkage platform binding MUST define this
 * additional platform-specific function. The TNC Client MUST call the function immediately after
 * calling TNC_IMC_Initialize to provide a pointer to the TNCC bind function. The IMC can then
 * use the TNCC bind function to obtain pointers to any other TNCC functions.
 *
 * In the imcID parameter, the TNCC MUST pass the value provided to TNC_IMC_Initialize. In
 * the bindFunction parameter, the TNCC MUST pass a pointer to the TNCC bind function. IMCs
 * MAY check if imcID matches the value previously passed to TNC_IMC_Initialize and return
 * TNC_RESULT_INVALID_PARAMETER if not, but they are not required to make this check.
 *
 * @param imcID - IMC ID assigned by TNCC
 * @param bindFunction - Pointer to TNC_TNCC_BindFunction
 */
TNC_IMC_API TNC_Result TNC_IMC_ProvideBindFunction(
/*in*/  TNC_IMCID imcID,
/*in*/  TNC_TNCC_BindFunctionPointer bindFunction) {
    TNC_Result rc = TNC_RESULT_SUCCESS;

    DEBUG("TNC_IMC_ProvideBindFunction\n");

    /* check internal status */
    if (!initialized)
        return TNC_RESULT_NOT_INITIALIZED;

    /* check ID */
    if (imcID != id)
        return TNC_RESULT_INVALID_PARAMETER;

    /* Bind  */
    if (bindFunction) {
        if ((*bindFunction)(imcID,
                            "TNC_TNCC_ReportMessageTypes",
                            (void**)&reportMessageTypesPtr)
                != TNC_RESULT_SUCCESS) {
            ERROR("bind function fails -TNC_TNCC_ReportMessageTypes\n");
            rc = TNC_RESULT_FATAL;
            return rc;
        }
        if ((*bindFunction)(imcID,
                            "TNC_TNCC_RequestHandshakeRetry",
                            (void**)&requestHandshakeRetryPtr)
                != TNC_RESULT_SUCCESS) {
            ERROR("bind function fails - TNC_TNCC_RequestHandshakeRetry\n");
            rc = TNC_RESULT_FATAL;
            return rc;
        }
        if ((*bindFunction)(imcID,
                            "TNC_TNCC_SendMessage",
                            (void**)&sendMessagePtr)
                != TNC_RESULT_SUCCESS) {
            ERROR("bind functionfails -  TNC_TNCC_SendMessage\n");
            rc = TNC_RESULT_FATAL;
            return rc;
        }
    }

    // TODO(munetoh) rc
    rc = reportMessageTypes(imcID,
                            messageTypes,
                            sizeof(messageTypes) / sizeof(TNC_MessageType));

    return rc;
}





