/*
 * This file is part of the OpenPTS project.
 *
 * The Initial Developer of the Original Code is International
 * Business Machines Corporation. Portions created by IBM
 * Corporation are Copyright (C) 2010 International Business
 * Machines Corporation. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Common Public License as published by
 * IBM Corporation; either version 1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Common Public License for more details.
 *
 * You should have received a copy of the Common Public License
 * along with this program; if not, a copy can be viewed at
 * http://www.opensource.org/licenses/cpl1.0.php.
 */

/**
 * \file src/verifier.c
 * \brief TCG IF-M Verifier
 * @author Seiji Munetoh <munetoh@users.sourceforge.jp>
 * @date 2010-04-06
 * cleanup 2011-04-26 SM
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>

#define __USE_GNU
#include <search.h>  // hash table

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <unistd.h>
#include <fcntl.h>

#include <openpts.h>


/**
 * enroll
 *
 * get target UUID
 * get target RMs
 * else?
 *
 * Function Test
 *
 *   file         test
 *   ----------------------------------
 *   
 */
int enroll(
    OPENPTS_CONTEXT *ctx,
    char *host,
    char *ssh_username,
    char *ssh_port,
    char *conf_dir,
    int force) {
    int sock;
    int rc = PTS_SUCCESS;
    PTS_IF_M_Attribute *read_tlv = NULL;
    char *str_verifier_uuid = NULL;
    char buf[BUF_SIZE];
    int i;
    struct stat st;
    pid_t ssh_pid;
    int ssh_status;

    // OPENPTS_CONFIG *conf;
    OPENPTS_CONFIG *target_conf;
    PTS_Byte *value;
    OPENPTS_IF_M_Capability *cap;

    /* check */
    // TODO

    DEBUG("enroll() - start, force = %d  (1:overwite) --------------------------------------\n", force);

    if (ctx->conf == NULL) {
        ERROR("ctx->conf\n");
        return PTS_INTERNAL_ERROR;
    }

    if (ctx->target_conf == NULL) {
        TODO("enroll() - ctx->target_conf must be set before\n");
        ctx->target_conf = newPtsConfig();
    }
    target_conf = ctx->target_conf;

    if (target_conf->uuid == NULL) {
        target_conf->uuid = newOpenptsUuid();
    }
    if (target_conf->rm_uuid == NULL) {
        target_conf->rm_uuid = newOpenptsUuid();
    }

    /* connect to the target collector */
    ssh_pid = ssh_connect(host,
                          ssh_username,
                          ssh_port,
                          NULL,
                          &sock);

    if (ssh_pid == -1) {
        ERROR("connection to %s failed.\n", host);
        rc = PTS_OS_ERROR;
        goto out;
    }

    /* V->C capability (hello) */
    rc = writePtsTlv(ctx, sock, OPENPTS_CAPABILITIES);

    /* C->V capability (hello) */
    read_tlv = readPtsTlv(sock);
    if (read_tlv == NULL) {
        ERROR("Can not get the message from collector\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }
    if (read_tlv->type != OPENPTS_CAPABILITIES) {
        ERROR("\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }
    if (read_tlv->length != sizeof(OPENPTS_IF_M_Capability)) {  // TODO set name
        ERROR("UUID length = %d != 36\n", read_tlv->length);
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }
    cap =  (OPENPTS_IF_M_Capability *)read_tlv->value;

    /* version */
    memcpy(&target_conf->pts_flag,    &cap->flag, 4);
    memcpy(&target_conf->tpm_version, &cap->tpm_version, 4);
    memcpy(&target_conf->tss_version, &cap->tss_version, 4);
    memcpy(&target_conf->pts_version, &cap->pts_version, 4);

    /* Collector - UUID */
    if (target_conf->uuid != NULL) {
        // DEBUG("realloc target_conf->uuid\n");  // TODO realloc happen
        freeOpenptsUuid(target_conf->uuid);
    }
    target_conf->uuid = newOpenptsUuid();
    target_conf->uuid->uuid = malloc(sizeof(PTS_UUID));
    memcpy(target_conf->uuid->uuid, &cap->platform_uuid, 16);
    target_conf->uuid->str = getStringOfUuid(target_conf->uuid->uuid);

    /* RM Set - UUID */
    if (target_conf->rm_uuid != NULL) {
        // DEBUG("realloc target_conf->rm_uuid\n");  // TODO realloc happen
        freeOpenptsUuid(target_conf->rm_uuid);
    }
    target_conf->rm_uuid = newOpenptsUuid();
    target_conf->rm_uuid->uuid = malloc(sizeof(PTS_UUID));
    memcpy(target_conf->rm_uuid->uuid, &cap->manifest_uuid, 16);
    target_conf->rm_uuid->str = getStringOfUuid(target_conf->rm_uuid->uuid);

    /* verifier */
    str_verifier_uuid = getStringOfUuid(ctx->conf->uuid->uuid);

    DEBUG("Verifier  UUID         : %s\n", str_verifier_uuid);
    DEBUG("Collector UUID         : %s\n", target_conf->uuid->str);

    /* Setup the dir for the collector */
    if (target_conf->config_dir != NULL) {
        // DEBUG("realloc target_conf->config_dir()\n");  // TODO realloc happen
        free(target_conf->config_dir);
    }
    target_conf->config_dir = getFullpathName(conf_dir, target_conf->uuid->str);

    if (target_conf->rm_basedir != NULL) {
        // DEBUG("realloc target_conf->rm_basedir\n");  // TODO realloc happen
        free(target_conf->rm_basedir);
    }
    target_conf->rm_basedir = getFullpathName(target_conf->config_dir, target_conf->rm_uuid->str);

    rc = makeDir(target_conf->config_dir);
    rc = makeDir(target_conf->rm_basedir);

    /* check the local storage of this collector */
    target_conf->uuid->filename = getFullpathName(target_conf->config_dir, "uuid");

    if (target_conf->aide_database_filename != NULL) {
        DEBUG("realloc ctx->conf->aide_database_filename\n");
        free(target_conf->aide_database_filename);
    }
    target_conf->aide_database_filename = getFullpathName(target_conf->config_dir, "aide.db.gz");
#ifdef CONFIG_SQLITE
    target_conf->aide_sqlite_filename = getFullpathName(target_conf->config_dir, "aide.sqlite.db");
#endif
    if (target_conf->aide_ignorelist_filename != NULL) {
        DEBUG("realloc ctx->conf->aide_ignorelist_filename\n");
        free(target_conf->aide_ignorelist_filename);
    }
    target_conf->aide_ignorelist_filename = getFullpathName(target_conf->config_dir, "aide.ignore");

    if (target_conf->config_file != NULL) {
        // DEBUG("realloc target_conf->config_file\n");  // TODO realloc happen
        free(target_conf->config_file);
    }
    target_conf->config_file = getFullpathName(target_conf->config_dir, "target.conf");

    DEBUG("conf dir               : %s\n", target_conf->config_dir);
    DEBUG("rm dir                 : %s\n", target_conf->rm_basedir);
    DEBUG("AIDE DB                : %s\n", target_conf->aide_database_filename);
#ifdef CONFIG_SQLITE
    DEBUG("AIDE SQLite DB         : %s\n", target_conf->aide_sqlite_filename);
#endif

    if (force == 1) {
        /* delete existing info */
        // DEBUG("enroll - force=1 NA. Sorry\n");
    } else {
        /* check existing info */
        struct stat st;
        if (lstat(target_conf->config_file , &st) == -1) {
            // Missing,
            DEBUG("%s is missing. Get RM from target\n", target_conf->config_file);
        } else {
            // EXIST -> Update
            fprintf(stderr, "%s exist. if you want to override, use -f option\n", target_conf->config_file);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    }


    /* free */
    // TODO free
    // read_tlv->value -= 16; // TODO
    freePtsTlv(read_tlv);

    /* get the Reference Manifest from target(collector) */


    /* V->C template RIMM req  */
    rc = writePtsTlv(ctx, sock, REQUEST_RIMM_SET);

    if (rc != 0) {
        ERROR("template RIMM req was failed\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    /* C->V template RIMM (RIMM embedded to CTX) */
    read_tlv = readPtsTlv(sock);
    if (read_tlv->type != RIMM_SET) {
        ERROR("");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }
    value = read_tlv->value;

    /* RIMM -> CTX */
    DEBUG("RIMM total size        : %d\n", read_tlv->length);
    {
        int num;
        int len;

        num = getUint32(value);
        DEBUG("RM num                 : %d\n", num);

        target_conf->rm_num = num;

        value += 4;

        // TODO remove n below - 20110112 SM
        /* Check RM DIR */
        snprintf(buf, BUF_SIZE, "%s/%s",
                target_conf->config_dir,
                target_conf->rm_uuid->str);

        if (lstat(buf, &st) == -1) {
            /* Missing conf dir => create */
            rc = mkdir(buf, S_IRUSR | S_IWUSR | S_IXUSR);
            if (rc != 0) {
                ERROR("create conf directory, %s was failed\n", buf);
                rc = PTS_INTERNAL_ERROR;
                goto close;
            }
            // DEBUG("RM DIR  : %s\n", buf);
        } else if ((st.st_mode & S_IFMT) != S_IFDIR) {
            ERROR("RM directory, %s is not a directory %x %x\n", buf, (st.st_mode & S_IFMT), S_IFDIR);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }

        /* Get RMs  */
        ctx->conf->rm_num = num;
        for (i = 0; i < num; i++) {
            snprintf(buf, BUF_SIZE, "%s/rm%d.xml",
                target_conf->rm_basedir,
                i);

            if (target_conf->rm_filename[i] != NULL) {
                DEBUG("enroll() - free conf->rm_filename[%d] %s\n", i, target_conf->rm_filename[i]);
                free(target_conf->rm_filename[i]);
            }

            target_conf->rm_filename[i] = smalloc(buf);


            len = getUint32(value);
            DEBUG("RM[%d] size             : %d\n", i, len);
            DEBUG("RM[%d] filename         : %s\n", i, target_conf->rm_filename[i]);

            value += 4;

            rc = saveToFile(target_conf->rm_filename[i], len, value);
            if (rc < 0) {
                ERROR("enroll - save RM[%d], %s failed\n", i, target_conf->rm_filename[i]);
                rc = PTS_INTERNAL_ERROR;
                goto close;
            }

            value += len;
        }

        /* RM UUID file */
        /* save to rm_uuid file */
        snprintf(buf, BUF_SIZE, "%s/rm_uuid",
            target_conf->config_dir);
        rc = writeUuidFile(target_conf->rm_uuid->str, buf, 1);  // TODO do not overwite?
    }

    /* free */
    freePtsTlv(read_tlv);

#ifdef CONFIG_AIDE
    // TODO(munetoh) capability defile validation mode of collector
    /* V->C AIDE_DATABASE req  */
    rc = writePtsTlv(ctx, sock, REQUEST_AIDE_DATABASE);

    if (rc != 0) {
        ERROR("template RIMM req was failed\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    /* C->V AIDE DATABASE */
    read_tlv = readPtsTlv(sock);
    if (read_tlv->type != AIDE_DATABASE) {
        if (read_tlv->type == OPENPTS_ERROR) {
            // TODO check msg?
            /* AIDE DB is missing */
            target_conf->ima_validation_mode = OPENPTS_VALIDATION_MODE_NONE;
            DEBUG("enroll - AIDE DB is missing. do not validate IMA's IMLs\n");
        } else {
            ERROR("");
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    } else {
        // Got AIDE DB?
        if (read_tlv->length > 0) {
            /* AIDE_DATABASE -> CTX */
            DEBUG("AIDE_DATABASE size     : %d\n", read_tlv->length);

            rc = saveToFile(target_conf->aide_database_filename, read_tlv->length, read_tlv->value);
            if (rc < 0) {
                ERROR("enroll - save AIDE DB failed\n");
                rc = PTS_INTERNAL_ERROR;
                goto close;
            }

#ifdef CONFIG_SQLITE
            DEBUG("conv to sqlite %s\n", target_conf->aide_sqlite_filename);
            rc = convertAideDbfileToSQLiteDbFile(
                    ctx->conf->aide_database_filename,
                    ctx->conf->aide_sqlite_filename);
            if (rc != PTS_SUCCESS) {
                ERROR("enroll - convert AIDE DB to SQLiteDB was failed\n");
                rc = PTS_INTERNAL_ERROR;
                goto close;
            }
#endif
            target_conf->ima_validation_mode = OPENPTS_VALIDATION_MODE_AIDE;
        } else {
            /* no AIDE DB */
            target_conf->ima_validation_mode = OPENPTS_VALIDATION_MODE_NONE;
            DEBUG("enroll - AIDE DB is missing. do not validate IMA's IMLs\n");
        }
    }

    /* free */
    freePtsTlv(read_tlv);

#endif  // CONFIG_AIDE



    /* V->C TPM PUBKEY req */
    rc = writePtsTlv(ctx, sock, REQUEST_TPM_PUBKEY);

    if (rc != 0) {
        ERROR("TPM PUBKEY req was failed\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    /* C->V TPM PUBKEY */
    read_tlv = readPtsTlv(sock);
    if (read_tlv->type == OPENPTS_ERROR) {
        // TODO Ignore now
        TODO("Target did not have TPM_PUBKEY");
    } else if (read_tlv->type != TPM_PUBKEY) {
        ERROR("read_tlv->type != TPM_PUBKEY, but %d", read_tlv->type);
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    if (read_tlv->length > 0) {
        /* TPM_PUBKEY -> CTX */
        DEBUG("TPM_PUBKEY size        : %d\n", read_tlv->length);

        // TODO used by two
        if (target_conf->pubkey != NULL) {
            DEBUG("enroll() - reset the PUBKEY\n");
            free(target_conf->pubkey);
        }

        target_conf->pubkey_length = read_tlv->length;
        target_conf->pubkey = malloc(target_conf->pubkey_length);
        // TODO check NULL
        memcpy(
            target_conf->pubkey,
            read_tlv->value,
            target_conf->pubkey_length);
        /* save to the target.conf */
    } else {
        DEBUG("enroll - TPM_PUBKEY is missing.\n");
    }

    /* free */
    freePtsTlv(read_tlv);

    /* save target conf */
    writeTargetConf(target_conf, target_conf->uuid->uuid, target_conf->config_file);  // ctx.c

    /* OK */
    rc = PTS_SUCCESS;

  close:
    close(sock);
    waitpid(ssh_pid, &ssh_status, 0);

  out:
    /* free */
    if (str_verifier_uuid != NULL) free(str_verifier_uuid);

    DEBUG("enroll() - done, force = %d  (1:overwite) --------------------------------------\n", force);

    return rc;
}



/**
 * updateRm
 *
 * get target NEW RMs
 *
 * Function Test
 *
 *   file         test
 *   ----------------------------------
 *   
 */
int updateRm(
    OPENPTS_CONTEXT *ctx,
    char *host,
    char *ssh_username,
    char *ssh_port,
    char *conf_dir) {
    int sock;
    int rc;
    PTS_IF_M_Attribute *read_tlv;
    char *rm_filename[MAX_RM_NUM];
    pid_t ssh_pid;
    int ssh_status;

    PTS_UUID *collector_uuid;
    char *str_collector_uuid;
    PTS_UUID *rm_uuid;
    char *str_rm_uuid;
    char *str_verifier_uuid;
    char buf[BUF_SIZE];
    int i;
    char * collector_dir;
    char * rm_dir;
    OPENPTS_CONFIG *conf;
    OPENPTS_IF_M_Capability *cap;

    /* check */
    if (ctx == NULL) {
        ERROR("ctx is null\n");
        return PTS_INTERNAL_ERROR;
    }
    conf = ctx->conf;
    if (conf == NULL) {
        ERROR("conf is null\n");
        return PTS_INTERNAL_ERROR;
    }

    /* connect to the target collector */
    ssh_pid = ssh_connect(host,
                          ssh_username,
                          ssh_port,
                          NULL,
                          &sock);

    if (ssh_pid == -1) {
        ERROR("connection to %s failed.\n", host);
        rc = PTS_OS_ERROR;
        goto out;
    }

    /* V->C capability (hello) */
    rc = writePtsTlv(ctx, sock, OPENPTS_CAPABILITIES);

    /* C->V capability (hello) */
    read_tlv = readPtsTlv(sock);
    if (read_tlv->type != OPENPTS_CAPABILITIES) goto error;
    if (read_tlv->length != sizeof(OPENPTS_IF_M_Capability)) {  // TODO set name
        ERROR("UUID length = %d != 36\n", read_tlv->length);
        goto error;
    }
    cap = (OPENPTS_IF_M_Capability *) read_tlv->value;

    /* version */
    // memcpy(&target_conf->tpm_version, &cap->tpm_version, 4);
    // memcpy(&target_conf->tss_version, &cap->tss_version, 4);
    // memcpy(&target_conf->pts_version, &cap->pts_version, 4);

    /* Collector - UUID */
    collector_uuid = malloc(sizeof(PTS_UUID));
    memcpy(collector_uuid, &cap->platform_uuid, 16);

    /* RM Set - UUID */
    rm_uuid = malloc(sizeof(PTS_UUID));
    memcpy(rm_uuid, &cap->manifest_uuid, 16);

    /* UUID strings */
    str_collector_uuid = getStringOfUuid(collector_uuid);
    str_rm_uuid = getStringOfUuid(rm_uuid);
    str_verifier_uuid = getStringOfUuid(ctx->conf->uuid->uuid);
    if ((str_collector_uuid == NULL) ||
        (str_rm_uuid == NULL) ||
        (str_verifier_uuid == NULL)) {
        ERROR("no memory\n");
        goto error;
    }

    DEBUG("Verifier  UUID    %s\n", str_verifier_uuid);
    DEBUG("Collector UUID    %s\n", str_collector_uuid);
    DEBUG("Collector RM UUID %s\n", str_rm_uuid);

    /* Setup the dir for the collector */
    collector_dir = getFullpathName(conf_dir, str_collector_uuid);
    rm_dir = getFullpathName(collector_dir, str_rm_uuid);

    rc = checkDir(collector_dir);
    if (rc != PTS_SUCCESS) {
        /* unknwon collector */
        ERROR("Unknown collector, UUID= %s\n", str_collector_uuid);
        addReason(ctx, "Missing collector configuration");
        addReason(ctx, "Collector hostname = %s", host);
        addReason(ctx, "Collector UUID = %s", str_collector_uuid);
        rc = PTS_NOT_INITIALIZED;
        goto error;
    }
    rc = checkDir(rm_dir);
    if (rc == PTS_SUCCESS) {
        /* ??? Already Exist */
        ERROR("Exist RM, UUID= %s\n", str_collector_uuid);
        addReason(ctx, "Already exist the Reference Manifest(RM)");
        addReason(ctx, "Collector hostname = %s", host);
        addReason(ctx, "Collector UUID = %s", str_collector_uuid);
        addReason(ctx, "Collector RM UUID = %s", str_rm_uuid);
        rc = PTS_FATAL;
        goto error;
    }

    /* create new RM dir */
    rc = makeDir(rm_dir);
    if (rc != PTS_SUCCESS) {
        /* unknwon collector */
        ERROR("Create RM dir failed, %s\n", rm_dir);
        rc = PTS_INTERNAL_ERROR;
        goto error;
    }

    conf->property_filename = getFullpathName(collector_dir, "vr.properties");
    conf->ir_filename = getFullpathName(collector_dir, "ir.xml");
    conf->prop_filename = getFullpathName(collector_dir, "target.conf");

    if (conf->rm_uuid == NULL) {
        // TODO gen at ead pts conf
        conf->rm_uuid = newOpenptsUuid();
    }
    conf->rm_uuid->filename = getFullpathName(collector_dir, "rm_uuid");

    DEBUG("conf dir       : %s\n", collector_dir);
    DEBUG("rm dir         : %s\n", rm_dir);
    DEBUG("RM UUID file   : %s\n", conf->rm_uuid->filename);

    /* free */
    freePtsTlv(read_tlv);

    /* get the Reference Manifest from target(collector) */

    /* V->C template RIMM req  */
    rc = writePtsTlv(ctx, sock, REQUEST_RIMM_SET);

    if (rc != 0) {
        ERROR("template RIMM req was failed\n");
        goto error;
    }

    /* C->V template RIMM (RIMM embedded to CTX) */
    read_tlv = readPtsTlv(sock);
    if (read_tlv->type != RIMM_SET) {
        ERROR("");
        goto error;
    }

    /* RIMM -> CTX */
    DEBUG("RIMM len %d\n", read_tlv->length);
    {
        int num;
        int len;

        num = getUint32(read_tlv->value);
        DEBUG("RM num %d\n", num);

        read_tlv->value += 4;

        /* Get RMs  */
        ctx->conf->rm_num = num;
        for (i = 0; i < num; i++) {
            /* RM file*/
            snprintf(buf, BUF_SIZE, "%s/%s/rm%d.xml",
                collector_dir,
                str_rm_uuid,
                i);
            rm_filename[i] = smalloc(buf);
            DEBUG("RM[%d]          : %s\n", i, rm_filename[i]);

            len = getUint32(read_tlv->value);
            DEBUG("RM[%d] len %d -> %s\n", i, len, rm_filename[i]);

            read_tlv->value += 4;

            rc = saveToFile(rm_filename[i], len, read_tlv->value);
            if (rc < 0) {
                ERROR("engage - save RM[%d], %s failed\n", i, rm_filename[i]);
                goto error;
            }
            ctx->conf->rm_filename[i] = smalloc(rm_filename[i]);

            read_tlv->value += len;
        }

        /* RM UUID file */
        /* save to rm_uuid file */
        snprintf(buf, BUF_SIZE, "%s/rm_uuid",
            collector_dir);
        rc = writeUuidFile(str_rm_uuid, buf, 1);  // overwite
    }

    /* save target conf */
    // TODO need to updade?
    // writeTargetConf(ctx->conf, collector_uuid, target_conf_filename);  // ctx.c

    rc = PTS_SUCCESS;

  error:
    close(sock);
    waitpid(ssh_pid, &ssh_status, 0);

  out:
    // DEBUG("error at verifier\n");
    return rc;
}


/**
 * Write policy.conf from current prop
 *
 * return num of polocy
 *
 * ignore ima.0.*=*
 *
 * TODO move to prop.c
 */
int  writePolicyConf(OPENPTS_CONTEXT *ctx, char *filename) {
    FILE *fp;
    OPENPTS_PROPERTY *prop;
    int i = 0;

    DEBUG("writePolicyConf       : %s\n", filename);

    if ((fp = fopen(filename, "w")) == NULL) {
        ERROR("File %s open was failed\n", filename);
        return -1;
    }

    /* top */
    prop = ctx->prop_start;

    fprintf(fp, "# OpenPTS validation policy, name=value\n");
    while (prop != NULL) {
        if (!strncmp(prop->name, "ima.aggregate", 13)) {
            /* IMA aggregate validation policy */
            fprintf(fp, "%s=%s\n", prop->name, prop->value);
            i++;
        } else if (!strncmp(prop->name, "ima.", 4)) {
            /* IMA measurement - SKIP */
        } else {
            fprintf(fp, "%s=%s\n", prop->name, prop->value);
            i++;
        }
        prop = prop->next;
    }
    fprintf(fp, "# %d reference props\n", i);
    fclose(fp);

    return i;
}


#ifdef CONFIG_AIDE
#define HASH_TABLE_SIZE ((size_t) 2048)

/**
 * Write writeAideIgnoreList from current prop
 * IMA measurment with OPENPTS_RESULT_UNKNOWN flag -> 
 *
 * Returm
 *   n  count of list
 *   -1 ERROR
 *
 * ima.0.integrty=unknown
 * ima.0.name=/init
 * 
 * TODO move to prop.c?
 * TODO use hash table, name:count?
 */
int  writeAideIgnoreList(OPENPTS_CONTEXT *ctx, char *filename) {
    FILE *fp;
    OPENPTS_SNAPSHOT * ss;
    OPENPTS_PCR_EVENT_WRAPPER *ew;
    TSS_PCR_EVENT *event;
    char *name;
    int cnt = 0;
    // int i;
    /* hash */
    struct hsearch_data hd;
    ENTRY e, *ep;
    void* ecnt = 0;
    int rc;

    DEBUG("writeAideIgnoreList     : %s\n", filename);

    if ((fp = fopen(filename, "w")) == NULL) {
        ERROR("File %s open was failed\n", filename);
        return -1;
    }

    /* top */
    ss = getSnapshotFromTable(ctx->ss_table, 10, 1);  // Linux-IMA, TODO define by CONF?
    ew = ss->start;


    /* look over the  event chain  */
    fprintf(fp, "# OpenPTS AIDE ignore name list\n");

    /* ew -> hash */
    memset(&hd, 0, sizeof(hd));

    rc = hcreate_r(HASH_TABLE_SIZE, &hd);
    if (rc == 0) {
        if (errno == ENOMEM) {
            ERROR("ENOMEM\n");
            cnt = -1;
            goto error;
        }
        ERROR("ERROR rc=%d\n", rc);
        // return -1;
        cnt = -1;
        goto error;
    }

    while (ew != NULL) {
        if (ew->status == OPENPTS_RESULT_UNKNOWN) {
            event = ew->event;
            name = (char *)event->rgbEvent;
            name += SHA1_DIGEST_SIZE;
            /* add '\n' */
            name = snmalloc(name, (event->ulEventLength - SHA1_DIGEST_SIZE));

            ecnt = 0;

            e.key = name;
            e.data = NULL;

            rc = hsearch_r(e, FIND, &ep, &hd);
            if (rc == 0) {
                /* miss */
                e.data = (void*) ecnt;
                rc = hsearch_r(e, ENTER, &ep, &hd);
                // TODO check error
                fprintf(fp, "# %d \n", cnt);
                fprintf(fp, "%s\n", name);
                cnt++;
            } else {
                /* hit, ++ */
                ecnt = ep->data;
                ecnt++;
                ep->data = ecnt;
            }
        }
        ew = ew->next_pcr;
    }

    hdestroy_r(&hd);

    /* close  */
    fprintf(fp, "# %d props\n", cnt);

  error:
    fclose(fp);

    return cnt;
}
#endif  // CONFIG_AIDE


/**
 * verifier
 *
 * @param ctx
 * @param host
 * @param port
 * @param conf_dir - base dir of collector configuration
 * @param mode   0:normal  1:sync (update policy, ignorelist)
 *     0  OPENPTS_VERIFY_MODE
 *     1  OPENPTS_UPDATE_MODE -- note) do not update the RM, use updateRm()
 *
 * Returm
 *   PTS_SUCCESS
 *   PTS_OS_ERROR
 *   PTS_INTERNAL_ERROR
 *   PTS_RULE_NOT_FOUND  RM not found
 *
 * Function Test
 *
 *   file         test
 *   ----------------------------------
 *   check_ifm.c  test_ifm
 */
int verifier(
    OPENPTS_CONTEXT *ctx,
    char *host,
    char *ssh_username,
    char *ssh_port,
    char *conf_dir,
    int mode) {
    int rc = PTS_SUCCESS;
    int result = OPENPTS_RESULT_VALID;
    /* sock */
    int sock;
    pid_t ssh_pid;
    int ssh_status;

    /* TLV/PTS */
    PTS_IF_M_Attribute *read_tlv = NULL;
    OPENPTS_UUID *rm_uuid = NULL;  // received from collector
    char *str_verifier_uuid = NULL;
    OPENPTS_CONFIG *conf;
    OPENPTS_CONFIG *target_conf = NULL;
    int i;
    char * collector_dir = NULL;
    char * rm_dir = NULL;
    OPENPTS_IF_M_Capability *cap;

    DEBUG("verifier() - start\n");
    DEBUG("  conf_dir             : %s\n", conf_dir);
    DEBUG("  mode                 : %d  (0:just verify, 1:update the policy)\n", mode);
    /* check */
    if (ctx == NULL) {
        ERROR("ctx is null\n");
        return PTS_INTERNAL_ERROR;
    }
    conf = ctx->conf;
    if (conf == NULL) {
        ERROR("conf is null\n");
        return PTS_INTERNAL_ERROR;
    }

    /* target_conf */
    if (ctx->target_conf == NULL) {
        TODO("must set the target conf before\n");
        ctx->target_conf = newPtsConfig();
    }
    target_conf = ctx->target_conf;


    /* connect to the target collector */
    ssh_pid = ssh_connect(host,
                          ssh_username,
                          ssh_port,
                          NULL,
                          &sock);

    if (ssh_pid == -1) {
        ERROR("connection failed (server = %s)\n", host);
        addReason(ctx, "connection failed (server = %s)\n", host);
        rc = PTS_OS_ERROR;
        goto out;
    }



    /* IF-M start */

    /* V->C capability (hello) */
    rc = writePtsTlv(ctx, sock, OPENPTS_CAPABILITIES);

    /* C->V capability (hello) */
    read_tlv = readPtsTlv(sock);
    if (read_tlv == NULL) {
        ERROR("can't connect to target, %s\n", host);
        rc = PTS_INTERNAL_ERROR;
        goto close;
    } else if (read_tlv->type != OPENPTS_CAPABILITIES) {
        goto close;
    } else if (read_tlv->length != sizeof(OPENPTS_IF_M_Capability)) {
        // TODO PTS_CAPABILITIES_SIZE
        ERROR("UUID length = %d != 36\n", read_tlv->length);
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }
    cap = (OPENPTS_IF_M_Capability *)read_tlv->value;

    /* version */
    memcpy(&target_conf->pts_flag, &cap->flag, 4);
    memcpy(&target_conf->tpm_version, &cap->tpm_version, 4);
    memcpy(&target_conf->tss_version, &cap->tss_version, 4);
    memcpy(&target_conf->pts_version, &cap->pts_version, 4);

    /* Collector - UUID (update target conf) */
    if (target_conf->uuid != NULL) {
        // DEBUG("realloc target_conf->uuid\n");  // TODO realloc happen
        freeOpenptsUuid(target_conf->uuid);
    }
    target_conf->uuid = newOpenptsUuid();
    target_conf->uuid->uuid = malloc(sizeof(PTS_UUID));
    memcpy(target_conf->uuid->uuid, &cap->platform_uuid, 16);
    target_conf->uuid->str = getStringOfUuid(target_conf->uuid->uuid);

    /* RM Set - UUID  (local variable) */
    rm_uuid = newOpenptsUuid();
    rm_uuid->uuid = malloc(sizeof(PTS_UUID));
    memcpy(rm_uuid->uuid, &cap->manifest_uuid, 16);
    rm_uuid->str = getStringOfUuid(rm_uuid->uuid);

    /* verifier */
    str_verifier_uuid = getStringOfUuid(ctx->conf->uuid->uuid);

    if ((target_conf->uuid->str == NULL) ||
        (rm_uuid->str == NULL) ||
        (str_verifier_uuid == NULL)) {
        ERROR("no memory\n");
        goto close;
    }

    DEBUG("Verifier  UUID         : %s\n", str_verifier_uuid);
    DEBUG("Collector UUID         : %s\n", target_conf->uuid->str);
    DEBUG("RM  UUID               : %s\n", rm_uuid->str);

    /* Setup the dir for the collector */
    collector_dir = getFullpathName(conf_dir, target_conf->uuid->str);
    rm_dir = getFullpathName(collector_dir, rm_uuid->str);

    if (conf->policy_filename != NULL) {
        // DEBUG("realloc conf->policy_filename  \n");   // TODO realloc happen
        free(conf->policy_filename);
    }
    conf->policy_filename = getFullpathName(collector_dir, "policy.conf");

    if (conf->property_filename != NULL) {
        // DEBUG("realloc conf->property_filename  \n");  // TODO realloc happen
        free(conf->property_filename);
    }
    conf->property_filename = getFullpathName(collector_dir, "vr.property");  // TODO ptoperty or prop

    // TODO if DIR is missing it was new collector or new RM set
    // DEBUG("collector_dir %s\n", collector_dir);

    /* check Collector  */
    rc = checkDir(collector_dir);
    if (rc != PTS_SUCCESS) {
        /* unknwon collector */
        ERROR("verifier() - Unknown collector, UUID= %s dir= %s, rc=%d\n",
            target_conf->uuid->str, collector_dir, rc);
        addReason(ctx, "Missing collector configuration");
        addReason(ctx, "Collector hostname = %s", host);
        addReason(ctx, "Collector UUID = %s", target_conf->uuid->str);
        rc = PTS_NOT_INITIALIZED;
        goto close;
    }

    /* check RM */
    rc = checkDir(rm_dir);
    if (rc != PTS_SUCCESS) {
        /* unknwon RM */
        ERROR("verifier() - Unknown RM, (RM dir = %s)\n", rm_dir);
        addReason(ctx, "Missing Reference Manifest(RM)");
        addReason(ctx, "Collector hostname = %s", host);
        addReason(ctx, "Collector UUID = %s", target_conf->uuid->str);
        addReason(ctx, "Collector RM UUID = %s", rm_uuid->str);
        addReason(ctx, "Missing RM dir = %s", rm_dir);
        rc = PTS_RULE_NOT_FOUND;
        goto close;
    }

    /* check the local storage */
    if (target_conf->config_file != NULL) {
        // DEBUG("realloc target_conf->config_file\n");  // TODO realloc happen
        free(target_conf->config_file);
    }
    target_conf->config_file = getFullpathName(collector_dir, "target.conf");

    rc = readTargetConf(target_conf, target_conf->config_file);
    if (rc != PTS_SUCCESS) {
        ERROR("verifier() - readTargetConf failed\n");
        // TODO so?
    }

    /* check and setup the RM */
    /* stored RM UUID */
    if (target_conf->rm_uuid == NULL) {
        ERROR("target_conf->rm_uuid == NULL\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }
    rc = readOpenptsUuidFile(target_conf->rm_uuid);
    if (rc != PTS_SUCCESS) {
        ERROR("read RM UUID file %s was failed, initialize ptscd first\n", target_conf->rm_uuid->filename);
        rc = PTS_INTERNAL_ERROR;
        goto close;
    } else {
        DEBUG("conf->str_rm_uuid      : %s\n", target_conf->rm_uuid->str);
    }

    /* stored NEW RM UUID */
    rc = readOpenptsUuidFile(target_conf->newrm_uuid);
    if (rc != PTS_SUCCESS) {
        /* not exist -> OK */
        DEBUG("conf->str_newrm_uuid   : missing (file:%s)\n", target_conf->newrm_uuid->filename);
        target_conf->newrm_uuid->uuid = NULL;
        target_conf->newrm_uuid->str = NULL;
        target_conf->newrm_uuid->time = NULL;
    } else {
        DEBUG("conf->str_newrm_uuid   : %s\n", target_conf->newrm_uuid->str);
    }

    //  Given UUID == newrm_uuid  => Reboot OK, change newrm_uuid to rm_uuid
    //  Given UUID == rm_uuid  => OK
    //  Given UUID == missing  => ASK

    /* compare stored NEWRM UUID and given RM UUID */
    if (target_conf->newrm_uuid != NULL) {
        if (target_conf->newrm_uuid->uuid != NULL) {
            if (memcmp(target_conf->newrm_uuid->uuid, rm_uuid->uuid, 16) != 0) {
                /* MISS */
                DEBUG("RM changed %s -> %s (not new rm)\n", target_conf->newrm_uuid->str, rm_uuid->str);
            } else {
                /* HIT */
                DEBUG("RM changed %s -> %s (good reboot)\n", target_conf->rm_uuid->str, rm_uuid->str);

                printf("Collector's manifest has been changed to new manifest (expected reboot)\n");
                printf("  old manifest UUID : %s\n", target_conf->rm_uuid->str);
                printf("  new manifest UUID : %s\n", rm_uuid->str);

                /* Change the rm_uuid file (for next run) */
                DEBUG("update : %s => %s\n", target_conf->rm_uuid->filename, rm_uuid->str);
                DEBUG("delete : %s\n", target_conf->newrm_uuid->filename);
                rc = writeUuidFile(rm_uuid->str, target_conf->rm_uuid->filename, 1);  // overwrite
                rc = remove(target_conf->newrm_uuid->filename);

                /* Change the rm_uuid */
                target_conf->rm_uuid->str  = target_conf->newrm_uuid->str;
                target_conf->rm_uuid->uuid = target_conf->newrm_uuid->uuid;
                target_conf->rm_uuid->time = target_conf->newrm_uuid->time;

                target_conf->newrm_uuid->str = NULL;
                target_conf->newrm_uuid->uuid = NULL;
                target_conf->newrm_uuid->time = NULL;
                // TODO free old UUID

                goto rm;
            }
        }
    }

    /* check given RM UUID and stored RM UUID */
    if (memcmp(target_conf->rm_uuid->uuid, rm_uuid->uuid, 16) != 0) {
        PTS_DateTime *t0;
        PTS_DateTime *t1;

        DEBUG("RM changed %s -> %s\n", target_conf->rm_uuid->str, rm_uuid->str);

        // TODO update RM
        addReason(ctx, "Collector uses another Reference Manifest(RM)");
        addReason(ctx, "Collector hostname = %s", host);
        addReason(ctx, "Collector UUID = %s", target_conf->uuid->str);

        t0 = getDateTimeOfUuid(target_conf->rm_uuid->uuid);
        t1 = getDateTimeOfUuid(rm_uuid->uuid);

        addReason(ctx, "Previous RM UUID = %s, timestamp = %04d-%02d-%02d-%02d:%02d:%02d",
            conf->rm_uuid->str,
            t0->year + 1900,
            t0->mon + 1,
            t0->mday,
            t0->hour,
            t0->min,
            t0->sec);

        addReason(ctx, "Current  RM UUID = %s, timestamp = %04d-%02d-%02d-%02d:%02d:%02d",
            rm_uuid->str,
            t1->year + 1900,
            t1->mon + 1,
            t1->mday,
            t1->hour,
            t1->min,
            t1->sec);

        rc = PTS_RULE_NOT_FOUND;  // TODO
        goto close;
    }

    // TODO search other (OLD) RMs

 rm:
    /* RM */
    rc = getRmSetDir(target_conf);  // ctx->conf);

    DEBUG("logging dir            : %s\n", collector_dir);
    for (i = 0; i < conf->rm_num; i++) {
        DEBUG("RM[%d]                  : %s\n", i, target_conf->rm_filename[i]);
    }
    DEBUG("AIDE DB                : %s\n", target_conf->aide_database_filename);
#ifdef CONFIG_SQLITE
    DEBUG("AIDE SQLITE DB         : %s\n", target_conf->aide_sqlite_filename);
#endif
    DEBUG("AIDE ignore list       : %s\n", target_conf->aide_ignorelist_filename);
    DEBUG("IR                     : %s\n", target_conf->ir_filename);
    DEBUG("Prop                   : %s\n", target_conf->prop_filename);
    DEBUG("Policy                 : %s\n", target_conf->policy_filename);

    /* check RM */
    for (i = 0; i< target_conf->rm_num; i++) {
        struct stat st;
        if (lstat(target_conf->rm_filename[i], &st) == -1) {
            ERROR("verifier - RM (%s) is missing. Get RM from target. enroll(init) first\n",
                target_conf->rm_filename[i]);
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    }


    /* V->C  D-H nonce param req ---------------------------------- */
    /*   setup req  */
    ctx->nonce->req->reserved = 0;
    ctx->nonce->req->min_nonce_len = 16;
    ctx->nonce->req->dh_group_set = DH_GROUP_2;

    /*   send req   */
    rc = writePtsTlv(ctx, sock, DH_NONCE_PARAMETERS_REQUEST);

    /* C->V  D-H nonce param res ---------------------------------- */

    freePtsTlv(read_tlv);

    read_tlv = readPtsTlv(sock);
    if (read_tlv == NULL) {
        ERROR("[IF-M] DH_NONCE_PARAMETERS_REQUEST was failed, check the collector");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    } else if (read_tlv->type != DH_NONCE_PARAMETORS_RESPONSE) {
        goto close;
    }
    // DEBUG("new read_tlv %p\n",read_tlv);

    /* res -> fin */
    ctx->nonce->res->reserved[0]         = read_tlv->value[0];
    ctx->nonce->res->reserved[1]         = read_tlv->value[1];
    ctx->nonce->res->reserved[2]         = read_tlv->value[2];
    ctx->nonce->res->nonce_length        = read_tlv->value[3];
    ctx->nonce->res->selected_dh_group   = (read_tlv->value[4]<<8) | read_tlv->value[5];
    ctx->nonce->res->hash_alg_set        = (read_tlv->value[6]<<8) | read_tlv->value[7];

    /* set pubkey length */
    setDhPubkeylength(ctx->nonce);

    /* nonce */
    ctx->nonce->res->dh_respondor_nonce = malloc(ctx->nonce->res->nonce_length);
    memcpy(
        ctx->nonce->res->dh_respondor_nonce,
        &read_tlv->value[8],
        ctx->nonce->res->nonce_length);

    /* pubkey */
    ctx->nonce->res->dh_respondor_public = malloc(ctx->nonce->pubkey_length);
    memcpy(
        ctx->nonce->res->dh_respondor_public,
        &read_tlv->value[8 + ctx->nonce->res->nonce_length],
        ctx->nonce->pubkey_length);
    ctx->nonce->pubkey = ctx->nonce->res->dh_respondor_public;  // link

    rc = calcDh(ctx->nonce);
    if (rc != 0) {
        ERROR("calcDh failed\n");
        goto close;
    }

    /* V->C D-H nonce finish  --------------------------------------------- */
    rc = writePtsTlv(ctx, sock, DH_NONCE_FINISH);

    /* V->C IR req -------------------------------------------------------- */
    rc = writePtsTlv(ctx, sock, REQUEST_INTEGRITY_REPORT);

    /* C->V IR ------------------------------------------------------------ */
    freePtsTlv(read_tlv);
    read_tlv = readPtsTlv(sock);
    if (read_tlv == NULL) {
        ERROR("REQUEST_INTEGRITY_REPORT was failed, check the collector");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    } else if (read_tlv->type != INTEGRITY_REPORT) {
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    /* save IR to file */
    if (read_tlv->length > 0) {
        rc = saveToFile(target_conf->ir_filename, read_tlv->length, read_tlv->value);
        if (rc < 0) {
            ERROR("verifier - save IR failed\n");
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    } else {
        ERROR("verifier - collector can not send IR\n");
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    /* load Reference Manifest (BIN-BHV) */
    DEBUG("Load RM  -------------------------------- \n");

    for (i = 0; i <  target_conf->rm_num; i++) {
        rc = readRmFile(ctx, target_conf->rm_filename[i], i);
        if (rc < 0) {
            ERROR("readRmFile fail\n");
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    }

    if (mode == 0) {
        /* Load Policy to validate properties */
        DEBUG("Load Policy  -------------------------------- \n");
        rc = loadPolicyFile(ctx, target_conf->policy_filename);
        if (rc < 0) {
            ERROR("loadPolicyFile fail\n");
            rc = PTS_INTERNAL_ERROR;
            goto close;
        }
    } else {
        /* remove */
        // ctx->conf->aide_ignorelist_filename = NULL;
    }

    /* Validate IR by FSM */
    result = validateIr(ctx, target_conf->ir_filename);  /* ir.c */

    if (mode == OPENPTS_VERIFY_MODE) {
        /* save properties */
        DEBUG("save property          : %s\n", target_conf->prop_filename);

        rc = saveProperties(ctx, target_conf->prop_filename);
        if (rc != PTS_SUCCESS) {
            ERROR("saveProperties was failed %s\n", target_conf->prop_filename);
            goto close;
        }
    } else if (mode == OPENPTS_UPDATE_MODE) {
        /* gen policy and ignore list */
        DEBUG("update policy and ignore list %s\n", target_conf->policy_filename);
        rc = writePolicyConf(ctx, target_conf->policy_filename);
        DEBUG("policy num          : %d policies\n", rc);
        if (ctx->ima_unknown > 0) {
#ifdef CONFIG_AIDE
            rc = writeAideIgnoreList(ctx, target_conf->aide_ignorelist_filename);
            DEBUG("%d ignore list of AIDE\n", rc);
#endif
        }
    } else {
        ERROR("unknown mode %d\n", mode);
        goto close;
    }

    /* V->C VR */
    rc = writePtsTlv(ctx, sock, VERIFICATION_RESULT);
    if (rc < 0) {
        rc = PTS_INTERNAL_ERROR;
        goto close;
    }

    /* return validateIr() result  */
    // TODO
    // OPENPTS_RESULT_INVALID
    if (result == OPENPTS_RESULT_VALID) {
        DEBUG("verifier() result      : VALID");
        rc = PTS_SUCCESS;        // 0 -> 0
    } else if (result == OPENPTS_RESULT_UNVERIFIED) {
        DEBUG("verifier() result      : UNVERIFIED");
        rc = PTS_VERIFY_FAILED;  // 101 -> 34
    } else if (result == OPENPTS_RESULT_INVALID) {
        DEBUG("verifier() result      : INVALID");
        rc = PTS_VERIFY_FAILED;  // 102 -> 34
    } else if (result == OPENPTS_RESULT_UNKNOWN) {
        DEBUG("verifier() result      : UNKNOWN");
        rc = PTS_VERIFY_FAILED;  // 104 -> 34
    } else if (result == OPENPTS_RESULT_IGNORE) {
        DEBUG("verifier() result      : IGNORE");
        rc = PTS_VERIFY_FAILED;  // 103 -> 34
    } else {
        DEBUG("verifier() result      : ERROR");
        rc = PTS_INTERNAL_ERROR;
    }

  close:
    /* close socket */
    close(sock);
    waitpid(ssh_pid, &ssh_status, 0);

  out:
    /* free */
    if (read_tlv != NULL) freePtsTlv(read_tlv);
    if (str_verifier_uuid != NULL) free(str_verifier_uuid);
    if (collector_dir != NULL) free(collector_dir);
    if (rm_dir != NULL) free(rm_dir);

    if (rm_uuid != NULL) freeOpenptsUuid(rm_uuid);

    if ((rc == PTS_VERIFY_FAILED) && (mode == 1)) {
        DEBUG("verifier() - update the policy");
        rc = PTS_SUCCESS;
    }

    DEBUG("verifier() - done (rc = %d)\n", rc);

    return rc;
}


