/*
Copyright (c) 2008, BBR Inc.  All rights reserved.
          (c) 2008 Tobias Hoffmann

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
/*
 pdftoijs.cc
 pdf to ijs filter
*/

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <goo/GooString.h>
#include <goo/gmem.h>
#include <Object.h>
#include <Stream.h>
#include <PDFDoc.h>
#include <SplashOutputDev.h>
#include <stdarg.h>
#include <Error.h>
#include <GlobalParams.h>
#include <splash/SplashTypes.h>
#include <splash/SplashBitmap.h>
extern "C" {
#include <ijs/ijs.h>
#include <ijs/ijs_client.h>
};

namespace {
  int exitCode = 0;
  const char *inputfile = NULL, *outputfile = NULL;
  const char *ijsserver = NULL;
  int job_id = 0;
  int resolution[2] = {0,0};
  enum { NONE=-1, COL_RGB, COL_CMYK, COL_BLACK1, COL_WHITE1, COL_BLACK8, COL_WHITE8 } colspace=NONE;
  const char *devManu=NULL, *devModel=NULL;
};

void CDECL myErrorFun(int pos, char *msg, va_list args)
{
  if (pos >= 0) {
    fprintf(stderr, "ERROR (%d): ", pos);
  } else {
    fprintf(stderr, "ERROR: ");
  }
  vfprintf(stderr, msg, args);
  fprintf(stderr, "\n");
  fflush(stderr);
}

/* parse   300x400  or  300  */
void parse_resolution(char *str)
{
  char *tmp=strchr(str,'x');
  if (tmp) {
    *tmp=0;
    resolution[0]=atoi(str);
    resolution[1]=atoi(tmp+1);
    *tmp='x';
  } else {
    resolution[0]=resolution[1]=atoi(str);
  }
}

/* parse  "cmyk" "grey" "rgb" */
void parse_colorspace(const char *str)
{
  if (strcasecmp(str,"rgb")==0) {
    colspace=COL_RGB;
  } else if (strcasecmp(str,"black1")==0) {
    colspace=COL_BLACK1;
  } else if (strcasecmp(str,"white1")==0) {
    colspace=COL_WHITE1;
  } else if (strcasecmp(str,"black8")==0) {
    colspace=COL_BLACK8;
  } else if (strcasecmp(str,"white8")==0) {
    colspace=COL_WHITE8;
#ifdef SPLASH_CMYK
  } else if (strcasecmp(str,"cmyk")==0) {
    colspace=COL_CMYK;
  } else {
    error(-1,"Unknown colorspace; supported are 'rgb', 'cmyk', 'white1', 'black1', 'white8', 'black8'");
#else
  } else {
    error(-1,"Unknown colorspace; supported are 'rgb', 'white1', 'black1', 'white8', 'black8'");
#endif
    exit(1);
  }
}

void parseOpts(int argc, char **argv)
{
  bool usage=false;
  int c;
  while ((c=getopt(argc,argv,"j:r:s:c:m:d:o:h"))!=-1) {
    switch (c) {
    case 'j':
      job_id=atoi(optarg);
      break;
    case 's':
      ijsserver=optarg;
      break;
    case 'r':
      parse_resolution(optarg);
      break;
    case 'c':
      parse_colorspace(optarg);
      break;
    case 'm':
      devManu=optarg;
      break;
    case 'd':
      devModel=optarg;
      break;
    case 'o':
      outputfile=optarg;
      break;
    case 'h':
    default:
      usage=true;
      break;
    }
  }
  if ( (!ijsserver)||(!resolution[0])||(!resolution[1])||(colspace==-1)||
       (!devManu)||(!devModel) ) {
    usage=true;
  } else if (argc==optind+1) {
    inputfile=argv[optind];
  }

  if (usage) {
    printf("pdftoijs filter\n\n"
           "Usage: %s -m manufacturer -d deviceModel -j jobid -r resolution\n"
           "   -c colorspace -s ijsserver [-o outputfile] [options] [pdffile]\n\n"

           "Required arguments:\n"
           "  -m manufacturer: the DeviceManufacturer of the printer, used by the ijs server\n"
           "  -d deviceModel: the DeviceModel string, used by the ijs server\n"
           "  -r resolution: the desired output resolution, e.g.\n"
           "                 -r 300  or  -r 1200x600\n"
           "  -c colorspace: the desired output colorspace, one of\n"
           "                 'rgb'\n"
           "                 'cmyk' (availability depending on poppler compile-options)\n"
           "                 'white1', 'black1':  1-bit normal/inverted\n"
           "                 'white8', 'black8':  8-bit greyscale normal/inverted\n"
           "  -s ijsserver: the ijsserver executable\n"
           "  [pdffile]: the pdf to process\n\n"
           "Optional arguments:\n"
           "  -j jobid: integer to idenitfy the job\n"
           "  -o outfile: output to [outfile] instead of stdout\n\n"
           ,argv[0]);
    exit(1);
  }
}

int main(int argc, char *argv[]) {
  PDFDoc *doc;
  SplashOutputDev *out;
  SplashColor paperColor;
  int i;
  int npages;
  IjsClientCtx *ctx=NULL;
  enum SplashColorMode cmode;
  int rowpad;
  GBool reverseVideo;

  setErrorFunction(::myErrorFun);
#ifdef GLOBALPARAMS_HAS_A_ARG
  globalParams = new GlobalParams(0);
#else
  globalParams = new GlobalParams();
#endif
  parseOpts(argc, argv);

  if (!inputfile) {
    /* stdin */
    Object obj;
    BaseStream *str;
    FILE *fp;
    int n;
    char buf[BUFSIZ];

    fp = tmpfile();
    if (fp==NULL) {
      error(-1,"Can't create temporary file");
      exit(1);
    }

    /* copy stdin to the tmp file */
    while ((n = read(0,buf,BUFSIZ)) > 0) {
      if ((int)fwrite(buf,1,n,fp) != n) {
        error(-1,"Can't copy stdin to temporary file");
        fclose(fp);
	exit(1);
      }
    }

    obj.initNull();
    rewind(fp);
//    parsePDFTOPDFComment(fp); // TODO?
//    rewind(fp);
    str = new FileStream(fp,0,gFalse,0,&obj);
    doc = new PDFDoc(str);
  } else {
    GooString *fileName = new GooString(inputfile);
    FILE *fp;

    if ((fp = fopen(inputfile,"rb")) == 0) {
        error(-1,"Can't open input file %s",inputfile);
	exit(1);
    }
//    parsePDFTOPDFComment(fp); // TODO?
    fclose(fp);
    doc = new PDFDoc(fileName,NULL,NULL);
  }

  if (!doc->isOk()) {
    exitCode = 1;
    goto err1;
  }

  char tmp[100];
  tmp[99]=0;
  // ... OutputFD=stdout .. needs to be done before forking
  int outfd;
  outfd=dup(fileno(stdout)); 

  ctx = ijs_invoke_server (ijsserver);
  ijs_client_open (ctx);
  ijs_client_begin_job (ctx,job_id);
  if (outputfile) {
    ijs_client_set_param(ctx,job_id,"OutputFile",outputfile,strlen(outputfile));
  } else {
    snprintf(tmp,99,"%d",outfd);
    ijs_client_set_param(ctx,job_id,"OutputFD",tmp,strlen(tmp));
    close(outfd);
  }
  ijs_client_set_param(ctx,job_id,"DeviceManufacturer",devManu,strlen(devManu));
  ijs_client_set_param(ctx,job_id,"DeviceModel",devModel,strlen(devModel));
  // TODO: get supported output-formats, pagesize from ijs-server, overriding commandline  (?!)

  /* set image's values */
  int numChan,bitsPerSample;
  const char *devName;
  reverseVideo = gFalse;
  switch (colspace) {
  case COL_RGB:
    numChan=3;
    bitsPerSample=8;
    cmode = splashModeRGB8;
    devName = "DeviceRGB";
    rowpad = 3;
    /* set paper color white */
    paperColor[0] = 255;
    paperColor[1] = 255;
    paperColor[2] = 255;
    break;
  case COL_BLACK1:
    reverseVideo = gTrue;
  case COL_WHITE1:
    numChan=1;
    bitsPerSample=1;
    cmode = splashModeMono1;
    devName = "DeviceGray";
    /* set paper color white */
    paperColor[0] = 255;
    rowpad = 1;
    break;
  case COL_BLACK8:
    reverseVideo = gTrue;
  case COL_WHITE8:
    numChan=1;
    bitsPerSample=8;
    cmode = splashModeMono8;
    devName = "DeviceGray";
    /* set paper color white */
    paperColor[0] = 255;
    rowpad = 1;
    break;
#ifdef SPLASH_CMYK
  case COL_CMYK:
    numChan=4;
    bitsPerSample=8;
    cmode = splashModeCMYK8;
    devName = "DevicCMYK";
    /* set paper color white */
    paperColor[0] = 0;
    paperColor[1] = 0;
    paperColor[2] = 0;
    paperColor[3] = 0;
    rowpad = 4;
    break;
#endif
  default:
    error(-1,"Specified ColorSpace is not supported");
    exit(1);
    break;
  }

  out = new SplashOutputDev(cmode,rowpad/* row padding */,
    reverseVideo,paperColor,gTrue,gFalse);
  out->startDoc(doc->getXRef());

  snprintf(tmp,99,"%d",numChan);
  ijs_client_set_param(ctx,job_id,"NumChan",tmp,strlen(tmp));
  snprintf(tmp,99,"%d",bitsPerSample);
  ijs_client_set_param(ctx,job_id,"BitsPerSample",tmp,strlen(tmp));
  ijs_client_set_param(ctx,job_id,"ColorSpace",devName,strlen(devName));
  snprintf(tmp,99,"%dx%d",resolution[0],resolution[1]);
  ijs_client_set_param(ctx,job_id,"Dpi",tmp,strlen(tmp));

  npages = doc->getNumPages();
  for (i = 1;i <= npages;i++) {
    SplashBitmap *bitmap;
    unsigned int size;

    doc->displayPage(out,i,resolution[0],resolution[1],0,gFalse,gFalse,gFalse);
    bitmap = out->getBitmap();

    /* set page parameters */
    snprintf(tmp,99,"%d",bitmap->getWidth());
    ijs_client_set_param(ctx,job_id,"Width",tmp,strlen(tmp));
    snprintf(tmp,99,"%d",bitmap->getHeight());
    ijs_client_set_param(ctx,job_id,"Height",tmp,strlen(tmp));
    ijs_client_begin_page(ctx,job_id);

    /* write page image */
    size = bitmap->getRowSize()*bitmap->getHeight();
    int status=ijs_client_send_data_wait(ctx,job_id,(const char *)bitmap->getDataPtr(),size);
    if (status) {
        error(-1,"Can't write page %d image: %d",i,status);
	exit(1);
    }

    status=ijs_client_end_page(ctx,job_id);
    if (status) {
        error(-1,"Can't finish page %d: %d",i,status);
	exit(1);
    }
  }
  ijs_client_end_job (ctx, job_id);
  ijs_client_close (ctx);

  ijs_client_begin_cmd (ctx, IJS_CMD_EXIT);
  ijs_client_send_cmd_wait (ctx);

  delete out;
err1:
  delete doc;

  // Check for memory leaks
  Object::memCheck(stderr);
  gMemReport(stderr);

  return exitCode;
}

/* replace memory allocation methods for memory check */

void * operator new(size_t size)
{
  return gmalloc(size);
}

void operator delete(void *p)
{
  gfree(p);
}

void * operator new[](size_t size)
{
  return gmalloc(size);
}

void operator delete[](void *p)
{
  gfree(p);
}
