/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.manager;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.List;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import jp.gr.java_conf.dangan.util.lha.LhaFile;
import jp.gr.java_conf.dangan.util.lha.LhaInputStream;
import jp.sf.orangesignal.csv.Csv;
import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.LhaEntryFilter;
import jp.sf.orangesignal.csv.ZipEntryFilter;
import jp.sf.orangesignal.csv.handlers.BeanListHandler;

/**
 * Java プログラム要素のリストと区切り文字形式データの統合入力インタフェースの実装クラスを提供します。
 *
 * @author 杉澤 浩二
 */
public class CsvBeanLoader<T> implements CsvLoader<T> {

	/**
	 * 区切り文字形式情報を保持します。
	 */
	private CsvConfig cfg;

	/**
	 * データアクセスハンドラを保持します。
	 */
	private BeanListHandler<T> handler;

	/**
	 * コンストラクタです。
	 *
	 * @param cfg 区切り文字形式情報
	 * @param beanClass JavaBean の型
	 * @throws IllegalArgumentException パラメータが不正な場合
	 */
	public CsvBeanLoader(final CsvConfig cfg, final Class<T> beanClass) {
		if (cfg == null) {
			throw new IllegalArgumentException("CsvConfig must not be null");
		}
		this.cfg = cfg;
		this.handler = new BeanListHandler<T>(beanClass);
	}

	/**
	 * Java プログラム要素へデータを設定する名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定する名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定しない名前群が存在する場合
	 * @see BeanListHandler#includes(String...)
	 */
	public CsvBeanLoader<T> includes(final String... names) {
		handler.includes(names);
		return this;
	}

	/**
	 * Java プログラム要素へデータを設定しない名前群を設定します。
	 * 
	 * @param names Java プログラム要素へデータを設定しない名前群
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException Java プログラム要素へデータを設定する名前群が存在する場合
	 * @see BeanListHandler#excludes(String...)
	 */
	public CsvBeanLoader<T> excludes(final String... names) {
		handler.excludes(names);
		return this;
	}

	@Override
	public List<T> from(final Reader reader) throws IOException {
		return Csv.load(reader, cfg, handler);
	}

	@Override
	public List<T> from(final InputStream in, final String encoding) throws IOException {
		return Csv.load(in, encoding, cfg, handler);
	}

	@Override
	public List<T> from(final InputStream in) throws IOException {
		return Csv.load(in, cfg, handler);
	}

	@Override
	public List<T> from(final File file, final String encoding) throws IOException {
		return Csv.load(file, encoding, cfg, handler);
	}

	@Override
	public List<T> from(final File file) throws IOException {
		return Csv.load(file, cfg, handler);
	}

	@Override
	public List<T> from(final LhaInputStream in, final LhaEntryFilter filter) throws IOException {
		return Csv.load(in, cfg, handler, filter);
	}

	@Override
	public List<T> from(final LhaInputStream in) throws IOException {
		return Csv.load(in, cfg, handler);
	}

	@Override
	public List<T> from(final LhaFile lhaFile, final LhaEntryFilter filter) throws IOException {
		return Csv.load(lhaFile, cfg, handler, filter);
	}

	@Override
	public List<T> from(final LhaFile lhaFile) throws IOException {
		return Csv.load(lhaFile, cfg, handler);
	}

	@Override
	public List<T> from(final ZipInputStream in, final ZipEntryFilter filter) throws IOException {
		return Csv.load(in, cfg, handler, filter);
	}

	@Override
	public List<T> from(final ZipInputStream in) throws IOException {
		return Csv.load(in, cfg, handler);
	}

	@Override
	public List<T> from(final ZipFile zipFile, final ZipEntryFilter filter) throws IOException {
		return Csv.load(zipFile, cfg, handler, filter);
	}

	@Override
	public List<T> from(final ZipFile zipFile) throws IOException {
		return Csv.load(zipFile, cfg, handler);
	}

}
