/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta;

import java.io.Serializable;
import java.util.Date;

/**
 * 非時系列データの値幅情報を提供します。
 * 
 * @author 杉澤 浩二
 */
public class Step implements Serializable {

	private static final long serialVersionUID = 4179274620367080517L;

	/**
	 * 初日を保持します。
	 */
	public Date openDate;

	/**
	 * 高値日を保持します。
	 */
	public Date highDate;

	/**
	 * 安値日を保持します。
	 */
	public Date lowDate;

	/**
	 * 終日を保持します。
	 */
	public Date closeDate;

	/**
	 * 始値を保持します。
	 */
	public double open;

	/**
	 * 高値を保持します。
	 */
	public double high;

	/**
	 * 安値を保持します。
	 */
	public double low;

	/**
	 * 終値を保持します。
	 */
	public double close;

	/**
	 * 期間を保持します。
	 */
	public int period;

	/**
	 * デフォルトコンストラクタです。
	 */
	protected Step() {}

	/**
	 * コンストラクタです。
	 * 
	 * @param date 日時
	 * @param price 価格
	 */
	public Step(final Date date, final double price) {
		this(date, price, 1);
	}

	/**
	 * コンストラクタです。
	 * 
	 * @param date 日時
	 * @param price 価格
	 * @param period 期間
	 */
	public Step(final Date date, final double price, final int period) {
		this.openDate	= date;
		this.highDate	= date;
		this.lowDate	= date;
		this.closeDate	= date;
		this.open		= price;
		this.high		= price;
		this.low		= price;
		this.close		= price;
		this.period		= period;
	}

	/**
	 * <p>指定された非時系列データから指定された日時を含む最初の位置を返します。</p>
	 * 指定された非時系列データが <code>null</code> の場合は <code>-1</code> を返します。
	 * 
	 * @param array 非時系列データ
	 * @param find 検索する日時
	 * @return 指定された日時を含む最初の位置。見つからない場合は <code>-1</code>
	 * @since 2.2
	 */
	public static int indexOf(final Step[] array, final Date find) {
		return indexOf(array, find, 0);
	}

	/**
	 * <p>指定された非時系列データから指定された日時を含む最初の位置を返します。</p>
	 * 指定された非時系列データが <code>null</code> の場合は <code>-1</code> を返します。
	 * 
	 * @param array 非時系列データ
	 * @param find 検索す日時
	 * @param start 開始位置
	 * @return 指定された日時を含む最初の位置。見つからない場合は <code>-1</code>
	 * @since 2.2
	 */
	public static int indexOf(final Step[] array, final Date find, final int start) {
		if (array == null)
			return -1;

		final int length = array.length;
		final int _start = start >= 0 ? start : 0;
		for (int i = _start; i < length; i++) {
			if (array[i] == null)
				continue;
			if (array[i].openDate.compareTo(find) <= 0 && array[i].closeDate.compareTo(find) >= 0)
				return i;
		}
		return -1;
	}

	/**
	 * <p>指定された非時系列データから指定された日時を含む最後の位置を返します。</p>
	 * 指定された非時系列データが <code>null</code> の場合は <code>-1</code> を返します。
	 * 
	 * @param array 非時系列データ
	 * @param find 検索する日時
	 * @return 指定された日時を含む最後の位置。見つからない場合は <code>-1</code>
	 * @since 2.2
	 */
	public static int lastIndexOf(final Step[] array, final Date find) {
		return lastIndexOf(array, find, Integer.MAX_VALUE);
	}

	/**
	 * <p>指定された非時系列データから指定された日時を含む最後の位置を返します。</p>
	 * 指定された非時系列データが <code>null</code> の場合は <code>-1</code> を返します。
	 * 
	 * @param array 非時系列データ
	 * @param find 検索する日時
	 * @param start 開始位置
	 * @return 指定された日時を含む最後の位置。見つからない場合は <code>-1</code>
	 * @since 2.2
	 */
	public static int lastIndexOf(final Step[] array, final Date find, final int start) {
		if (array == null)
			return -1;

		final int length = array.length;
		final int _start = start >= length ? length - 1 : start;
		for (int i = _start; i >= 0; i--) {
			if (array[i] == null)
				continue;
			if (array[i].openDate.compareTo(find) <= 0 && array[i].closeDate.compareTo(find) >= 0)
				return i;
		}
		return -1;
	}

}
