/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.text.Format;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvListHandler;
import jp.sf.orangesignal.csv.filters.BeanFilter;

/**
 * Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラの基底クラスを提供します。
 * 
 * @author 杉澤 浩二
 */
public abstract class BeanListHandlerSupport<T, H extends BeanListHandlerSupport<T, H>> implements CsvListHandler<T> {

	/**
	 * Java プログラム要素の型を保持します。
	 */
	private Class<T> type;

	/**
	 * Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップを保持します。
	 * 
	 * @since 1.2
	 */
	private Map<String, Format> valueParserMapping = new HashMap<String, Format>();

	/**
	 * 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップを保持します。
	 * 
	 * @since 1.2
	 */
	private Map<Object, Format> valueFormatterMapping = new HashMap<Object, Format>();

	/**
	 * 区切り文字形式データの項目値コンバータを保持します。
	 */
	private CsvValueConverter valueConverter = new SimpleCsvValueConverter();

	/**
	 * Java プログラム要素フィルタを保持します。
	 */
	protected BeanFilter beanFilter;

	/**
	 * 取得データの開始位置を保持します。
	 */
	protected int offset;

	/**
	 * 取得データの限度数を保持します。
	 */
	protected int limit;

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	protected BeanListHandlerSupport(final Class<T> type) {
		if (type == null) {
			throw new IllegalArgumentException("Class must not be null");
		}
		this.type = type;
	}

	// ------------------------------------------------------------------------

	/**
	 * Java プログラム要素の型を返します。
	 * 
	 * @return Java プログラム要素の型
	 */
	public Class<T> getType() { return type; }

	/**
	 * Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップを設定します。
	 * 
	 * @param valueParserMapping Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップ
	 * @throws IllegalArgumentException <code>valueParserMapping</code> が <code>null</code> の場合
	 * @since 1.2.4
	 */
	public void setValueParserMapping(final Map<String, Format> valueParserMapping) {
		if (valueParserMapping == null) {
			throw new IllegalArgumentException("CSV value parser mapping must not be null");
		}
		this.valueParserMapping = valueParserMapping;
	}

	/**
	 * Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップを設定します。
	 * 
	 * @param valueParserMapping Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>valueParserMapping</code> が <code>null</code> の場合
	 * @since 1.2
	 */
	@SuppressWarnings("unchecked")
	public H valueParserMapping(final Map<String, Format> valueParserMapping) {
		setValueParserMapping(valueParserMapping);
		return (H) this;
	}

	/**
	 * 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップを設定します。
	 * 
	 * @param valueFormatterMapping 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップ
	 * @throws IllegalArgumentException <code>valueFormaterMapping</code> が <code>null</code> の場合
	 * @since 1.2.4
	 */
	public void setValueFormatterMapping(final Map<Object, Format> valueFormatterMapping) {
		if (valueFormatterMapping == null) {
			throw new IllegalArgumentException("CSV value formatter mapping must not be null");
		}
		this.valueFormatterMapping = valueFormatterMapping;
	}

	/**
	 * 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップを設定します。
	 * 
	 * @param valueFormatterMapping 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>valueFormaterMapping</code> が <code>null</code> の場合
	 * @since 1.2
	 */
	@SuppressWarnings("unchecked")
	public H valueFormatterMapping(final Map<Object, Format> valueFormatterMapping) {
		setValueFormatterMapping(valueFormatterMapping);
		return (H) this;
	}

	/**
	 * 区切り文字形式データの項目値コンバータを設定します。
	 * 
	 * @param valueConverter 区切り文字形式データの項目値コンバータ
	 * @throws IllegalArgumentException <code>valueConverter</code> が <code>null</code> の場合
	 * @since 1.2.4
	 */
	public void setValueConverter(final CsvValueConverter valueConverter) {
		if (valueConverter == null) {
			throw new IllegalArgumentException("CsvValueConverter must not be null");
		}
		this.valueConverter = valueConverter;
	}

	/**
	 * 区切り文字形式データの項目値コンバータを設定します。
	 * 
	 * @param valueConverter 区切り文字形式データの項目値コンバータ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>valueConverter</code> が <code>null</code> の場合
	 * @since 1.2
	 */
	@SuppressWarnings("unchecked")
	public H valueConverter(final CsvValueConverter valueConverter) {
		setValueConverter(valueConverter);
		return (H) this;
	}

	/**
	 * 区切り文字形式データの項目値コンバータを設定します。
	 * 
	 * @param converter 区切り文字形式データの項目値コンバータ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>converter</code> が <code>null</code> の場合
	 * @deprecated このメソッドの使用を廃止します。1.3 で削除予定。
	 */
	public H converter(final CsvValueConverter converter) {
		return valueConverter(converter);
	}

	/**
	 * 区切り文字形式データの項目値コンバータを返します。
	 * 
	 * @return 区切り文字形式データの項目値コンバータ
	 * @deprecated このメソッドの使用を廃止します。1.3 で削除予定。
	 */
	public CsvValueConverter getConverter() { return valueConverter; }

	/**
	 * Java プログラム要素フィルタを設定します。
	 * 
	 * @param beanFilter Java プログラム要素フィルタ
	 * @return このオブジェクトへの参照
	 * @since 1.2.3
	 */
	@SuppressWarnings("unchecked")
	public H filter(final BeanFilter beanFilter) {
		this.beanFilter = beanFilter;
		return (H) this;
	}

	/**
	 * 取得データの開始位置を設定します。
	 * 
	 * @param offset 取得データの開始位置
	 * @since 1.2.4
	 */
	public void setOffset(final int offset) {
		this.offset = offset;
	}

	/**
	 * 取得データの開始位置を設定します。
	 * 
	 * @param offset 取得データの開始位置
	 * @return このオブジェクトへの参照
	 * @since 1.2.1
	 */
	@SuppressWarnings("unchecked")
	public H offset(final int offset) {
		setOffset(offset);
		return (H) this;
	}

	/**
	 * 取得データの限度数を設定します。
	 * 
	 * @param limit 取得データの限度数
	 * @since 1.2.4
	 */
	public void setLimit(final int limit) {
		this.limit = limit;
	}

	/**
	 * 取得データの限度数を設定します。
	 * 
	 * @param limit 取得データの限度数
	 * @return このオブジェクトへの参照
	 * @since 1.2.1
	 */
	@SuppressWarnings("unchecked")
	public H limit(final int limit) {
		setLimit(limit);
		return (H) this;
	}

	// ------------------------------------------------------------------------

	/**
	 * <p>指定された Java プログラム要素のフィールド名と項目値を解析するオブジェクトをマップへ追加します。</p>
	 * <p>
	 * 指定されたフィールド名に既に項目値を解析するオブジェクトが設定されている場合、
	 * 既存の項目値解析オブジェクトへパラメータで指定された項目値解析オブジェクトのパターン文字列を追加します。
	 * </p>
	 * 
	 * @param field Java プログラム要素のフィールド名
	 * @param parser 項目値を解析するオブジェクト
	 * @since 1.2
	 */
	protected void setValueParser(final String field, final Format parser) {
		final Format _parser = valueParserMapping.get(field);
		if (_parser != null) {
			valueParserMapping.put(field, mergeFormatPattern(_parser, parser));
		} else {
			valueParserMapping.put(field, parser);
		}
	}

	/**
	 * 指定された項目名 (または項目位置) と項目値へ書式化するオブジェクトをマップへ追加します。
	 * 
	 * @param column 項目名 (または項目位置)
	 * @param formatter 項目値へ書式化するオブジェクト
	 * @since 1.2
	 */
	protected void setValueFormatter(final Object column, final Format formatter) {
		valueFormatterMapping.put(column, formatter);
	}

	/**
	 * Java プログラム要素の型が表すクラスの新しいインスタンスを生成します。
	 * 
	 * @return Java プログラム要素の型が表す、クラスの新しく割り当てられたインスタンス
	 * @throws IOException Java プログラム要素のインスタンス化に失敗した場合
	 */
	protected T createBean() throws IOException {
		try {
			return type.newInstance();
		} catch (IllegalAccessException e) {
			throw new IOException("Cannot create " + type.getName() + ": " + e.getMessage(), e);
		} catch (InstantiationException e) {
			throw new IOException("Cannot create " + type.getName() + ": " + e.getMessage(), e);
		}
	}

	/**
	 * 指定された項目名 (または項目位置) と Java プログラム要素のフィールド名のマップと Java プログラム要素の型から、
	 * Java プログラム要素のフィールド名と項目名群のマップを構築して返します。
	 * 
	 * @param map 項目名 (または項目位置) と Java プログラム要素のフィールド名のマップ
	 * @return Java プログラム要素のフィールド名と項目名群のマップ
	 * @since 1.2
	 */
	protected Map<String, Object[]> createFieldAndColumnsMap(final Map<?, String> map) {
		final Map<String, Object[]> results = new HashMap<String, Object[]>();
		for (final Field f : type.getDeclaredFields()) {
			final String fieldName = f.getName();
			final List<Object> list = new ArrayList<Object>();
			for (final Map.Entry<?, String> e : map.entrySet()) {
				if (fieldName.equals(e.getValue())) {
					list.add(e.getKey());
				}
			}
			if (list.size() > 0) {
				results.put(fieldName, list.toArray());
			}
		}
		
		return results;
	}

	/**
	 * 指定された項目値を指定されたフィールドのオブジェクトへ変換して返します。
	 * この実装は、指定されたフィールドに対応する項目値を解析するオブジェクトが存在する場合は、{@link Format#parseObject(String)} で得られたオブジェクトを返します。
	 * それ以外の場合は、項目値コンバータを使用して得られたオブジェクトを返します。
	 * 
	 * @param field フィールド
	 * @param value 項目値
	 * @return 変換された項目値
	 * @since 1.2
	 */
	protected Object stringToObject(final Field field, final String value) {
		final Format format = valueParserMapping.get(field.getName());
		if (format != null) {
			if (value == null || value.isEmpty()) {
				return null;
			}
			try {
				return format.parseObject(value);
			} catch (ParseException e) {
				throw new IllegalArgumentException(String.format("Unable to parse the %s: %s", field.getName(), value), e);
			}
		}
		return valueConverter.convert(value, field.getType());
	}

	/**
	 * 指定されたオブジェクトを項目値へ変換して返します。
	 * この実装は、指定された項目に対応する項目値へ書式化するオブジェクトが存在する場合は、{@link Format#format(Object)} で得られた文字列を返します。
	 * それ以外の場合は、項目値コンバータを使用して得られた文字列を返します。
	 * 
	 * @param column 項目名 (または項目位置)
	 * @param obj オブジェクト
	 * @return 文字列の項目値
	 * @since 1.2
	 */
	protected String objectToString(final Object column, final Object obj) {
		final Format format = valueFormatterMapping.get(column);
		if (format != null) {
			if (obj == null) {
				return null;
			}
			return format.format(obj);
		}
		return valueConverter.convert(obj);
	}

	// ------------------------------------------------------------------------
	// フィールド操作用静的メソッド群

	/**
	 * 指定された Java プログラム要素の型が表すクラスの指定された宣言フィールドをリフレクトする {@link Field} オブジェクトを返します。
	 * 
	 * @param type Java プログラム要素の型
	 * @param name フィールド名
	 * @return 指定された Java プログラム要素の {@link Field} オブジェクト
	 * @throws IOException 指定された名前のフィールドが見つからない場合
	 * @throws NullPointerException <code>name</code> が <code>null</code> の場合
	 * @throws SecurityException 
	 */
	public static Field getField(final Class<?> type, final String name) throws IOException {
		try {
			return type.getDeclaredField(name);
		} catch (NoSuchFieldException e) {
			throw new IOException("Field " + name + " not found in " + type.getName() + ": " + e.getMessage(), e);
		}
	}

	/**
	 * 指定された Java プログラム要素の指定されたフィールドを、指定された新しい値に設定します。
	 * 基本となるフィールドにプリミティブ型が指定されている場合、新しい値は自動的にラップ解除されます。
	 * 
	 * @param bean フィールドを変更する Java プログラム要素
	 * @param field フィールド
	 * @param value 変更中の Java プログラム要素の新しいフィールド値
	 * @throws IOException 基本となるフィールドにアクセスできない場合。または指定されたオブジェクトが基本となるフィールド (またはそのサブクラスか実装側) を宣言するクラスまたはインタフェースのインスタンスではない場合、あるいはラップ解除変換が失敗した場合
	 * @throws NullPointerException 指定されたオブジェクトが null で、フィールドがインスタンスフィールドの場合
	 * @throws SecurityException 
	 */
	public static void setFieldValue(final Object bean, final Field field, final Object value) throws IOException {
		if (!field.isAccessible()) {
			field.setAccessible(true);
		}
		try {
			field.set(bean, value);
		} catch (IllegalAccessException e) {
			throw new IOException("Cannot set " + field.getName() + ": " + e.getMessage(), e);
		} catch (IllegalArgumentException e) {
			throw new IOException("Cannot set " + field.getName() + ": " + e.getMessage(), e);
		}
	}

	/**
	 * 指定された Java プログラム要素について、指定された {@link Field} によって表されるフィールドの値を返します。
	 * プリミティブ型の場合、オブジェクト内に自動的に格納されてから返されます。
	 * 
	 * @param bean Java プログラム要素
	 * @param field フィールド
	 * @return Java プログラム要素 <code>bean</code> 内で表現される値。プリミティブ値は適切なオブジェクト内にラップされてから返される
	 * @throws IOException 基本となるフィールドにアクセスできない場合。指定されたオブジェクトが基本となるフィールド (またはそのサブクラスか実装側) を宣言するクラスまたはインタフェースのインスタンスではない場合
	 * @throws NullPointerException 指定されたオブジェクトが null で、フィールドがインスタンスフィールドの場合
	 * @throws SecurityException 
	 */
	public static Object getFieldValue(final Object bean, final Field field) throws IOException {
		if (!field.isAccessible()) {
			field.setAccessible(true);
		}
		try {
			return field.get(bean);
		} catch (IllegalAccessException e) {
			throw new IOException("Cannot get " + field.getName() + ": " + e.getMessage(), e);
		} catch (IllegalArgumentException e) {
			throw new IOException("Cannot get " + field.getName() + ": " + e.getMessage(), e);
		}
	}

	// ------------------------------------------------------------------------
	// 書式文字列操作用静的メソッド群

	private static Format mergeFormatPattern(final Format format, final Format... formats) {
		try {
			final StringBuilder buf = new StringBuilder();
			buf.append(getFormatPattern(format));
			for (final Format fmt : formats) {
				buf.append(getFormatPattern(fmt));
			}
			
			final Format result = (Format) format.clone();
			result.getClass().getMethod("applyPattern", String.class).invoke(result, buf.toString());
			return result;
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	private static String getFormatPattern(final Format format) {
		try {
			return (String) format.getClass().getMethod("toPattern").invoke(format);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

}
