/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.blog.bean;

import java.text.DateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.faces.component.UIViewRoot;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.model.SelectItem;

import jp.sf.pal.blog.BlogConstants;
import jp.sf.pal.blog.converter.Converter;
import jp.sf.pal.blog.importer.Importer;
import jp.sf.pal.blog.model.BlogMessage;
import jp.sf.pal.blog.util.BlogDaoUtil;
import jp.sf.pal.blog.util.BlogMessaging;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;
import org.seasar.portlet.util.PortletResourceBundleUtil;

import com.marevol.utils.faces.application.FacesMessageUtil;
import com.marevol.utils.faces.util.LocaleUtil;

public class BlogEditPageBean
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(BlogEditPageBean.class);

    private Long id;

    private String title;

    private String message;

    private String formattype;

    private Date createdTime;

    private Date updatedTime;

    public BlogEditPageBean()
    {
        Long id = null;
        try
        {
            id = (Long) BlogMessaging
                    .receiveForEdit(BlogConstants.BLOG_MESSAGE_ID);
        }
        catch (ClassCastException e)
        {
            log.warn(e);
        }

        if (id != null)
        {
            if (log.isDebugEnabled())
            {
                log.debug("BlogEditPageBean() -  : id=" + id.toString());
            }

            setId(id);

            BlogMessage blogMsg = BlogDaoUtil.getBlogMessageDao()
                    .getBlogMessageById(id.longValue());
            if (blogMsg != null)
            {
                setTitle(blogMsg.getTitle());
                setMessage(blogMsg.getMessage());
                setFormattype(blogMsg.getFormattype());
                setCreatedTime(blogMsg.getCreatedtime());
                setUpdatedTime(blogMsg.getUpdatedtime());
            }
            else
            {
                if (log.isDebugEnabled())
                {
                    // i18n
                    Object[] params = new Object[] { id.toString() };
                    log.debug(PortletResourceBundleUtil.getString(LocaleUtil
                            .getLocale(), "blogedit.IDIsNotFound", params));
                }
                setId(null);
            }
        }
        else
        {
            if (log.isDebugEnabled())
            {
                log.debug("BlogEditPageBean() - id = null");
            }
        }
    }

    public String save()
    {
        if (log.isDebugEnabled())
        {
            log.debug("save() - start");
        }

        ExternalContext externalContext = FacesContext.getCurrentInstance()
                .getExternalContext();

        if (externalContext.getRemoteUser() == null)
        {
            // i18n
            FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(), "blogedit.UnknownUser"));

            if (log.isDebugEnabled())
            {
                log.debug("save() - end");
            }
            return null;
        }

        if (log.isDebugEnabled())
        {
            log.debug("save() - getId()=" + getId());
        }

        if (getId() == null)
        {
            // create a new blog message
            try
            {
                BlogMessage blogMsg = new BlogMessage();
                blogMsg.setOwner(externalContext.getRemoteUser());
                Date now = Calendar.getInstance().getTime();
                blogMsg.setCreatedtime(now);
                blogMsg.setUpdatedtime(now);
                blogMsg.setFormattype(getFormattype());
                blogMsg.setTitle(getTitle());
                blogMsg.setMessage(getMessage());

                BlogDaoUtil.getBlogMessageDao().save(blogMsg);

                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(), "blogedit.SaveMsg"));
            }
            catch (RuntimeException e)
            {
                log.error(e);
                // i18n
                FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogedit.DatabaseAccessError"), e.toString());
            }

        }
        else
        {
            // update the blog message
            try
            {
                BlogMessage blogMsg = BlogDaoUtil.getBlogMessageDao()
                        .getBlogMessageById(id.longValue());
                blogMsg.setOwner(externalContext.getRemoteUser());
                Date now = Calendar.getInstance().getTime();
                // blogMsg.setCreatedtime(now);
                blogMsg.setUpdatedtime(now);
                blogMsg.setFormattype(getFormattype());
                blogMsg.setTitle(getTitle());
                blogMsg.setMessage(getMessage());

                BlogDaoUtil.getBlogMessageDao().saveOrUpdate(blogMsg);

                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogedit.UpdatedMsg"));
            }
            catch (HibernateException e)
            {
                log.error(e);
                // i18n
                FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogedit.DatabaseAccessError"), e.toString());
            }

        }

        if (log.isDebugEnabled())
        {
            log.debug("save() - end");
        }
        return null;
    }

    public String reset()
    {
        if (getId() != null)
        {
            BlogMessaging.cancelForEdit(BlogConstants.BLOG_MESSAGE_ID);
            // i18n
            FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(),
                            "blogedit.ReadyToCreateANewData"));
        }
        return null;
    }

    public String delete()
    {
        if (getId() != null)
        {
            try
            {
                BlogMessage blogMsg = BlogDaoUtil.getBlogMessageDao()
                        .getBlogMessageById(id.longValue());

                BlogDaoUtil.getBlogMessageDao().delete(blogMsg);

                setId(null);

                // i18n
                FacesMessageUtil.addInfoMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogedit.DeletedMsg"));
            }
            catch (HibernateException e)
            {
                log.error(e);
                // i18n
                FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                        .getString(LocaleUtil.getLocale(),
                                "blogedit.DatabaseAccessError"), e.toString());
            }

        }
        else
        {
            // i18n
            FacesMessageUtil.addErrorMessage(PortletResourceBundleUtil
                    .getString(LocaleUtil.getLocale(),
                            "blogedit.TheSpecifiedDataIDIsNull"));
        }
        return null;
    }

    // Setter and Getter

    /**
     * @return Returns the message.
     */
    public String getMessage()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getMessage() - start");
        }

        if (log.isDebugEnabled())
        {
            log.debug("getMessage() - end");
        }
        return message;
    }

    /**
     * @param message The message to set.
     */
    public void setMessage(String message)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setMessage(String) - start");
        }

        this.message = message;

        if (log.isDebugEnabled())
        {
            log.debug("setMessage(String) - end");
        }
    }

    /**
     * @return Returns the title.
     */
    public String getTitle()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getTitle() - start");
        }

        if (log.isDebugEnabled())
        {
            log.debug("getTitle() - end");
        }
        return title;
    }

    /**
     * @param title The title to set.
     */
    public void setTitle(String title)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setTitle(String) - start");
        }

        this.title = title;

        if (log.isDebugEnabled())
        {
            log.debug("setTitle(String) - end");
        }
    }

    /**
     * @return Returns the id.
     */
    public Long getId()
    {
        if (log.isDebugEnabled())
        {
            log.debug("getId() - start");
        }

        if (log.isDebugEnabled())
        {
            log.debug("getId() - end");
        }
        return id;
    }

    /**
     * @param id The id to set.
     */
    public void setId(Long id)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setId(Long) - start");
        }

        this.id = id;

        if (log.isDebugEnabled())
        {
            log.debug("setId(Long) - end");
        }
    }

    /**
     * @return Returns the createdTime.
     */
    public Date getCreatedTime()
    {
        return createdTime;
    }

    public String getFormattedCreatedTime()
    {
        if (getCreatedTime() == null)
        {
            return BlogConstants.NONEXISTENT_VALUE;
        }

        UIViewRoot viewRoot = FacesContext.getCurrentInstance().getViewRoot();
        Locale locale = null;
        if (viewRoot != null)
        {
            locale = viewRoot.getLocale();
        }
        else
        {
            locale = Locale.ENGLISH;
        }

        DateFormat df = DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
                DateFormat.MEDIUM, locale);
        return df.format(getCreatedTime());
    }

    /**
     * @param createdTime The createdTime to set.
     */
    public void setCreatedTime(Date createdDate)
    {
        this.createdTime = createdDate;
    }

    /**
     * @return Returns the updatedTime.
     */
    public Date getUpdatedTime()
    {
        return updatedTime;
    }

    public String getFormattedUpdatedTime()
    {
        if (getUpdatedTime() == null)
        {
            return BlogConstants.NONEXISTENT_VALUE;
        }

        UIViewRoot viewRoot = FacesContext.getCurrentInstance().getViewRoot();
        Locale locale = null;
        if (viewRoot != null)
        {
            locale = viewRoot.getLocale();
        }
        else
        {
            locale = Locale.ENGLISH;
        }

        DateFormat df = DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
                DateFormat.MEDIUM, locale);
        return df.format(getUpdatedTime());
    }

    /**
     * @param updatedTime The updatedTime to set.
     */
    public void setUpdatedTime(Date updatedDate)
    {
        this.updatedTime = updatedDate;
    }

    /**
     * @return Returns the formattype.
     */
    public String getFormattype()
    {
        if (formattype == null || formattype.equals(""))
        {
            S2Container container = SingletonS2ContainerFactory.getContainer();

            List list = (List) container
                    .getComponent(BlogConstants.CONVERTER_LIST);
            Iterator ite = list.iterator();
            if (ite.hasNext())
            {
                Object obj = ite.next();
                if (obj instanceof Converter)
                {
                    return ((Converter) obj).getFormatType();
                }
            }
            throw new IllegalArgumentException("Converter does not exist.");
        }
        return formattype;
    }

    /**
     * @param formattype The formattype to set.
     */
    public void setFormattype(String formattype)
    {
        this.formattype = formattype;
    }

    public SelectItem[] getFormattypeItems()
    {
        S2Container container = SingletonS2ContainerFactory.getContainer();

        List list = (List) container.getComponent(BlogConstants.CONVERTER_LIST);
        Iterator ite = list.iterator();
        SelectItem[] items = new SelectItem[list.size()];
        Locale locale = FacesContext.getCurrentInstance().getViewRoot()
                .getLocale();
        int i = 0;
        while (ite.hasNext())
        {
            Object obj = ite.next();
            if (obj instanceof Converter)
            {
                Converter conveter = (Converter) obj;
                Map localizedNames = conveter.getLocalizedName();
                Iterator keys = localizedNames.keySet().iterator();
                while (keys.hasNext())
                {
                    String localeString = (String) keys.next();
                    if (com.marevol.utils.util.LocaleUtil.fallback(locale,
                            com.marevol.utils.util.LocaleUtil
                                    .convert(localeString)) != null)
                    {
                        items[i] = new SelectItem(conveter.getFormatType(),
                                (String) localizedNames.get(localeString));
                        i++;
                        break;
                    }
                }
            }
            else
            {
                throw new IllegalArgumentException("Invalid conveter: " + obj);
            }
        }
        //TODO i18n
        return items;

    }
}
