/*
 * Copyright 2004-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.tomahawk.resourcehandler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;

import javax.portlet.PortletContext;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

/**
 * This class has basic methods for ResourceHandler.
 * 
 * @author <a href="mailto:shinsuke@yahoo.co.jp">Shinsuke Sugaya</a>
 */
public class DefaultResourceHandler implements ResourceHandler
{
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(DefaultResourceHandler.class);

    private PortletContext portletContext;

    protected String resourcesInHead;

    protected String resourcesBeforeContents;

    protected String resourcesAfterContents;

    public DefaultResourceHandler()
    {
        resourcesInHead = "";
        resourcesBeforeContents = "";
        resourcesAfterContents = "";
    }

    public void init()
    {

    }

    /**
     * @return Returns the portletContext.
     */
    public PortletContext getPortletContext()
    {
        return portletContext;
    }

    /**
     * @param portletContext The portletContext to set.
     */
    public void setPortletContext(PortletContext portletContext)
    {
        this.portletContext = portletContext;
    }

    public void parse(String htmlContents)
    {
        // in <head>
        resourcesInHead = getStringInTag(htmlContents, "<head>");
        if (log.isDebugEnabled())
        {
            log.debug("parse(String) - resourcesInHead=" + resourcesInHead);
        }
        if (resourcesInHead == null)
        {
            resourcesInHead = "";
        }

        // in <body>
        String body = getStringInTag(htmlContents, "<body>");
        if (log.isDebugEnabled())
        {
            log.debug("parse(String) - body=" + body);
        }
        if (body != null)
        {
            int startIndex = body.indexOf(ResourceHandler.BODY_CONTENTS);
            if (startIndex > 0)
            {
                resourcesBeforeContents = body.substring(0, startIndex);
                resourcesAfterContents = body.substring(startIndex
                        + ResourceHandler.BODY_CONTENTS.length());
            }
            else
            {
                resourcesAfterContents = body
                        .substring(ResourceHandler.BODY_CONTENTS.length());
            }
        }

        if (log.isDebugEnabled())
        {
            log.debug("parse(String) - resourcesBeforeContents="
                    + resourcesBeforeContents);
            log.debug("parse(String) - resourcesAfterContents="
                    + resourcesAfterContents);
        }
    }

    protected String getStringInTag(String htmlContents, String startTag)
    {
        int startIndex = htmlContents.indexOf(startTag);
        if (startIndex != -1)
        {
            startIndex += startTag.length();
            String endTag = "</" + startTag.substring(1);
            int lastIndex = htmlContents.indexOf(endTag);
            if (lastIndex != -1)
            {
                return htmlContents.substring(startIndex, lastIndex);
            }
        }
        return null;
    }

    public void addResourcesToHead(RenderRequest request,
            RenderResponse response)
    {
        try
        {
            response.getWriter().write(resourcesInHead);
            response.getWriter().flush();
        }
        catch (IOException e)
        {
            log.error("cannot render resources before contents.", e);
        }
    }

    public void renderResourcesBeforeContents(RenderRequest request,
            RenderResponse response)
    {
        try
        {
            response.getWriter().write(resourcesBeforeContents);
            response.getWriter().flush();
        }
        catch (IOException e)
        {
            log.error("cannot render resources before contents.", e);
        }
    }

    public void renderResourcesAfterContents(RenderRequest request,
            RenderResponse response)
    {
        try
        {
            response.getWriter().write(resourcesAfterContents);
            response.getWriter().flush();
        }
        catch (IOException e)
        {
            log.error("cannot render resources after contents.", e);
        }
    }

}
