/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Get Weather Info from Livedoor Weather Web Service
 */
package jp.sf.pal.weather ;

import java.util.* ;
import java.io.IOException;
import org.xml.sax.*;
import javax.xml.parsers.*;
import org.w3c.dom.* ;

public class WeatherWebService {
	
	public static final String WEATHER_URI = 
		"http://weather.livedoor.com/forecast/webservice/rest/v1?" ;
	public static final String TODAY = "today" ;
	public static final String TOMORROW = "tomorrow" ;
	public static final String DAYAFTERTOMORROW ="dayaftertomorrow" ;
	
	public static final String FORCAST_MAP_URI =
		"http://weather.livedoor.com/forecast/rss/forecastmap.xml" ;
	
	public static final String NODE_TITLE = "title" ;
	public static final String NODE_TELOP = "telop" ;
	public static final String NODE_LOCATION = "location" ;
	public static final String ATTR_AREA = "area" ;
	public static final String ATTR_PREF = "pref" ;
	public static final String ATTR_CITY = "city" ;
	public static final String NODE_LINK = "link" ;
	public static final String NODE_PUBLICTIME = "publictime" ;
	public static final String NODE_IMAGE = "image" ;
	public static final String NODE_IMAGE_TITLE = "title" ;
	public static final String NODE_IMAGE_LINK = "link" ;
	public static final String NODE_IMAGE_URL = "url" ;
	public static final String NODE_IMAGE_WIDTH = "width" ;
	public static final String NODE_IMAGE_HEIGHT = "height" ;
	public static final String NODE_DESCRIPTION = "description" ;
	public static final String NODE_COPYRIGHT = "copyright" ;
	public static final String NODE_COPYRIGHT_TITLE = "title" ;
	public static final String NODE_COPYRIGHT_LINK = "link" ;
    public static final String NODE_TEMPERATURE = "temperature";
    public static final String NODE_TEMP_MAX = "max";
    public static final String NODE_TEMP_MIN = "min";
    public static final String NODE_TEMP_CELSIUS = "celsius";
	
	public static final String NODE_CITY = "city" ;
	public static final String ATTR_CITY_TITLE = "title" ;
	public static final String ATTR_CITY_ID = "id" ;
	
	/** 天気 */
	private		Weather		weather ;
	
	/** Child Node of <lwws> */ 
	private		Map		lwwsChildMap ;
	
	/** 都市一覧 */
	private		Map		cityMap ;
    
    /** 都市 */
    private     String  city ;
    
    /** 日 */
    private     String  day ;
    
    public WeatherWebService() {
        this.day = TODAY ;
        this.cityMap = new TreeMap() ;
        this.weather = new Weather() ;
    }
    
    /**
     * @param city 都市ID
     */
    public WeatherWebService(String city) {
        this();
        this.city = city ;
    }
	
	/**
	 * @param city	都市ID	
	 * @param day	ͽ予報日
	 */
	public WeatherWebService(String city, String day) {
        this(city);
        this.day = day ;
	}
    
    /**
     * 天気の取得
     * @return 天気
     */
    public Weather getWeather() {
        String  uri = WEATHER_URI + "city=" + this.city + "&day=" + this.day ;
        this.getWeatherInfo(uri) ;
        return this.weather ;
    }
    
    /**
     * 都市一覧の Map の取得
     * @return 都市Map
     */
    public Map getCityMap() {
        this.getCityInfo() ;        
        return this.cityMap ;
    }
    
    /**
     * 天気を取得する日のセット
     * @param day 日(TODAY, TOMORROW, DATAFTERTOMORROW)
     */
    public void setDay(String day) {
        this.day = day;
    }

    /**
     * 取得する都市のセット
     * @param city 都市ID
     */
    public void setCity(String city) {
        this.city = city;
    }
	
	/**
	 * 天気情報をウェブサービスから取得
	 */
	private void getWeatherInfo(String uri) {
		
		try {
			
			DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance() ;
			DocumentBuilder builder = dbFactory.newDocumentBuilder() ;
			
			// Get Document
			Document	weather_doc = builder.parse(uri) ;
			
			// Get <lwws> Node
			Node lwws = weather_doc.getFirstChild() ;
			
			// Get child node of <lwws>
			lwwsChildMap = this.getElement(lwws) ;
			
			// Weather に取得した情報をセット
			this.setWeather() ;
			
		} catch (FactoryConfigurationError fce) {
			fce.printStackTrace() ;
		} catch (ParserConfigurationException pce) {
			pce.printStackTrace() ;
		} catch (IOException ioe) {
			ioe.printStackTrace() ;
		} catch (SAXException saxe) {
			saxe.printStackTrace() ;
		}
	}

	/**
	 * 都市名と id の一覧を得る
	 */
	private void getCityInfo() {
		
		try {
			
			DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance() ;
			DocumentBuilder builder = dbFactory.newDocumentBuilder() ;
			
			// Get Document
			Document	city_doc = builder.parse(FORCAST_MAP_URI) ;
			
			// Get rss Element
			Element rss = city_doc.getDocumentElement() ;
			
			// Get Channel Element (channel は一つという前提)
			Element channel = (Element)rss.getElementsByTagName("channel").item(0) ; 

			// Get "ldWeather.source" Element
			Element sourceNode = (Element)channel.getElementsByTagName("ldWeather:source").item(0) ;
			
			this.setCityMap(sourceNode) ;
			
		} catch (FactoryConfigurationError fce) {
			fce.printStackTrace() ;
		} catch (ParserConfigurationException pce) {
			pce.printStackTrace() ;
		} catch (IOException ioe) {
			ioe.printStackTrace() ;
		} catch (SAXException saxe) {
			saxe.printStackTrace() ;
		}		

	}

    /**
	 * 都市一覧の Map に値をセット
	 * @param node
	 */
	private void setCityMap(Element node) {
		NodeList nodeList = node.getChildNodes() ;
		String id = null ;    // 都市の id
		String title = null ; // 都市名 
		
        // XML の階層構造をたどって，都市情報を取得し，Map に設定する
		if (nodeList.getLength() > 0) {
			for (int i=0; i<nodeList.getLength(); i++) {
				Node childNode = nodeList.item(i) ;
				if (childNode.getNodeType() == Node.ELEMENT_NODE) {
					this.setCityMap((Element)childNode) ;
				}
			}
		} else {
			String nodeName = node.getNodeName() ;
			if (nodeName.equals(NODE_CITY)) {
				id = node.getAttribute(ATTR_CITY_ID) ;
				title = node.getAttribute(ATTR_CITY_TITLE) ;
				this.cityMap.put(Integer.parseInt(id), title) ;
			}
		}

	}

	/**
	 * <lwws> の子ノードの値を Weather にセット
	 */
	private void setWeather() {
		
		// タイトル
		Node titleNode = (Node)lwwsChildMap.get(NODE_TITLE) ;
		String title = titleNode.getFirstChild().getNodeValue() ;
		weather.setTitle(title) ;
		
		// テロップ
		Node telopNode = (Node)lwwsChildMap.get(NODE_TELOP) ;
		String telop = telopNode.getFirstChild().getNodeValue() ;
		weather.setTelop(telop) ;
		
		// Link
		Node linkNode = (Node)lwwsChildMap.get(NODE_LINK) ;
		String link = linkNode.getFirstChild().getNodeValue() ;
		weather.setLink(link) ;
		
		// 発表日時
		Node publictimeNode = (Node)lwwsChildMap.get(NODE_PUBLICTIME) ;
		String publictime = publictimeNode.getFirstChild().getNodeValue() ;
		weather.setPublictime(publictime) ;
		
		// Location
		this.getLocationInfo() ;
		
		// Image
		this.getImageInfo() ;
		
		// Copyright
		this.getCopyrightInfo() ;
        
        // Temperature
        this.getTemperatureInfo();
		
		// 天気概況
		Node descriptionNode = (Node)lwwsChildMap.get(NODE_DESCRIPTION) ;
		String description = descriptionNode.getFirstChild().getNodeValue() ;
		weather.setDescription(description) ;
		
	}
	
	/**
	 * Location 情報の取得
	 */
	private void getLocationInfo() {
		
		// Location
		Node locationNode = (Node)lwwsChildMap.get(NODE_LOCATION);
		NamedNodeMap locationAttr = locationNode.getAttributes() ;
		String area = locationAttr.getNamedItem(ATTR_AREA).getNodeValue() ;
		String pref = locationAttr.getNamedItem(ATTR_PREF).getNodeValue() ;
		String city = locationAttr.getNamedItem(ATTR_CITY).getNodeValue() ;
		weather.setArea(area) ;
		weather.setPref(pref) ;
		weather.setCity(city) ;
	}
	
	/**
	 * 画像情報の取得
	 */
	private void getImageInfo() {
		
		Node imageNode = (Node)lwwsChildMap.get(NODE_IMAGE) ;
		Map imageMap = this.getElement(imageNode) ;
		
		Node titleNode = (Node)imageMap.get(NODE_IMAGE_TITLE) ;
		Node linkNode = (Node)imageMap.get(NODE_IMAGE_LINK) ;
		Node urlNode = (Node)imageMap.get(NODE_IMAGE_URL) ;
		Node widthNode = (Node)imageMap.get(NODE_IMAGE_WIDTH) ;
		Node heightNode = (Node)imageMap.get(NODE_IMAGE_HEIGHT) ;
		
		String title = titleNode.getFirstChild().getNodeValue() ;
		String link = linkNode.getFirstChild().getNodeValue() ;
		String url = urlNode.getFirstChild().getNodeValue() ;
		String width = widthNode.getFirstChild().getNodeValue() ;
		String height = heightNode.getFirstChild().getNodeValue() ;
		
		weather.setImgTitle(title) ;
		weather.setImgLink(link) ;
		weather.setImgUrl(url) ;
		weather.setImgHeight(height) ;
		weather.setImgWidth(width) ;
	}
	
	/**
	 * Copyright 情報の取得
	 */
	private void getCopyrightInfo() {
		
		Node copyrightNode = (Node)lwwsChildMap.get(NODE_COPYRIGHT) ;
		Map copyrightMap = this.getElement(copyrightNode) ;
		
		Node titleNode = (Node)copyrightMap.get(NODE_COPYRIGHT_TITLE) ;
		Node linkNode = (Node)copyrightMap.get(NODE_COPYRIGHT_LINK) ;

		String title = titleNode.getFirstChild().getNodeValue() ;
		String link = linkNode.getFirstChild().getNodeValue() ;
		
		weather.setCopyrightTitle(title) ;
		weather.setCopyrightLink(link) ;
	}
    
    private void getTemperatureInfo() {
        Node tempNode = (Node)lwwsChildMap.get(NODE_TEMPERATURE);
        Map tempMap = this.getElement(tempNode);
        
        // 最高気温
        Map maxMap = this.getElement((Node)tempMap.get(NODE_TEMP_MAX));
        Node maxCNode = (Node)maxMap.get(NODE_TEMP_CELSIUS);
        if (maxCNode.getFirstChild() != null) {
            weather.setTempMax(maxCNode.getFirstChild().getNodeValue());
        }
        
        // 最低気温
        Map minMap = this.getElement((Node)tempMap.get(NODE_TEMP_MIN));
        Node minCNode = (Node)minMap.get(NODE_TEMP_CELSIUS);
        if (minCNode.getFirstChild() != null) {
            weather.setTempMin(minCNode.getFirstChild().getNodeValue());
        }
    }
	
	/**
	 * Ϳ与えられたノードの子ノードのエレメントを得る
	 * @param node　取得したいノード
	 * @return ノード名をキーに、ノードを値に入れた Map
	 */
	private Map getElement(Node node) {
		
		Map		elementMap = new HashMap() ;
		
		NodeList childNodeList = node.getChildNodes() ;
			
		for (int i=0; i<childNodeList.getLength(); i++) {
			Node childNode = childNodeList.item(i) ;
				
			// 現在のノードが Element の時だけ Map に放りこむ
			if (childNode.getNodeType() == Node.ELEMENT_NODE) {
				String nodeName = childNode.getNodeName() ;
				elementMap.put(nodeName, childNode) ;
			}
		}
			
		return	elementMap ;
	}


}
