/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.vfs.web.navigator;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import jp.sf.pal.vfs.VFSConstants;
import jp.sf.pal.vfs.util.VFSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;
import org.apache.commons.vfs.FileSystemManager;
import org.apache.myfaces.custom.fileupload.UploadedFile;
import org.seasar.framework.container.S2Container;

import com.marevol.utils.faces.application.FacesMessageUtil;

public class UploadFilePage extends DefaultPage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(UploadFilePage.class);

    private static final int BLOCK_SIZE = 4096;

    private UploadedFile uploadedFile;

    public UploadFilePage(S2Container container) {
        super(container);
    }

    public String upload() {
        if (!VFSUtil.checkPath(getUri())) {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("access denied.");
            setUri(VFSUtil.getDefaultUri());
            return VFSConstants.DISPLAY_FOLDER_LIST_PAGE;
        }

        if (log.isDebugEnabled()) {
            log.debug("upload() - start");
            log.debug("upload() - upload: getUploadedFile()="
                    + getUploadedFile());
            log.debug("upload() - upload: filename="
                    + getUploadedFile().getName());
        }
        try {
            // For Windows
            String filename = getUploadedFile().getName();
            int index = filename.lastIndexOf("\\");
            if (index > 0) {
                filename = filename.substring(index + 1);
            }
            if (log.isDebugEnabled()) {
                log.debug("upload() - filename=" + filename);
            }

            FileSystemManager fsManager = VFSUtil.getFileSystemManager();
            FileObject newFileObject = fsManager.resolveFile(getUri() + "/"
                    + filename);
            // TODO add confirmation
            if (newFileObject.exists()) {
                // TODO I18N
                FacesMessageUtil
                        .addWarnMessage(filename
                                + " exists. You need to delete it before uploading it. ");
                return goDisplayFolderListPage();
            }
            // TODO should catch exception to check if it's created
            newFileObject.createFile();
            newFileObject.getContent().close();
            drain(getUploadedFile().getInputStream(), newFileObject
                    .getContent().getOutputStream());

            // TODO I18N
            FacesMessageUtil.addInfoMessage("Uploaded file: " + filename);

            setUploadedFile(null);

            if (log.isDebugEnabled()) {
                log.debug("upload() - end");
            }
            return goDisplayFolderListPage();
        } catch (FileSystemException e) {
            // TODO I18N
            FacesMessageUtil.addErrorMessage("Failed to update the file.", e
                    .toString());
            log.error("Failed to update the file.", e);
        } catch (IOException e) {
            // TODO I18N
            FacesMessageUtil.addErrorMessage("Failed to update the file.", e
                    .toString());
            log.error("Failed to update the file.", e);
        }

        if (log.isDebugEnabled()) {
            log.debug("upload() - end");
        }
        return null;
    }

    private void drain(InputStream r, OutputStream w) throws IOException {
        byte[] bytes = new byte[BLOCK_SIZE];
        try {
            int length = r.read(bytes);
            while (length != -1) {
                if (length != 0) {
                    w.write(bytes, 0, length);
                    w.flush();
                }
                length = r.read(bytes);
            }
        } finally {
            bytes = null;
        }
    }

    /**
     * @return Returns the uploadedFile.
     */
    public UploadedFile getUploadedFile() {
        return uploadedFile;
    }

    /**
     * @param uploadedFile
     *            The uploadedFile to set.
     */
    public void setUploadedFile(UploadedFile uploadedFile) {
        this.uploadedFile = uploadedFile;
    }

}
