/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.dao.impl;

import java.util.Calendar;

import javax.jcr.AccessDeniedException;
import javax.jcr.InvalidItemStateException;
import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.UnsupportedRepositoryOperationException;
import javax.jcr.ValueFormatException;
import javax.jcr.lock.LockException;
import javax.jcr.nodetype.ConstraintViolationException;
import javax.jcr.nodetype.NoSuchNodeTypeException;
import javax.jcr.query.InvalidQueryException;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.version.VersionException;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FolderNodeDao;
import jp.sf.pal.cms.dto.FolderNodeDto;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;
import org.seasar.jcr.SessionFactory;

/**
 * @author shinsuke
 * 
 */
public class FolderNodeDaoImpl implements FolderNodeDao {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(FolderNodeDaoImpl.class);

    private SessionFactory sessionFactory_;

    private S2Container container;

    public FolderNodeDaoImpl(SessionFactory sessionFactory) {
        sessionFactory_ = sessionFactory;
    }

    public FolderNodeDto getFolderNode(String scopeName, String path)
            throws CMSException {
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = path.split(CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        if (log.isDebugEnabled()) {
            log.debug("getFileNode(String, String) - q=" + q);
        }

        try {
            Query query = sessionFactory_.getSession().getWorkspace()
                    .getQueryManager().createQuery(q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Node node = ni.nextNode();
                if (log.isDebugEnabled()) {
                    // TODO remove
                    log.debug("getFolderNode(String, String) - node.getName()="
                            + node.getName());
                }
                if (node.getName().equals(CMSConstants.FOLDER_NODE)) {
                    FolderNodeDto folderNode = (FolderNodeDto) container
                            .getComponent(FolderNodeDto.class);
                    // FolderNodeDto folderNode = new FolderNodeDto();
                    folderNode.setScope(scopeName);
                    folderNode.setParentPath(CMSUtil.getParentPath(path));
                    folderNode.setName(node.getProperty(
                            CMSConstants.FOLDER_NODE_NAME_ATTR).getString());
                    folderNode.setCreatedTime(node.getProperty(
                            CMSConstants.FOLDER_NODE_CREATED_ATTR).getDate());
                    folderNode.setLastModifiedTime(node.getProperty(
                            CMSConstants.FOLDER_NODE_LASTMODIFIED_ATTR)
                            .getDate());
                    return folderNode;
                } else {
                    log.warn("Undefined object:" + node);
                }
            }
            return null;
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void insert(FolderNodeDto folderNode) throws CMSException {
        if (folderNode.getName() == null || folderNode.getName().equals("")) {
            throw new CMSException("Invalid folder name.");
        }
        String scopeName = folderNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = folderNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        if (log.isDebugEnabled()) {
            log.debug("insert(FolderNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Calendar now = Calendar.getInstance();
                Node folder = ni.nextNode();

                // cms:folder
                Node newFolderNode = folder.addNode(CMSConstants.FOLDER_NODE);
                CMSUtil.setProperty(newFolderNode,
                        CMSConstants.FOLDER_NODE_NAME_ATTR, folderNode
                                .getName());
                CMSUtil.setProperty(newFolderNode,
                        CMSConstants.FOLDER_NODE_CREATED_ATTR, folderNode
                                .getCreatedTime(), now);
                CMSUtil.setProperty(newFolderNode,
                        CMSConstants.FOLDER_NODE_LASTMODIFIED_ATTR, folderNode
                                .getLastModifiedTime(), now);

                session.save();
            } else {
                throw new CMSException(
                        "Cannot find the target folder: scopeName=" + scopeName
                                + ", path=" + folderNode.getParentPath());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (UnsupportedRepositoryOperationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void update(FolderNodeDto folderNode) throws CMSException {
        if (folderNode.getName() == null || folderNode.getName().equals("")) {
            throw new CMSException("Invalid folder name.");
        }
        if (folderNode.getOldName() == null
                || folderNode.getOldName().equals("")) {
            throw new CMSException("Invalid previous folder name.");
        }
        String scopeName = folderNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = folderNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE + "[@"
                + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                + CMSUtil.escapeAttributeValue(folderNode.getOldName()) + "']");
        // q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
        // + CMSConstants.FILE_NODE_NAME_ATTR + "='"
        // + CMSUtil.escapeAttributeValue(objectNode.getOldName()) + "']");

        if (log.isDebugEnabled()) {
            log.debug("updateFile(FileNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Calendar now = Calendar.getInstance();
                Node updatedFolderNode = ni.nextNode();
                CMSUtil.setProperty(updatedFolderNode,
                        CMSConstants.FOLDER_NODE_NAME_ATTR, folderNode
                                .getName());
                CMSUtil.setProperty(updatedFolderNode,
                        CMSConstants.FOLDER_NODE_CREATED_ATTR, folderNode
                                .getCreatedTime());
                // else {
                // updatedFileNode.setProperty(
                // CMSConstants.FOLDER_NODE_CREATED_ATTR, now);
                // }
                // TODO file name update time
                CMSUtil.setProperty(updatedFolderNode,
                        CMSConstants.FOLDER_NODE_LASTMODIFIED_ATTR, now);

                session.save();
            } else {
                throw new CMSException(
                        "Cannot find the target folder: scopeName=" + scopeName
                                + ", path=" + folderNode.getParentPath()
                                + ", name=" + folderNode.getName());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (UnsupportedRepositoryOperationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (PathNotFoundException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ValueFormatException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    public void delete(FolderNodeDto folderNode) throws CMSException {
        String scopeName = folderNode.getScope();
        StringBuffer q = new StringBuffer(CMSConstants.PATH_SEPARATOR
                + CMSConstants.PATH_SEPARATOR + CMSConstants.SCOPE_NODE + "[@"
                + CMSConstants.SCOPE_NODE_NAME_ATTR + " = '"
                + CMSUtil.escapeAttributeValue(scopeName) + "']");

        String[] nodes = folderNode.getParentPath().split(
                CMSConstants.PATH_SEPARATOR);
        for (int i = 1; i < nodes.length; i++) {
            q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE
                    + "[@" + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                    + CMSUtil.escapeAttributeValue(nodes[i]) + "']");
        }
        q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FOLDER_NODE + "[@"
                + CMSConstants.FOLDER_NODE_NAME_ATTR + "='"
                + CMSUtil.escapeAttributeValue(folderNode.getName()) + "']");
        // q.append(CMSConstants.PATH_SEPARATOR + CMSConstants.FILE_NODE + "[@"
        // + CMSConstants.FILE_NODE_NAME_ATTR + "='"
        // + CMSUtil.escapeAttributeValue(objectNode.getName()) + "']");

        if (log.isDebugEnabled()) {
            log.debug("delete(FileNodeDto) - q=" + q);
        }
        try {
            Session session = sessionFactory_.getSession();
            Query query = session.getWorkspace().getQueryManager().createQuery(
                    q.toString(), Query.XPATH);
            QueryResult result = query.execute();
            NodeIterator ni = result.getNodes();
            if (ni.hasNext()) {
                Node deletedFileNode = ni.nextNode();
                deletedFileNode.remove();
                session.save();
            } else {
                throw new CMSException(
                        "Cannot find the target file: scopeName=" + scopeName
                                + ", path=" + folderNode.getParentPath()
                                + ", name=" + folderNode.getName());
            }
        } catch (InvalidQueryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (VersionException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (LockException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ConstraintViolationException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (AccessDeniedException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (ItemExistsException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (InvalidItemStateException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (NoSuchNodeTypeException e) {
            throw new CMSException("Exception: Query=" + q, e);
        } catch (RepositoryException e) {
            throw new CMSException("Exception: Query=" + q, e);
        }
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }
}
