/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.helper.impl;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.faces.context.FacesContext;
import javax.portlet.PortletSession;
import javax.portlet.RenderResponse;
import javax.servlet.http.HttpSession;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.helper.DownloadHelper;

import org.seasar.framework.container.S2Container;

/**
 * @author shinsuke
 * 
 */
public class DownloadHelperImpl implements DownloadHelper {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(DownloadHelperImpl.class);

    private static String DOWNLOAD_FILE_MAP = CMSConstants.CONSTANT_PREFIX
            + "DownloadFileMap";

    private static int DEFAULT_MAP_SIZE = 4;

    private static int DEFAULT_KEY_LENGTH = 64;

    private int mapSize;

    private int keyLength;

    private S2Container container;

    public DownloadHelperImpl() {
        this.mapSize = DEFAULT_MAP_SIZE;
        this.keyLength = DEFAULT_KEY_LENGTH;
    }

    public DownloadHelperImpl(int mapSize, int keyLength) {
        this.mapSize = mapSize;
        this.keyLength = keyLength;
    }

    public String consume(String key) {
        String path = consumeFromSessionAttribute(key);
        return path;
    }

    public String publish(String value) {
        if (value == null) {
            return null;
        }
        String namespace = getNamespace();
        String key = namespace
                + RandomStringUtils.randomAlphanumeric(keyLength);
        publishToSessionAttribute(key, value);
        return key;
    }

    public void reset() {
        removeSessionAttribute(DOWNLOAD_FILE_MAP);
    }

    private String getNamespace() {
        if (getContainer().getExternalContext().getResponse() instanceof RenderResponse) {
            return (String) ((RenderResponse) getContainer()
                    .getExternalContext().getResponse()).getNamespace();
        }
        return "";
    }

    private void publishToSessionAttribute(String key, String value) {
        Object obj = getContainer().getExternalContext().getSession();
        if (obj instanceof PortletSession) {
            PortletSession session = (PortletSession) obj;
            Map map = (Map) session.getAttribute(DOWNLOAD_FILE_MAP,
                    PortletSession.APPLICATION_SCOPE);
            if (map == null) {
                map = new HashMap(mapSize);
                session.setAttribute(DOWNLOAD_FILE_MAP, map,
                        PortletSession.APPLICATION_SCOPE);
            }
            map.put(key, value);
        } else if (obj instanceof HttpSession) {
            HttpSession session = (HttpSession) obj;
            Map map = (Map) session.getAttribute(DOWNLOAD_FILE_MAP);
            if (map == null) {
                map = new HashMap(mapSize);
                session.setAttribute(DOWNLOAD_FILE_MAP, map);
            }
            map.put(key, value);
        } else {
            log.error("Invalid session object: " + obj);
            throw new IllegalStateException("Invalid session object: " + obj);
        }
    }

    private String consumeFromSessionAttribute(String key) {
        Object obj = getContainer().getExternalContext().getSession();
        if (obj instanceof PortletSession) {
            PortletSession session = (PortletSession) obj;
            Map map = (Map) session.getAttribute(DOWNLOAD_FILE_MAP,
                    PortletSession.APPLICATION_SCOPE);
            if (map == null) {
                return null;
            }
            String value = (String) map.get(key);
            map.remove(key);
            return value;
        } else if (obj instanceof HttpSession) {
            HttpSession session = (HttpSession) obj;
            Map map = (Map) session.getAttribute(DOWNLOAD_FILE_MAP);
            if (map == null) {
                return null;
            }
            String value = (String) map.get(key);
            map.remove(key);
            // TODO size check
            return value;
        } else {
            log.error("Invalid session object: " + obj);
            throw new IllegalStateException("Invalid session object: " + obj);
        }

    }

    private void removeSessionAttribute(String key) {
        Object obj = getContainer().getExternalContext().getSession();
        if (obj instanceof PortletSession) {
            PortletSession session = (PortletSession) obj;
            Map map = (Map) session.getAttribute(DOWNLOAD_FILE_MAP,
                    PortletSession.APPLICATION_SCOPE);
            if (map != null) {
                String namespace = getNamespace();
                if ("".equals(namespace)) {
                    log.warn("The namespace is not defined.");
                    return;
                }
                for (Iterator ite = map.keySet().iterator(); ite.hasNext();) {
                    String name = (String) ite.next();
                    if (name.startsWith(namespace)) {
                        map.remove(name);
                    }
                }
            }
        } else if (obj instanceof HttpSession) {
            HttpSession session = (HttpSession) obj;
            session.removeAttribute(DOWNLOAD_FILE_MAP);
        } else {
            log.error("Invalid session object: " + obj);
            throw new IllegalStateException("Invalid session object: " + obj);
        }
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }
}
