/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.servlet;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.dto.FileNodeDto;
import jp.sf.pal.cms.helper.DownloadHelper;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;

/**
 * @author shinsuke
 * 
 */
public class FileServlet extends HttpServlet {

    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(FileServlet.class);

    private static final int BLOCK_SIZE = 4096;

    public void doGet(HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {

        String fileId = (String) request.getParameter(CMSConstants.FILE_ID);
        if (log.isDebugEnabled()) {
            log.debug("doGet() - fileId=" + fileId);
        }
        if (fileId == null) {
            log.error("The file id is emtpy. ");
            // TODO i18n
            response.sendError(HttpServletResponse.SC_NOT_FOUND,
                    "The file id is emtpy. ");
            return;
        }

        S2Container container = SingletonS2ContainerFactory.getContainer();
        DownloadHelper downloadHelper = (DownloadHelper) container
                .getComponent(DownloadHelper.class);
        String value = downloadHelper.consume(fileId);
        if (value == null) {
            log.error("Could not find the specified object. ");
            // TODO i18n
            response.sendError(HttpServletResponse.SC_NOT_FOUND,
                    "Could not find the specified object. ");
            return;
        }
        int index = value.indexOf(CMSConstants.KEY_SEPARATOR);
        if (index < 0) {
            log.error("Could not find the path. " + value);
            // TODO i18n
            response.sendError(HttpServletResponse.SC_NOT_FOUND,
                    "Could not find the path. " + value);
            return;
        }
        String scopeName = value.substring(0, index);
        String path = value.substring(index + 1);
        if (log.isDebugEnabled()) {
            log.debug("doGet(HttpServletRequest, HttpServletResponse) - value="
                    + value + ", index=" + index + ", scopeName=" + scopeName
                    + ", path=" + path);
        }

        try {
            FileNodeDao fileNodeDao = (FileNodeDao) container
                    .getComponent(FileNodeDao.class);
            FileNodeDto fileNode = fileNodeDao.getFileNode(scopeName, path);
            if (fileNode == null) {
                log.error("Could not find scope=" + scopeName + ", path="
                        + value);
                // TODO i18n
                response
                        .sendError(HttpServletResponse.SC_NOT_FOUND,
                                "Could not find scope=" + scopeName + ", path="
                                        + value);
                return;
            }

            InputStream in = fileNode.getInputStream();
            if (in == null) {
                log.error("The input stream is null: scope=" + scopeName
                        + ", path=" + value);
                // TODO i18n
                response.sendError(HttpServletResponse.SC_NOT_FOUND,
                        "The input stream is null: scope=" + scopeName
                                + ", path=" + value);
                return;
            }

            // MIME Type
            StringBuffer contentType = new StringBuffer();
            if (fileNode.getMimeType() != null) {
                contentType.append(fileNode.getMimeType());
            } else {
                contentType.append("application/octet-stream");
            }
            if (fileNode.getEncoding() != null
                    && !fileNode.getEncoding().equals("")) {
                contentType.append(";charset=");
                contentType.append(fileNode.getEncoding());
            }
            response.setContentType(contentType.toString());

            // File Name
            response.setHeader("Content-disposition", "attachment; filename=\""
                    + fileNode.getName() + "\"");

            drain(in, response.getOutputStream());
        } catch (CMSException e) {
            log.error("Could not load the target object: " + path, e);
            // TODO i18n
            response.sendError(HttpServletResponse.SC_NOT_FOUND,
                    "Could not load the target object: " + path);
        }
    }

    public static void drain(InputStream r, OutputStream w) throws IOException {
        byte[] bytes = new byte[BLOCK_SIZE];
        try {
            int length = r.read(bytes);
            while (length != -1) {
                if (length != 0) {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        } finally {
            bytes = null;
        }

    }
}
