/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.web.editor;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.portlet.RenderResponse;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.ObjectNodeDao;
import jp.sf.pal.cms.dao.PageNodeDao;
import jp.sf.pal.cms.dto.ObjectNodeDto;
import jp.sf.pal.cms.dto.PageNodeDto;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.ComponentNotFoundRuntimeException;
import org.seasar.framework.container.CyclicReferenceRuntimeException;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.TooManyRegistrationRuntimeException;
import org.seasar.jcr.SessionFactory;

import com.marevol.utils.faces.application.FacesMessageUtil;

public class DisplayFolderListPage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory
            .getLog(DisplayFolderListPage.class);

    private String path;

    private String defaultPath;

    private S2Container container;

    public DisplayFolderListPage(S2Container container) {
        if (log.isDebugEnabled()) {
            log.debug("DisplayFolderListPage()");
        }
        Object obj = FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap().get(CMSConstants.PATH);
        if (obj != null) {
            setPath((String) obj);
        } else {
            String namespace = CMSUtil.getNamespace();
            try {
                PageNodeDao pageNodeDao = (PageNodeDao) container
                        .getComponent(PageNodeDao.class);
                PageNodeDto pageNodeDto = pageNodeDao.getPageNode(namespace);
                if (pageNodeDto != null && pageNodeDto.getPath() != null) {
                    setPath(CMSUtil.getParentPath(pageNodeDto.getPath()));
                } else {
                    setPath(CMSConstants.ROOT_FOLDER);
                }
            } catch (CMSException e) {
                log.warn("Could not get a page node: " + namespace);
                setPath(CMSConstants.ROOT_FOLDER);
            }
        }

    }

    // Action
    public String goCreateNewFile() {
        CMSUtil.clearSession();
        return CMSConstants.CREATE_NEW_FILE_PAGE;
    }

    public String goCreateNewFolder() {
        CMSUtil.clearSession();
        return CMSConstants.CREATE_NEW_FOLDER_PAGE;
    }

    public String storeXmlFile() {
        // TODO
        // S2Container container = SingletonS2ContainerFactory.getContainer();
        SessionFactory sessionFactory = (SessionFactory) container
                .getComponent(SessionFactory.class);

        File outputFile = new File("/tmp/cms.xml");
        try {
            FileOutputStream out = new FileOutputStream(outputFile);
            sessionFactory.getSession()
                    .exportSystemView("/", out, false, false);
        } catch (FileNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (PathNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (RepositoryException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return null;
    }

    public String go() {
        setPathToSession(getPath());
        return null;
    }

    public String goParentFolder() {
        setPathToSession(CMSUtil.getParentPath(getPath()));
        return null;
    }

    public String select() {
        CMSUtil.clearSession();

        FacesContext context = FacesContext.getCurrentInstance();
        ExternalContext exContext = context.getExternalContext();
        Map map = exContext.getRequestMap();
        ObjectNodeDto objectNode = (ObjectNodeDto) map.get("objectNode");
        if (objectNode.getNodeType().equals(CMSConstants.FOLDER_NODE)) {
            // folder
            setPathToSession(CMSUtil.normalizedPath(objectNode.getParentPath()
                    + CMSConstants.PATH_SEPARATOR)
                    + objectNode.getName());
            return null;
        } else if (objectNode.getNodeType().equals(CMSConstants.FILE_NODE)) {
            // file
            FacesContext.getCurrentInstance().getExternalContext()
                    .getSessionMap().put(CMSConstants.SELECTED_OBJECT_NAME,
                            objectNode.getName());
            return CMSConstants.DISPLAY_FILE_DETAIL_PAGE;
        } else {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("Invalid node type.");
            return null;
        }
    }

    private void setPathToSession(String path) {
        FacesContext.getCurrentInstance().getExternalContext().getSessionMap()
                .put(CMSConstants.PATH, path);
        setPath(path);
    }

    public String editFolder() {
        CMSUtil.clearSession();

        FacesContext context = FacesContext.getCurrentInstance();
        ExternalContext exContext = context.getExternalContext();
        Map map = exContext.getRequestMap();
        ObjectNodeDto objectNode = (ObjectNodeDto) map.get("objectNode");
        if (objectNode.getNodeType().equals(CMSConstants.FOLDER_NODE)) {
            // folder
            FacesContext.getCurrentInstance().getExternalContext()
                    .getSessionMap().put(CMSConstants.SELECTED_OBJECT_NAME,
                            objectNode.getName());
            return CMSConstants.DISPLAY_FOLDER_DETAIL_PAGE;
        } else {
            // TODO i18n
            FacesMessageUtil.addErrorMessage("Invalid node type.");
            return null;
        }
    }

    // Setter and Getter
    /**
     * @return the path
     */
    public String getPath() {
        return path;
    }

    /**
     * @param path
     *            the path to set
     */
    public void setPath(String path) {
        this.path = CMSUtil.normalizedPath(path);
    }

    public List getObjectNodes() {
        try {
            // S2Container container =
            // SingletonS2ContainerFactory.getContainer();
            ObjectNodeDao objectNodeDao = (ObjectNodeDao) container
                    .getComponent(ObjectNodeDao.class);
            // TODO insert service layer?
            if (log.isDebugEnabled()) {
                log
                        .debug("DisplayFolderListPage() - scope="
                                + CMSUtil.getScopeFromRequest() + ", path="
                                + getPath());
            }
            return objectNodeDao.getObjectNodes(CMSUtil.getScopeFromRequest(),
                    getPath());
        } catch (ComponentNotFoundRuntimeException e) {
            log.error(e);
        } catch (TooManyRegistrationRuntimeException e) {
            log.error(e);
        } catch (CyclicReferenceRuntimeException e) {
            log.error(e);
        } catch (CMSException e) {
            log.error(e);
        }
        // TODO empty List needed?
        return null;
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }

    /**
     * @return Returns the defaultPath.
     */
    public String getDefaultPath() {
        if (defaultPath != null) {
            return defaultPath;
        }
        String namespace = CMSUtil.getNamespace();
        PageNodeDao pageNodeDao = (PageNodeDao) getContainer().getComponent(
                PageNodeDao.class);
        try {
            PageNodeDto pageNodeDto = pageNodeDao.getPageNode(namespace);
            if (pageNodeDto != null && pageNodeDto.getPath() != null) {
                defaultPath = pageNodeDto.getPath();
                if (log.isDebugEnabled()) {
                    log.debug("getDefaultPath() - defaultPath=" + defaultPath);
                }
                return defaultPath;
            }
        } catch (CMSException e) {
            log.warn("Could not call PageNodeDao#getPageNode(" + namespace
                    + ") in getDefaultPath().");
        }
        defaultPath = "";
        if (log.isDebugEnabled()) {
            log.debug("getDefaultPath() - defaultPath=" + defaultPath);
        }
        return defaultPath;
    }

    public String getNamespace() {
        Object response = FacesContext.getCurrentInstance()
                .getExternalContext().getResponse();
        if (response instanceof RenderResponse) {
            return ((RenderResponse) response).getNamespace();
        }
        return null;
    }
}
