/*
 * Copyright 2005-2006 The Portal Application Laboratory Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.pal.cms.web.viewer;

import java.io.InputStream;

import javax.faces.context.FacesContext;

import jp.sf.pal.cms.CMSConstants;
import jp.sf.pal.cms.CMSException;
import jp.sf.pal.cms.dao.FileNodeDao;
import jp.sf.pal.cms.dao.PageNodeDao;
import jp.sf.pal.cms.dto.FileNodeDto;
import jp.sf.pal.cms.dto.PageNodeDto;
import jp.sf.pal.cms.rewriter.HtmlRewriter;
import jp.sf.pal.cms.util.CMSUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.container.S2Container;
import org.xml.sax.InputSource;

/**
 * @author shinsuke
 * 
 */
public class DisplayContentPage {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(DisplayContentPage.class);

    private String contentPath;

    private S2Container container;

    public DisplayContentPage(S2Container container) {
        if (log.isDebugEnabled()) {
            log.debug("DisplayContentPage(S2Container container)");
        }
        setContainer(container);
        Object obj = FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap().get(CMSConstants.CONTENT_PATH);
        if (obj != null) {
            setContentPath((String) obj);
        } else {
            String namespace = CMSUtil.getNamespace();
            if (log.isDebugEnabled()) {
                log.debug("DisplayContentPage(S2Container) - namespace="
                        + namespace);
            }
            try {
                PageNodeDao pageNodeDao = (PageNodeDao) container
                        .getComponent(PageNodeDao.class);
                PageNodeDto pageNodeDto = pageNodeDao.getPageNode(namespace);
                if (pageNodeDto != null && pageNodeDto.getPath() != null) {
                    setContentPath(pageNodeDto.getPath());
                } else {
                    setContentPath(CMSConstants.DEFAULT_CONTENT_PATH);
                }
            } catch (CMSException e) {
                log.error("Could not get a page node: " + namespace);
                setContentPath(CMSConstants.DEFAULT_CONTENT_PATH);
            }
        }
    }

    /**
     * @return Returns the contentPath.
     */
    public String getContentPath() {
        return contentPath;
    }

    /**
     * @param contentPath
     *            The contentPath to set.
     */
    public void setContentPath(String contentPath) {
        this.contentPath = contentPath;
    }

    /**
     * @return Returns the content.
     */
    public String getContent() {
        if (contentPath != null) {
            String content = retrieveContent(contentPath);
            if (content != null) {
                return content;
            }
            // TODO i18n
            return "Could not find the target content: " + contentPath;
        } else {
            // TODO i18n
            return "Need to specify the page path.";
        }
    }

    protected String retrieveContent(String path) {
        FileNodeDao fileNodeDao = (FileNodeDao) container
                .getComponent(FileNodeDao.class);
        try {
            FileNodeDto fileNode = fileNodeDao.getFileNode(CMSUtil
                    .getScopeFromRequest(), path);
            if (fileNode == null) {
                return null;
            }
            // set title
            if (fileNode.getTitle() != null) {
                setPageTitle(fileNode.getTitle());
            }
            return rewriteContent(fileNode.getInputStream(), fileNode
                    .getEncoding());
        } catch (CMSException e) {
            log.error("Could not find the target file: " + path, e);
        }

        return null;
    }

    protected void setPageTitle(String title) {
        CMSUtil.getRenderResponse().setTitle(title);
    }

    protected String rewriteContent(InputStream inputStream, String encoding)
            throws CMSException {
        if (inputStream == null) {
            throw new CMSException("The input stream is null.");
        }
        HtmlRewriter htmlRewriter = (HtmlRewriter) container
                .getComponent(HtmlRewriter.class);
        InputSource inputSource = new InputSource(inputStream);
        inputSource.setEncoding(encoding);
        return htmlRewriter.parse(inputSource);
    }

    /**
     * @param content
     *            The content to set.
     */
    public void setContent(String content) {
        // None
    }

    /**
     * @return Returns the container.
     */
    public S2Container getContainer() {
        return container;
    }

    /**
     * @param container
     *            The container to set.
     */
    public void setContainer(S2Container container) {
        this.container = container;
    }

}
