# testresponse.rb: testcases for additional methods on WEBrick::HTTPResponse

require 'test/unit'
load 'blazer-ja-proxy'

require 'zlib'
require 'stringio'

require 'nkf'
class TestResponse < Test::Unit::TestCase
	include WEBrick

	def test_on_charset
		{
			'text/html; charset=iso-8859-1' => NKF::ASCII,
			'text/html; charset=EUC-JP' => NKF::EUC,
			'text/html; charset=utf-8' => NKF::UTF8,
			'text/plain; charset=utf-8' => NKF::UTF8,
			'text/html;CHARSET=Shift_JIS' => NKF::SJIS,
			'text/html' => nil,
			'text/plain' => nil,
			'text/css' => nil,
			'image/gif' => NKF::BINARY,
			nil => nil,
		}.each_pair do |content_type, charset|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = content_type
			assert_equal(charset, res.charset, "Content-type: " + content_type.inspect)
		end
	end

	def test_on_html_header
		{
			'<html><head><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=UTF-8"><title>Google</title>' => NKF::UTF8,
			<<'_TDIARY_HEADER' => NKF::EUC,
<meta http-equiv="Content-Type" content="text/html; charset=EUC-JP">
<meta name="generator" content="tDiary 2.1.3.20051021">
<meta http-equiv="Last-Modified" content="Tue, 21 Mar 2006 20:00:20 GMT">
<meta http-equiv="Content-Script-Type" content="text/javascript; charset=UTF-8">
_TDIARY_HEADER
			<<'_SPONICHI' => NKF::SJIS,
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN"
"http://www.w3.org/TR/html4/loose.dtd">
<html>
<head>
<title>ݥ˥ Sponichi Annex ®</title>
<meta http-equiv="Content-Type" content="text/html; charset=Shift_JIS">
</head>
_SPONICHI
		}.each_pair do |header, charset|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.body = header
			assert_equal(charset, res.charset, "header: " + header)
		end
	end

	def test_on_xml_header
		{
			"\xFE\xFFhoge hoge" => NKF::UTF16,
			"\xEF\xBB\xBFhoge hoge" => NKF::UTF8,
			<<'_HEAD' => NKF::SJIS,
<?xml version="1.0" encoding="Shift_JIS"?>
<!DOCTYPE wml PUBLIC "-//WAPFORUM//DTD WML 1.2//EN" 
	"http://www.wapforum.org/DTD/wml12.dtd">
_HEAD
			'<?xml>unknown charset' => NKF::UTF8,
		}.each_pair do |header, charset|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = 'xml'
			res.body = header
			assert_equal(charset, res.charset, "header: " + header.inspect)
		end
	end

	def test_body
		{
			[nil, nil, '<html><head><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=UTF-8"><title>Google</title>'] => NKF::UTF8,
			[nil, nil, '<html><head><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=EUC-JP"><title>Google</title>'] => NKF::EUC,
			['text/html', nil, '<html><head><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=UTF-8"><title>Google</title>'] => NKF::UTF8,
			['text/html', nil, '<html><head><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=EUC-JP"><title>Google</title>'] => NKF::EUC,
			['text/html', nil, '<?xml version="1.0" encording="UTF-8"?><meta http-equiv="content-type" content="application/xhtml+xml; charset=UTF-8" />'] => NKF::UTF8,
			# http://www.radiumsoftware.com/
			['text/html', 'gzip', '<html><head><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=EUC-JP"><title>Google</title>'] => NKF::EUC,
			[nil, nil, 'dead beaf'] => NKF::ASCII,
			[nil, nil, NKF::nkf('-e', 'ˤϡ')] => NKF::EUC,
			[nil, nil, NKF::nkf('-s', 'ˤϡ')] => NKF::SJIS,
			[nil, nil, NKF::nkf('-j', 'ˤϡ')] => NKF::JIS,
			[nil, nil, NKF::nkf('-w', 'ˤϡ')] => NKF::UTF8,
			[nil, nil, "\211PNG\r\n\032\n\000\000\000\rIHDR\000\000\0000\000\000\000\020\004\003\000\000\000\242\200\341\202\000\000\000\022PLTE\377\377\377\377\377\377\354\377\331\277\233k\255\337dc\2009\326\373d\205\000\000\000\001tRNS\000@\346\330f\000\000\000\tpHYs\000\000\v\022\000\000\v\022\001\322\335~\374\000\000\000\atIME\a\322\006\030\0239\006pH\250\362\000\000\000\214IDATx\234u\220\321\r\003!\fC\221*\026\211\330 \352\002'\363\217P\262\377*MH\350\365\256m>\300\370\371\203\270\224\357\251\252wKu\230/\300\233DH\201>\212\034\r\030\351O\262P=\250\241\227i\376\006Bd\272\222\203J\016z\200\271\2008\200\230\334\300\242\366\352\360\023\202\vp\275\234\eh'\320\023<\236\241#;\326e\277b\346\0045\242\t\330't\202\334\234\331\367\210Pl\376\331\325\256guu\231\037\205\376\233\027\177\2378\025A\257\001\216\000\000\000\000IEND\256B`\202"] => nil,
			# a PNG file
		}.each_pair do |a, charset|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = a[0]
			res['content-encoding'] = a[1]
			case a[1]
			when 'gzip'
				gzipped = StringIO.new
				gzfile = Zlib::GzipWriter.new(gzipped)
				gzfile.write(a[2])
				gzfile.close
				res.body = gzipped.string
			else
				res.body = a[2]
			end
			assert_equal(charset, res.charset, "body: " + a[1].inspect)
		end
	end

	def test_convert
		src = NKF::nkf('-w8', <<'_END')
<html>
<head>
<meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=UTF-8">
<title>Hello world</title>
</head>
<p>ˤϡ</p>
</html>
_END
		dst = NKF::nkf('-e', <<'_END')
<html>
<head>
<meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=EUC-JP">
<title>Hello world</title>
</head>
<p>ˤϡ</p>
</html>
_END
		dst_char = NKF::EUC
		res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
		res.content_type = 'text/html; charset=utf-8'
		res.body = src

		res.convert_to!(dst_char)

		assert_equal(dst_char, res.charset)
		assert_equal(dst, res.body)
	end

	def test_http_header_convert
		{
			'text/html' => 'text/html; charset=EUC-JP',
			'text/html; charset=utf-8' => 'text/html; charset=EUC-JP',
			'text/html;CHARSET=Shift_JIS' => 'text/html; charset=EUC-JP',
			'application/xhtml+xml; charset=UTF-8' => 'application/xhtml+xml; charset=EUC-JP',
		}.each_pair do |src, dst|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = src
			res.convert_to!(NKF::EUC)
			assert_equal(dst, res.content_type)
		end
	end

	def test_meta_insert
		{
			'<?xml?><html><head><title>Hello world</title></head><body>hello</body></html>' =>
			'<?xml?><html><head><title>Hello world</title><meta http-equiv="content-type" content="text/html; charset=EUC-JP"/></head><body>hello</body></html>',

			'<html><head><title>Hello world</title><meta HTTP-EQUIV="content-type" CONTENT="text/html;CHARSET=Shift_JIS"></head><body>hello</body></html>' =>
			'<html><head><title>Hello world</title><meta HTTP-EQUIV="content-type" CONTENT="text/html; charset=EUC-JP"></head><body>hello</body></html>',
			'<html><head><title>Hello world</title></head><body>hello</body></html>' =>
			'<html><head><title>Hello world</title><meta http-equiv="content-type" content="text/html; charset=EUC-JP"></head><body>hello</body></html>',
			"\n<html>\n<head>\n<meta http-equiv=Content-Type content=\"text/html; charset=UTF-8\">\n<title>test</title>\n</head></html>" =>
			"\n<html>\n<head>\n<meta http-equiv=Content-Type content=\"text/html; charset=EUC-JP\">\n<title>test</title>\n</head></html>",

			'<html><body>hello</body></html>' =>
			'<html><head><meta http-equiv="content-type" content="text/html; charset=EUC-JP"></head><body>hello</body></html>',

			'<html>hello</html>' =>
			'<html><head><meta http-equiv="content-type" content="text/html; charset=EUC-JP"></head>hello</html>',

			'hello' =>
			'<head><meta http-equiv="content-type" content="text/html; charset=EUC-JP"></head>hello',
			'<head><meta http-equiv="content-type" content="text/html; charset=EUC_JP"></head>hello' =>
			'<head><meta http-equiv="content-type" content="text/html; charset=EUC-JP"></head>hello',
		}.each_pair do |src, dst|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = 'text/html'
			res.body = src.dup
			res.convert_to!(NKF::EUC)
			assert_equal(dst, res.body)
		end
	end

	def test_xhtml_convert
		{
			<<'_END' => <<'_END',
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html PUBLIC "-//WAPFORUM//DTD XHTML Mobile 1.0//EN" "http://www.wapforum.org/DTD/xhtml-mobile10.dtd"><html xmlns="http://www.w3.org/1999/xhtml"><head><title>test</title></head><body><p>hello world</p></body></html>
_END
<?xml version="1.0" encoding="EUC-JP"?>
<!DOCTYPE html PUBLIC "-//WAPFORUM//DTD XHTML Mobile 1.0//EN" "http://www.wapforum.org/DTD/xhtml-mobile10.dtd"><html xmlns="http://www.w3.org/1999/xhtml"><head><title>test</title></head><body><p>hello world</p></body></html>
_END
			<<'_END' => <<'_END',	# http://www.radiumsoftware.com/
<meta http-equiv="content-type" content="application/xhtml+xml; charset=UTF-8" />
_END
<meta http-equiv="content-type" content="application/xhtml+xml; charset=EUC-JP" />
_END
		}.each_pair do |src, dst|
			dst_char = NKF::EUC
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = 'application/xhtml+xml'
			res.body = src.dup

			res.convert_to!(dst_char)

			assert_equal(dst_char, res.charset)
			assert_equal(dst, res.body)
		end
	end

	def test_gzip
		{
			'hello' => "\037\213\010\000mI7D\000\003\363H\315\311\311\a\000\202\211\321\367\005\000\000\000"
		}.each_pair do |src, dst|
			res = HTTPResponse.new( {:HTTPVersion => '1.1'} )
			res.content_type = 'text/plain'
			res.body = src
			res.gzip!
			#assert_equal(dst, res.body)
			assert_equal('gzip', res['content-encoding'])
			#assert_equal(dst.size, res['content-length'])
			# result seems to change with mtime
		end
	end

end
