/*
 * smtpauth.c
 * $Id: smtpauth.c,v 1.14 2009/06/12 08:55:50 taizo Exp $
 * Copyright (C) 2009 HDE, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Emacs; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/utsname.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <net/if.h>

#include "smtpauth.h"

#ifdef USE_SSL
#include <openssl/err.h>
#include <openssl/md5.h>
#else
#include "global.h"
#include "md5.h"
#endif

#define EHLO_CMD ("EHLO ")	/* ESMTP ehlo command */
#define AUTH_CMD ("AUTH ")	/* ESMTP auth command */
#define QUIT_CMD ("QUIT ")	/* ESMTP quit command */

#define RESP_LEN 1000
#define RESP_IERROR	 "internal error"
#define RESP_UNAVAILABLE "remote authentication server is currently unavailable"
#define RESP_UNEXPECTED	 "unexpected response from remote authentication server"
#define RESP_SYNCERROR   "error synchronizing with remote authentication server"
#define RESP_CREDERROR   "remote authentication server rejected your credentials"

#define AUTH_NG 0
#define AUTH_OK 1

#define AUTH_PLAIN      1 << 0
#define AUTH_LOGIN      1 << 1
#define AUTH_CRAM_MD5   1 << 2
#define AUTH_DIGEST_MD5 1 << 3

#define B64(c)  (isascii(c) ? base64val[(int)(c)] : -1)

extern int retry_writev(socket_t *sock, struct iovec *iov, int iovcnt);
extern int socket_read(socket_t *sock, char *buf, size_t len);
extern int socket_close(socket_t *sock);
extern void socket_perror(const char *func, socket_t *sock, int ret);

void md5_hex_hmac(char *hexdigest, unsigned char *text, unsigned int text_len, unsigned char *key, unsigned int key_len);
void hmac_md5(unsigned char *text, unsigned int text_len, unsigned char *key, unsigned int key_len, unsigned char *digest);
int start_tls(smtp_t *smtp, config_t *cfg);


int smtp_quit(socket_t *sock, config_t *cfg);
int auth_plain(socket_t *sock, config_t *cfg);
int auth_login(socket_t *sock, config_t *cfg);
int auth_cram_md5(socket_t *sock, config_t *cfg);
int auth_digest_md5(socket_t *sock, config_t *cfg);

config_t global;
static const char base64char[64] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
static const char base64val[128] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1
};


void
md5_hex_hmac(char *hexdigest, unsigned char *text, unsigned int text_len, unsigned char *key, unsigned int key_len) {

    unsigned char digest[16];
    int cnt;

    hmac_md5(text, text_len, key, key_len, digest);
    for(cnt=0; cnt<16; cnt++) {
        sprintf(hexdigest + 2 * cnt, "%02x", digest[cnt]);
    }
}


void
hmac_md5(unsigned char *text, unsigned int text_len, unsigned char *key, unsigned int key_len, unsigned char *digest) {

    MD5_CTX context;
    unsigned char k_ipad[64];
    unsigned char k_opad[64];
    int cnt;

    memset(k_ipad, 0, sizeof(k_ipad));
    memset(k_opad, 0, sizeof(k_opad));
    if(key_len > 64) {
        MD5_CTX tctx;

#ifdef USE_SSL
        MD5_Init(&tctx);
        MD5_Update(&tctx, key, key_len);
        MD5_Final(k_ipad, &tctx);
        MD5_Final(k_opad, &tctx);
#else
        MD5Init(&tctx);
        MD5Update(&tctx, key, key_len);
        MD5Final(k_ipad, &tctx);
        MD5Final(k_opad, &tctx);
#endif
    } else {
        memcpy(k_ipad, key, key_len);
        memcpy(k_opad, key, key_len);
    }

    for(cnt=0; cnt<64; cnt++) {
        k_ipad[cnt] ^= 0x36;
        k_opad[cnt] ^= 0x5c;
    }

#ifdef USE_SSL
    MD5_Init(&context);
    MD5_Update(&context, k_ipad, 64);
    MD5_Update(&context, text, text_len);
    MD5_Final(digest, &context);

    MD5_Init(&context);
    MD5_Update(&context, k_opad, 64);
    MD5_Update(&context, digest, 16);
    MD5_Final(digest, &context);
#else
    MD5Init(&context);
    MD5Update(&context, k_ipad, 64);
    MD5Update(&context, text, text_len);
    MD5Final(digest, &context);

    MD5Init(&context);
    MD5Update(&context, k_opad, 64);
    MD5Update(&context, digest, 16);
    MD5Final(digest, &context);
#endif
}


void
base64_encode(char *out, const char *in, int inlen) {

    const char *inp = in;
    char *outp = out;

    while(inlen >= 3) {
        *outp++ = base64char[(inp[0] >> 2) & 0x3f];
        *outp++ = base64char[((inp[0] & 0x03) << 4) | ((inp[1] >> 4) & 0x0f)];
        *outp++ = base64char[((inp[1] & 0x0f) << 2) | ((inp[2] >> 6) & 0x03)];
        *outp++ = base64char[inp[2] & 0x3f];
        inp += 3;
        inlen -= 3;
    }
    if(inlen > 0) {
        *outp++ = base64char[(inp[0] >> 2) & 0x3f];
        if(inlen == 1) {
            *outp++ = base64char[(inp[0] & 0x03) << 4];
            *outp++ = '=';
        }
        else {
            *outp++ = base64char[((inp[0] & 0x03) << 4) | ((inp[1] >> 4) & 0x0f)];
            *outp++ = base64char[((inp[1] & 0x0f) << 2)];
        }
        *outp++ = '=';
    }
    *outp = '\0';
}


int
base64_decode(char *out, const char *in, int inlen) {

    const char *inp = in;
    char *outp = out;
    char buf[4];

    if( inlen < 0) {
        inlen = 2100000000;
    }
    while(inlen >=4 && *inp != '\0') {
        buf[0] = *inp++;
        inlen--;
        if( B64(buf[0]) == -1) break;

        buf[1] = *inp++;
        inlen--;
        if(B64(buf[1]) == -1) break;

        buf[2] = *inp++;
        inlen--;
        if(buf[2] != '=' && B64(buf[2]) == -1) break;

        buf[3] = *inp++;
        inlen--;
        if(buf[3] != '=' && B64(buf[3]) ==  -1) break;

        *outp++ = ((B64(buf[0]) << 2) & 0xfc) | ((B64(buf[1]) >> 4) & 0x03);
        if(buf[2] != '=') {
           *outp++ = ((B64(buf[1]) & 0x0f) << 4) | ((B64(buf[2]) >> 2) & 0x0f);
           if(buf[3] != '=') {
                *outp++ = ((B64(buf[2]) & 0x03) << 6) | (B64(buf[3]) & 0x3f);
           }
        }
    }
    return outp - out;
}


smtp_t *
smtp_auth(config_t *cfg) {

    int s;
    struct sockaddr_in addr;
    struct hostent *he;
    smtp_t *smtp = NULL;
    char msgbuf[256];

    struct iovec iov[5];
    char *c;
    int rc;
    char rbuf[RESP_LEN];
    int auth = 0;
    int avail_auth_type = 0;
    char *tbuf;
    struct utsname  h_name[1];
    char *myhostname;

    int                n;
    struct sockaddr_in taddr;
    int                sd;
    struct ifconf      ifconf;
    struct ifreq       *ifr, ifreq;
    unsigned char      *ifptr;
    int                iflen;
#ifdef USE_SSL
    int use_ssl;
#endif

    if(!cfg->password) {
        if(!global.password) {
            global.password = getpass("Password:");
            if(!global.password) {
                return 0;
            }
            if(!*global.password) {
                global.password = NULL;
                goto bail;
            }
            global.password = strdup(global.password);
        }
        cfg->password = strdup(global.password);
    }

    assert(cfg->username != NULL);
    assert(cfg->password != NULL);

    smtp = calloc(1, sizeof(smtp_t));
    smtp->sock = calloc(1, sizeof(socket_t));
    smtp->buf = calloc(1, sizeof(buffer_t));
    smtp->buf->sock = smtp->sock;
    smtp->sock->fd = -1;
    smtp->error = 0;

    /* open connection to SMTP server */
    memset(&addr, 0, sizeof(addr));
    addr.sin_port = htons(cfg->port);
    addr.sin_family = AF_INET;
    he = gethostbyname(cfg->host);
    if(!he) {
        smtp->error = 1;
        strcpy(msgbuf, "Error: resolving hostname ");
        strcat(msgbuf, cfg->host);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }

    if((sd = socket(PF_INET, SOCK_DGRAM, 0)) != -1) {
        bzero(&ifconf, sizeof(struct ifconf));
        bzero(&ifreq, sizeof(struct ifreq));
        iflen = 10 * sizeof(struct ifreq);
        ifptr = malloc(iflen);
        ifconf.ifc_len = iflen;
        ifconf.ifc_ifcu.ifcu_req = (struct ifreq *)ifptr;
        if(ioctl(sd, SIOCGIFCONF, &ifconf) != -1) {
            for(iflen=sizeof(struct ifreq); iflen<=ifconf.ifc_len; iflen+=sizeof(struct ifreq)) {
                ifr = (struct ifreq *)ifptr;
                strcpy(ifreq.ifr_ifrn.ifrn_name, ifr->ifr_name);
                if(ioctl(sd, SIOCGIFADDR, &ifreq) != -1) {
                    n = 0;
                    while(he->h_addr_list[n]) {
                        if(he->h_addrtype == AF_INET) {
                            memset((char*)&taddr, 0, sizeof(taddr));
                            memcpy((char*)&taddr.sin_addr, he->h_addr_list[n], he->h_length);
#ifdef DEBUG
                            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): my ip: %s",
                              inet_ntoa(((struct sockaddr_in *)&ifreq.ifr_addr)->sin_addr));
                            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): smtp ip: %s",
                              inet_ntoa(taddr.sin_addr));
#endif
                            if(((struct sockaddr_in *)&ifreq.ifr_addr)->sin_addr.s_addr == taddr.sin_addr.s_addr) {
                                smtp->error = 1;
                                strcpy(msgbuf, "Error: this host is specified. ");
                                strcat(msgbuf, inet_ntoa(taddr.sin_addr));
                                smtp->error_message = malloc(strlen(msgbuf) + 1);
                                strcpy(smtp->error_message, msgbuf);
                                goto bail;
                            }
                        }
                        n++;
                    }
                }
                ifptr += sizeof(struct ifreq);
            }
        }
    }
    addr.sin_addr.s_addr = *((int *)he->h_addr_list[0]);
    s = socket(PF_INET, SOCK_STREAM, 0);
    if(connect(s, (struct sockaddr *) &addr, sizeof(addr))) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): connection error = %s",strerror(errno));
#endif
        smtp->error = 1;
        strcpy(msgbuf, "Error: connecting to ");
        strcat(msgbuf, inet_ntoa(addr.sin_addr));
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }
    smtp->sock->fd = s;

#ifdef USE_SSL
    use_ssl = 0;
    if(cfg->use_smtps) {
      if(start_tls(smtp, cfg)) {
        smtp->error = 1;
        strcpy(msgbuf, "Error: start_tls");
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
      }
      use_ssl = 1;
    }
#endif

    /* CLAIM: we now have a TCP connection to the remote SMTP server */
    alarm(cfg->timeout);

    rc = socket_read(smtp->sock, rbuf, sizeof(rbuf));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (banner): %m");
#endif
        smtp->error = 1;
        strcpy(msgbuf, RESP_SYNCERROR);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }
    rbuf[rc] = '\0';
    c = strpbrk(rbuf, "\r\n");
    if(c != NULL) {
        *c = '\0';
    }

    if(strncmp(rbuf, "220 ", sizeof("220 ")-1)) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): unexpected response during initial handshake: %s", rbuf);
#endif
        smtp->error = 1;
        strcpy(msgbuf, RESP_UNEXPECTED);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }

    if((uname(h_name)) < 0){
        myhostname = "localhost.localdomain";
    } else {
        myhostname = h_name->nodename;
    }

    iov[0].iov_base = EHLO_CMD;
    iov[0].iov_len  = sizeof(EHLO_CMD) - 1;
    iov[1].iov_base = myhostname;
    iov[1].iov_len  = strlen(myhostname);
    iov[2].iov_base = "\r\n";
    iov[2].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s%s", EHLO_CMD, myhostname);
#endif
    alarm(cfg->timeout);
    rc = retry_writev(smtp->sock, iov, 3);
    memset(iov, 0, sizeof(iov));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): writev: %m");
#endif
        smtp->error = 1;
        strcpy(msgbuf, RESP_IERROR);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }

    /* read and parse the EHLO response */
    alarm(cfg->timeout);
    rc = socket_read(smtp->sock, rbuf, sizeof(rbuf));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
        smtp->error = 1;
        strcpy(msgbuf, RESP_IERROR);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }

    if((tbuf = strstr(rbuf, "250-STARTTLS"))) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): STARTTLS not supported.");
#endif
    }

    if((tbuf = strstr(rbuf, "250-AUTH"))) {
        if(strncmp(tbuf, "250", sizeof("250")-1) == 0) {
            char *p = tbuf;
            p += 3;
            if(*p == '-' || *p == ' ') p++;
            if(strncasecmp(p, "AUTH", sizeof("AUTH")-1) == 0) {
                p += 5;
                if(strcasestr(p, "PLAIN"))
                    avail_auth_type |= AUTH_PLAIN;
                if(strcasestr(p, "LOGIN"))
                    avail_auth_type |= AUTH_LOGIN;
                if(strcasestr(p, "CRAM-MD5"))
                    avail_auth_type |= AUTH_CRAM_MD5;
                if(strcasestr(p, "DIGEST-MD5"))
                    avail_auth_type |= AUTH_DIGEST_MD5;
            }
        }
    }

    if(avail_auth_type == 0) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): smtp authentication is not implemented: %s", rbuf);
#endif
        smtp->error = 1;
        strcpy(msgbuf, RESP_UNEXPECTED);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }
#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): auth_type: %d", avail_auth_type);
#endif

    /* build the AUTH command */
    if(avail_auth_type & AUTH_CRAM_MD5) {
        auth = auth_cram_md5(smtp->sock,&global);
    }
    else if((avail_auth_type & AUTH_LOGIN) != 0) {
        auth = auth_login(smtp->sock,&global);
    }
    else if((avail_auth_type & AUTH_PLAIN) != 0) {
        auth = auth_plain(smtp->sock,&global);
    }
    else if((avail_auth_type & AUTH_DIGEST_MD5) != 0) {
        auth = auth_digest_md5(smtp->sock,&global);
    }
    else {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): smtp authentication is not implemented: %s", rbuf);
#endif
        smtp->error = 1;
        strcpy(msgbuf, RESP_UNEXPECTED);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth) auth: [%d]", auth);
#endif
    if(auth == 0) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth) rejected: [%s]", global.username);
#endif
        smtp->error = 2;
        strcpy(msgbuf, RESP_CREDERROR);
        smtp->error_message = malloc(strlen(msgbuf) + 1);
        strcpy(smtp->error_message, msgbuf);
        goto bail;
    }

    smtp_quit(smtp->sock,&global);
    return smtp;

    bail:
        smtp_quit(smtp->sock,&global);
        if(smtp->error == 1)
            return smtp;
        else if(smtp->error == 2)
            return smtp;
        return smtp;
}


int
smtp_quit(socket_t *sock, config_t *cfg) {

    struct iovec iov[3];
    int rc;

    iov[0].iov_base = QUIT_CMD;
    iov[0].iov_len  = sizeof(QUIT_CMD) - 1;
    iov[1].iov_base = "\r\n";
    iov[1].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s", QUIT_CMD);
#endif
    alarm(cfg->timeout);
    rc = retry_writev(sock, iov, 2);
    memset(iov, 0, sizeof(iov));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): quit writev: %m");
#endif
    }
    (void)socket_close(sock);
    return 1;
}


int
auth_cram_md5(socket_t *sock, config_t *cfg) {

    struct iovec iov[3];
    int rc;
    char rbuf[RESP_LEN];
    char buf[RESP_LEN];

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): AUTH CRAM-MD5");
#endif
    iov[0].iov_base = AUTH_CMD;
    iov[0].iov_len  = sizeof(AUTH_CMD) - 1;
    iov[1].iov_base = "CRAM-MD5";
    iov[1].iov_len  = strlen("CRAM-MD5");
    iov[2].iov_base = "\r\n";
    iov[2].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s%s", AUTH_CMD, "CRAM-MD5");
#endif
    alarm(cfg->timeout);
    rc = retry_writev(sock, iov, 3);
    memset(iov, 0, sizeof(iov));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): cram-md5 writev: %m");
#endif
        return AUTH_NG;
    }

    alarm(cfg->timeout);
    rc = socket_read(sock, rbuf, sizeof(rbuf));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
        return AUTH_NG;
    }

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %s",rbuf);
#endif
    if(strncmp(rbuf, "334 ", sizeof("334 ")-1) == 0) {
        char *response;
        char *response64;
        char *challenge;
        int challengelen;
        unsigned char hexdigest[33];

        challenge = malloc(strlen(rbuf + 4) + 1);
        challengelen = base64_decode(challenge, rbuf + 4, -1);
        challenge[challengelen] = '\0';
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): challenge=%s", challenge);
#endif

        snprintf(buf, sizeof(buf), "%s", cfg->password);
        md5_hex_hmac(hexdigest, challenge, challengelen, buf, strlen(cfg->password));
        free(challenge);

        response = malloc(sizeof(char)*128);
        sprintf(response, "%s %s", cfg->username, hexdigest);
        response64 = malloc((strlen(response) + 3) * 2 + 1);
        base64_encode(response64, response, strlen(response));
        free(response);

        iov[0].iov_base = response64;
        iov[0].iov_len  = strlen(response64);
        iov[1].iov_base = "\r\n";
        iov[1].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s", response64);
#endif
        alarm(cfg->timeout);
        rc = retry_writev(sock, iov, 2);
        memset(iov, 0, sizeof(iov));
        alarm(0);
        if(rc == -1) {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): cram-md5 writev: %m");
#endif
            return AUTH_NG;
        }

        alarm(cfg->timeout);
        rc = socket_read(sock, rbuf, sizeof(rbuf));
        alarm(0);
        if(rc == -1) {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
            return AUTH_NG;
        }

#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %s",rbuf);
#endif
        if(strncmp(rbuf, "235 ", sizeof("235 ")-1) != 0) {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): auth failure.");
#endif
            return AUTH_NG;
        }
        free(response64);
    } else {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): it seems cram-md5 mech is not implemented.");
#endif
        return AUTH_NG;
    }
    return AUTH_OK;
}


int
auth_login(socket_t *sock, config_t *cfg) {

    struct iovec iov[3];
    int rc;
    char rbuf[RESP_LEN];
    //char buf[RESP_LEN];
    char *buf;

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): AUTH LOGIN");
#endif
    iov[0].iov_base = AUTH_CMD;
    iov[0].iov_len  = sizeof(AUTH_CMD) - 1;
    iov[1].iov_base = "LOGIN";
    iov[1].iov_len  = strlen("LOGIN");
    iov[2].iov_base = "\r\n";
    iov[2].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s%s", AUTH_CMD, "LOGIN");
#endif
    alarm(cfg->timeout);
    rc = retry_writev(sock, iov, 3);
    memset(iov, 0, sizeof(iov));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): login writev: %m");
#endif
        return AUTH_NG;
    }

    alarm(cfg->timeout);
    rc = socket_read(sock, rbuf, sizeof(rbuf));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
        return AUTH_NG;
    }

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %s",rbuf);
#endif
    if(strncmp(rbuf, "334 ", sizeof("334 ")-1) == 0) {
        buf = malloc(sizeof(char)*128);
        base64_encode(buf, cfg->username, strlen(cfg->username));

        iov[0].iov_base = buf;
        iov[0].iov_len  = strlen(buf);
        iov[1].iov_base = "\r\n";
        iov[1].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s", buf);
#endif
        alarm(cfg->timeout);
        rc = retry_writev(sock, iov, 2);
        memset(iov, 0, sizeof(iov));
        alarm(0);
        if(rc == -1) {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): login writev: %m");
#endif
            return AUTH_NG;
        }

        alarm(cfg->timeout);
        rc = socket_read(sock, rbuf, sizeof(rbuf));
        alarm(0);
        if(rc == -1) {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
            return AUTH_NG;
        }

#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %s",rbuf);
#endif
        if(strncmp(rbuf, "334 ", sizeof("334 ")-1) == 0) {
            buf = malloc(sizeof(char)*128);
            base64_encode(buf, cfg->password, strlen(cfg->password));

            iov[0].iov_base = buf;
            iov[0].iov_len  = strlen(buf);
            iov[1].iov_base = "\r\n";
            iov[1].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s", buf);
#endif
            alarm(cfg->timeout);
            rc = retry_writev(sock, iov, 2);
            memset(iov, 0, sizeof(iov));
            alarm(0);
            if(rc == -1) {
#ifdef DEBUG
                syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): login writev: %m");
#endif
                return AUTH_NG;
            }

            alarm(cfg->timeout);
            rc = socket_read(sock, rbuf, sizeof(rbuf));
            alarm(0);
            if(rc == -1) {
#ifdef DEBUG
                syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
                return AUTH_NG;
            }

#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %s",rbuf);
#endif
            if(strncmp(rbuf, "235 ", sizeof("235 ")-1) != 0) {
#ifdef DEBUG
                syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): auth failure.");
#endif
                return AUTH_NG;
            }
        } else {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): it seems login mech is not implemented.");
#endif
            return AUTH_NG;
        }
    } else {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): it seems login mech is not implemented.");
#endif
        return AUTH_NG;
    }
    return AUTH_OK;
}


int
auth_plain(socket_t *sock, config_t *cfg) {

    struct iovec iov[3];
    int rc;
    char rbuf[RESP_LEN];
    //char buf[RESP_LEN];
    char *buf;
    int cnt, len;
    char phrase[512];

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): AUTH PLAIN");
#endif
    sprintf(phrase,"%s^%s^%s", cfg->username, cfg->username, cfg->password);
    len = strlen(phrase);
    for(cnt=len-1; cnt>=0; cnt--) {
        if(phrase[cnt] == '^') {
            phrase[cnt] = '\0';
        }
    }
    buf = malloc(sizeof(char)*128);
    base64_encode(buf, phrase, len);

    iov[0].iov_base = AUTH_CMD;
    iov[0].iov_len  = sizeof(AUTH_CMD) - 1;
    iov[1].iov_base = "PLAIN ";
    iov[1].iov_len  = strlen("PLAIN ");
    iov[2].iov_base = buf;
    iov[2].iov_len  = strlen(buf);
    iov[3].iov_base = "\r\n";
    iov[3].iov_len  = sizeof("\r\n") - 1;

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): sending %s%s %s", AUTH_CMD, "PLAIN", buf);
#endif
    alarm(cfg->timeout);
    rc = retry_writev(sock, iov, 4);
    memset(iov, 0, sizeof(iov));
    free(buf);
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): plain writev: %m");
#endif
        return AUTH_NG;
    }

    alarm(cfg->timeout);
    rc = socket_read(sock, rbuf, sizeof(rbuf));
    alarm(0);
    if(rc == -1) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %m");
#endif
        return AUTH_NG;
    }

    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): read (response): %s",rbuf);
    if(strncmp(rbuf, "235 ", sizeof("235 ")-1) != 0) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): auth failure.");
#endif
        return AUTH_NG;
    }
    return AUTH_OK;
}


int
auth_digest_md5(socket_t *sock, config_t *cfg) {

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): AUTH DIGEST-MD5");
#endif
    syslog(LOG_WARNING, "pam_smtpauth(smtpauth): auth_digest_md5 is not implemented.");
    return AUTH_NG;
}

#ifdef USE_SSL
SSL_CTX *SSLContext = 0;

#ifdef VERYIFY_CERT
static int
verify_cert(SSL *ssl) {

    X509 *cert;
    int err;
    char buf[256];
    int ret = -1;
    BIO *bio;

    cert = SSL_get_peer_certificate(ssl);
    if(!cert) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): Error: no server certificate.");
#endif
        return -1;
    }

    err = SSL_get_verify_result(ssl);
    if(err == X509_V_OK) {
        return 0;
    }

#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): Error:  can't verify certificate: %s (%d).",
             X509_verify_cert_error_string(err), err);
#endif
    X509_NAME_oneline(X509_get_subject_name(cert), buf, sizeof(buf));
    fprintf(stderr,"\nSubject: %s\n", buf);
    X509_NAME_oneline(X509_get_issuer_name(cert), buf, sizeof(buf));
    fprintf(stderr,"Issuer:  %s\n", buf);
    bio = BIO_new(BIO_s_mem());
    ASN1_TIME_print(bio, X509_get_notBefore(cert));
    memset(buf, 0, sizeof(buf));
    BIO_read(bio, buf, sizeof(buf) - 1);
    fprintf(stderr,"Valid from: %s\n", buf);
    ASN1_TIME_print(bio, X509_get_notAfter(cert));
    memset(buf, 0, sizeof(buf));
    BIO_read(bio, buf, sizeof(buf) - 1);
    BIO_free(bio);
    fprintf(stderr,"      to:   %s\n", buf);

    fprintf(stderr, 
        "\nThere is no way to verify this certificate.\n"
         " It is possible that a hostile attacker has replaced the server certificate.\n"
         " Continue at your own risk!\n"
         "\nAccept this certificate anyway? [no]: ");
    if(fgets(buf, sizeof(buf), stdin) && (buf[0] == 'y' || buf[0] == 'Y')) {
        ret = 0;
        fprintf(stderr, "\nFine, but don't say I didn't warn you!\n\n");
    }
    return ret;
}
#endif

static int
init_ssl(config_t *conf) {

    SSL_METHOD *method;
    int options = 0;

    if(!conf->certfile) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): Error: SSLCertificateFile not defined.");
#endif
        return -1;
    }
    SSL_load_error_strings();
    SSL_library_init();
    if(conf->use_tlsv1 && !conf->use_sslv2 && !conf->use_sslv3)
        method = TLSv1_client_method();
    else
        method = SSLv23_client_method();

    SSLContext = SSL_CTX_new(method);

    if(access(conf->certfile, F_OK)) {
        if(errno != ENOENT) {
#ifdef DEBUG
            syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): Error: SSLCertificateFile is not accessible.");
#endif
            return -1;
        }
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): Warning: SSLCertificateFile doesn't exist, can't verify server certificates.");
#endif
    } else if(!SSL_CTX_load_verify_locations(SSLContext, conf->certfile, NULL)) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): Error: SSL_CTX_load_verify_locations: %s.",ERR_error_string(ERR_get_error(), 0));
#endif
        SSL_CTX_free(SSLContext);
        return -1;
    }

    if(!conf->use_sslv2) {
        options |= SSL_OP_NO_SSLv2;
    }
    if(!conf->use_sslv3) {
        options |= SSL_OP_NO_SSLv3;
    }
    if(!conf->use_tlsv1) {
        options |= SSL_OP_NO_TLSv1;
    }

    SSL_CTX_set_options(SSLContext, options);

    /* we check the result of the verification after SSL_connect() */
    SSL_CTX_set_verify(SSLContext, SSL_VERIFY_NONE, 0);
    return 0;
}

int
start_tls(smtp_t *smtp, config_t *cfg) {

    int ret;
    /* initialize SSL */
    if(init_ssl(cfg)) {
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): failed to initialize ssl session.");
#endif
        return 1;
    }

    smtp->sock->ssl = SSL_new(SSLContext);
    SSL_set_fd(smtp->sock->ssl, smtp->sock->fd);
    if((ret = SSL_connect(smtp->sock->ssl)) <= 0) {
        socket_perror("connect", smtp->sock, ret);
#ifdef DEBUG
        syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): failed to connect ssl session.");
#endif
        return 1;
    }
#ifdef VERIFY_CERT
    /* verify the server certificate */
    if(verify_cert(smtp->sock->ssl)) {
        return 1;
    }
#endif
    smtp->sock->use_ssl = 1;
#ifdef DEBUG
    syslog(LOG_DEBUG, "pam_smtpauth(smtpauth): SSL support enabled.");
#endif
    return 0;
}
#endif

