#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <sys/types.h>

#ifndef _WIN32
#include <unistd.h>
#else
#include <windows.h>
#include <process.h>
#endif

#ifdef MPI_USE
#include <mpi.h>
#endif

#define _MAIN_DEF

#include "Imputation.h"


/** MPIの初期化処理
 *  MPI用の引数が処理されて、削除される。
 *  @param argc [IN/OUT] 引数の数
 *  @param argv [IN/OUT] 引数
 */
void setupMPI(int *argc, char ***argv)
{
#ifdef MPI_USE
    MPI_Init(argc, argv);
    MPI_Comm_size(MPI_COMM_WORLD, &MyMpiSize);
    MPI_Comm_rank(MPI_COMM_WORLD, &MyMpiRank);
#else
    MyMpiSize = 1;
    MyMpiRank = 0;
#endif /* MPI_USE */
}


/****************************************************************/
/*                                                              */
/* メイン関数                                                   */
/*                                                              */
/****************************************************************/
int main(int argc, char* argv[])
{
    int i = 0;
    InputImputation inputImputation={"", "", "", "", "", "", 0, 0};

    /* MPIによる初期化処理 */
    setupMPI(&argc, &argv);

#ifdef MPI_USE
    /* 時間計測用変数の初期化 */
    for (i = 0; i < NUM_TM; i++){
        tm[i] = 0.0;
    }

    /* 時間計測 */
    tm[TM_START] = MPI_Wtime();
#endif /* MPI_USE */

    if(argc != 9) {
        printf("[usage]imputation.exe [HapmapDataFile] [PhasedHapmapDataFile] [LDblockFile] [RecombinationFile] [ErrorRateFile] [OutputFile] [SamplePerDivision] [EstimateSampleNum]\n");
        return 255;
    }

    /* 引数の値を取得 */
    strcpy(inputImputation.hapmapDataFile,      argv[1]);
    strcpy(inputImputation.phasedHapmapDataFile,argv[2]);
    strcpy(inputImputation.ldBlockFile,         argv[3]);
    strcpy(inputImputation.recombinationFile,   argv[4]);
    strcpy(inputImputation.errorRateFile,       argv[5]);
    strcpy(inputImputation.outputFile,          argv[6]);
    inputImputation.samplePerDivision = atol(argv[7]);
    inputImputation.estimateSampleNum = atol(argv[8]);

    /* 中間データファイル名を決定 */
    sprintf(inputImputation.simpleDataFile, "%s.dat", inputImputation.hapmapDataFile);
    sprintf(inputImputation.pedigreeFile, "%s.ped", inputImputation.hapmapDataFile);
    sprintf(inputImputation.snpListFile, "%s.snps", inputImputation.phasedHapmapDataFile);
    sprintf(inputImputation.phasedHaplotypeFile, "%s.haplos", inputImputation.phasedHapmapDataFile);

    /* Imputationを実行 */
    imputation(&inputImputation);

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_END] = MPI_Wtime();


    printf("%4d\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\n",
           MyMpiRank,
           tm[TM_READ]        - tm[TM_START],       /* 初期化 */
           tm[TM_TRANS]       - tm[TM_READ],        /* データ読み込み */
           tm[TM_EST_SPLIT]   - tm[TM_TRANS],       /* 入力データフォーマット変換 */
           tm[TM_EST_BCAST]   - tm[TM_EST_SPLIT],   /* LDブロックデータ分割 */
           tm[TM_ESTIMATE]    - tm[TM_EST_BCAST],   /* ブロードキャスト */
           tm[TM_EST_Barrier] - tm[TM_ESTIMATE],    /* パラメータ推定 */
           tm[TM_EST_MERGE]   - tm[TM_EST_Barrier], /* 他ランクのパラメータ推定待ち */
           tm[TM_SPLIT]       - tm[TM_EST_MERGE],   /* 推定パラメータ結合 */
           tm[TM_BCAST]       - tm[TM_SPLIT],       /* Pedデータ分割 */
           tm[TM_MACH]        - tm[TM_BCAST],       /* ブロードキャスト */
           tm[TM_Barrier]     - tm[TM_MACH],        /* Imputation実行 */
           tm[TM_MERGE]       - tm[TM_Barrier],     /* 他ランクのImputation待ち */
           tm[TM_END]         - tm[TM_MERGE],       /* Imputation結果結合 */
           tm[TM_END]         - tm[TM_START]);      /* 全処理時間 */

    /* MPI終了処理 */
    MPI_Finalize();

#endif /* MPI_USE */

    return 0;
}


/****************************************************************/
/*                                                              */
/* Imputationを実行                                             */
/*                                                              */
/****************************************************************/
void imputation(InputImputation* inputImputation)
{
    int retval = 0;

    long hapmapFileLine = 0;        /*  */
    long phasedFileLine = 0;        /*  */
    long impHapmapSnpNum = 0;       /*  */
    long impPhasedSnpNum = 0;       /*  */
    long impPhasedStartNum = 0;     /*  */
    long pedigreeFileLine = 0;      /* pedigreeFileライン数 */
    long divideNum = 0;             /* pedigreeFile分割数 */
    long hapmapSampleNum = 0;       /* HapmapDataのサンプルデータ人数 */
    long ldBlockNum = 0;            /* LDブロック数 */
    long* ldBlock = NULL;           /* LDブロック境界格納用 */
    pid_t pid = 0;                  /* プロセスID */

    FILE *fpHapmap = NULL;          /* 入力のHapmapDataファイルポインタ */
    FILE *fpPhased = NULL;          /* 入力のPhasedHapmapDataファイルポインタ */
    FILE *erate = NULL;             /* 入力のerateファイルポインタ */

    SnpData *hapmapTmpSnpData = NULL;   /* HapmapData一時格納用構造体 */
    SnpData *hapmapSnpData = NULL;      /* Imputation用HapmapData格納用構造体 */
    SnpData *phasedSnpData = NULL;      /* Imputation用PhasedHapmapData格納用構造体 */

/****************************************************************/
/* 入力データ読み込み                                           */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_READ] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* ファイルオープン */
        retval = InputFileOpen(&fpHapmap, inputImputation->hapmapDataFile);
        if (retval != 0) {
            goto finalize;
        }
        retval = InputFileOpen(&fpPhased, inputImputation->phasedHapmapDataFile);
        if (retval != 0) {
            goto finalize;
        }

        /* LDブロックを読み込み、配列の格納する */
        retval = setLDBlock(inputImputation, &ldBlockNum, &ldBlock);
        if (retval != 0) {
            goto finalize;
        }

        /* HapmapDataFileを読み込む */
        /* 入力ファイルのライン数を取得 */
        hapmapFileLine = DataReaderCountFileLine(fpHapmap);
        /* ファイルポインタを先頭に戻す */
        fseek(fpHapmap, 0L, SEEK_SET);
        /* データ格納用構造体のメモリ確保 */
        hapmapTmpSnpData = (SnpData*)malloc1Dim(sizeof(SnpData), hapmapFileLine);
        if (NULL == hapmapTmpSnpData) { goto finalize; }
        /* データをファイルから読み込み構造体に収める */
        DataReaderSetAllData(fpHapmap, hapmapTmpSnpData, hapmapFileLine, TYPE_HAPMAP);

        /* PhasedHapmapDataFileを読み込む */
        /* 入力ファイルのライン数を取得 */
        phasedFileLine = DataReaderCountFileLine(fpPhased);
        /* ファイルポインタを先頭に戻す */
        fseek(fpPhased, 0L, SEEK_SET);
        /* データ格納用構造体のメモリ確保 */
        phasedSnpData = (SnpData*)malloc1Dim(sizeof(SnpData), phasedFileLine);
        if (NULL == phasedSnpData) { goto finalize; }
        /* データをファイルから読み込み構造体に収める */
        DataReaderSetAllData(fpPhased, phasedSnpData, phasedFileLine, TYPE_PHASED);
        /* 入力LDブロック範囲内の最初のSnp位置を取得する */
        impPhasedStartNum = getFirstSnpNum(phasedSnpData, phasedFileLine, ldBlock, ldBlockNum);

        /* ファイルクローズ */
        FileClose(fpHapmap);  fpHapmap = NULL;
        FileClose(fpPhased);  fpPhased = NULL;

        /* 入力データの整合性をチェックしてImputation用データを作成する */
        /* 新たに作成するデータはhapmapSnpDataのみであることに注意する */
        impHapmapSnpNum = MakeImputationData(hapmapTmpSnpData, phasedSnpData, hapmapFileLine, phasedFileLine, &hapmapSnpData, ldBlock, ldBlockNum);

        /* これ以降不要な確保したメモリを開放する */
        DataReaderSnpDataMemoryFree(hapmapTmpSnpData, hapmapFileLine); hapmapTmpSnpData = NULL;

        /* サンプルデータ人数を取得（遺伝子データ数でないことに注意） */
        hapmapSampleNum = hapmapSnpData[1].sampleNum;

        /* PhasedHapmapDataは染色体情報を持っていないので、HapmapDataから取得する */
        strcpy( phasedSnpData[1].chrom, hapmapSnpData[1].chrom );

    } /* rank0のみ実行ここまで */

/****************************************************************/
/* 入力データをImputation処理用にフォーマット変換する           */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_TRANS] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* フォーマット変換実施 */
        retval = convertToImputation(inputImputation, hapmapSnpData, phasedSnpData, impHapmapSnpNum, phasedFileLine, ldBlock, ldBlockNum);
        if (retval != 0) { goto finalize; }
    } /* rank0のみ実行ここまで */

/****************************************************************/
/* 入力データ分割処理（LDブロックで分割）                       */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_EST_SPLIT] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* ランク０のプロセスIDを取得 */
        pid = getpid();

        /* パラメータ推定サンプル数が1以上の場合、処理を実施する */
        if (inputImputation->estimateSampleNum > 0) {
            /* パラメータ推定用に入力データをLDブロックで分割する */
            retval = divideInputDataForEstimate(inputImputation, hapmapSnpData, phasedSnpData, impHapmapSnpNum, phasedFileLine, pid, ldBlock, ldBlockNum);
            if (retval != 0) {
                goto finalize;
            }
        }

        /* これ以降不要な確保したメモリを開放する */
        DataReaderSnpDataMemoryFree(hapmapSnpData, impHapmapSnpNum); hapmapSnpData = NULL;
        //DataReaderSnpDataMemoryFree(phasedSnpData, phasedFileLine); phasedSnpData = NULL;

    } /* rank0のみ実行ここまで */

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_EST_BCAST] = MPI_Wtime();

    /* LDブロック数の送信 */
    MPI_Bcast(&ldBlockNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);
    /* プロセスIDの送信 */
    MPI_Bcast(&pid, 1, MPI_INT, 0, MPI_COMM_WORLD);

    if (MyMpiSize > ldBlockNum) {
        /* rank0のみ実行 */
        if (MyMpiRank == 0) {
            fprintf(stderr, "プロセッサ数(%d)が分割数(%ld)よりも多く設定されています.\n", MyMpiSize, ldBlockNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }
#endif /* MPI_USE */

/****************************************************************/
/* パラメータ推定                                               */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_ESTIMATE] = MPI_Wtime();
#endif /* MPI_USE */

    /* パラメータ推定サンプル数が1以上の場合、推定を実施する */
    if (inputImputation->estimateSampleNum > 0) {
        /* パラメータを推定する */
        retval = estimateParameter(inputImputation, ldBlockNum, pid);
        if (retval != 0) {
            goto finalize;
        }
    }

/****************************************************************/
/* 推定パラメータ結合（LDブロック毎の推定結果を結合）           */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_EST_Barrier] = MPI_Wtime();

    /* 全rankのImputation処理が終了するまで待機 */
    MPI_Barrier(MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_EST_MERGE] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* パラメータ推定サンプル数が1以上の場合、処理を実施する */
        if (inputImputation->estimateSampleNum > 0) {
            /* パラメータ推定結果を統合する */
            retval = margeEstimateResult(inputImputation, ldBlockNum, pid, &impPhasedSnpNum);
            if (retval != 0) {
                goto finalize;
            }
        }
        else {
            /* impPhasedSnpNumを決定する */
            /* ファイルオープン */
            retval = InputFileOpen(&erate, inputImputation->errorRateFile);
            if (retval != 0) {
                goto finalize;
            }
            /* 入力ファイルのライン数を取得 */
            impPhasedSnpNum = DataReaderCountFileLine(erate) - 1;
            /* ファイルクローズ */
            FileClose(erate);  erate = NULL;
        }
    } /* rank0のみ実行ここまで */

/****************************************************************/
/* Pedigreeデータ分割処理                                       */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_SPLIT] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* Pedigreeデータを分割する */
        retval = dividePedigree(inputImputation, &divideNum, pid);
        if (retval != 0) {
            goto finalize;
        }
    } /* rank0のみ実行ここまで */

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_BCAST] = MPI_Wtime();

    /* 分割数の送信 */
    MPI_Bcast(&divideNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);
    /* プロセスIDの送信 */
    //MPI_Bcast(&pid, 1, MPI_INT, 0, MPI_COMM_WORLD);

    if (MyMpiSize > divideNum) {
        /* rank0のみ実行 */
        if (MyMpiRank == 0) {
            fprintf(stderr, "プロセッサ数(%d)が分割数(%ld)よりも多く設定されています.\n", MyMpiSize, divideNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }
#endif /* MPI_USE */

/****************************************************************/
/* Imputation処理                                               */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_MACH] = MPI_Wtime();
#endif /* MPI_USE */

    /* Imputation処理を実行する */
    retval = executeImputation(inputImputation, divideNum, pid);
    if (retval != 0) {
        goto finalize;
    }

/****************************************************************/
/* Imputation結果統合処理                                       */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_Barrier] = MPI_Wtime();

    /* 全rankのImputation処理が終了するまで待機 */
    MPI_Barrier(MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_MERGE] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* Genotypeデータを統合する */
        retval = margeGenotype(inputImputation, divideNum, pid, phasedSnpData, impPhasedSnpNum, impPhasedStartNum, hapmapSampleNum);
        if (retval != 0) {
            goto finalize;
        }
    } /* rank0のみ実行ここまで */

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {

        /* ファイルクローズ */
        FileClose(fpHapmap);
        FileClose(fpPhased);

        /* 確保したメモリを開放する */
        DataReaderSnpDataMemoryFree(hapmapSnpData, impHapmapSnpNum);
        DataReaderSnpDataMemoryFree(phasedSnpData, phasedFileLine);
        free1Dim(ldBlock);

        /* 中間ファイルを削除する */

    } /* rank0のみ実行ここまで */

    return;
}


/****************************************************************/
/*                                                              */
/* LDブロックを読み込み、配列に格納する                         */
/*                                                              */
/****************************************************************/
int setLDBlock(
    InputImputation* inputImputation,
    long*            ldBlockNum,
    long**           ldBlock)
{
    int retval = 0;
    long ldBlockFileLine = 0;   /* LDブロックファイルのライン数 */

    FILE *fpLD = NULL;    /* LDブロックファイルポインタ */

    /* ファイルオープン */
    retval = InputFileOpen(&fpLD, inputImputation->ldBlockFile);
    if (retval != 0) {
        goto finalize;
    }

    /* LDブロックファイルのライン数を取得 */
    ldBlockFileLine = DataReaderCountFileLine(fpLD);
    /* LDブロック格納用配列のメモリ確保 */
    *ldBlock = (long*)malloc1Dim(sizeof(long), ldBlockFileLine);
    if (NULL == ldBlock) { goto finalize; }
    /* ファイルポインタを先頭に戻す */
    fseek(fpLD, 0L, SEEK_SET);
    /* LDブロックを配列に収める */
    DataReaderSetLDBlock(fpLD, *ldBlock);
    /* 入力LDブロック数 */
    *ldBlockNum = ldBlockFileLine - 1;

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpLD);

    return retval;
}


/****************************************************************/
/*                                                              */
/* 入力データをImputation処理用にフォーマット変換する           */
/*                                                              */
/****************************************************************/
int convertToImputation(
    InputImputation*    inputImputation,
    SnpData*            hapmapSnpData,
    SnpData*            phasedSnpData,
    long                impHapmapSnpNum,
    long                impPhasedSnpNum,
    long*               ldBlock,
    long                ldBlockNum)
{
    int retval = 0;
    long i = 0;
    long j = 0;
    long startPos = 0;
    long endPos = 0;
    long hapmapSampleDataNum = 0;
    long phasedSampleDataNum = 0;
    long sampleIndex = 0;
    long haplotypeIndex = 0;

    FILE *fpDat = NULL;     /* SimpleDataFileのファイルポインタ */
    FILE *fpPed = NULL;     /* PedigreeFileのファイルポインタ */
    FILE *fpSnps = NULL;    /* SnpListFileのファイルポインタ */
    FILE *fpHaplos = NULL;  /* PhasedHaplotypeFileのファイルポインタ */

    /* ファイルオープン */
    retval = OutputFileOpen(&fpDat, inputImputation->simpleDataFile);
    if (retval != 0) {
        goto finalize;
    }
    retval = OutputFileOpen(&fpPed, inputImputation->pedigreeFile);
    if (retval != 0) {
        goto finalize;
    }
    retval = OutputFileOpen(&fpSnps, inputImputation->snpListFile);
    if (retval != 0) {
        goto finalize;
    }
    retval = OutputFileOpen(&fpHaplos, inputImputation->phasedHaplotypeFile);
    if (retval != 0) {
        goto finalize;
    }

    /* サンプルデータ数の取得 */
    hapmapSampleDataNum = hapmapSnpData[1].sampleDataNum;
    phasedSampleDataNum = phasedSnpData[1].sampleDataNum;

/****************************************************************/
/* 変換出力処理                                                 */
/****************************************************************/

    /* 入力LDブロックで指定された範囲のみ変換出力する */
    startPos = ldBlock[0];
    endPos = ldBlock[ldBlockNum];

    /* SimpleDataFile出力 */
    for (j = 0; j < impHapmapSnpNum; j++) {
        if (startPos <= (hapmapSnpData+j)->pos) {
            if ((hapmapSnpData+j)->pos < endPos) {
                /* SimpleDataFile出力 */
                fprintf(fpDat, "M %s\n", (hapmapSnpData+j)->rsNumber);
            }
        }
    }

    /* PedigreeFile出力 */
    sampleIndex = 0;
    for (i = 0; i < hapmapSampleDataNum; i+=2) {
        sampleIndex++;
        fprintf(fpPed, "Sample%d\tSample%d\t0\t0\t1\t", sampleIndex, sampleIndex);
        for (j = 0; j < impHapmapSnpNum; j++) {
            if (startPos <= (hapmapSnpData+j)->pos) {
                if ((hapmapSnpData+j)->pos < endPos) {
                    /* PedigreeFile出力 */
                    fprintf(fpPed, "%c %c ", (hapmapSnpData+j)->SNPdata[i], (hapmapSnpData+j)->SNPdata[i+1]);
                }
            }
        }
        fprintf(fpPed, "\n");
    }

    /* SnpListFile出力 */
    for (j = 1; j < impPhasedSnpNum; j++) {
        if (startPos <= (phasedSnpData+j)->pos) {
            if ((phasedSnpData+j)->pos < endPos) {
                /* SnpListFile出力 */
                fprintf(fpSnps, "%s\n", (phasedSnpData+j)->rsNumber);
            }
        }
    }

    /* PhasedHaplotypeFile出力 */
    sampleIndex = 0;
    for (i = 0; i < phasedSampleDataNum; i++) {
        if ( (i % 2) == 0 ) {
            sampleIndex++;
            haplotypeIndex = 1;
        }
        else {
            haplotypeIndex = 2;
        }
        fprintf(fpHaplos, "Sample%d->Sample%d HAPLO%d ", sampleIndex, sampleIndex, haplotypeIndex);
        for (j = 1; j < impPhasedSnpNum; j++) {
            if (startPos <= (phasedSnpData+j)->pos) {
                if ((phasedSnpData+j)->pos < endPos) {
                    /* PhasedHaplotypeFile出力 */
                    fprintf(fpHaplos, "%c", (phasedSnpData+j)->SNPdata[i]);
                }
            }
        }
        fprintf(fpHaplos, "\n");
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpDat);
    FileClose(fpPed);
    FileClose(fpSnps);
    FileClose(fpHaplos);
    /* 確保したメモリを開放する */

    return retval;
}


/****************************************************************/
/*                                                              */
/* パラメータ推定用に入力データをLDブロックで分割する           */
/*                                                              */
/****************************************************************/
int divideInputDataForEstimate(
    InputImputation*    inputImputation,
    SnpData*            hapmapSnpData,
    SnpData*            phasedSnpData,
    long                impHapmapSnpNum,
    long                impPhasedSnpNum,
    pid_t               pid,
    long*               ldBlock,
    long                ldBlockNum)
{
    int retval = 0;
    long i = 0;
    long j = 0;
    long k = 0;
    long hapmapSampleNum = 0;
    long subsetSampleNum = 0;
    long startPos = 0;
    long endPos = 0;
    long hapmapSampleDataNum = 0;
    long phasedSampleDataNum = 0;
    long sampleIndex = 0;
    long haplotypeIndex = 0;
    long subsetCount = 0;
    long subsetIndex = 0;
    int* subsetFlag = NULL;

    char tmpDatFileName[MAX_LEN] = "";      /* SimpleDataFileのテンポラリファイル名 */
    char tmpPedFileName[MAX_LEN] = "";      /* PedigreeFileのテンポラリファイル名 */
    char tmpSnpsFileName[MAX_LEN] = "";     /* SnpListFileのテンポラリファイル名 */
    char tmpHaplosFileName[MAX_LEN] = "";   /* PhasedHaplotypeFileのテンポラリファイル名 */

    FILE *fpDat = NULL;     /* SimpleDataFileのファイルポインタ */
    FILE *fpPed = NULL;     /* PedigreeFileのファイルポインタ */
    FILE *fpSnps = NULL;    /* SnpListFileのファイルポインタ */
    FILE *fpHaplos = NULL;  /* PhasedHaplotypeFileのファイルポインタ */

    /* 抽出サンプル数を取得 */
    subsetSampleNum = inputImputation->estimateSampleNum;
    /* HapmapDataのサンプル人数 */
    hapmapSampleNum = hapmapSnpData[1].sampleNum;
    /* サブセット作成用のサンプルを選ぶ */
    if (hapmapSampleNum > subsetSampleNum) {
        /* 乱数の初期化 */
        srand(time(NULL));
        /* サブセット作成用配列のメモリ確保 */
        subsetFlag = (int*)malloc1Dim(sizeof(int), hapmapSampleNum);
        if (NULL == subsetFlag) {
            goto finalize;
        }
        /* サブセット作成用のサンプルを決定 */
        while (subsetCount < subsetSampleNum) {
            /* C言語組み込みのランダム関数 0～Pedigreeデータのサンプル数の間の乱数取得 */
            subsetIndex = (long)(((double)rand() / (RAND_MAX + 1.0) ) * hapmapSampleNum);
            if (subsetFlag[subsetIndex] == 0) {
                subsetFlag[subsetIndex] = 1;
                subsetCount++;
            }
        }
    }

    /* サンプルデータ数の取得 */
    hapmapSampleDataNum = hapmapSnpData[1].sampleDataNum;
    phasedSampleDataNum = phasedSnpData[1].sampleDataNum;

    /* 入力データをLDブロック毎に分割する */
    for (k = 0; k < ldBlockNum; k++) { /* 将来、領域の重複を許す場合も考える */
        startPos = ldBlock[k];
        endPos = ldBlock[k+1];

        /* テンポラリファイル名決定 */
        sprintf(tmpDatFileName, "%s_input%ld_%d", inputImputation->simpleDataFile, k, pid);
        sprintf(tmpPedFileName, "%s_input%ld_%d", inputImputation->pedigreeFile, k, pid);
        sprintf(tmpSnpsFileName, "%s_input%ld_%d", inputImputation->snpListFile, k, pid);
        sprintf(tmpHaplosFileName, "%s_input%ld_%d", inputImputation->phasedHaplotypeFile, k, pid);
        /* ファイルオープン */
        retval = OutputFileOpen(&fpDat, tmpDatFileName);
        if (retval != 0) { goto finalize; }
        retval = OutputFileOpen(&fpPed, tmpPedFileName);
        if (retval != 0) { goto finalize; }
        retval = OutputFileOpen(&fpSnps, tmpSnpsFileName);
        if (retval != 0) { goto finalize; }
        retval = OutputFileOpen(&fpHaplos, tmpHaplosFileName);
        if (retval != 0) { goto finalize; }

        /* SimpleDataFile出力 */
        for (j = 0; j < impHapmapSnpNum; j++) {
            if (startPos <= (hapmapSnpData+j)->pos) {
                if ((hapmapSnpData+j)->pos < endPos) {
                    /* SimpleDataFile出力 */
                    fprintf(fpDat, "M %s\n", (hapmapSnpData+j)->rsNumber);
                }
                /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                else {
                    break;
                }
            }
        }
        /* 次のLDブロックのSNPをSimpleDataFile出力　LDブロック境界のSNPを重複させる */
        //if (k < ldBlockNum-1) {
        //    fprintf(fpDat, "M %s\n", (hapmapSnpData+j)->rsNumber);
        //}

        /* PedigreeFile出力 */
        sampleIndex = 0;
        for (i = 0; i < hapmapSampleDataNum; i+=2) {
            sampleIndex++;
            if (subsetFlag[sampleIndex-1] == 1) {
                fprintf(fpPed, "Sample%d\tSample%d\t0\t0\t1\t", sampleIndex, sampleIndex);
                for (j = 0; j < impHapmapSnpNum; j++) {
                    if (startPos <= (hapmapSnpData+j)->pos) {
                        if ((hapmapSnpData+j)->pos < endPos) {
                            /* PedigreeFile出力 */
                            fprintf(fpPed, "%c %c ", (hapmapSnpData+j)->SNPdata[i], (hapmapSnpData+j)->SNPdata[i+1]);
                        }
                        /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                        else {
                            break;
                        }
                    }
                }
                /* 次のLDブロックのSNPをPedigreeFile出力　LDブロック境界のSNPを重複させる */
                //if (k < ldBlockNum-1) {
                //    fprintf(fpPed, "%c %c ", (hapmapSnpData+j)->SNPdata[i], (hapmapSnpData+j)->SNPdata[i+1]);
                //}
                fprintf(fpPed, "\n");
            }
        }

        /* SnpListFile出力 */
        for (j = 1; j < impPhasedSnpNum; j++) {
            if (startPos <= (phasedSnpData+j)->pos) {
                if ((phasedSnpData+j)->pos < endPos) {
                    /* SnpListFile出力 */
                    fprintf(fpSnps, "%s\n", (phasedSnpData+j)->rsNumber);
                }
                /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                else {
                    break;
                }
            }
        }
        /* 次のLDブロックのSNPをSnpListFile出力　LDブロック境界のSNPを重複させる */
        if (k < ldBlockNum-1) {
            fprintf(fpSnps, "%s\n", (phasedSnpData+j)->rsNumber);
        }

        /* PhasedHaplotypeFile出力 */
        sampleIndex = 0;
        for (i = 0; i < phasedSampleDataNum; i++) {
            if ( (i % 2) == 0 ) {
                sampleIndex++;
                haplotypeIndex = 1;
            }
            else {
                haplotypeIndex = 2;
            }
            fprintf(fpHaplos, "Sample%d->Sample%d HAPLO%d ", sampleIndex, sampleIndex, haplotypeIndex);
            for (j = 1; j < impPhasedSnpNum; j++) {
                if (startPos <= (phasedSnpData+j)->pos) {
                    if ((phasedSnpData+j)->pos < endPos) {
                        /* PhasedHaplotypeFile出力 */
                        fprintf(fpHaplos, "%c", (phasedSnpData+j)->SNPdata[i]);
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
            /* 次のLDブロックのSNPをPhasedHaplotypeFile出力　LDブロック境界のSNPを重複させる */
            if (k < ldBlockNum-1) {
                fprintf(fpHaplos, "%c", (phasedSnpData+j)->SNPdata[i]);
            }
            fprintf(fpHaplos, "\n");
        }

        /* ファイルクローズ */
        FileClose(fpDat);       fpDat = NULL;
        FileClose(fpPed);       fpPed = NULL;
        FileClose(fpSnps);      fpSnps = NULL;
        FileClose(fpHaplos);    fpHaplos = NULL;
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpDat);
    FileClose(fpPed);
    FileClose(fpSnps);
    FileClose(fpHaplos);
    /* 確保したメモリを開放する */

    return retval;
}


/****************************************************************/
/*                                                              */
/* パラメータを推定する                                         */
/*                                                              */
/****************************************************************/
int estimateParameter(
    InputImputation* inputImputation,
    long             divideNum,
    pid_t            pid)
{
    int retval = 0;
    char estimateOutputFileName[MAX_LEN] = "";
    char divideDatFileName[MAX_LEN] = "";
    char dividePedFileName[MAX_LEN] = "";
    char divideSnpsFileName[MAX_LEN] = "";
    char divideHaplosFileName[MAX_LEN] = "";
    char tmpFileName[MAX_LEN] = "";
    char tmpFileName2[MAX_LEN] = "";
    char cmd[CMD_LEN] = "";
    long i = 0;
    long dividePartNum = 0;
    long dividePartStart = 0;

    /* プロセッサごとの担当分割数計算 */
    dividePartNum = divideNum / MyMpiSize;
    if (MyMpiRank < divideNum % MyMpiSize) {
        dividePartNum++;
    }
    /* 各プロセッサにおけるデータのオフセット値の計算 */
    if (MyMpiRank < divideNum % MyMpiSize) {
        dividePartStart = dividePartNum * MyMpiRank;
    }
    else {
        dividePartStart = dividePartNum * MyMpiRank + (divideNum % MyMpiSize);
    }

    /* Pedigreeデータ分割単位でループ */
    for (i = dividePartStart; i < dividePartStart + dividePartNum; i++) {
        /* 分割した入力データファイル名を決定 */
        sprintf(divideDatFileName, "%s_input%ld_%d", inputImputation->simpleDataFile, i, pid);
        sprintf(dividePedFileName, "%s_input%ld_%d", inputImputation->pedigreeFile, i, pid);
        sprintf(divideSnpsFileName, "%s_input%ld_%d", inputImputation->snpListFile, i, pid);
        sprintf(divideHaplosFileName, "%s_input%ld_%d", inputImputation->phasedHaplotypeFile, i, pid);
        /* パラメータ推定時の出力ファイル名セット */
        sprintf(estimateOutputFileName, "%s_estimate%ld_%d", inputImputation->outputFile, i, pid);

        /* MACH（パラメータ推定）処理実行 */
#ifndef _WIN32
        //sprintf(cmd, "mach1 -d %s -p %s -h %s -s %s --rounds 50 --greedy --geno --prefix %s --compact >& /dev/null",
        sprintf(cmd, "mach1 -d %s -p %s -h %s -s %s --rounds 50 --greedy --geno --prefix %s --compact 1> /dev/null 2> /dev/null",
#else
        sprintf(cmd, "D:\\MachTest\\mach1.exe -d %s -p %s -h %s -s %s --rounds 50 --greedy --geno --prefix %s",
#endif
                    divideDatFileName,
                    dividePedFileName,
                    divideHaplosFileName,
                    divideSnpsFileName,
                    estimateOutputFileName);
        system(cmd);
        sprintf(tmpFileName, "%s.info", estimateOutputFileName);
        remove(tmpFileName);
#ifndef _WIN32 
        sprintf(tmpFileName, "%s.geno.gz", estimateOutputFileName);
        remove(tmpFileName);
#else
        sprintf(tmpFileName, "%s.geno", estimateOutputFileName);
        remove(tmpFileName);
#endif

        /* 不要なテンポラリのファイルを削除 */
        remove(divideDatFileName);
        remove(dividePedFileName);
        remove(divideSnpsFileName);
        remove(divideHaplosFileName);

        /* 推定したパラメータファイルをリネーム（指定したファイル名）する */
        sprintf(tmpFileName, "%s.erate", estimateOutputFileName);
        sprintf(tmpFileName2, "%s_%ld_%d", inputImputation->errorRateFile, i, pid);
        remove(inputImputation->errorRateFile);
        rename(tmpFileName, tmpFileName2);
        sprintf(tmpFileName, "%s.rec", estimateOutputFileName);
        sprintf(tmpFileName2, "%s_%ld_%d", inputImputation->recombinationFile, i, pid);
        remove(inputImputation->recombinationFile);
        rename(tmpFileName, tmpFileName2);
    }


/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

//finalize:;
    /* ファイルクローズ */

    return retval;
}


/****************************************************************/
/*                                                              */
/* パラメータ推定結果を結合する                                 */
/*                                                              */
/****************************************************************/
int margeEstimateResult(
    InputImputation* inputImputation,
    long             divideNum,
    pid_t            pid,
    long*            impPhasedSnpNum)
{
    int retval = 0;
    char buf[MAX_LEN] = "";
    char divideErateFileName[MAX_LEN] = "";
    char divideRecFileName[MAX_LEN] = "";
    char eraeteMarker[MAX_LEN] = "";
    char recInterval[MAX_LEN] = "";
    long i = 0;
    long erateIndex = 0;
    long recIndex = 1;
    long recLineNum = 0;
    double erateAve = 0;        /* ErateファイルのAveRate */
    double erateLast = 0;       /* ErateファイルのLastRate */
    double erateAveOld = 0;     /* １つ前のErateファイルのAveRate */
    double erateLastOld = 0;    /* １つ前のErateファイルのLastRate */
    double recAve = 0;          /* RecファイルのAveRate */
    double recLast = 0;         /* RecファイルのLastRate */


    FILE *fpErate = NULL;       /* 出力Erateファイルポインタ */
    FILE *fpRec = NULL;         /* 出力Recファイルポインタ */
    FILE *fpDivErate = NULL;    /* 分割して推定したErateファイルポインタ */
    FILE *fpDivRec = NULL;      /* 分割して推定したRecファイルポインタ */


    /* 出力ファイルオープン */
    retval = OutputFileOpen(&fpErate, inputImputation->errorRateFile);
    if (retval != 0) { goto finalize; }
    retval = OutputFileOpen(&fpRec, inputImputation->recombinationFile);
    if (retval != 0) { goto finalize; }

    /* LDブロック分割単位でループ */
    for (i = 0; i < divideNum; i++){

        /* 分割してパラメータ推定した結果ファイル名決定 */
        sprintf(divideErateFileName, "%s_%ld_%d", inputImputation->errorRateFile, i, pid);
        sprintf(divideRecFileName, "%s_%ld_%d", inputImputation->recombinationFile, i, pid);
        /* 分割してパラメータ推定した結果ファイルオープン */
        retval = InputFileOpen(&fpDivErate, divideErateFileName);
        if (retval != 0) { goto finalize; }
        retval = InputFileOpen(&fpDivRec, divideRecFileName);
        if (retval != 0) { goto finalize; }

        /* 最初のLDブロックのみ先頭行のヘッダをファイル出力し、それ以外の場合ヘッダを捨てる */
        if (i != 0) {
            fgets(buf, MAX_LEN, fpDivErate);
            fgets(buf, MAX_LEN, fpDivRec);
        }
        else {
            /* Eraeteファイルにヘッダ情報出力 */
            fgets(buf, MAX_LEN, fpDivErate);
            fprintf(fpErate, "%s", buf);
            /* Recファイルにヘッダ情報出力 */
            fgets(buf, MAX_LEN, fpDivRec);
            fprintf(fpRec, "%s", buf);

        }

        recLineNum = 0;
        /* Recファイルの結合 まず分割Recファイルを特定文字数ずつ読み込む */
        while ( fgets(buf, MAX_LEN, fpDivRec) != NULL ) {
            /* 各列の要素に分解 */
            sscanf(buf, "%s %lf %lf", recInterval, &recAve, &recLast); 
            /* Recファイルに出力 */
            fprintf(fpRec, "%ld-%ld %.4lf %.4lf\n", recIndex, recIndex+1, recAve, recLast);
            recIndex++;
            recLineNum++;
        }

        erateIndex = 0;
        /* Erateファイルの結合 まず分割Erateファイルを特定文字数ずつ読み込む */
        while ( fgets(buf, MAX_LEN, fpDivErate) != NULL ) {
            /* 各列の要素に分解 */
            sscanf(buf, "%s %lf %lf", eraeteMarker, &erateAve, &erateLast); 
            /* 重複データ（１つまえの分割Erateファイルの最終行データ）を比較し、大きいほうを採用 */
            if ( (erateIndex == 0) && (erateAveOld > erateAve) ) {
                erateAve = erateAveOld;
                erateLast = erateLast;               
            }
            /* 分割Erateファイルの最終行は重複データになるので、ここでは出力しない */
            /* ただし、最後のLDブロックの場合は出力する */
            if ( (erateIndex != recLineNum) || (i == divideNum-1) ) {
                /* Eraeteファイルに出力 */
                fprintf(fpErate, "%s %.4lf %.4lf\n", eraeteMarker, erateAve, erateLast);
                *impPhasedSnpNum += 1;
            }
            erateIndex++;
        }
        /* 分割Erateファイルの最終行のデータを保持 */
        erateAveOld = erateAve;
        erateLastOld = erateLast;


        /* 分割データ出力用のファイルクローズ */
        FileClose(fpDivErate);  fpDivErate = NULL;
        FileClose(fpDivRec);    fpDivRec = NULL;

        /* 不要な中間データの削除 */
        remove(divideErateFileName);
        remove(divideRecFileName);
    }


/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpErate);
    FileClose(fpRec);
    FileClose(fpDivErate);
    FileClose(fpDivRec);

    return retval;
}


/****************************************************************/
/*                                                              */
/* Pedigreeデータを分割する                                     */
/*                                                              */
/****************************************************************/
int dividePedigree(
    InputImputation* inputImputation,
    long*            divideNum,
    pid_t            pid)
{
    int retval = 0;
    char buf[MAX_LEN] = "";
    char divideFileName[MAX_LEN] = "";
    long index = 0;
    long pedFileNum = 0;
    long divideLine = 0;

    FILE *fpIn = NULL;      /* 入力とするPedigreeデータファイルポインタ */
    FILE *fpOut = NULL;     /* 分割後のPedigreeデータ出力ファイルポインタ */

    /* 分割後のサンプル数指定が0以下の場合は分割せず、入力データをそのまま使用 */
    if (inputImputation->samplePerDivision <= 0) {
        /* 分割しないため、分割数は１に設定 */
        *divideNum = 1;
        return retval;
    }

    /* ファイルオープン */
    retval = InputFileOpen(&fpIn, inputImputation->pedigreeFile);
    if (retval != 0) {
        goto finalize;
    }
    /* 最初の分割データ出力用のファイルオープン */
    sprintf(divideFileName, "%s_input%ld_%d", inputImputation->pedigreeFile, pedFileNum, pid);
    retval = OutputFileOpen(&fpOut, divideFileName);
    if (retval != 0) {
        goto finalize;
    }

    /* 特定文字数ずつ読み込む */
    while ( fgets(buf, MAX_LEN, fpIn) != NULL ){
        if (fpOut == NULL) {
            /* 次の分割データ出力用ファイルの番号を更新 */
            pedFileNum++;
            /* 分割後のライン数カウンタをクリア */
            divideLine = 0;
            /* 次の分割データ出力用のファイルオープン */
            sprintf(divideFileName, "%s_input%ld_%d", inputImputation->pedigreeFile, pedFileNum, pid);
            retval = OutputFileOpen(&fpOut, divideFileName);
            if (retval != 0) {
                goto finalize;
            }
        }
        /* 分割データをファイル出力 */
        fprintf(fpOut, "%s", buf);
        /* 改行コードを検査 */
        index = strlen(buf);
        if ('\n' == buf[index-1]) {
            /* 分割後のライン数をカウント */
            divideLine++;
            if (divideLine == inputImputation->samplePerDivision) {
                /* 最初の分割データ出力用のファイルクローズ */
                FileClose(fpOut);
                fpOut = NULL;
            }
        }
    }
    /* 分割数を設定 */
    *divideNum = pedFileNum + 1;

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpIn);
    FileClose(fpOut);

    return retval;
}


/****************************************************************/
/*                                                              */
/* Imputation処理を実行する                                     */
/*                                                              */
/****************************************************************/
int executeImputation(
    InputImputation* inputImputation,
    long             divideNum,
    pid_t            pid)
{
    int retval = 0;
    char divideInputFileName[MAX_LEN] = "";
    char divideOutputFileName[MAX_LEN] = "";
    char tmpFileName[MAX_LEN] = "";
    char cmd[CMD_LEN] = "";
    long i = 0;
    long dividePartNum = 0;
    long dividePartStart = 0;

    /* プロセッサごとの担当分割数計算 */
    dividePartNum = divideNum / MyMpiSize;
    if (MyMpiRank < divideNum % MyMpiSize) {
        dividePartNum++;
    }
    /* 各プロセッサにおけるデータのオフセット値の計算 */
    if (MyMpiRank < divideNum % MyMpiSize) {
        dividePartStart = dividePartNum * MyMpiRank;
    }
    else {
        dividePartStart = dividePartNum * MyMpiRank + (divideNum % MyMpiSize);
    }

    /* Pedigreeデータ分割単位でループ */
    for (i = dividePartStart; i < dividePartStart + dividePartNum; i++) {
        /* 分割した入力データファイル名を決定 */
        if (inputImputation->samplePerDivision > 0) {
            /* 入力とするPedigreeデータ分割ファイル名セット */
            sprintf(divideInputFileName, "%s_input%ld_%d", inputImputation->pedigreeFile, i, pid);
            /* 分割データでの実行結果出力ファイル名セット */
            sprintf(divideOutputFileName, "%s_output%ld_%d", inputImputation->outputFile, i, pid);
        }
        /* 分割後のサンプル数指定が0以下の場合、指定した入力データをそのまま使用 */
        else {
            /* 入力とするPedigreeデータ分割ファイル名セット */
            sprintf(divideInputFileName, "%s", inputImputation->pedigreeFile);
            /* 分割データでの実行結果出力ファイル名セット */
            sprintf(divideOutputFileName, "%s_output%ld_%d", inputImputation->outputFile, i, pid);
        }

        /* MACH処理実行 */
#ifndef _WIN32
        //sprintf(cmd, "mach1 -d %s -p %s -h %s -s %s --crossovermap %s --errormap %s --greedy --mle --mldetails --prefix %s --compact >& /dev/null",
        sprintf(cmd, "mach1 -d %s -p %s -h %s -s %s --crossovermap %s --errormap %s --greedy --mle --mldetails --prefix %s --compact 1> /dev/null 2> /dev/null",
#else
        sprintf(cmd, "D:\\MachTest\\mach1.exe -d %s -p %s -h %s -s %s --crossovermap %s --errormap %s --greedy --mle --mldetails --prefix %s",
#endif
                      inputImputation->simpleDataFile,
                      divideInputFileName,
                      inputImputation->phasedHaplotypeFile,
                      inputImputation->snpListFile,
                      inputImputation->recombinationFile,
                      inputImputation->errorRateFile,
                      divideOutputFileName);
        system(cmd);
        /* 不要な中間データの削除 */
        /* 入力データファイルは分割したものだけ削除する */
        if (inputImputation->samplePerDivision > 0) {
            remove(divideInputFileName);
        }

        sprintf(tmpFileName, "%s.erate", divideOutputFileName);
        remove(tmpFileName);
        sprintf(tmpFileName, "%s.rec", divideOutputFileName);
        remove(tmpFileName);
        sprintf(tmpFileName, "%s.mlinfo", divideOutputFileName);
        remove(tmpFileName);
#ifndef _WIN32
        sprintf(tmpFileName, "%s.mldose.gz", divideOutputFileName);
        remove(tmpFileName);
        sprintf(tmpFileName, "%s.mlprob.gz", divideOutputFileName);
        remove(tmpFileName);
        sprintf(tmpFileName, "%s.mlqc.gz", divideOutputFileName);
        remove(tmpFileName);
#else
        sprintf(tmpFileName, "%s.mldose", divideOutputFileName);
        remove(tmpFileName);
        sprintf(tmpFileName, "%s.mlprob", divideOutputFileName);
        remove(tmpFileName);
        sprintf(tmpFileName, "%s.mlqc", divideOutputFileName);
        remove(tmpFileName);
#endif

    }

    return retval;
}


/****************************************************************/
/*                                                              */
/* Genotypeデータを統合する                                     */
/*                                                              */
/****************************************************************/
int margeGenotype(
    InputImputation* inputImputation,
    long             divideNum,
    pid_t            pid,
    SnpData*         phasedSnpData,
    long             impPhasedSnpNum,
    long             impPhasedStartNum,
    long             hapmapSampleNum)
{
    int retval = 0;
    char buf[MAX_LEN] = "";
    char divideFileName[MAX_LEN];
    char tmpFileName[MAX_LEN] = "";
    char cmd[CMD_LEN] = "";
    long genoFileNum = 0;
    long i = 0;
    long j = 0;
    long maxIndex = 0;
    long sampleIndex = 0;
    long snpIndex = 0;
    long column = 1;
    char** snpsData = NULL; /* Imputation結果のSNPデータ格納用 */

    FILE *fpIn = NULL;      /* Genotypeデータ分割ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */
    FILE *fpHapmap = NULL;  /* 入力HapmapDataファイルポインタ */


    /* 出力ファイルオープン */
    retval = OutputFileOpen(&fpOut, inputImputation->outputFile);
    if (retval != 0) {
        goto finalize;
    }

    /* Imputation結果のSNPデータ格納用のメモリ確保　１座位に２つの遺伝子データがあることに注意 */
    snpsData = (char**)mallocChar2Dim(hapmapSampleNum, impPhasedSnpNum*2);
    if (NULL == snpsData) { goto finalize; }

    /* Genotypeデータ分割単位でループ */
    for (i = 0; i < divideNum; i++){

#ifndef _WIN32
        /* Linux/Unix版の場合、Machの出力がgz形式なので解凍する */
        sprintf(cmd, "gzip -d %s_output%ld_%d.mlgeno.gz", inputImputation->outputFile, i, pid);
        system(cmd);
#endif

        /* Genotypeデータ分割ファイルオープン */
        sprintf(divideFileName, "%s_output%ld_%d.mlgeno", inputImputation->outputFile, i, pid);
        retval = InputFileOpen(&fpIn, divideFileName);
        if (retval != 0) {
            goto finalize;
        }

        /* Genotypeデータの遺伝子データをメモリ上に保持する */
        /* 特定文字数ずつ読み込む */
        while ( fgets(buf, MAX_LEN, fpIn) != NULL ) {
            maxIndex = strlen(buf);
            j = 0;
            /* １行読み込んだら次の行読み込み */
            for (j = 0; j <= maxIndex; j++){
                if (isspace(buf[j]) != 0){
                    column++;
                }
                else if (column >= 3) {
                    if (isalpha(buf[j]) != 0){
                        /* 遺伝子データを格納 */
                        snpsData[sampleIndex][snpIndex] = buf[j];
                        snpIndex++;
                    }
                }
                if ('\n' == buf[j]){
                    column = 1;
                    sampleIndex++;
                    snpIndex = 0;
                    break;
                }
            }
        }

        /* 分割データ出力用のファイルクローズ */
        FileClose(fpIn);
        fpIn = NULL;
        /* 不要な中間データの削除 */
        remove(divideFileName);
    }

    /* 入力HapmapDataファイルオープン */
    retval = InputFileOpen(&fpHapmap, inputImputation->hapmapDataFile);
    if (retval != 0) {
        goto finalize;
    }
    /* HapmapDataのヘッダ部をそのままコピー */
    while ( fgets(buf, MAX_LEN, fpHapmap) != NULL ) {
        /* ファイル出力 */
        fprintf(fpOut, "%s", buf);
        maxIndex = strlen(buf);
        if ('\n' == buf[maxIndex-1]) {
            break;
        }
    }
    /* ファイルクローズ */
    FileClose(fpHapmap);

    /* GenotypeデータをHapmapData形式に変換して出力 */
    for (snpIndex = 0; snpIndex < impPhasedSnpNum; snpIndex++) {
        fprintf(fpOut, "%s %s %s %ld + 0 0 0 0 0 0 ", 
                    phasedSnpData[impPhasedStartNum].rsNumber,
                    phasedSnpData[impPhasedStartNum].alleles,
                    phasedSnpData[1].chrom,
                    phasedSnpData[impPhasedStartNum].pos);
        impPhasedStartNum++;
        for (sampleIndex = 0; sampleIndex < hapmapSampleNum; sampleIndex++) {
            fprintf(fpOut, " %c%c", snpsData[sampleIndex][2*snpIndex], snpsData[sampleIndex][2*snpIndex+1]);
        }
        fprintf(fpOut, "\n");
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpIn);
    FileClose(fpOut);
    /* 確保したメモリを開放する */
    freeChar2Dim(snpsData, hapmapSampleNum);

    return retval;
}


/****************************************************************/    
/* 入力データの整合性をチェックしてImputation用データを作成する */
/****************************************************************/
long MakeImputationData(
    SnpData *hapmapTmpSnpData,
    SnpData *phasedSnpData,
    long hapmapFileLine,
    long phasedFileLine,
    SnpData **hapmapSnpData,
    long*               ldBlock,
    long                ldBlockNum)
{
    int retval = 0;
    long startPos = 0;
    long endPos = 0;
    long hapmapSampleDataNum = 0;
    long index = 0;
    long dataCount = 0;
    long hapmapIndex = 1;
    long phasedIndex = 1;

    /* 入力LDブロックで指定された範囲のみ変換出力する */
    startPos = ldBlock[0];
    endPos = ldBlock[ldBlockNum];

    /* サンプルデータ数を取得 */
    hapmapSampleDataNum = hapmapTmpSnpData[1].sampleDataNum;

    while ( hapmapIndex < hapmapFileLine || phasedIndex < phasedFileLine ){
        /* データのposを比較し、同じ箇所の遺伝子であることを確認 */
        if (hapmapTmpSnpData[hapmapIndex].pos == phasedSnpData[phasedIndex].pos){
            /* 入力LDブロックで指定された範囲を超えたらループを抜ける */
            if (hapmapTmpSnpData[hapmapIndex].pos > endPos) {
                break;
            }
            /* 入力LDブロックで指定された範囲のデータ数をカウント */
            else if (startPos <= hapmapTmpSnpData[hapmapIndex].pos) {
                /* Data数をカウント */
                dataCount++;
            }

            /* hapmapTmpSnpDataの次のデータを参照する */
            hapmapIndex++;
            /* phasedTmpSnpDataの次のデータを参照する */
            phasedIndex++;
        }
        else if (hapmapTmpSnpData[hapmapIndex].pos < phasedSnpData[phasedIndex].pos){
            /* hapmapSnpDataの次のデータを参照する */
            hapmapIndex++;
        }
        else if (hapmapTmpSnpData[hapmapIndex].pos > phasedSnpData[phasedIndex].pos){
            /* phasedSnpDataの次のデータを参照する */
            phasedIndex++;
        }
    }

    /* データ格納用構造体のメモリ確保 */
    *hapmapSnpData = (SnpData*)malloc1Dim(sizeof(SnpData), dataCount);

    hapmapIndex = 1;
    phasedIndex = 1;
    while ( hapmapIndex < hapmapFileLine || phasedIndex < phasedFileLine ){
        /* データのposを比較し、同じ箇所の遺伝子であることを確認 */
        if (hapmapTmpSnpData[hapmapIndex].pos == phasedSnpData[phasedIndex].pos){
            /* 入力LDブロックで指定された範囲を超えたらループを抜ける */
            if (hapmapTmpSnpData[hapmapIndex].pos > endPos) {
                break;
            }
            /* 入力LDブロックで指定された範囲のデータ数を保持する */
            else if (startPos <= hapmapTmpSnpData[hapmapIndex].pos) {
                /* Imputation用データの作成 */
                /* SNPデータ格納用配列のメモリ確保 */
                (*hapmapSnpData)[index].SNPdata = (char*)malloc1Dim(sizeof(char), hapmapSampleDataNum);
                /* HapmapData */
                strcpy( (*hapmapSnpData)[index].rsNumber, hapmapTmpSnpData[hapmapIndex].rsNumber );
                strcpy( (*hapmapSnpData)[index].alleles, hapmapTmpSnpData[hapmapIndex].alleles );
                strcpy( (*hapmapSnpData)[index].chrom, hapmapTmpSnpData[hapmapIndex].chrom );
                (*hapmapSnpData)[index].pos = hapmapTmpSnpData[hapmapIndex].pos;
                strncpy( (*hapmapSnpData)[index].SNPdata, hapmapTmpSnpData[hapmapIndex].SNPdata, hapmapSampleDataNum);
                (*hapmapSnpData)[index].sampleDataNum = hapmapTmpSnpData[hapmapIndex].sampleDataNum;
                (*hapmapSnpData)[index].sampleNum = hapmapTmpSnpData[hapmapIndex].sampleNum;
                index++;
            }

            /* hapmapTmpSnpDataの次のデータを参照する */
            hapmapIndex++;
            /* phasedTmpSnpDataの次のデータを参照する */
            phasedIndex++;
        }
        else if (hapmapTmpSnpData[hapmapIndex].pos < phasedSnpData[phasedIndex].pos){
            /* hapmapSnpData(case)の次のデータを参照する */
            hapmapIndex++;
        }
        else if (hapmapTmpSnpData[hapmapIndex].pos > phasedSnpData[phasedIndex].pos){
            /* phasedSnpData(control)の次のデータを参照する */
            phasedIndex++;
        }
    }
    
    return dataCount;
}

/****************************************************************/
/*                                                              */
/* 入力LDブロック範囲内の最初のSnp位置を取得する                */
/*                                                              */
/****************************************************************/
int getFirstSnpNum(
    SnpData*            snpData,
    long                snpNum,
    long*               ldBlock,
    long                ldBlockNum)
{
    int retval = 0;
    long j = 0;
    long startPos = 0;
    long endPos = 0;
    long snpStartNum = 0;

    /* 入力LDブロックで指定された範囲 */
    startPos = ldBlock[0];
    endPos = ldBlock[ldBlockNum];


    /* 入力LDブロック範囲内の最初のSnp位置を取得 */
    for (j = 1; j < snpNum; j++) {
        if (startPos <= (snpData+j)->pos) {
            if ((snpData+j)->pos < endPos) {
                snpStartNum = j;
                break;
            }
        }
    }

    return snpStartNum;
}
