#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "snp_CalcScore.h"

int main(int argc, char* argv[])
{
    InputScore inputScore={"", "", "", 0, "", 0, 0};

    if(argc != 8){
        printf("[usage]main.exe [InputFile1] [InputFile2] [OutputFile] [AreaFileType] [BlockAreaFile] [Score] [dataType] \n");
        return 255;
    }
    strcpy(inputScore.inputFile1, argv[1]);
    strcpy(inputScore.inputFile2, argv[2]);
    strcpy(inputScore.outputFile1, argv[3]);
    inputScore.areaFileType = atoi(argv[4]);
    strcpy(inputScore.blockAreaFile, argv[5]);
    inputScore.score = atoi(argv[6]);
    inputScore.dataType = atoi(argv[7]);

    /* スコア計算方法の指定 */
    iWay = inputScore.score;

    /* SNPのScore計算を行う */
    MainProgramScore(&inputScore);

}


/* ScoreのScore計算を行う */
void MainProgramScore(InputScore *inputScore)
{
    int retval = 0;
    int flag = 0;
    long i = 0;
	long j = 0;
    long index = 0;
    long fileLine1 = 0; /* 入力ファイル（case）のライン数 */
    long fileLine2 = 0; /* 入力ファイル（contorl）のライン数 */
    long areaFileLine = 0; /* haplotypeブロック領域指定ファイルのライン数 */
    long a = 0;         /* number of haplotype copies (=sequences) in case */
    long b = 0;         /* number of haplotype copies (=sequences) in control */
    long n = 0;
    long dataNum;       /* case、controlデータ整合後の総SNP数 */
    long jStart = 0;    /* haplotypeブロックの最初のSNPを示す座位 */
    long jEnd = 0;      /* haplotypeブロックの最後のSNPを示す座位 */
    long blockNum = 0;  /* haplotypeブロック数 */
    long startPos = 0;
    long endPos = 0;

    int **T = NULL;             /* 偶現表 */
    int *populationType = NULL;
    long *blockArea = NULL;     /* 各haplotypeブロック領域格納 */
    long *linkSNPNum = NULL;    /* 各haplotypeブロックのSNP数 */
    long *linkSNPStart = NULL;  /* 各haplotypeブロックの最初のSNPを示す座位 */
    double *Sobs = NULL;        /* 各SNPのスコア */

    FILE *fpCase = NULL;    /* 入力（case）ファイルポインタ */
    FILE *fpCntl = NULL;    /* 入力（control）ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */
    FILE *fpArea = NULL;    /* haplotypeブロック領域指定ファイルポインタ */

    char *caseData = NULL;      /* サンプルデータ（case）格納用 */
    char *controlData = NULL;   /* サンプルデータ（control）格納用 */

    SnpData *snpTmpData1 = NULL;
    SnpData *snpTmpData2 = NULL;
    SnpData *snpData1 = NULL;
    SnpData *snpData2 = NULL;


    /* ファイルオープン */
    retval = InputFileOpen(&fpCase, inputScore->inputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpCntl, inputScore->inputFile2);
    if (retval != 0){
        goto finalize;
    }
    retval = OutputFileOpen(&fpOut, inputScore->outputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpArea, inputScore->blockAreaFile);
    if (retval != 0){
        goto finalize;
    }

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/
 
    /* haplotypeブロック領域指定ファイルのライン数を取得 */
    areaFileLine = DataReaderCountFileLine(fpArea);
    /* haplotypeブロック領域格納用配列のメモリ確保 */
    blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
    if (NULL == blockArea){ goto finalize; }
    /* ファイルポインタを先頭に戻す */
    fseek(fpArea, 0L, SEEK_SET);
    /* haplotypeブロック領域を配列に収める */
    DataReaderSetHaplotypeBlockArea(fpArea, blockArea);

    /* 入力ファイルのライン数を取得 */
    fileLine1 = DataReaderCountFileLine(fpCase);
    fileLine2 = DataReaderCountFileLine(fpCntl);

    /* ファイルポインタを先頭に戻す */
    fseek(fpCase, 0L, SEEK_SET);
    fseek(fpCntl, 0L, SEEK_SET);

    /* データ一時格納用構造体のメモリ確保 */
    snpTmpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine1);
    if (NULL == snpTmpData1){ goto finalize; }
    snpTmpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine2);
    if (NULL == snpTmpData2){ goto finalize; }

    /* データをファイルから読み込み構造体に収める */
    DataReaderSetAllData(fpCase, snpTmpData1, fileLine1, inputScore->dataType);
    DataReaderSetAllData(fpCntl, snpTmpData2, fileLine2, inputScore->dataType);

    /* 入力データの整合性をチェックして並列化用の入力データを作成する */
    /* MPI_Bcastの回数を減らすためにサンプルデータは別配列（caseData, controlData）で保持する */
    dataNum = DataReaderMakeParallelData(snpTmpData1, snpTmpData2, fileLine1, fileLine2, &snpData1, &snpData2, &caseData, &controlData);

    /* 使用しない配列のメモリ開放 */
    /* 構造体SnpDataメンバのメモリを開放する */
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    snpTmpData1 = NULL;
    snpTmpData2 = NULL;

    /* 入力データのサンプル数取得 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    n = a + b;

    /* 構造体SnpDataにサンプルデータをコピーする */
    DataReaderDataCopyToSnpData(snpData1, caseData, dataNum, a);
    DataReaderDataCopyToSnpData(snpData2, controlData, dataNum, b);

/****************************************************************/
/* メモリ確保                                                   */
/****************************************************************/

    /* haplotypeブロック数 */
    if (inputScore->areaFileType == 0){
        blockNum = areaFileLine - 1;
    }
    else {
        /* haplotypeブロックの個数を計算する */
        if (blockArea[0] > dataNum){
            blockNum = 1;
            blockArea[1] = dataNum; /* linkSNPNumの値がデータ数だけになる */
        }
        else{
            blockNum = (dataNum - (blockArea[0] - blockArea[1]) ) / blockArea[1];
        }
    }
    /* 各haplotypeブロック領域内のSNP数格納用配列のメモリ確保 */
    linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPNum){ goto finalize; }
    /* 各haplotypeブロック領域内の最初のSNPを示す座位格納用配列のメモリ確保 */
    linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPStart){ goto finalize; }

    /* 各haplotypeブロック領域に該当するSNPデータを決定 */
    if (inputScore->areaFileType == 0){
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = blockArea[i];
            endPos = blockArea[i+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < dataNum; j++){
                if (startPos <= snpData1[j].pos){
                    if (snpData1[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
        }
    }
     else {
        for (i = 0; i < blockNum; i++){
            linkSNPNum[i] = blockArea[0];
            linkSNPStart[i] = i * blockArea[1];
        }
    }

    /* 偶現表Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), a + b);
    if (NULL == populationType) { goto finalize; }
    /* 各SNPのスコア格納用配列のメモリ確保 */
    Sobs = (double*)malloc1Dim(sizeof(double), dataNum);
    if (NULL == Sobs) { goto finalize; }

/****************************************************************/
/* Score計算                                                    */
/****************************************************************/

    /* haplotypeブロック単位でループ */
    jStart = 0;
    index = 0;
    for (i = 0; i < blockNum; i++){
        jStart = linkSNPStart[i];
        jEnd = jStart + linkSNPNum[i];
        /* 各haplotypeブロック内でループ */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderPopulationType(&snpData1[j], &snpData2[j], populationType);
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], populationType, T);
            /* スコアを計算　スコア計算中0割になってしまう場合は-1を返す */
            Sobs[index] = TableCalcScore(T);
            index++;
        }
    }

/****************************************************************/
/* Score計算結果出力                                            */
/****************************************************************/

    index = 0;
    fprintf(fpOut, "CaseData    = %s\n", inputScore->inputFile1);
    fprintf(fpOut, "ControlData = %s\n", inputScore->inputFile2);
    fprintf(fpOut, "BlockArea\tSNPNum\trsNumber\tPosition\tScore\tP\n");
    if (inputScore->areaFileType == 0){
        for (i = 0; i < blockNum; i++){
            jStart = linkSNPStart[i];
            jEnd = jStart + linkSNPNum[i];
            /* 各haplotypeブロック内でループ */
            for (j = jStart; j < jEnd; j++){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t \n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i],
                    snpData1[j].rsNumber,
                    snpData1[j].pos,
                    Sobs[index]);
                index++;
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            jStart = linkSNPStart[i];
            jEnd = jStart + linkSNPNum[i];
            /* 各haplotypeブロック内でループ */
            for (j = jStart; j < jEnd; j++){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.10lf\t \n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i],
                    snpData1[j].rsNumber,
                    snpData1[j].pos,
                    Sobs[j]);
            }
        }
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpCase);
    FileClose(fpCntl);
    FileClose(fpOut);
    FileClose(fpArea);
    /* 確保したメモリを開放する */
    free1Dim(blockArea);
    free1Dim(linkSNPNum);
    free1Dim(linkSNPStart);
    freeInt2Dim(T, ROW);
    free1Dim(populationType);
    free1Dim(Sobs);
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    free1Dim(caseData);
    free1Dim(controlData);
    free1Dim(snpData1);
    free1Dim(snpData2);

    return;
}
