#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "snp_Factorial.h"
#include "snp_Allele.h"
#include "snp_Kuriage.h"
#include "snp_BunkaiOld.h"
#include "snp_MultiNomial.h"
#include "snp_Range.h"
#include "snp_DataReader.h"
#include "snp_MemoryControl.h"

/* 正確法によるType I errorの確率計算を実行する */
double ExactExecute(double *h, char **haplotype, long L, long caseSample, long controlSample);

/* 入力データ格納用構造体 */
typedef struct struct_InputDataForExact {
    char inputFile1[MAX_LEN];   /* 入力ファイル名 */
    char inputFile2[MAX_LEN];   /* 入力ファイル名 */
    char outputFile1[MAX_LEN];  /* 出力ファイル名 */
    int areaFileType;           /* 領域指定ファイルのタイプ */
    char blockAreaFile[MAX_LEN];/* haplotypeブロック領域指定ファイル名 */
    int score;                      /* スコア計算方法の指定 */
    int dataType;                   /* 入力データフォーマットの指定 */
} InputExact;


int main(int argc, char* argv[])
{
    int retval = 0;
    int flag = 0;
    long i = 0;
	long j = 0;
    long fileLine1 = 0; /* 入力ファイルのライン数 */
    long fileLine2 = 0; /* 入力ファイルのライン数 */
    long areaFileLine = 0; /* haplotypeブロック領域指定ファイルのライン数 */
    long a = 0;
    long b = 0;
    long n = 0;
    long dataNum;       /* case、controlデータ整合後の総SNP数 */
    long index = 0;
    long jStart = 0;    /* haplotypeブロックの最初のSNPを示す座位 */
    long jEnd = 0;      /* haplotypeブロックの最後のSNPを示す座位 */
    long blockNum = 0;  /* Haplotypeブロック数 */
    long repeat = 0;
    long startPos = 0;
    long endPos = 0;
    double S = 0;

    int **T = NULL;             /* 偶現表 */
    int *populationType = NULL;
    long *L = NULL;             /* Haplotypeブロック毎のHaplotypeパターン数 */
    long *blockArea = NULL;     /* 各haplotypeブロック領域格納 */
    long *linkSNPNum = NULL;    /* 各haplotypeブロックのSNP数 */
    long *linkSNPStart = NULL;  /* 各haplotypeブロックの最初のSNPを示す座位 */
    long *maxScoreIndex = NULL;  /* Haplotypeブロック内のスコア最大値の座位 */
    char ***haplotype = NULL;    /* haplotypeデータ */
    double **h = NULL;        /* 頻度 */
    double *typeIError = NULL;  /* type I error 格納配列 */
    double *Sobs = NULL;        /* Haplotypeブロック毎のスコア最大値 */

    FILE *fpCase = NULL;    /* 入力（case）ファイルポインタ */
    FILE *fpCntl = NULL;    /* 入力（control）ファイルポインタ */
    FILE *fpOut = NULL;     /* 出力ファイルポインタ */
    FILE *fpArea = NULL;    /* haplotypeブロック領域指定ファイルポインタ */

    char *caseData = NULL;      /* サンプルデータ（case）格納用 */
    char *controlData = NULL;   /* サンプルデータ（control）格納用 */

    SnpData *snpTmpData1 = NULL;
    SnpData *snpTmpData2 = NULL;
    SnpData *snpData1 = NULL;
    SnpData *snpData2 = NULL;


    InputExact inputExact={"", "", "", 0, 0, 0, 0};

    if(argc != 8){
        printf("[usage]main.exe [InputFile1] [InputFile2] [outputFile] [AreaFileType] [BlockAreaFile] [Score] [dataType] \n");
        return 255;
    }
    strcpy(inputExact.inputFile1, argv[1]);
    strcpy(inputExact.inputFile2, argv[2]);
    strcpy(inputExact.outputFile1, argv[3]);
    inputExact.areaFileType = atoi(argv[4]);
    strcpy(inputExact.blockAreaFile, argv[5]);
    inputExact.score = atoi(argv[6]);
    inputExact.dataType = atoi(argv[7]);

    /* ファイルオープン */
    retval = InputFileOpen(&fpCase, inputExact.inputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpCntl, inputExact.inputFile2);
    if (retval != 0){
        goto finalize;
    }
    retval = OutputFileOpen(&fpOut, inputExact.outputFile1);
    if (retval != 0){
        goto finalize;
    }
    retval = InputFileOpen(&fpArea, inputExact.blockAreaFile);
    if (retval != 0){
        goto finalize;
    }

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/
 
    /* haplotypeブロック領域指定ファイルのライン数を取得 */
    areaFileLine = DataReaderCountFileLine(fpArea);
    /* haplotypeブロック領域格納用配列のメモリ確保 */
    blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
    if (NULL == blockArea){ goto finalize; }
    /* ファイルポインタを先頭に戻す */
    fseek(fpArea, 0L, SEEK_SET);
    /* haplotypeブロック領域を配列に収める */
    DataReaderSetHaplotypeBlockArea(fpArea, blockArea);

    /* 入力ファイルのライン数を取得 */
    fileLine1 = DataReaderCountFileLine(fpCase);
    fileLine2 = DataReaderCountFileLine(fpCntl);

    /* ファイルポインタを先頭に戻す */
    fseek(fpCase, 0L, SEEK_SET);
    fseek(fpCntl, 0L, SEEK_SET);

    /* データ一時格納用構造体のメモリ確保 */
    snpTmpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine1);
    snpTmpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine2);

    /* データをファイルから読み込み構造体に収める */
    DataReaderSetAllData(fpCase, snpTmpData1, fileLine1, inputExact.dataType);
    DataReaderSetAllData(fpCntl, snpTmpData2, fileLine2, inputExact.dataType);

    /* 入力データの整合性をチェックして並列化用の入力データを作成する */
    /* MPI_Bcastの回数を減らすためにサンプルデータは別配列（caseData, controlData）で保持する */
    dataNum = DataReaderMakeParallelData(snpTmpData1, snpTmpData2, fileLine1, fileLine2, &snpData1, &snpData2, &caseData, &controlData);

    /* 使用しない配列のメモリ開放 */
    /* 構造体SnpDataメンバのメモリを開放する */
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    snpTmpData1 = NULL;
    snpTmpData2 = NULL;

    /* 入力データのサンプル数取得 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    n = a + b;

    /* 構造体SnpDataにサンプルデータをコピーする */
    DataReaderDataCopyToSnpData(snpData1, caseData, dataNum, a);
    DataReaderDataCopyToSnpData(snpData2, controlData, dataNum, b);

/****************************************************************/
/* メモリ確保                                                   */
/****************************************************************/

    /* haplotypeブロック数 */
    if (inputExact.areaFileType == 0){
        blockNum = areaFileLine - 1;
    }
    else {
        /* haplotypeブロックの個数を計算する */
        if (blockArea[0] > dataNum){
            blockNum = 1;
            blockArea[1] = dataNum; /* linkSNPNumの値がデータ数だけになる */
        }
        else{
            blockNum = (dataNum - (blockArea[0] - blockArea[1]) ) / blockArea[1];
        }
    }
    /* 各haplotypeブロック領域内のSNP数格納用配列のメモリ確保 */
    linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPNum){ goto finalize; }
    /* 各haplotypeブロック領域内の最初のSNPを示す座位格納用配列のメモリ確保 */
    linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPStart){ goto finalize; }

    /* 各haplotypeブロック領域に該当するSNPデータを決定 */
    if (inputExact.areaFileType == 0){
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = blockArea[i];
            endPos = blockArea[i+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < dataNum; j++){
                if (startPos <= snpData1[j].pos){
                    if (snpData1[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            linkSNPNum[i] = blockArea[0];
            linkSNPStart[i] = i * blockArea[1];
        }
    }

    /* 偶現表Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == populationType) { goto finalize; }
    /* type I error 格納配列のメモリ確保 */
    typeIError = (double*)malloc1Dim(sizeof(double), blockNum);
    if (NULL == typeIError) { goto finalize; }
    /* haplotypeデータのメモリ確保 */
    haplotype = (char***)malloc1Dim(sizeof(char**), blockNum);
    if (NULL == haplotype) { goto finalize; }
    /* パターン頻度のメモリ確保 */
    h = (double**)malloc1Dim(sizeof(double*), blockNum);
    if (NULL == h) { goto finalize; }
    /* パターン数のメモリ確保 */
    L = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == L) { goto finalize; }
    /* Haplotypeブロック内のスコア最大値の座位のメモリ確保 */
    maxScoreIndex = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == maxScoreIndex) { goto finalize; }
    /* Haplotypeブロック毎のスコア最大値のメモリ確保 */
    Sobs = (double*)malloc1Dim(sizeof(double), blockNum);
    if (NULL == Sobs) { goto finalize; }

/****************************************************************/
/* 正確法による検定処理                                         */
/****************************************************************/

    /* haplotypeブロック単位でループ */
    jStart = 0;
    for (i = 0; i < blockNum; i++){
        jStart = linkSNPStart[i];
        jEnd = jStart + linkSNPNum[i];
        /* 各haplotypeブロック内でループ */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderPopulationType(&snpData1[j], &snpData2[j], populationType);
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], populationType, T);
            /* スコアを計算　スコア計算中0割になってしまう場合を-1を返す */
            S = TableCalcScore(T);
            /* 最大スコアを決定する */
            if (S > Sobs[i]){
                Sobs[i] = S;
                maxScoreIndex[i] = j;
            }
        }
        /* 各Haplotypeの頻度を計算 */
        L[i] = DataReaderHaplotypeFrequency(snpData1, snpData2, jStart, jEnd, &haplotype[i], &h[i]);

        /* 正確法による確率計算 */
        typeIError[i] = ExactExecute(h[i], haplotype[i], L[i], a, b);

    }

/****************************************************************/
/* 検定結果出力                                                 */
/****************************************************************/

    index = 0;
    fprintf(fpOut, "CaseData    = %s\n", inputExact.inputFile1);
    fprintf(fpOut, "ControlData = %s\n", inputExact.inputFile2);
    fprintf(fpOut, "BlockArea\tSNPNum\trsNumber\tPosition\tScore\tTypeIerror\n");
    if (inputExact.areaFileType == 0){
        for (i = 0; i < blockNum; i++){
            if (Sobs[i] != 0){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.3e\t%.3e\n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i],
                    snpData1[ maxScoreIndex[i] ].rsNumber,
                    snpData1[ maxScoreIndex[i] ].pos,
                    Sobs[i],
                    typeIError[i]);
            }
            else {
                fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                    blockArea[i],
                    blockArea[i+1],
                    linkSNPNum[i]);
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            if (Sobs[i] != 0){
                fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.3e\t%.3e\n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i],
                    snpData1[ maxScoreIndex[i] ].rsNumber,
                    snpData1[ maxScoreIndex[i] ].pos,
                    Sobs[i],
                    typeIError[i]);
            }
            else {
                fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                    snpData1[ linkSNPStart[i] ].pos,
                    snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                    linkSNPNum[i]);
            }
        }
    }

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpCase);
    FileClose(fpCntl);
    FileClose(fpOut);
    FileClose(fpArea);
    /* 確保したメモリを開放する */
    // haplotype変数はchar型の３次元配列
    // 各列の要素数は配列Lからわかる。文字数はlinkedSNPNumである。
    // 専用の開放の関数を作成する必要がある。とりあえず普通にfree
    for (i = blockNum-1; i >= 0; i--){
        for (index = L[i]-1; index >= 0; index--){
            free1Dim(haplotype[i][index]);
        }
        free1Dim(haplotype[i]);
    }
    free1Dim(haplotype);

    free1Dim(blockArea);
    free1Dim(linkSNPNum);
    free1Dim(linkSNPStart);
    free1Dim(typeIError);
    freeDouble2Dim(h,blockNum);
    freeInt2Dim(T, ROW);
    free1Dim(populationType);
    free1Dim(L);
    free1Dim(maxScoreIndex);
    free1Dim(Sobs);
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    free1Dim(caseData);
    free1Dim(controlData);
    free1Dim(snpData1);
    free1Dim(snpData2);

    return 0;
}


/* 正確法によるType I errorの確率計算を実行する */
double ExactExecute(double *h, char **haplotype, long L, long caseSample, long controlSample)
{
    Allele* A = NULL;
    int* X[2];
    double total=0.0;
    double result = -1.0;
    int* lower = NULL;
    int* upper = NULL;
    BunkaiOld* a = NULL;
    BunkaiOld* b = NULL;
 
    int samplesize = 5;
    int copyNumber = samplesize*2;
    int haba = 5; 
    int retval = 0;
    long i = 0;

    haba = (int)L;
    //case,controlで区別する必要がある
    samplesize = caseSample;
    copyNumber = caseSample + controlSample;

    X[0] = NULL;
    X[1] = NULL;

    A = (Allele*)malloc(sizeof(Allele));
    if(NULL == A) { goto finalize; }
    AlleleInit(A, haplotype, L, copyNumber);

    if(1 == FactorialSetFactorial(copyNumber+10)) { goto finalize; }

    lower = (int*)malloc(sizeof(int)*L);
    if(NULL == lower) { goto finalize; }
    upper = (int*)malloc(sizeof(int)*L);
    if(NULL == lower) { goto finalize; }
    for(i=0;i<L;i++){
	    //*
        lower[i] = RangeLowerLimit(copyNumber,h[i],haba);
        upper[i] = RangeUpperLimit(copyNumber,h[i],haba);
	    /*/
	    lower[i] = 0;
	    upper[i] = copyNumber;
	    /**/
    }

    a = (BunkaiOld*)malloc(sizeof(BunkaiOld));
    if(NULL == a) { goto finalize; }

    for( BunkaiOldInit(a, lower, L,upper,copyNumber); !(BunkaiOldDone(a)); BunkaiOldCount(a)){
        if(	BunkaiOldAllSum(a)==copyNumber) {
            b = (BunkaiOld*)malloc(sizeof(BunkaiOld));
            if(NULL == b) { goto finalize; }
            for( BunkaiOldInit(b, lower, L,upper,copyNumber); !(BunkaiOldDone(b)); BunkaiOldCount(b)){
                if( BunkaiOldAllSum(b)==copyNumber) {
                    /*
                    data = (int**)malloc(sizeof(int*)*2);
                    data[0] = BunkaiOldBunkai(a); // case
                    data[1] = BunkaiOldBunkai(b); // control
                    X[0]=data[0];
                    X[1]=data[1];
                    */
                    X[0]=BunkaiOldBunkai(a); // case
                    X[1]=BunkaiOldBunkai(b); // control
                    if ( AlleleF(A, X, L)==0 ){
                        //if ( ChiTest.significant2(samplesize,X,h)==0 ){
                        total+=MultiNomialMultinomialP2(copyNumber, X, L, h);
                    }
                }
            }
            BunkaiOldFinalize(b);
            free(b);
        }
    }
    BunkaiOldFinalize(a);

  printf("1-total=%g\n", 1.0-total);
    result = 1.0 - total;


finalize:;
    FactorialDeleteFactorial();
    if(A) AlleleFinalize(A, L);
    if(A) free(A);
    if(lower) free(lower);
    if(upper) free(upper);
    if(a) free(a);

    //printf("memory allocate error!\n");
    //exit(9);

    return result;

}
