#include <stdio.h>
#include <malloc.h>
#include <math.h>
// #include "stdafx.h"

#include "snp_Table.h"
#include "snp_Factorial.h"
#include "snp_Config.h"
#include "snp_MemoryControl.h"
#include "snp_Random.h"

int TableMain()
{
	int a1 = 0, a2 = 0;  /* case */
	int b1 = 0, b2 = 0;  /* control */
	SnpTable Tbl;
	int **T = NULL;
	int l = 0, u = 0;
	double Sobs = 0.0;
	int v = 0, w = 0;
	int i = 0;
	int **result;
	int *pos = NULL;
    int retval = 0;

	/* 設定 */
	a1 = 66, a2 = 34;  /* case */
	b1 = 90, b2 = 10;  /* control */
	Sobs = 11;

	//T = TableGetTable(Tbl);

    if (FactorialSetFactorial(a1 + a2 + b1 + b2) != 0){
        return 1;
    }

	retval = TableMakeTable4Int(&Tbl, a1, a2, b1, b2);
	l = TableCalcL(Tbl);
	u = TableCalcU(Tbl);
	v = TableCalcV(Tbl, Sobs);
	w = TableCalcW(Tbl, Sobs);

    result = (int**)calloc(2, sizeof(int*));
    if (NULL == result){
        return 1;
    }
	for (i = 0; i < 2; i++) {
		result[i] = (int*)calloc(2, sizeof(int));
        if (NULL == result[i]){
            return 1;
        }
	}

	for (i = l; i <= u; i++) {
		TableNewTable(Tbl, i, result);
		printf("%d\t%.14f", i, TableCalcScore(result));
		if ( (i < v) || (i > w) ) {
			printf("\t%f", TableCalcNu(result, Sobs));
		}
		printf("\n");
		//				showMatrix(result);
	}			

    /* logテーブルクリア */
    FactorialDeleteFactorial();
    /* 構造体tableのメモリ開放 */
    TableFinalTable(&Tbl);

#ifdef DBG
    //test
    printf("%d\n", getMallocCount());
#endif

	return 0;
}


/************************************************************************/
/*	テーブル初期化														*/
/*	tbl		O		テーブル											*/
/*	返り値	O		テーブル											*/
/************************************************************************/
int TableInitTable(SnpTable *tbl)
{
	tbl->horizontalSum = NULL;
	tbl->verticalSum = NULL;
//	tbl->fact;
	tbl->table = NULL;

	return 0;
}


/************************************************************************/
/*	テーブル作成（初期）												*/
/*	tbl			I	テーブル											*/
/*	numHori		I	水平方向のデータ数									*/
/*	numVert		I	垂直方向のデータ数									*/
/*	返り値		NORMAL_STATUS:		正常終了							*/
/*				ERROR_MEMORY_ALLOC:	メモリの確保に失敗した				*/
/************************************************************************/
int TableCreateTable(SnpTable *tbl, int numHori, int numVert)
{
	int *pos = NULL;
	int retVal = 0;

	TableInitTable(tbl);	/* Tableの初期化 */

    /* テーブルのメモリ確保 */
    tbl->table = (int**)mallocInt2Dim(numHori, numVert);
	if (NULL == tbl->table) {
		retVal = ERROR_MEMORY_ALLOC;
		goto finalize;
	}

	/* 水平方向及び垂直方向の和の配列 */
	tbl->horizontalSum = (int*)malloc1Dim(sizeof(int), numHori);
	if (NULL == tbl->horizontalSum) {
		retVal = ERROR_MEMORY_ALLOC;
		goto finalize;
	}
	tbl->verticalSum = (int*)malloc1Dim(sizeof(int), numVert);
	if (NULL == tbl->verticalSum) {
		retVal = ERROR_MEMORY_ALLOC;
		goto finalize;
	}

finalize:;
	switch (retVal) {
	case NORMAL_STATUS:
		return retVal;	/* 成功時にはここで処理を終える */
	case ERROR_MEMORY_ALLOC:
		printf("ERROR [TableCreateTable]: Cannot create Table.\n");
		break;
	}

	TableFinalTable(tbl);	/*　エラー時のメモリ領域解放　*/

	return retVal;
}


/************************************************************************/
/*	テーブル終了処理													*/
/*	tbl		I	テーブル												*/
/************************************************************************/
int TableFinalTable(SnpTable *tbl)
{
    if (tbl->table != NULL){
        freeInt2Dim(tbl->table, ROW);
    }
    if (tbl->horizontalSum != NULL){
        free1Dim(tbl->horizontalSum);
    }
    if (tbl->verticalSum != NULL){
        free1Dim(tbl->verticalSum);
    }

	tbl->horizontalSum = NULL;
	tbl->verticalSum = NULL;
//	tbl->fact;
	tbl->table = NULL;

	return 0;
}


/************************************************************************/
/*	４つの数値（整数型）からのテーブル作成								*/
/*	tbl		I	テーブル												*/
/*	a1		I	a1（Genotype1, Case）									*/
/*	a2		I	a2（Genotype2, Case）									*/
/*	b1		I	b1（Genotype1, Control）								*/
/*	b2		I	b2（Genotype2, Control）								*/
/*	返り値	0:	正常処理												*/
/************************************************************************/
int TableMakeTable4Int(SnpTable *tbl, int a1, int a2, int b1, int b2)
{
	int n;

    /* テーブルの初期化 */
    TableInitTable(tbl);
    TableCreateTable(tbl, ROW, COLUMN);


	tbl->table[0][0] = a1;		tbl->table[0][1] = a2;	/* case */
	tbl->table[1][0] = b1;		tbl->table[1][1] = b2;	/* control */

	tbl->horizontalSum[0] = a1 + a2;
	tbl->horizontalSum[1] = b1 + b2;

	tbl->verticalSum[0] = a1 + b1;
	tbl->verticalSum[1] = a2 + b2;

	n = a1 + a2 + b1 + b2;
//	tbl->fact = new Factorial(n);

	return 0;
}



/************************************************************************/
/*	2*2の配列（整数型）からのテーブル作成								*/
/*	tbl		O	テーブル												*/
/*	T		I	テーブル内の値											*/
/************************************************************************/
int TableMakeTableIntArray(SnpTable *tbl, int **T)
{
	int a = 0, b = 0, n = 0;
	int a1 = 0, a2 = 0, b1 = 0, b2 = 0;
	int n1 = 0, n2 = 0;

    /* テーブルの初期化 */
    TableInitTable(tbl);
    TableCreateTable(tbl, ROW, COLUMN);

	a1 = T[0][0];		a2 = T[0][1];		a = a1 + a2;/* case */
	b1 = T[1][0];		b2 = T[1][1];		b = b1 + b2;/* control */
	n1 = a1 + b1;		n2 = a2 + b2;		n = n1 + n2;

	tbl->table[0][0] = a1;		tbl->table[0][1] = a2;/* case */
	tbl->table[1][0] = b1;		tbl->table[1][1] = b2;/* control */

    tbl->horizontalSum[0] = a1 + a2;
	tbl->horizontalSum[1] = b1 + b2;

	tbl->verticalSum[0] = a1 + b1;
	tbl->verticalSum[1] = a2 + b2;

    return 0;
}


int **TableGetTable(SnpTable T)
{
	return T.table;
}


/************************************************************************/
/*	新しい候補															*/
/*	tbl		I	テーブル												*/
/*	a1		I															*/
/*	result	O	結果													*/
/*																		*/
/************************************************************************/
int TableNewCandidate(SnpTable tbl, int a1, int **result)
{
	/*　データサイズのチェック　*/
	if (a1 < 0) {
		printf("negative value\n"); /* out of bounds */
		return 1;
	}
	if (a1 > tbl.horizontalSum[0]) {
		printf("too large\n");	/* out of bounds */
		return 1;
	}
	if (a1 > tbl.verticalSum[0]) {
		printf("too large\n"); /* out of bounds */
		return 1;
	}

	result[0][0] = a1;
	result[0][1] = tbl.horizontalSum[0] - a1;
	result[1][0] = tbl.verticalSum[0] - a1;
	result[1][1] = tbl.horizontalSum[1] - result[1][0];

	return 0;
}



int TableNewTable(SnpTable tbl, int a1, int **result)
{
	int retVal;

	retVal = TableNewCandidate(tbl, a1, result);

	return retVal;
}



/************************************************************************/
/*	L値の計算															*/
/*	tbl		I	テーブル												*/
/*	返り値		L値														*/
/************************************************************************/
int TableCalcL(SnpTable tbl)	/* lower limit */
{	int result = 0;

	result = tbl.horizontalSum[0] - tbl.verticalSum[1];
	if (result < 0) {
		result = 0; 
	}

	return result;
}


/************************************************************************/
/*	U値の計算															*/
/*	tbl		I	テーブル												*/
/*	返り値		U値														*/
/************************************************************************/
int TableCalcU(SnpTable tbl)	/* upper limit */
{
	int result = 0;

	result = tbl.horizontalSum[0];
	if (result > tbl.verticalSum[0]) {
		result = tbl.verticalSum[0];
	}

	return result;
}


/************************************************************************/
/*	Pearsonのスコアの計算												*/
/*	T		I		テーブル（2x2）										*/
/*	返り値			スコア												*/
/************************************************************************/
double TableCalcPearsonScore(int **T)		/* 式12) Pearsonのscore */
{
	int a1 = 0, a2 = 0, a = 0;
	int b1 = 0, b2 = 0, b = 0;
	int n1 = 0, n2 = 0, n = 0;
	double val = 0;
    double denominator = 0;

	a1 = T[0][0];		a2 = T[0][1];		a = a1 + a2;/* case */
	b1 = T[1][0];		b2 = T[1][1];		b = b1 + b2;/* control */
	n1 = a1 + b1;		n2 = a2 + b2;		n = n1 + n2;

    /* 0割り回避のため、分母が0ならば-1を返す */
    denominator = (double)a * b * n1 * n2;
    if (denominator == 0){
        return -1;
    }

	val = (double)n * (a1 * b2 - a2 * b1) * (a1 * b2 - a2 * b1);
	val /= denominator;

    return val;
}

/************************************************************************/
/*	vの計算																*/
/*	tbl		I	テーブル												*/
/*	Sobs	I	テーブル												*/
/*	返り値		vの値													*/
/************************************************************************/
int TableCalcV(SnpTable tbl, double Sobs) /* lower boundary of X where S>Sobs */
{
	int a = 0, b = 0, n1 = 0;
	double n = 0.0;
	int result = 0;

	a = tbl.horizontalSum[0];
	b = tbl.horizontalSum[1];
	n1 = tbl.verticalSum[0];
	n = (double)(a + b);

	result = (int)(n1 * a / n - TableRootD(tbl, Sobs) / n);
	if (result < 0) {
		result = 0; 
	}

	return result;
}


/************************************************************************/
/*	tbl		I	テーブル												*/
/*	Sobs	I	テーブル												*/
/*	返り値		W値														*/
/************************************************************************/
int TableCalcW(SnpTable tbl, double Sobs) /* upper boundary of X where S>Sobs */
{
	int a = 0, b = 0;
	int n1 = 0;
	double n = 0.0;
	int result = 0;

	a = tbl.horizontalSum[0];
	b = tbl.horizontalSum[1];
	n1 = tbl.verticalSum[0];
	n = a + b;

	result = 1 + (int)(n1 * a / n + TableRootD(tbl, Sobs) / n);
	if (result < 0) {
		result = 0; 
	}

	return result;
}


/************************************************************************/
/*	判別値のルートの計算												*/
/*	tbl		I	テーブル												*/
/*	Sobs	I															*/
/*	返り値	判別値のルート												*/
/************************************************************************/
double TableRootD(SnpTable tbl, double Sobs)
{
	return sqrt(TableDiscriminant(tbl, Sobs));
}


/************************************************************************/
/*	判別値の計算														*/
/*	tbl		I	テーブル												*/
/*	Sobs	I															*/
/*	返り値	判別値														*/
/************************************************************************/
double TableDiscriminant(SnpTable tbl, double Sobs) /* 判別式 */
{
	int a = 0, b = 0;
	int n1 = 0, n2 = 0, n = 0;

	a = tbl.horizontalSum[0];
	b = tbl.horizontalSum[1];

	n1 = tbl.verticalSum[0];
	n2 = tbl.verticalSum[1];
	n = a + b;

	return Sobs * a * b * n1 * n2 /(double)n;
}



/************************************************************************/
/*	判別値の計算														*/
/*	tbl			I	テーブル											*/
/************************************************************************/
int TableTwoParts(SnpTable tbl, double Sobs)
{
	if (TableDiscriminant(tbl, Sobs) <= 0) { /* imaginary */
		return FLAG_FALSE;
	}
	if ( (TableCalcV(tbl, Sobs) > TableCalcL(tbl))
			&& (TableCalcW(tbl, Sobs) < TableCalcU(tbl)) ) {
		return FLAG_TRUE;
	}
	return FLAG_FALSE;
}


/************************************************************************/
/*	Fst値の計算															*/
/*	T		I	テーブル												*/
/*	返り値	Fst値														*/
/************************************************************************/
double TableCalcFst(int **T)		/* 式12) Pearsonのscore */
{
	double a1 = 0.0, a2 = 0.0, a = 0.0;
	double b1 = 0.0, b2 = 0.0, b = 0.0;
	double n1 = 0.0, n2 = 0.0, n = 0.0;
	double p1 = 0.0, p2 = 0.0, p = 0.0;
	double Hs = 0.0, Ht = 0.0;
	double Fst = 0.0;


	a1 = T[0][0];		a2 = T[0][1];		a = a1 + a2;/* case */
	b1 = T[1][0];		b2 = T[1][1];		b = b1 + b2;/* control */
	n1 = a1 + b1;		n2 = a2 + b2;		n = n1 + n2;

	p = a / n;
	p1 = a1 / n1;  /* gene frequency at population 1 */
	p2 = a2 / n2;  /* gene frequency at population 2 */

	Hs = 0.5 * (p1 * (1.0 - p1) + p2 * (1.0 - p2)); /* average heterozygosity */
	Ht = p * (1.0 - p);  /* total heterozygosity */
	Fst = 1.0 - (Hs / Ht); /* Fst */

	return Fst;
}



/************************************************************************/
/*	データ行列の表示													*/
/*	data	I	表示する行列											*/
/*	len1	I															*/
/*	len2	I															*/
/*	返り値	なし														*/
/************************************************************************/
void TableShowMatrix(int **data, int len1, int len2)
{
	int i = 0;

	for (i = 0; i < len1; i++){
		TableShowVector(data[i], len2);
	}
	printf("\n");
}



/************************************************************************/
/*	データ列の表示														*/
/*	data	I	表示する配列											*/
/*	len		I															*/
/*	返り値	なし														*/
/************************************************************************/
void TableShowVector(int *data, int len)
{
	int h;

	for (h = 0; h < len; h++) {
		printf("%d\t", data[h]);
	}
	printf("\n");
}


/************************************************************************/
/*	muの値の計算														*/
/*	T	I	テーブル													*/
/*																		*/
/*																		*/
/*																		*/
/************************************************************************/
double TableCalcMu(int **T)
{
	int a1 = 0, a2 = 0, a = 0;
	int b1 = 0, b2 = 0, b = 0;
	int n1 = 0, n2 = 0, n = 0;
	double logComb1 = 0.0, logComb2 = 0.0;	/* 組み合わせ数のLog */

	a1 = T[0][0];		a2 = T[0][1];		a = a1 + a2;	/* case */
	b1 = T[1][0];		b2 = T[1][1];		b = b1 + b2;	/* control */
	n1 = a1 + b1;		n2 = a2 + b2;		n = n1 + n2;

	logComb1 = FactorialGetLogCombination(n1, a1);

	logComb2 = FactorialGetLogCombination(n2, a2);

	return exp(logComb1 + logComb2);
}




/************************************************************************/
/*															*/
/************************************************************************/
double TableCalcNu(int **T, double Sobs)
{
	double S;

	S = TableCalcScore(T);
	if (S > Sobs) {
		return TableCalcMu(T);
	}

	return 0.0;
}


/************************************************************************/
/*															*/
/************************************************************************/
int TableInCj(double Sobs, int **T, int len1, int len2)
{
	int i = 0, j = 0;
	double S;

	for (i = 0; i < len1; i++) {
		for (j = 0; j < len2; j++) {
			if (T[i][j] < 0) {
				 return FLAG_FALSE; /* out of bounds */
			}
		}
	}
	S = TableCalcScore(T);

	if (S >= Sobs) {
		return FLAG_TRUE;
	}
	return FLAG_FALSE;
}


/************************************************************************/
/*															*/
/************************************************************************/
int TableJudgeCj(double Sobs, int **T, int len1, int len2)
{
	/* if X is out of range return 1 */
	/* if Score(T(X))>=Sobs return 2 */
	/* if X is within range and Score(T(X))<Sobs return 0 */
	int i = 0, j = 0;
    double S = 0;

	for (i = 0; i < len1; i++) {
		for (j = 0; j < len2; j++){
			if (T[i][j] < 0) {
				return OUT_OF_RANGE; /* out of bounds */
			}
		}
	}

	S = TableCalcScore(T);
	if (S >= Sobs) {
		return WITH_IN_RANGE;
	}

	return IN_BETWEEN;
}


/************************************************************************/
/*															*/
/*	T1: 	I													*/
/*	numI:	I												*/
/*	numJ:	I												*/
/*	T2:		I												*/
/*															*/
/*															*/
/************************************************************************/
double TableCalcR(int **T1, int len1, int len2, int **T2)
{
	/* 将来的に一度計算すれば十分というアルゴリズムに変更する */
	double logNum = 0.0; /* 分子 */
	double logDen = 0.0; /* 分母 */
	int i = 0, j = 0;
	double logResult = 0.0;
	double retVal = 0.0;

	for (i = 0; i < len1; i++) {
		for(j = 0; j < len2; j++) {
			logNum += FactorialGetLogFactorial(T1[i][j]);
			logDen += FactorialGetLogFactorial(T2[i][j]);
		}
	}
	logResult = logNum - logDen;
	retVal = exp(logResult);

	return retVal;
}

/* 偶現表のスコア計算 */
double TableCalcScore(int **T)
{
    double score = 0;
                
    /* スコアを計算 */
    switch(iWay){
        case 0: /* PeasonScore計算 */
                score = TableCalcPearsonScore(T);
                break;
        case 1: /* Fst計算 */
                score = TableCalcFst(T);
                break;
        default:/* 定義していない値の場合 */
                score = -1;
                break;
    }

    return score; 
}

/* muのlog値の計算 */
double TableCalcLogMu(int **T)
{
	int a1 = 0, a2 = 0, a = 0;
	int b1 = 0, b2 = 0, b = 0;
	int n1 = 0, n2 = 0, n = 0;
	double logComb1 = 0.0, logComb2 = 0.0;	/* 組み合わせ数のLog */

	a1 = T[0][0];		a2 = T[0][1];		a = a1 + a2;	/* case */
	b1 = T[1][0];		b2 = T[1][1];		b = b1 + b2;	/* control */
	n1 = a1 + b1;		n2 = a2 + b2;		n = n1 + n2;

	logComb1 = FactorialGetLogCombination(n1, a1);

	logComb2 = FactorialGetLogCombination(n2, a2);

	return logComb1 + logComb2;
}

/* one region */
void TableMarkov1(SnpTable T, double Score, int **table, dsfmt_t *snp_dsfmt_data)
{
    int a1 = 0,
        a2 = 0;
    int b1 = 0,
        b2 = 0;
    int newA1 = 0;
    int u = 0;
    int l = 0;
    int i = 0;
    int j = 0;
    int retval = 0;

    double y = 0,
           y2 = 0;

    int **X = NULL;
    int **newX = NULL;

    X = TableGetTable(T);
    a1 = X[0][0];
    a2 = X[0][1];
    b1 = X[1][0];
    b2 = X[1][1];
    
    i = TablePlusMinusOne(snp_dsfmt_data);
    newA1 = a1 + i;

    u = TableCalcU(T);
    l = TableCalcL(T);

    /* new candidate */
    if ( (newA1 < l) || (newA1 > u) ){
        return; //unchanged
    }
    /* newXのメモリ確保 */
    newX = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == newX) { goto finalize; }

    retval = TableNewCandidate(T, newA1, newX);

    if (FLAG_TRUE == TableInCj(Score, newX, ROW, COLUMN)){ /* condition 5 */
        if (1 == i){
            y = (b1 * a2) / ( (double) (a1 + 1) * (b2 + 1) );
        }
        if (-1 == i){
            y = (a1 * b2) / ( (double) (b1 + 1) * (a2 + 1) );
        }
        /* mcmc transition */
        if (y >= 1){ /* procedure 6 */
            retval = TableNewTable(T, newA1, table);
        }
        else{
            y2 = myRand(snp_dsfmt_data);
            if (y >= y2){ /* procedere 7 */
                retval = TableNewTable(T, newA1, table);
            }
        }
    }

finalize:;
    /* 確保したメモリを開放する */
    freeInt2Dim(newX, ROW);
}


/* two regions */
void TableMarkov2(SnpTable T, double Score, int **table, dsfmt_t *snp_dsfmt_data)
{
    int a1 = 0,
        a2 = 0;
    int b1 = 0,
        b2 = 0;
    int u = 0;
    int l = 0;
    int i = 0;
    int j = 0;
    int flag = 0;
    int oldA1 = 0,
        newA1 = 0;
    int retval = 0;

    double y = 0,
           y2 = 0;
    double z = 0,
           z2 = 0;

    int **X = NULL;
    int **newX = NULL;
    int **oldA1X = NULL;
    int **newA1X = NULL;

    X = TableGetTable(T);
    a1 = X[0][0];
    a2 = X[0][1];
    b1 = X[1][0];
    b2 = X[1][1];

    i = TablePlusMinusOne(snp_dsfmt_data);
    newA1 = a1 + i;

    u = TableCalcU(T);
    l = TableCalcL(T);

    if ( (newA1 < l) || (newA1 > u) ){
        return; //unchanged
    }

    /* new candidate */
    /* newXのメモリ確保 */
    newX = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == newX) { goto finalize; }

    retval = TableNewCandidate(T, a1, newX);
    flag = TableJudgeCj(Score, newX, ROW, COLUMN);

    if (WITH_IN_RANGE == flag){ /* condition 5 */
        if (1 == i){
            y = (b1 * a2) / ( (double) (a1 + 1) * (b2 + 1) );
        }
        if (-1 == i){
            y = (a1 * b2) / ( (double) (b1 + 1) * (a2 + 1) );
        }
        /* mcmc transition */
        if (y >= 1){ /* procedure 6 */
            retval = TableNewTable(T, newA1, table);
        }
        else{
            y2 = myRand(snp_dsfmt_data);
            if (y >= y2){ /* procedere 7 */
                retval = TableNewTable(T, newA1, table);
            }
        }
    }
    if (IN_BETWEEN == flag){ /* leap to the other island */
        oldA1 = T.table[0][0];
        newA1 = oldA1;

        /* oldA1Xのメモリ確保 */
        oldA1X = (int**)mallocInt2Dim(ROW, COLUMN);
        if (NULL == oldA1X) { goto finalize; }
        /* newA1Xのメモリ確保 */
        newA1X = (int**)mallocInt2Dim(ROW, COLUMN);
        if (NULL == newA1X) { goto finalize; }


        if (1 == i){ /* leap from w to v */
            newA1 = TableCalcW(T, Score);
            retval = TableNewCandidate(T, oldA1, oldA1X);
            retval = TableNewCandidate(T, newA1, newA1X);
            z = TableCalcR(oldA1X, ROW, COLUMN, newA1X);
        }
        if (-1 == i){ /* leap from v to w */
            newA1 = TableCalcV(T, Score);
            retval = TableNewCandidate(T, oldA1, oldA1X);
            retval = TableNewCandidate(T, newA1, newA1X);
            z = TableCalcR(oldA1X, ROW, COLUMN, newA1X);
        }
        
        if (z > 1){
            retval = TableNewTable(T, newA1, table);
        }
        else{
            z2 = myRand(snp_dsfmt_data);
            if (z >= z2){ /* leap */
                retval = TableNewTable(T, newA1, table);
            }
        }
    }

finalize:;
    /* 確保したメモリを開放する */
    freeInt2Dim(newX, ROW);
    freeInt2Dim(oldA1X, ROW);
    freeInt2Dim(newA1X, ROW);
}

int TablePlusMinusOne(dsfmt_t *snp_dsfmt_data)
{
    /* return -1 or 1 with probability 0.5 */
    int result = 0;

    result = (int)(myRand(snp_dsfmt_data) * 2) * 2 - 1;

    return result;
}
