#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef MPI_USE
#include <mpi.h>
#endif

#ifdef _OPENMP
#include <omp.h>
#endif

#define _MAIN_DEF

#include "snp_typeI.h"

dsfmt_t snp_dsfmt_global_data;
#pragma omp threadprivate(snp_dsfmt_global_data)

#ifdef MPI_USE
/* プロトタイプ宣言 */
int Define_MPI_InputStruct(MPI_Datatype *mpi_inputStruct);
#endif

/** MPIの初期化処理
 *  MPI用の引数が処理されて、削除される。
 *  @param argc [IN/OUT] 引数の数
 *  @param argv [IN/OUT] 引数
 */
void setupMPI(int *argc, char ***argv)
{
#ifdef MPI_USE
    MPI_Init(argc, argv);
    MPI_Comm_size(MPI_COMM_WORLD, &MyMpiSize);
    MPI_Comm_rank(MPI_COMM_WORLD, &MyMpiRank);
#else
    MyMpiSize = 1;
    MyMpiRank = 0;
#endif /* MPI_USE */
}


int main(int argc, char* argv[])
{
    InputTypeI inputTypeI={"", "", "", 0, "", 0, 0, 0, 0};

    int i = 0;

    /* MPIによる初期化処理 */
    setupMPI(&argc, &argv);

#ifdef MPI_USE
    /* 時間計測用変数の初期化 */
    for (i = 0; i < NUM_TM; i++){
        tm[i] = 0.0;
    }

    /* 時間計測 */
    tm[TM_START] = MPI_Wtime();
#endif /* MPI_USE */

    if(argc != 10){
        printf("[usage]main.exe [InputFile1] [InputFile2] [outputFile] [AreaFileType] [BlockAreaFile] [Score] [Repeat] [Generation] [dataType] \n");
        return 255;
    }
    strcpy(inputTypeI.inputFile1, argv[1]);
    strcpy(inputTypeI.inputFile2, argv[2]);
    strcpy(inputTypeI.outputFile1, argv[3]);
    inputTypeI.areaFileType = atoi(argv[4]);
    strcpy(inputTypeI.blockAreaFile, argv[5]);
    inputTypeI.score = atoi(argv[6]);
    inputTypeI.repeat = atol(argv[7]);
    inputTypeI.gen = atol(argv[8]);
    inputTypeI.dataType = atoi(argv[9]);

    if (MyMpiRank == 0) {
        printf("inputFile1: %s\n", inputTypeI.inputFile1);
        printf("inputFile2: %s\n", inputTypeI.inputFile2);
        printf("outputFile1: %s\n", inputTypeI.outputFile1);
        printf("blockAreaFile: %s\n", inputTypeI.blockAreaFile);
        printf("score: %d\n", inputTypeI.score);
        printf("repeat: %ld\n", inputTypeI.repeat);
        printf("gen: %ld\n", inputTypeI.gen);
        printf("dataType: %d\n", inputTypeI.dataType);
        printf("MyMpiSize: %d\n", MyMpiSize);
    }

    /* スコア計算方法の指定 */
    iWay = inputTypeI.score;

#pragma omp parallel
{
    /* 乱数発生関数初期化 */
    initMyRand(&snp_dsfmt_global_data);
}

    /* MultiLocus検定を行う */
    MainProgramMulti(&inputTypeI);

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_END] = MPI_Wtime();

    MPI_Finalize();    /* MPI終了処理 */

    printf("%4d\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\n",
           MyMpiRank,
           tm[TM_B_F_BCAST]  - tm[TM_START],
           tm[TM_B_BCAST]    - tm[TM_B_F_BCAST],
           tm[TM_B_CALC]     - tm[TM_B_BCAST],
           tm[TM_A_CALC]     - tm[TM_B_CALC],
           tm[TM_A_GATHER]   - tm[TM_A_CALC],
           tm[TM_END]        - tm[TM_A_GATHER],
           tm[TM_END]        - tm[TM_START]);

#if 0
    printf("%4d  1 time Init-READ      %lf (sec)\n", MyMpiRank, tm[TM_B_READ]     - tm[TM_START]);
    printf("%4d  2 time READ           %lf (sec)\n", MyMpiRank, tm[TM_A_READ]     - tm[TM_B_READ]);
    printf("%4d  3 time READ-FBcast    %lf (sec)\n", MyMpiRank, tm[TM_B_F_BCAST]  - tm[TM_A_READ]);
    printf("%4d  4 time FBcast         %lf (sec)\n", MyMpiRank, tm[TM_A_F_BCAST]  - tm[TM_B_F_BCAST]);
    printf("%4d  5 time FBcast-Bcast   %lf (sec)\n", MyMpiRank, tm[TM_B_BCAST]    - tm[TM_A_F_BCAST]);
    printf("%4d  6 time Bcast          %lf (sec)\n", MyMpiRank, tm[TM_A_BCAST]    - tm[TM_B_BCAST]);
    printf("%4d  7 time Bcast-Calc     %lf (sec)\n", MyMpiRank, tm[TM_B_CALC]     - tm[TM_A_BCAST]);
    printf("%4d  8 time Calculation    %lf (sec)\n", MyMpiRank, tm[TM_A_CALC]     - tm[TM_B_CALC]);
    printf("%4d  9 time Calc-FGather   %lf (sec)\n", MyMpiRank, tm[TM_B_F_GATHER] - tm[TM_A_CALC]);
    printf("%4d 10 time FGather        %lf (sec)\n", MyMpiRank, tm[TM_A_F_GATHER] - tm[TM_B_F_GATHER]);
    printf("%4d 11 time FGather-Gather %lf (sec)\n", MyMpiRank, tm[TM_B_GATHER]   - tm[TM_A_F_GATHER]);
    printf("%4d 12 time Gather         %lf (sec)\n", MyMpiRank, tm[TM_A_GATHER]   - tm[TM_B_GATHER]);
    printf("%4d 13 time Gather-Write   %lf (sec)\n", MyMpiRank, tm[TM_B_WRITE]    - tm[TM_A_GATHER]);
    printf("%4d 14 time Write          %lf (sec)\n", MyMpiRank, tm[TM_A_WRITE]    - tm[TM_B_WRITE]);
    printf("%4d 15 time Write-End      %lf (sec)\n", MyMpiRank, tm[TM_END]        - tm[TM_A_WRITE]);

    printf("%4d 20 time Init           %lf (sec)\n", MyMpiRank, tm[TM_B_F_BCAST]  - tm[TM_START]);
    printf("%4d 21 time FBcast         %lf (sec)\n", MyMpiRank, tm[TM_B_BCAST]    - tm[TM_B_F_BCAST]);
    printf("%4d 22 time Bcast          %lf (sec)\n", MyMpiRank, tm[TM_B_CALC]     - tm[TM_B_BCAST]);
    printf("%4d 23 time Calculation    %lf (sec)\n", MyMpiRank, tm[TM_A_CALC]     - tm[TM_B_CALC]);
    printf("%4d 24 time Gather         %lf (sec)\n", MyMpiRank, tm[TM_A_GATHER]   - tm[TM_A_CALC]);
    printf("%4d 25 time End            %lf (sec)\n", MyMpiRank, tm[TM_END]        - tm[TM_A_GATHER]);

    printf("%4d 30 time Calculation    %lf (sec)\n", MyMpiRank, tm[TM_A_CALC]     - tm[TM_B_CALC]);
    printf("%4d 31 time F_Bcast-Gather %lf (sec)\n", MyMpiRank, tm[TM_A_GATHER]   - tm[TM_B_F_BCAST]);
    printf("%4d 32 time A_Read-B_Write %lf (sec)\n", MyMpiRank, tm[TM_A_WRITE]    - tm[TM_B_READ]);
    printf("%4d 33 time Total          %lf (sec)\n", MyMpiRank, tm[TM_END]        - tm[TM_START]);
#endif
#endif /* MPI_USE */

    return 0;
}


/* MultiLocus検定を行う */
void MainProgramMulti(InputTypeI *inputTypeI)
{
    int retval = 0;
    int flag = 0;
    int numOutput = 0;
    int outDataOffset = 0;
    int numPartOutput = 0;
    int numTIEPartOutput = 0;
    long i = 0;
    long j = 0;
    long k = 0;
    long fileLine1 = 0; /* 入力ファイルのライン数 */
    long fileLine2 = 0; /* 入力ファイルのライン数 */
    long areaFileLine = 0; /* haplotypeブロック領域指定ファイルのライン数 */
    long a = 0;         /* number of haplotype copies (=sequences) in case */
    long b = 0;         /* number of haplotype copies (=sequences) in control */
    long n = 0;
    long dataNum;       /* case、controlデータ整合後の総SNP数 */
    long index = 0;
    long jStart = 0;    /* haplotypeブロックの最初のSNPを示す座位 */
    long jEnd = 0;      /* haplotypeブロックの最後のSNPを示す座位 */
    long blockNum = 0;  /* haplotypeブロック数 */
    long repeat = 0;
    long startPos = 0;
    long endPos = 0;
    long blockPartNum = 0;
    long blockPartStart = 0;
    double S = 0;

    /* Gather用 */
    int *numOutputs = NULL;
    int *outDataOffsets = NULL;
    int *outTIEOutputs = NULL;
    int *outTIEDataOffsets = NULL;
    int *numTIEOutputs = NULL;
    int **T = NULL;                 /* 偶現表 */
    int *populationType = NULL;
    //char ***haplotype = NULL;       /* haplotypeデータ */
    char **haplotype = NULL;       /* haplotypeデータ */
    char *caseData = NULL;          /* サンプルデータ（case）格納用 */
    char *controlData = NULL;       /* サンプルデータ（control）格納用 */
    //long *L = NULL;                 /* 各haplotypeブロック毎のhaplotypeパターン数 */
    long L = 0;                 /* haplotypeブロックのhaplotypeパターン数 */
    long *blockArea = NULL;         /* 各haplotypeブロック領域格納 */
    long *linkSNPNum = NULL;        /* 各haplotypeブロックのSNP数 */
    long *linkSNPStart = NULL;      /* 各haplotypeブロックの最初のSNPを示す座位 */
    long *maxScoreIndex = NULL;     /* 各haplotypeブロック内のスコア最大値の座位 */
    long *maxScoreIndexAll = NULL;  /* Haplotypeブロック内のスコア最大値の座位 */
    //double **freq = NULL;           /* 頻度 */
    double *freq = NULL;           /* 頻度 */
    double *typeIError = NULL;      /* type I error 格納配列 */
    double *typeIErrorAll = NULL;   /* type I error 格納配列 */
    double *Sobs = NULL;            /* 各haplotypeブロック毎のスコア最大値 */
    double *SobsAll = NULL;         /* Haplotypeブロック毎のスコア最大値 */

    FILE *fpCase = NULL;            /* 入力（case）ファイルポインタ */
    FILE *fpCntl = NULL;            /* 入力（control）ファイルポインタ */
    FILE *fpOut = NULL;             /* 出力ファイルポインタ */
    FILE *fpArea = NULL;            /* haplotypeブロック領域指定ファイルポインタ */

    SnpData *snpTmpData1 = NULL;
    SnpData *snpTmpData2 = NULL;
    SnpData *snpData1 = NULL;
    SnpData *snpData2 = NULL;

#ifdef MPI_USE
    /* MPI用構造体設定 */
    MPI_Datatype MPI_INPUT_STRUCT;
#endif /* MPI_USE */

    repeat = inputTypeI->repeat;

/****************************************************************/
/* データ入力                                                   */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_B_READ] = MPI_Wtime();
#endif /* MPI_USE */

    /* rank0のみ実行 */
    if (MyMpiRank == 0) {
        /* ファイルオープン */
        retval = InputFileOpen(&fpCase, inputTypeI->inputFile1);
        if (retval != 0){
            goto finalize;
        }
        retval = InputFileOpen(&fpCntl, inputTypeI->inputFile2);
        if (retval != 0){
            goto finalize;
        }
        retval = OutputFileOpen(&fpOut, inputTypeI->outputFile1);
        if (retval != 0){
            goto finalize;
        }
        retval = InputFileOpen(&fpArea, inputTypeI->blockAreaFile);
        if (retval != 0){
            goto finalize;
        }

        /* 入力ファイルのライン数を取得 */
        fileLine1 = DataReaderCountFileLine(fpCase);
        fileLine2 = DataReaderCountFileLine(fpCntl);

        /* ファイルポインタを先頭に戻す */
        fseek(fpCase, 0L, SEEK_SET);
        fseek(fpCntl, 0L, SEEK_SET);

        /* データ一時格納用構造体のメモリ確保 */
        snpTmpData1 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine1);
        if (NULL == snpTmpData1){ goto finalize; }
        snpTmpData2 = (SnpData*)malloc1Dim(sizeof(SnpData), fileLine2);
        if (NULL == snpTmpData2){ goto finalize; }

        /* データをファイルから読み込み構造体に収める */
        DataReaderSetAllData(fpCase, snpTmpData1, fileLine1, inputTypeI->dataType);
        DataReaderSetAllData(fpCntl, snpTmpData2, fileLine2, inputTypeI->dataType);

        /* 入力データの整合性をチェックして並列化用の入力データを作成する */
        /* MPI_Bcastの回数を減らすためにサンプルデータは別配列（caseData, controlData）で保持する */
        dataNum = DataReaderMakeParallelData(snpTmpData1, snpTmpData2, fileLine1, fileLine2, &snpData1, &snpData2, &caseData, &controlData);

        /* 使用しない配列のメモリ開放 */
        /* 構造体SnpDataメンバのメモリを開放する */
        DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
        DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
        snpTmpData1 = NULL;
        snpTmpData2 = NULL;
 
        /* haplotypeブロック領域指定ファイルのライン数を取得 */
        areaFileLine = DataReaderCountFileLine(fpArea);
        /* haplotypeブロック領域格納用配列のメモリ確保 */
        blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
        if (NULL == blockArea){ goto finalize; }
        /* ファイルポインタを先頭に戻す */
        fseek(fpArea, 0L, SEEK_SET);
        /* haplotypeブロック領域を配列に収める */
        DataReaderSetHaplotypeBlockArea(fpArea, blockArea);
        /* haplotypeブロック数 */
        if (inputTypeI->areaFileType == 0){
            blockNum = areaFileLine - 1;
        }
        else {
            /* haplotypeブロックの個数を計算する */
            if (blockArea[0] > dataNum){
                blockNum = 1;
                blockArea[1] = dataNum; /* linkSNPNumの値がデータ数だけになる */
            }
            else{
                blockNum = (dataNum - (blockArea[0] - blockArea[1]) ) / blockArea[1];
            }
        }
    }/* rank0のみ実行ここまで */

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_A_READ] = MPI_Wtime();
    /* 時間計測 */
    tm[TM_B_F_BCAST] = MPI_Wtime();

    /* 全データ数とブロック数の送信 */
    MPI_Bcast(&dataNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);
    MPI_Bcast(&blockNum, 1, MPI_LONG, 0, MPI_COMM_WORLD);
    MPI_Bcast(&areaFileLine, 1, MPI_LONG, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_F_BCAST] = MPI_Wtime();

    if (MyMpiSize > blockNum) {
        /* rank0のみ実行 */
        if (MyMpiRank == 0) {
            fprintf(stderr, "プロセッサ数(%d)がデータブロック数(%ld)よりも多く設定されています.\n", MyMpiSize, blockNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }
    if (MyMpiSize > dataNum) {
        /* rank0のみ実行 */
        if (MyMpiRank == 0) {
            fprintf(stderr, "プロセッサ数(%d)がデータ数(%ld)よりも多く設定されています.\n", MyMpiSize, dataNum);
        }
        MPI_Abort(MPI_COMM_WORLD, 1);
        goto finalize;
    }

    /* ランク0以外の入力データのメモリ確保 */
    if (MyMpiRank != 0) {
        /* snpData1（case）のメモリ確保 */
        snpData1 = (SnpData *)malloc1Dim(sizeof(SnpData), dataNum);
        if (snpData1 == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", MyMpiRank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* snpData2（control）のメモリ確保 */
        snpData2 = (SnpData *)malloc1Dim(sizeof(SnpData), dataNum);
        if (snpData2 == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", MyMpiRank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* haplotypeブロック領域のメモリ確保 */
        blockArea = (long*)malloc1Dim(sizeof(long), areaFileLine);
        if (blockArea == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", MyMpiRank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
    }

    /* MPI用型の定義 */
    Define_MPI_InputStruct(&MPI_INPUT_STRUCT);

    /* 時間計測 */
    tm[TM_B_BCAST] = MPI_Wtime();

    /* 入力データ(構造体)のブロードキャスト */
    MPI_Bcast(snpData1, dataNum, MPI_INPUT_STRUCT, 0, MPI_COMM_WORLD);
    MPI_Bcast(snpData2, dataNum, MPI_INPUT_STRUCT, 0, MPI_COMM_WORLD);
    MPI_Bcast(blockArea, areaFileLine, MPI_LONG, 0, MPI_COMM_WORLD);
#endif /* MPI_USE */

    /* 入力データのサンプル数取得 */
    a = snpData1[0].dataNum;
    b = snpData2[0].dataNum;
    n = a + b;

#ifdef MPI_USE
    /* ランク0以外の入力データのメモリ確保 */
    if (MyMpiRank != 0) {
        /* サンプルデータ（case）格納用のメモリ確保 */
        caseData = (char *)malloc1Dim(sizeof(char), dataNum * a);
        if (caseData == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", MyMpiRank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* サンプルデータ（control）格納用のメモリ確保 */
        controlData = (char *)malloc1Dim(sizeof(char), dataNum * b);
        if (controlData == NULL) {
            fprintf(stderr, "rank:%d メモリを確保できませんでした.\n", MyMpiRank);
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
    }

    /* 入力データ(データ部)のブロードキャスト */
    MPI_Bcast(caseData, dataNum * a, MPI_CHAR, 0, MPI_COMM_WORLD);
    MPI_Bcast(controlData, dataNum * b, MPI_CHAR, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_BCAST] = MPI_Wtime();
#endif /* MPI_USE */

    /* 構造体SnpDataにサンプルデータをコピーする */
    DataReaderDataCopyToSnpData(snpData1, caseData, dataNum, a);
    DataReaderDataCopyToSnpData(snpData2, controlData, dataNum, b);

/****************************************************************/
/* メモリ確保                                                   */
/****************************************************************/

    /* プロセッサごとのデータ個数計算 */
    blockPartNum = blockNum / MyMpiSize;
    if (MyMpiRank < blockNum % MyMpiSize) {
        blockPartNum++;
    }
    /* 各プロセッサにおけるデータのオフセット値の計算 */
    if (MyMpiRank < blockNum % MyMpiSize) {
        blockPartStart = blockPartNum * MyMpiRank;
    }
    else{
        blockPartStart = (blockPartNum + 1) * (blockNum % MyMpiSize)
                       + blockPartNum * (MyMpiRank - blockNum % MyMpiSize);
    }

    /* 各haplotypeブロック領域内のSNP数格納用配列のメモリ確保 */
    linkSNPNum = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPNum){ goto finalize; }
    /* 各haplotypeブロック領域内の最初のSNPを示す座位格納用配列のメモリ確保 */
    linkSNPStart = (long*)malloc1Dim(sizeof(long), blockNum);
    if (NULL == linkSNPStart){ goto finalize; }

    /* 各haplotypeブロック領域に該当するSNPデータを決定 */
    if (inputTypeI->areaFileType == 0){
        for (i = 0; i < blockNum; i++){ /* 将来、領域の重複を許す場合も考える */
            startPos = blockArea[i];
            endPos = blockArea[i+1];
            flag = 0;
            /* 入力データはポジションでソートされていると仮定 */
            for (j = 0; j < dataNum; j++){
                if (startPos <= snpData1[j].pos){
                    if (snpData1[j].pos < endPos){
                        /* 領域内のSNP数カウント */
                        linkSNPNum[i]++;
                        /* 領域内の最初のSNPを示す座位を保持 */
                        if (0 == flag ){
                            linkSNPStart[i] = j;
                            flag = 1;
                        }
                    }
                    /* これ以降、領域に該当するデータは出現しないので次のブロックを調べる */
                    else {
                        break;
                    }
                }
            }
        }
    }
    else {
        for (i = 0; i < blockNum; i++){
            linkSNPNum[i] = blockArea[0];
            linkSNPStart[i] = i * blockArea[1];
        }
    }

    /* 偶現表Tのメモリ確保 */
    T = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == T){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), n);
    if (NULL == populationType) { goto finalize; }
    /* type I error 格納配列のメモリ確保 */
    typeIError = (double*)malloc1Dim(sizeof(double), blockPartNum * repeat);
    if (NULL == typeIError) { goto finalize; }
    /* mod 2010.12..13 メモリ節約のため使いますように変更 */
    /* haplotypeデータのメモリ確保 */
    //haplotype = (char***)malloc1Dim(sizeof(char**), blockPartNum);
    //haplotype = (char**)malloc1Dim(sizeof(char*), blockPartNum);
    //if (NULL == haplotype) { goto finalize; }
    /* パターン頻度のメモリ確保 */
    //freq = (double**)malloc1Dim(sizeof(double*), blockPartNum);
    //freq = (double*)malloc1Dim(sizeof(double), blockPartNum);
    //if (NULL == freq) { goto finalize; }
    /* パターン数のメモリ確保 */
    //L = (long*)malloc1Dim(sizeof(long), blockPartNum);
    //if (NULL == L) { goto finalize; }
    /* Haplotypeブロック内のスコア最大値の座位のメモリ確保 */
    maxScoreIndex = (long*)malloc1Dim(sizeof(long), blockPartNum);
    if (NULL == maxScoreIndex) { goto finalize; }
    /* Haplotypeブロック毎のスコア最大値のメモリ確保 */
    Sobs = (double*)malloc1Dim(sizeof(double), blockPartNum);
    if (NULL == Sobs) { goto finalize; }

/****************************************************************/
/* 検定処理                                                     */
/****************************************************************/

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_B_CALC] = MPI_Wtime();
#endif /* MPI_USE */

    /* haplotypeブロック単位でループ */
    jStart = 0;
    for (i = 0, k = blockPartStart; i < blockPartNum; i++, k++){
        jStart = linkSNPStart[k];
        jEnd = jStart + linkSNPNum[k];
        /* 各haplotypeブロック内でループ */
        for (j = jStart; j < jEnd; j++){
            /* 観測値から偶現表を作成する */
            DataReaderPopulationType(&snpData1[j], &snpData2[j], populationType);
            DataReaderMakeTableDi(&snpData1[j], &snpData2[j], populationType, T);
            /* スコアを計算　スコア計算中0割になってしまう場合を-1を返す */
            S = TableCalcScore(T);
            /* 最大スコアを決定する */
            if (S > Sobs[i]){
                Sobs[i] = S;
                maxScoreIndex[i] = j;
            }
        }
        /* 各Haplotypeの頻度を計算 */
        //L[i] = DataReaderHaplotypeFrequency(snpData1, snpData2, jStart, jEnd, &haplotype[i], &freq[i]);
        /* mod 2010.12..13 メモリ節約のため使いますように変更 */
        L = DataReaderHaplotypeFrequency(snpData1, snpData2, jStart, jEnd, &haplotype, &freq);

        /* ハプロタイプ関連解析のType I errorの確率計算を実行する */
        if (Sobs[i] > 0.0) {
            //TypeIExecute(freq[i], haplotype[i], L[i], Sobs[i], a, b, linkSNPNum[k], &typeIError[i*repeat], repeat, inputTypeI->gen);
            //TypeIExecuteHybrid(freq[i], haplotype[i], L[i], Sobs[i], a, b, linkSNPNum[k], &typeIError[i*repeat], repeat, inputTypeI->gen);
            /* mod 2010.12..13 メモリ節約のため使いますように変更 */
            TypeIExecuteHybrid(freq, haplotype, L, Sobs[i], a, b, linkSNPNum[k], &typeIError[i*repeat], repeat, inputTypeI->gen);
        }

        /* 使用済みメモリ領域を解放 */
        /* mod 2010.12..13 メモリ節約のため使いますように変更 */
        freeChar2Dim(haplotype, L);
        free1Dim(freq);
    }

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_A_CALC] = MPI_Wtime();

    if (MyMpiRank == 0) {
        /* 各ランクのブロック数を格納する配列のメモリを確保 */
        numOutputs = (int *)malloc1Dim(sizeof(int), MyMpiSize);
        if (numOutputs == NULL) {
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* 各ランクデータのオフセット値を格納する配列のメモリを確保 */
        outDataOffsets = (int *)malloc1Dim(sizeof(int), MyMpiSize);
        if (outDataOffsets == NULL) {
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* 各ランクのTypeIerror計算結果数を格納する配列のメモリを確保 */
        numTIEOutputs = (int *)malloc1Dim(sizeof(int), MyMpiSize);
        if (numTIEOutputs == NULL) {
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
        /* 各ランクのTypeIerro計算結果のオフセット値を格納するメモリを確保 */
        outTIEDataOffsets = (int *)malloc1Dim(sizeof(int), MyMpiSize);
        if (outTIEDataOffsets == NULL) {
            MPI_Abort(MPI_COMM_WORLD, 1);
            goto finalize;
        }
    }

    /* 時間計測 */
    tm[TM_B_F_GATHER] = MPI_Wtime();

    numOutput = blockPartNum;

    /* 各ランクが計算した出力値の個数 */
    MPI_Gather(&numOutput, 1, MPI_INT, numOutputs, 1,MPI_INT, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_F_GATHER] = MPI_Wtime();

    if (MyMpiRank == 0) {
        /* TypeIErrorの個数設定 */
        for (i = 0; i < MyMpiSize; i++) {
            numTIEOutputs[i] = numOutputs[i] * repeat;
        }

        /* ランクごとのデータのオフセット値の計算 */
        outDataOffsets[0] = 0;
        outTIEDataOffsets[0] = 0;
        for (i = 1; i < MyMpiSize; i++) {
            outDataOffsets[i] = outDataOffsets[i-1] + numOutputs[i-1];
            outTIEDataOffsets[i] = outTIEDataOffsets[i-1] + numOutputs[i-1] * repeat;
        }

        /* 出力用データを格納する構造体配列のメモリ確保 */
        SobsAll = (double *)malloc1Dim(sizeof(double), blockNum);
        maxScoreIndexAll = (long *)malloc1Dim(sizeof(long), blockNum);
        typeIErrorAll = (double *)malloc1Dim(sizeof(double), blockNum * repeat);
    }

    /* 時間計測 */
    tm[TM_B_GATHER] = MPI_Wtime();

    /* 各プロセッサからのデータの収集 */
    MPI_Gatherv(Sobs, numOutput, MPI_DOUBLE, SobsAll,
               numOutputs, outDataOffsets, MPI_DOUBLE, 0, MPI_COMM_WORLD);
    MPI_Gatherv(maxScoreIndex, numOutput, MPI_LONG, maxScoreIndexAll,
               numOutputs, outDataOffsets, MPI_LONG, 0, MPI_COMM_WORLD);
    MPI_Gatherv(typeIError, numOutput * repeat, MPI_DOUBLE, typeIErrorAll,
               numTIEOutputs, outTIEDataOffsets, MPI_DOUBLE, 0, MPI_COMM_WORLD);

    /* 時間計測 */
    tm[TM_A_GATHER] = MPI_Wtime();


    /* 時間計測 */
    tm[TM_B_WRITE] = MPI_Wtime();
#else
    /* ポインタ代入（MPI_Gatherv処理のかわり） */
    SobsAll = Sobs;
    maxScoreIndexAll = maxScoreIndex;
    typeIErrorAll = typeIError;
#endif /* MPI_USE */

/****************************************************************/
/* 検定結果出力                                                 */
/****************************************************************/

    if (MyMpiRank == 0) {
        fprintf(fpOut, "CaseData    = %s\n", inputTypeI->inputFile1);
        fprintf(fpOut, "ControlData = %s\n", inputTypeI->inputFile2);
        fprintf(fpOut, "Repeat      = %ld\n", inputTypeI->repeat);
        fprintf(fpOut, "Genoration  = %ld\n", inputTypeI->gen);
        fprintf(fpOut, "BlockArea\tSNPNum\trsNumber\tPosition\tScore\tTypeIerror...\n");
        if (inputTypeI->areaFileType == 0){
            for (i = 0; i < blockNum; i++){
                if (SobsAll[i] != 0){
                    fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.3e\t", 
                            blockArea[i],
                            blockArea[i+1],
                            linkSNPNum[i],
                            snpData1[ maxScoreIndexAll[i] ].rsNumber,
                            snpData1[ maxScoreIndexAll[i] ].pos,
                            SobsAll[i]);
                    /* type I errorを出力 */
                    for (index = 0; index < repeat; index++){
                        fprintf(fpOut, "%.3e\t", typeIErrorAll[i*repeat+index]);
                    }
                    fprintf(fpOut, "\n");
                }
                else {
                    fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                            blockArea[i],
                            blockArea[i+1],
                            linkSNPNum[i]);
                }
            }
        }
        else {
            for (i = 0; i < blockNum; i++){
                if (SobsAll[i] != 0){
                    fprintf(fpOut, "%ld-%ld\t%ld\t%s\t%ld\t%.3e\t", 
                            snpData1[ linkSNPStart[i] ].pos,
                            snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                            linkSNPNum[i],
                            snpData1[ maxScoreIndexAll[i] ].rsNumber,
                            snpData1[ maxScoreIndexAll[i] ].pos,
                            SobsAll[i]);
                    /* type I errorを出力 */
                    for (index = 0; index < repeat; index++){
                        fprintf(fpOut, "%.3e\t", typeIErrorAll[i*repeat+index]);
                    }
                    fprintf(fpOut, "\n");
                }
                else {
                    fprintf(fpOut, "%ld-%ld\t%ld\tNoData\n", 
                            snpData1[ linkSNPStart[i] ].pos,
                            snpData1[ linkSNPStart[i] + linkSNPNum[i] - 1 ].pos,
                            linkSNPNum[i]);
                }
            }
        }
    }

#ifdef MPI_USE
    /* 時間計測 */
    tm[TM_A_WRITE] = MPI_Wtime();
#endif /* MPI_USE */

/****************************************************************/
/* 終了処理                                                     */
/****************************************************************/

finalize:;
    /* ファイルクローズ */
    FileClose(fpCase);
    FileClose(fpCntl);
    FileClose(fpOut);
    FileClose(fpArea);
    /* 確保したメモリを開放する */
    /* mod 2010.12..13 メモリ節約のため使いますように変更のためコメントアウト */
    // haplotype変数はchar型の３次元配列
    // 各列の要素数は配列Lからわかる。文字数はlinkedSNPNumである。
    // 専用の開放の関数を作成する必要がある。とりあえず普通にfree
    //for (i = blockPartNum-1; i >= 0; i--){
    //    for (index = L[i]-1; index >= 0; index--){
    //        free1Dim(haplotype[i][index]);
    //    }
    //    free1Dim(haplotype[i]);
    //}
    //free1Dim(haplotype);

#ifdef MPI_USE
    free1Dim(numOutputs);
    free1Dim(outDataOffsets);
    free1Dim(outTIEOutputs);
    free1Dim(outTIEDataOffsets);
    free1Dim(numTIEOutputs);

    free1Dim(SobsAll);
    free1Dim(typeIErrorAll);
    free1Dim(maxScoreIndexAll);
#endif /* MPI_USE */

    free1Dim(blockArea);
    free1Dim(linkSNPNum);
    free1Dim(linkSNPStart);
    free1Dim(typeIError);
    //freeDouble2Dim(freq,blockPartNum);
    freeInt2Dim(T, ROW);
    free1Dim(populationType);
    //free1Dim(L);
    free1Dim(maxScoreIndex);
    free1Dim(Sobs);
    DataReaderSnpDataMemoryFree(snpTmpData1, fileLine1);
    DataReaderSnpDataMemoryFree(snpTmpData2, fileLine2);
    free1Dim(caseData);
    free1Dim(controlData);
    free1Dim(snpData1);
    free1Dim(snpData2);

    return;
}

/* ハプロタイプ関連解析のType I errorの確率計算を実行する */
int TypeIExecute(double *freq, char **haplotype, long L, double Sobs, long a, long b, long dataNum, double *typeIError, long repeat, long gen)
{
    int burnin = 100000; //10万に変更
    int **X = NULL;
    int flag = 0;
    long j = 0;
    long m = 0;
    long n = 0;
    long pop = 0;
    long copyNumber = 0;

    double S = 0;
    double total = 0;
    double diff = 0;

    // 引数で指定するように変更
    //double gen = 100000;

    int **Tresample = NULL;             /* 偶現表 */
    int *populationType = NULL;
    char **sequences = NULL;

    /* add 2010.11.10 使いまわし用メモリ領域格納変数追加 */
    int *data = NULL;
    int *data1 = NULL;
    int *data2 = NULL;

    SnpData *resampledData1 = NULL;
    SnpData *resampledData2 = NULL;

    copyNumber = a + b;

    /* add 2010.11.10 使いまわし用メモリ領域確保 */
    /* dataのメモリ確保 */
    data = (int*)malloc1Dim(sizeof(int), a + b);
    if (NULL == data) { goto finalize; }
    /* data1のメモリ確保 */
    data1 = (int*)malloc1Dim(sizeof(int), a);
    if (NULL == data1) { goto finalize; }
    /* data2のメモリ確保 */
    data2 = (int*)malloc1Dim(sizeof(int), b);
    if (NULL == data2) { goto finalize; }

    /* データ格納用構造体のメモリ確保 */
    resampledData1 = (SnpData*)malloc1Dim(sizeof(SnpData), dataNum);
    resampledData2 = (SnpData*)malloc1Dim(sizeof(SnpData), dataNum);
    DataReaderSnpDataMemoryAllocate(resampledData1, dataNum, a);
    DataReaderSnpDataMemoryAllocate(resampledData2, dataNum, b);
    /* sequencesのメモリ確保 */
    sequences = (char**)mallocChar2Dim(a + b, dataNum);
    if (NULL == sequences) { goto finalize; }
    /* 偶現表Tresampleのメモリ確保 */
    Tresample = (int**)mallocInt2Dim(ROW, COLUMN);
    if (NULL == Tresample){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int*)malloc1Dim(sizeof(int), a + b);
    if (NULL == populationType) { goto finalize; }

    //for (burnin = 0; burnin < 1 + gen * 0; burnin += 100000){ //burninでループしないように変更
        /* メモリ確保 */
        X = (int**)mallocInt2Dim(2, L);
        if (NULL == X) { goto finalize; }

        /* procedure 2 */
        //TypeIStartSet2(a, L, X[0]);
        //TypeIStartSet2(b, L, X[1]);
        TypeIStartSet(a, freq, L, X[0], &snp_dsfmt_global_data);
        TypeIStartSet(b, freq, L, X[1], &snp_dsfmt_global_data);

        /* start Markov */
        /* burn in */
        for (m = 0; m < burnin; m++){
            TypeIMarkov(X, freq, L, &snp_dsfmt_global_data);
        }

        for (n = 0; n < repeat; n++){
            diff = 0;
            for (m = 0; m < gen; m++){
                total++;
                TypeIMarkov(X, freq, L, &snp_dsfmt_global_data);

                /* ハプロタイプからXで表されている数の配列を作成 */
                TypeISampling(X[0], haplotype, dataNum, L, sequences);
                /* DNA配列データをSnpDataの配列に変換 */
                DataReaderSequenceToSnpData(sequences, resampledData1, dataNum, a);
                /* ハプロタイプからXで表されている数の配列を作成 */
                TypeISampling(X[1], haplotype, dataNum, L, sequences);
                /* DNA配列データをSnpDataの配列に変換 */
                DataReaderSequenceToSnpData(sequences, resampledData2, dataNum, b);

                /* フラグ初期化 */
                flag = 0;
                for (j = 0; j < dataNum; j++){
                    /* 観測値から偶現表を作成する */
                    DataReaderPopulationType(&resampledData1[j], &resampledData2[j], populationType);
                    /* mod 2010.11.10 メモリ使いまわす版の関数を呼び出す */
                    DataReaderMakeTableDiMemoryGiven(&resampledData1[j], &resampledData2[j], populationType, Tresample, data1, data2, data);
                    //DataReaderMakeTableDi(&resampledData1[j], &resampledData2[j], populationType, Tresample);
                    /* スコアを計算　スコア計算中0割になってしまう場合を-1を返す */
                    S = TableCalcScore(Tresample);
                    /* 最大スコアを決定する */
                    if (S >= Sobs) {
                        diff++;
                        /* S>=Sobsを判別するフラグをたてる */
                        flag = 1;
                    }
                }
                /* 一個以上違いがあるときに一個カウント */
                if (flag > 0){
                    diff++;
                }
            }
            typeIError[n] = diff/gen;
            /* 確認用標準出力 */
            //printf("%d\t%d\t%d\t%d\t%d\t%lf\n", n+1, a, b, copyNumber, burnin, diff/gen);
        }
    //}

finalize:;
    /* 確保したメモリを開放する */
    freeInt2Dim(X, 2);
    freeChar2Dim(sequences, a + b);
    free1Dim(resampledData1);
    free1Dim(resampledData2);
    
    /* add 2010.11.10 確保した使いまわし用メモリ領域解放 */
    free1Dim(data);
    free1Dim(data1);
    free1Dim(data2);

    return 0;
}


#if 1
/* ハプロタイプ関連解析のType I errorの確率計算を実行する（OpenMP対応） */
int TypeIExecuteHybrid(double *freq, char **haplotype, long L, double Sobs, long a, long b, long dataNum, double *typeIError, long repeat, long gen)
{
    int burnin = 100000; //10万に変更
    int flag = 0;
    long i = 0;
    long j = 0;
    long m = 0;
    long n = 0;
    long pop = 0;
    long copyNumber = 0;

    double S = 0;
    double total = 0;
    double diff = 0;

    int threadNum = 1;      /* スレッド数 */
    int iam = 0;            /* スレッド番号 */

    int **data = NULL;
    int **data1 = NULL;
    int **data2 = NULL;
    int ***X = NULL;
    int ***Tresample = NULL;             /* 偶現表 */
    int **populationType = NULL;
    char ***sequences = NULL;

    SnpData **resampledData1 = NULL;
    SnpData **resampledData2 = NULL;

#ifdef _OPENMP
    /* 使用可能なプロセッサ数を取得 */
    threadNum = omp_get_num_procs();
#endif

    copyNumber = a + b;

    /* 各スレッド専用のメモリ領域を確保 */ 
    X = (int***)malloc1Dim(sizeof(int**), threadNum);
    if (NULL == X) { goto finalize; }
    /* dataのメモリ確保 */
    data = (int**)malloc1Dim(sizeof(int*), threadNum);
    if (NULL == data) { goto finalize; }
    /* data1のメモリ確保 */
    data1 = (int**)malloc1Dim(sizeof(int*), threadNum);
    if (NULL == data1) { goto finalize; }
    /* data2のメモリ確保 */
    data2 = (int**)malloc1Dim(sizeof(int*), threadNum);
    if (NULL == data2) { goto finalize; }
    /* データ格納用構造体のメモリ確保 */
    resampledData1 = (SnpData**)malloc1Dim(sizeof(SnpData*), threadNum);
    resampledData2 = (SnpData**)malloc1Dim(sizeof(SnpData*), threadNum);
    /* sequencesのメモリ確保 */
    sequences = (char***)malloc1Dim(sizeof(char**), threadNum);
    if (NULL == sequences) { goto finalize; }
    /* 偶現表Tresampleのメモリ確保 */
    Tresample = (int***)malloc1Dim(sizeof(int**), threadNum);
    if (NULL == Tresample){ goto finalize; }
    /* populationTypeのメモリ確保 */
    populationType = (int**)malloc1Dim(sizeof(int*), threadNum);
    if (NULL == populationType) { goto finalize; }

    /* 各スレッド専用に確保したメモリ領域の内部メモリを確保 */
    for (i = 0; i < threadNum; i++) {
        X[i] = (int**)mallocInt2Dim(2, L);
        if (NULL == X[i]) { goto finalize; }
        /* dataのメモリ確保 */
        data[i] = (int*)malloc1Dim(sizeof(int), a + b);
        if (NULL == data[i]) { goto finalize; }
        /* data1のメモリ確保 */
        data1[i] = (int*)malloc1Dim(sizeof(int), a);
        if (NULL == data1[i]) { goto finalize; }
        /* data2のメモリ確保 */
        data2[i] = (int*)malloc1Dim(sizeof(int), b);
        if (NULL == data2[i]) { goto finalize; }
        /* データ格納用構造体のメモリ確保 */
        resampledData1[i] = (SnpData*)malloc1Dim(sizeof(SnpData), dataNum);
        if (NULL == resampledData1[i]) { goto finalize; }
        resampledData2[i] = (SnpData*)malloc1Dim(sizeof(SnpData), dataNum);
        if (NULL == resampledData1[i]) { goto finalize; }
        DataReaderSnpDataMemoryAllocate(resampledData1[i], dataNum, a);
        DataReaderSnpDataMemoryAllocate(resampledData2[i], dataNum, b);
        /* sequencesのメモリ確保 */
        sequences[i] = (char**)mallocChar2Dim(a + b, dataNum);
        if (NULL == sequences[i]) { goto finalize; }
        /* 偶現表Tresampleのメモリ確保 */
        Tresample[i] = (int**)mallocInt2Dim(ROW, COLUMN);
        if (NULL == Tresample[i]){ goto finalize; }
        /* populationTypeのメモリ確保 */
        populationType[i] = (int*)malloc1Dim(sizeof(int), a + b);
        if (NULL == populationType[i]) { goto finalize; }
    }

    //for (burnin = 0; burnin < 1 + gen * 0; burnin += 100000){ //burninでループしないように変更

// OpenMP スレッド並列開始
#pragma omp parallel private(m, iam)
    {

#ifdef _OPENMP
        /* 自分のスレッド番号を取得 */
        iam = omp_get_thread_num();
#endif

        /* procedure 2 */
        //TypeIStartSet2(a, L, X[0]);
        //TypeIStartSet2(b, L, X[1]);
        TypeIStartSet(a, freq, L, X[iam][0], &snp_dsfmt_global_data);
        TypeIStartSet(b, freq, L, X[iam][1], &snp_dsfmt_global_data);
        /* start Markov */
        /* burn in */
        for (m = 0; m < burnin; m++){
            TypeIMarkov(X[iam], freq, L, &snp_dsfmt_global_data);
        }

#pragma omp for private(j, flag, diff, S)
        for (n = 0; n < repeat; n++){
            diff = 0;
            for (m = 0; m < gen; m++){
                total++;
                TypeIMarkov(X[iam], freq, L, &snp_dsfmt_global_data);

                /* ハプロタイプからXで表されている数の配列を作成 */
                TypeISampling(X[iam][0], haplotype, dataNum, L, sequences[iam]);
                /* DNA配列データをSnpDataの配列に変換 */
                DataReaderSequenceToSnpData(sequences[iam], resampledData1[iam], dataNum, a);
                /* ハプロタイプからXで表されている数の配列を作成 */
                TypeISampling(X[iam][1], haplotype, dataNum, L, sequences[iam]);
                /* DNA配列データをSnpDataの配列に変換 */
                DataReaderSequenceToSnpData(sequences[iam], resampledData2[iam], dataNum, b);

                /* フラグ初期化 */
                flag = 0;
                for (j = 0; j < dataNum; j++){
                    /* 観測値から偶現表を作成する */
                    DataReaderPopulationType(&resampledData1[iam][j], &resampledData2[iam][j], populationType[iam]);
                    /* mod 2010.11.10 メモリ使いまわす版の関数を呼び出す */
                    DataReaderMakeTableDiMemoryGiven(&resampledData1[iam][j], &resampledData2[iam][j], populationType[iam], Tresample[iam], data1[iam], data2[iam], data[iam]);
                    //DataReaderMakeTableDi(&resampledData1[j], &resampledData2[j], populationType, Tresample);
                    /* スコアを計算　スコア計算中0割になってしまう場合を-1を返す */
                    S = TableCalcScore(Tresample[iam]);
                    /* 最大スコアを決定する */
                    if (S >= Sobs) {
                        diff++;
                        /* S>=Sobsを判別するフラグをたてる */
                        flag = 1;
                    }
                }
                /* 一個以上違いがあるときに一個カウント */
                if (flag > 0){
                    diff++;
                }
            }
            typeIError[n] = diff/gen;
            /* 確認用標準出力 */
            //printf("%d\t%d\t%d\t%d\t%d\t%lf\n", n+1, a, b, copyNumber, burnin, diff/gen);
        }
    } // OpenMP スレッド並列終了

    //}

finalize:;
    /* 各スレッド専用に確保したメモリ領域の内部メモリを確保 */
    for (i = 0; i < threadNum; i++) {
        freeInt2Dim(X[i], 2);
        free1Dim(data[i]);
        free1Dim(data1[i]);
        free1Dim(data2[i]);
        DataReaderSnpDataMemoryFree(resampledData1[i], dataNum);
        DataReaderSnpDataMemoryFree(resampledData2[i], dataNum);
        //free1Dim(resampledData1[i]);  /* DataReaderSnpDataMemoryFree関数内で解放している */
        //free1Dim(resampledData2[i]);
        freeChar2Dim(sequences[i], a + b);
        freeInt2Dim(Tresample[i], ROW);
        free1Dim(populationType[i]);
    }
    /* 確保したメモリを開放する */
    free1Dim(X);
    free1Dim(data);
    free1Dim(data1);
    free1Dim(data2);
    free1Dim(resampledData1);
    free1Dim(resampledData2);
    free1Dim(sequences);
    free1Dim(Tresample);
    free1Dim(populationType);

    return 0;
}
#endif

#if 0
/* ハプロタイプ関連解析のType I errorの確率計算を実行する（OpenMP対応） */
int TypeIExecuteHybrid(double *ifreq, char **haplotype, long L, double Sobs, long a, long b, long dataNum, double *typeIError, long repeat, long gen)
{
    int burnin = 100000; //10万に変更
    int flag = 0;
    long i = 0;
    long j = 0;
    long m = 0;
    long n = 0;
    long pop = 0;
    long copyNumber = 0;

    double S = 0;
    double total = 0;
    double diff = 0;

    int iam = 0;            /* スレッド番号 */

    int *data = NULL;
    int *data1 = NULL;
    int *data2 = NULL;
    int **X = NULL;
    int **Tresample = NULL;             /* 偶現表 */
    int *populationType = NULL;
    char **sequences = NULL;
    double *freq = NULL;

    SnpData *resampledData1 = NULL;
    SnpData *resampledData2 = NULL;

    copyNumber = a + b;

    //for (burnin = 0; burnin < 1 + gen * 0; burnin += 100000){ //burninでループしないように変更

// OpenMP スレッド並列開始
#pragma omp parallel private(m, iam, data, data1, data2, X, Tresample, populationType, sequences, resampledData1, resampledData2, freq)
    {

#ifdef _OPENMP
        /* 自分のスレッド番号を取得 */
        iam = omp_get_thread_num();
#endif

        /* メモリ確保 */
        X = (int**)mallocInt2Dim(2, L);
        if (NULL == X) { goto finalize; }
        /* dataのメモリ確保 */
        data = (int*)malloc1Dim(sizeof(int), a + b);
        if (NULL == data) { goto finalize; }
        /* data1のメモリ確保 */
        data1 = (int*)malloc1Dim(sizeof(int), a);
        if (NULL == data1) { goto finalize; }
        /* data2のメモリ確保 */
        data2 = (int*)malloc1Dim(sizeof(int), b);
        if (NULL == data2) { goto finalize; }
        /* データ格納用構造体のメモリ確保 */
        resampledData1 = (SnpData*)malloc1Dim(sizeof(SnpData), dataNum);
        if (NULL == resampledData1) { goto finalize; }
        resampledData2 = (SnpData*)malloc1Dim(sizeof(SnpData), dataNum);
        if (NULL == resampledData2) { goto finalize; }
        DataReaderSnpDataMemoryAllocate(resampledData1, dataNum, a);
        DataReaderSnpDataMemoryAllocate(resampledData2, dataNum, b);
        /* sequencesのメモリ確保 */
        sequences = (char**)mallocChar2Dim(a + b, dataNum);
        if (NULL == sequences) { goto finalize; }
        /* 偶現表Tresampleのメモリ確保 */
        Tresample = (int**)mallocInt2Dim(ROW, COLUMN);
        if (NULL == Tresample){ goto finalize; }
        /* populationTypeのメモリ確保 */
        populationType = (int*)malloc1Dim(sizeof(int), a + b);
        if (NULL == populationType) { goto finalize; }
        /* frepのメモリ確保 */
        freq = (double*)malloc1Dim(sizeof(double), L);
        if (NULL == freq) { goto finalize; }

        /* private freq に値を代入 */
        for (m = 0; m < L; m++){
            freq[m] = ifreq[m];
        }

        /* procedure 2 */
        //TypeIStartSet2(a, L, X[0]);
        //TypeIStartSet2(b, L, X[1]);
        TypeIStartSet(a, freq, L, X[0]);
        TypeIStartSet(b, freq, L, X[1]);

        /* burn in */
        for (m = 0; m < burnin; m++){
            TypeIMarkov(X, freq, L);
        }

#pragma omp for private(j, flag, diff, S)
        for (n = 0; n < repeat; n++){
            diff = 0;
            for (m = 0; m < gen; m++){
                total++;
                TypeIMarkov(X, freq, L);

                /* ハプロタイプからXで表されている数の配列を作成 */
                TypeISampling(X[0], haplotype, dataNum, L, sequences);
                /* DNA配列データをSnpDataの配列に変換 */
                DataReaderSequenceToSnpData(sequences, resampledData1, dataNum, a);
                /* ハプロタイプからXで表されている数の配列を作成 */
                TypeISampling(X[1], haplotype, dataNum, L, sequences);
                /* DNA配列データをSnpDataの配列に変換 */
                DataReaderSequenceToSnpData(sequences, resampledData2, dataNum, b);

                /* フラグ初期化 */
                flag = 0;
                for (j = 0; j < dataNum; j++){
                    /* 観測値から偶現表を作成する */
                    DataReaderPopulationType(&resampledData1[j], &resampledData2[j], populationType);
                    /* mod 2010.11.10 メモリ使いまわす版の関数を呼び出す */
                    DataReaderMakeTableDiMemoryGiven(&resampledData1[j], &resampledData2[j], populationType, Tresample, data1, data2, data);
                    //DataReaderMakeTableDi(&resampledData1[j], &resampledData2[j], populationType, Tresample);
                    /* スコアを計算　スコア計算中0割になってしまう場合を-1を返す */
                    S = TableCalcScore(Tresample);
                    /* 最大スコアを決定する */
                    if (S >= Sobs) {
                        diff++;
                        /* S>=Sobsを判別するフラグをたてる */
                        flag = 1;
                    }
                }
                /* 一個以上違いがあるときに一個カウント */
                if (flag > 0){
                    diff++;
                }
            }
            typeIError[n] = diff/gen;
            /* 確認用標準出力 */
            //printf("%d\t%d\t%d\t%d\t%d\t%lf\n", n+1, a, b, copyNumber, burnin, diff/gen);
        }

finalize:;
        /* 確保したメモリの解放 */
        freeInt2Dim(X, 2);
        free1Dim(data);
        free1Dim(data1);
        free1Dim(data2);
        DataReaderSnpDataMemoryFree(resampledData1, dataNum);
        DataReaderSnpDataMemoryFree(resampledData2, dataNum);
        freeChar2Dim(sequences, a + b);
        freeInt2Dim(Tresample, ROW);
        free1Dim(populationType);
        free1Dim(freq);

    } // OpenMP スレッド並列終了

    //}

    return 0;
}
#endif


/* MCMCを実行 */
void TypeIMarkov(int **X, double *h, long L, dsfmt_t *snp_dsfmt_data)
{
    int j = 0;
    long u = 0;
    long v = 0;
    int XjuStar = 0;
    int XjvStar = 0;

    double c = 0;
    double tmp = 0;

    /* procedure 3 */
    j = TypeIZeroOne(snp_dsfmt_data);

    /* procedure 4 */
    u = (int)(myRand(snp_dsfmt_data) * L);

    if (0 == X[j][u]){ /* condition 5 */
        /* invariant */
    }
    else{ /* condition 6 */
        v = 0;
        do{
            v = (int)(myRand(snp_dsfmt_data) * L);
        } while(u == v);
        
        /* new candidate */
        XjuStar = X[j][u] - 1;
        XjvStar = X[j][v] + 1;

        /* transition probability */
        c = h[v] * X[j][u] / (h[u] * (X[j][v] +1));

        if (c >= 1){ /* condition 8 */
            X[j][u] = XjuStar;
            X[j][v] = XjvStar;
        }
        else{
            tmp = myRand(snp_dsfmt_data);
            if (tmp < c){
                X[j][u] = XjuStar;
                X[j][v] = XjvStar;
            }
        }

        /* procedure 10 */
    }
}

/* 最初のhaplotypeが全部というセット */
int TypeIStartSet2(long sampleSize, long L, int *result)
{
    long i = 0;
    long sum = 0;

    /* 配列初期化 */
    for(i = 0; i < L; i++){
        result[i] = 0;
    }

    result[0] = sampleSize;

    return 0;
}

/* 初期頻度に比例させた最初のセットを作成 */
int TypeIStartSet(long sampleSize, double *frequency, long L, int *result, dsfmt_t *snp_dsfmt_data)
{
    long i = 0;
    long sum = 0;
    long index = 0;

    /* 配列初期化 */
    for(i = 0; i < L; i++){
        result[i] = 0;
    }

    i = 0;
    for (i = 0; i < sampleSize; i++){
        index = TypeIDiscreteRandomValue(frequency, L, snp_dsfmt_data);
        result[index]++;
        //result[ TypeIDiscreteRandomValue(frequency, L) ]++;
    }

    return 0;
}

/* 初期頻度に比例した値をランダムに返す */
long TypeIDiscreteRandomValue(double *xlist, long len, dsfmt_t *snp_dsfmt_data)
{
    long i = 0;
    long num = 0;
    double x = 0;
    double all = 0;
    double tmp = 0;

    for (i = 0; i < len; i++){
        all += xlist[i];
    }
    x = myRand(snp_dsfmt_data) * all;

    for (i = 0; i < len; i++){
        tmp += xlist[i];
        if (x > tmp){
            num++;
        }
    }

    return num;
}

/* 確率1/2で0か1を返す */
int TypeIZeroOne(dsfmt_t *snp_dsfmt_data)
{
    /* return 0 or 1 with probability 0.5 */
    int result = 0;

    result = (int)(myRand(snp_dsfmt_data) * 2);

    return result;
}



/* ハプロタイプからXで表されている数の配列を作成 */
int TypeISampling(int *X, char **haplotype, long dataNum, long L, char **sequences)
{
    /* dataNum：SNP数 */
    /* L：ハプロタイプの数（種類の数） */
    long tmp = 0;
    long h = 0;
    long h2 = 0;

    for (h = 0; h < L; h++){
        for (h2 = 0; h2 < X[h]; h2++){
            strncpy(sequences[tmp], haplotype[h], dataNum);
            tmp++;
        }
    }

    return 0;
}

#ifdef MPI_USE
/**********************************************************************/
/* 入力データのMPIに対応した型の定義                                  */
/* mpi_InputStruct  O   MPI対応の型                                   */
/* 返り値： 0: 正常終了                                               */
/**********************************************************************/
int Define_MPI_InputStruct(MPI_Datatype *mpi_inputStruct)
{
    int i = 0;

/* MPI関係 */
#define MAX_BLOCK_NUM 10    /* MPI用構造体を作成するときの型のブロック数 */

    int blockcount[MAX_BLOCK_NUM];
    MPI_Aint address[MAX_BLOCK_NUM];
    MPI_Datatype type[MAX_BLOCK_NUM];

    SnpData sti;  // 元となる型（構造体）

    /* MPI用の構造体を作るパラメータの初期化 */
    for (i = 0; i < MAX_BLOCK_NUM; i++) {
        blockcount[i] = 0;
        address[i] = (MPI_Aint)NULL;
        type[i] = MPI_INT;
    }

    /* 入力データ用のMPI用の構造体定義 */
    blockcount[0] = RS_NUM_LEN;
    blockcount[1] = SNP_NUM;
    blockcount[2] = CHROM_LEN;
    blockcount[3] = 1;
    blockcount[4] = 3 + SNP_COUNT;
    MPI_Address(sti.rsNumber, &address[0]);
    MPI_Address(sti.SNPalleles, &address[1]);
    MPI_Address(sti.chrom, &address[2]);
    MPI_Address(&sti.pos, &address[3]);
    MPI_Address(&sti.allelesNum, &address[4]);
    for (i = 4; i >= 0; i--) {
        address[i] -= address[0];
    }
    /* 型指定 */
    type[0] = MPI_CHAR;
    type[1] = MPI_CHAR;
    type[2] = MPI_CHAR;
    type[3] = MPI_LONG;
    type[4] = MPI_INT;
    MPI_Type_struct(5, blockcount, address, type, mpi_inputStruct);
    MPI_Type_commit(mpi_inputStruct);

#undef MAX_BLOCK_NUM

	return 0;
}

/* デバッグ用 */
void CheckCopySNPData(SnpData *snpData, long dataNum, int MyMpiRank, int IDNum)
{
    long i = 0, j = 0;
    static long len = 0;
    static char *buf = NULL;

    if (len < 22 + snpData[i].dataNum * 2 + 2) {
        len = 22 + snpData[i].dataNum * 2 + 2;
        if (buf != NULL)  free(buf);
        buf = (char*)malloc(sizeof(char) * len);
        for (i = 0; i < len; i++)  buf[i] = '\0';
    }

    for (i = 0; i < dataNum; i++) {
        printf("%-4d  %-4d  %-6ld    ", IDNum, MyMpiRank, i); // 22文字
        for (j = 0; j < snpData[i].dataNum; j+=2) {
            printf("%c%c  ", snpData[i].SNPdata[j],
                             snpData[i].SNPdata[j+1]);
        }
        printf("\n");
    }

	return;
}
#endif /* MPI_USE */
