<?php

/* -------------------------------------------------------------------------
	ClientDetect class
	a part of PC2M Website Transcoder for Mobile Clients
	Copyright (C) 2005-2006 ucb.rcdtokyo and the contributors
	http://www.rcdtokyo.com/pc2m/note/
------------------------------------------------------------------------- */

define('CLIENTDETECT_USE_DNS_TO_RESOLVE_HOSTNAME', true);
// define('CLIENTDETECT_PATH_TO_WURFL_CONFIG_FILE', './wurfl/wurfl_config.php');

/* SAMPLE USER-AGENTS */

/* DoCoMo i-mode */
// $_SERVER['HTTP_USER_AGENT'] = 'DoCoMo/1.0/SH251iS/c10/TB/W24H12';
// $_SERVER['HTTP_USER_AGENT'] = 'DoCoMo/1.0/SH251iS/c10/TB/W24H12/serA1B2C3D4E5F';
// $_SERVER['HTTP_USER_AGENT'] = 'DoCoMo/2.0 N900i(c100;TB;W24H12)';
// $_SERVER['HTTP_USER_AGENT'] = 'DoCoMo/2.0 N900i(c100;TB;W24H12;serA1B2C3D4E5F6G7H;iccA1B2C3D4E5F6G7H8I9J0)';

/* DoCoMo PCSV */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.08 (N901iS;FOMA;c300;TB;W24H12)';

/* J-PHONE */
// $_SERVER['HTTP_USER_AGENT'] = 'J-PHONE/2.0/J-SH02';
// $_SERVER['HTTP_USER_AGENT'] = 'J-PHONE/5.0/V801SH/SNABCD1234567 SH/0005aa Profile/MIDP-1.0 Configuration/CLDC-1.0 Ext-Profile/JSCL-1.1.0';
// $_SERVER['HTTP_USER_AGENT'] = 'Vodafone/1.0/V902SH/SHJ001/SNABCD1234567 Browser/UP.Browser/7.0.2.1 Profile/MIDP-2.0 Configuration/CLDC-1.1 Ext-J-Profile/JSCL-1.2.2 Ext-V-Profile/VSCL-2.0.0';
// $_SERVER['HTTP_USER_AGENT'] = 'Vodafone/1.0/V702NK2/NKJ001/SNABCD1234567 Series60/2.6 Nokia6680/4.04.28 Profile/MIDP-2.0 Configuration/CLDC-1.1';
// $_SERVER['HTTP_USER_AGENT'] = 'MOT-C980/80.2F.2E. MIB/2.2.1 Profile/MIDP-2.0 Configuration/CLDC-1.1';
// $_SERVER['HTTP_X_JPHONE_MSNAME'] = 'V702sMO';
// $_SERVER['HTTP_X_JPHONE_DISPLAY'] = '176*220';
// $_SERVER['HTTP_X_JPHONE_COLOR'] = 'C65536';

/* UP.Browser */
// $_SERVER['HTTP_USER_AGENT'] = 'UP.Browser/3.04-ST13 UP.Link/3.4.5.9';
// $_SERVER['HTTP_USER_AGENT'] = 'OPWV-SDK/62K UP.Browser/6.2.0.5.136 (GUI) MMP/2.0';
// $_SERVER['HTTP_USER_AGENT'] = 'KDDI-SA31 UP.Browser/6.2.0.7.3.129 (GUI) MMP/2.0';
// $_SERVER['HTTP_USER_AGENT'] = 'SIE-M65/50 UP.Browser/7.0.2.2.d.3(GUI) MMP/2.0 Profile/MIDP-2.0 Configuration/CLDC-1.1';
// $_SERVER['HTTP_X_UP_DEVCAP_MAX_PDU'] = '131072';
// $_SERVER['HTTP_X_UP_DEVCAP_SCREENPIXELS'] = '230,242';
// $_SERVER['HTTP_X_UP_DEVCAP_SCREENDEPTH'] = '16,RGB565';
// $_SERVER['HTTP_X_UP_SUBNO'] = 'ABCD0123456789_UP.Link/3.4.5.9';

/* Willcom PCSV (K-TAI mode) except W-ZERO3 */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/3.0(DDIPOCKET;KYOCERA/AH-K3001V/1.1.16.65.000000/0.1/C100) Opera 7.0';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/3.0(WILLCOM;KYOCERA/WX310K/2;1.1.5.15.000000/0.1/C100) Opera 7.0';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/3.0(WILLCOM;JRC/WX310J/2;1/1/C128) NetFront 3.3';

/* Willcom PCSV (Browser's native mode) */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.0 (compatible; MSIE 6.0; KYOCERA/AH-K3001V/1.4.1.67.000000/0.1) Opera 7.0 [ja]';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.0 (compatible; MSIE 6.0; KYOCERA/WX310K/2;1.1.5.15.000000/0.1) Opera 7.0 [ja]';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.08(MobilePhone;NMCS/3.3) NetFront/3.3';

/* Generic Web browsers */
/* Opera for W-ZERO3 */
// $_SERVER['HTTP_USER_AGENT'] = 'Opera/8.50 (Windows CE; U) [SHARP/WS003SH; PPC; 480x640]';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.78 (Windows CE; U) Opera 8.50 [SHARP/WS003SH; PPC; 480x640]';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; SHARP/WS003SH; PPC; 480x640) Opera 8.50 [ja]';
/* Internet Explorer Mobile (Pocket IE) */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/2.0 (compatible; MSIE 3.02; Windows CE)';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/2.0 (compatible; MSIE 3.02; Windows CE; PPC; 240x320)';
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.0 (compatible; MSIE 4.01; Windows CE; PPC; 240x320)';
/* Opera Mini */
// $_SERVER['HTTP_USER_AGENT'] = 'Opera/8.01 (J2ME/MIDP; Opera Mini/1.2.2958; en; U; ssr)';
/* Nokia S60 */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/5.0 (SymbianOS/9.1; U; en-us) AppleWebKit/109 (KHTML, like Gecko) Safari/109';

/* Sample user agents at http://www.osnews.com/story.php?news_id=13446 */
// $_SERVER['HTTP_PROFILE'] = 'http://www.example.com/example.xml';
/* Opera Mobile for Nokia 6600 Series 60 */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.0 (compatible; MSIE 6.0; Symbian OS; Nokia 6600/4.09.1; 6936) Opera 8.50 [en]';
/* NetFront for Sanyo MM-8300 */
// $_SERVER['HTTP_USER_AGENT'] = 'Mozilla/4.0 (MobilePhone MM-8300/US/1.0) NetFront/3.1 MMP/2.0';
/* Teleca Obigo for Samsung SPHa920 */
// $_SERVER['HTTP_USER_AGENT'] = 'Samsung-SPHA920 AU-MIC-A920/2.0 MMP/2.0';
/* Sony Ericsson SEMC-Browser for Sony Ericsson K750i */
// $_SERVER['HTTP_USER_AGENT'] = 'SonyEricssonK750i/R1BC Browser/SEMC-Browser/4.2 Profile/MIDP-2.0 Configuration/CLDC-1.1';
/* Nokia S40 */
// $_SERVER['HTTP_USER_AGENT'] = 'Nokia6230i/2.0 (03.23) Profile/MIDP-2.0 Configuration/CLDC-1.1';
/* Motorola E398 */
// $_SERVER['HTTP_USER_AGENT'] = 'MOT-E398/0E.20.59R MIB/2.2.1 Profile/MIDP-2.0 Configuration/CLDC-1.0';

class ClientDetect
{
	var $user_agent = '';
	var $dev_type = 'mozilla';
	var $cache_size = 100000;
	var $screen = array(
		'width' => 240,
		'color' => 65536,
		'gif_support' => true,
		'jpg_support' => true,
		'png_support' => true
	);
	var $serial_id = '';
	var $hostname = '';
	var $is_mobile = false;
	var $is_postcapable = true;
	var $preferred_markup = 'xhtml';
	var $chtml = '2.0';
	var $accept = array();
	var $accept_charset = array();

	function ClientDetect()
	{
		ClientDetect::__construct();
	}

	function __construct()
	{
		$this->user_agent = isset($_SERVER['HTTP_USER_AGENT'])? $_SERVER['HTTP_USER_AGENT']: '';
		if (0 === strpos($this->user_agent, 'DoCoMo/')) {
			$this->_detectDoCoMo();
		} elseif (isset($_SERVER['HTTP_X_JPHONE_MSNAME'])) {
			$this->_detectJPhone();
		} elseif (preg_match('/\bUP\.Browser\b/s', $this->user_agent)) {
			$this->_detectUPBrowser();
		} elseif (preg_match('/\b(?:WILLCOM|DDIPOCKET)\b/s', $this->user_agent)) {
			$this->_detectWillcomPCSV();
		} elseif (isset($_SERVER['HTTP_X_WAP_PROFILE'])
			or isset($_SERVER['HTTP_PROFILE'])) {
			$this->_detectMiscMobile();
		} else {
			$this->_detectMisc();
		}
		$this->_detectHost();
		$this->_detectAcceptCharset();
		$this->_detectAccept();
		$this->_overrideGraphicSupport();
		$this->_overrideScreenSize();
		$this->_overridePreferredMarkup();
	}

	/**
	 * DoCoMo mova:      ser[serial number (11 digit alphanumeric)]
	 * DoCoMo FOMA:      icc[FOMA card serial number (20 digits alphanumeric)]
	 * EZweb:            [vendor id (4 digit numeric)][terminal id (10 digit numeric)]_[UP.Link server]
	 * Vodafone/J-Phone: SN[4 digit alphabet][7 digit numeric]
	 *
	 * @return string
	 * @access public
	 */
	function getSerialId()
	{
		return $this->serial_id;
	}

	function getCacheSize()
	{
		return $this->cache_size;
	}

	function getScreenInfo()
	{
		return $this->screen;
	}

	function getHostname()
	{
		return $this->hostname;
	}

	function _detectDoCoMo()
	{
		$this->dev_type = 'docomo';
		$this->preferred_markup = 'chtml';
		if (preg_match('/^(DoCoMo\/[\d\.]+)[\s\/]([^\/\(\s]+)[\/\(\s](.+)/', $this->user_agent, $matches)) {
			$this->_retrieveDevInfoDoCoMo($matches[2]);
			switch ($matches[1]) {
				case 'DoCoMo/1.0':
					$array = explode('/', $matches[3]);
					break;
				default:
					$array = explode(';', preg_replace('/^(.+)\)$/', '$1', $matches[3]));
			}
			array_walk($array, array(&$this, '_trimExploded'));
			// Cache Size:
			// Returns the minimum spec if the UA value does not contain cache size
			$this->cache_size = (preg_match('/^c(\d+)$/', $array[0], $matches))?
				$matches[1] * 1000:
				5000;
			// Serial ID:
			// mova: ser[serial number (11 digit alphanumeric)]
			// FOMA: icc[FOMA card serial number (20 digits alphanumeric)]
			foreach (array_reverse($array) as $value) {
				if (preg_match('/^(?:ser\w{11}|icc\w{20})$/', $value)) {
					$this->serial_id = $value;
					break;
				}
			}
		}
	}

	function _retrieveDevInfoDoCoMo($name)
	{
		// Reading device information file.
		if (false !== @include 'DevInfo_DoCoMo.inc.php') {
			if (isset($_dev_info[$name])) {
				$this->screen = array(
					'width' => $_dev_info[$name][0],
					'color' => $_dev_info[$name][1],
					'gif_support' => $_dev_info[$name][2],
					'jpg_support' => $_dev_info[$name][3],
					'png_support' => $_dev_info[$name][4]
				);
				$this->chtml = $_dev_info[$name][5];
			// If the model name is not in the list, it is supposed to be a new one.
			// So we return average spec of 'recent' devices.
			} else {
				$this->screen = array(
					'width' => 240,
					'color' => 65536,
					'gif_support' => true,
					'jpg_support' => true,
					'png_support' => false
				);
				$this->chtml = '6.0';
			}
		// If the device information file is not available,
		// we return the minimum spec of the known color device.
		} else {
			$this->screen = array(
				'width' => 96,
				'color' => 256,
				'gif_support' => true,
				'jpg_support' => false,
				'png_support' => false
			);
		}
	}

	function _detectJPhone()
	{
		$this->dev_type = 'vodafone';
		if ('' === $this->user_agent) {
			$version = 'J-PHONE/2.0';
		} else {
			$array = explode(' ', $this->user_agent);
			$array = explode('/', $array[0]);
			array_walk($array, array(&$this, '_trimExploded'));
			$version = array_shift($array).'/'.array_shift($array);
			// Serial ID:
			// ID format is supposed to be SN[4 digit alphabet][7 digit numeric]
			// But there seems no official statement about the number of digits.
			foreach ($array as $value) {
				if (preg_match('/^SN\w+$/', $value)) {
					$this->serial_id = $value;
					break;
				}
			}
		}
		// Cache size is 6KB for 'J-PHONE/2.0' and 'J-PHONE/3.0' (Packet incompatible),
		// 12KB for 'J-PHONE/4.x', 200KB for 'J-PHONE/5.0', 300KB for others.
		// 'J-PHONE/*.*' does not support GIF and 'J-PHONE/2.0' does not support JPEG.
		// 'J-PHONE/4.0' and earlier does not accept XHTML.
		if (preg_match('/^J-PHONE\/(\d)\.\d$/', $version, $matches)) {
			$this->screen['gif_support'] = false;
			switch ($matches[1]) {
				case 5:
					$this->cache_size = 200000;
					break;
				case 4:
					$this->cache_size = 12000;
					$this->preferred_markup = 'chtml';
					break;
				case 3:
					$this->cache_size = 6000;
					$this->preferred_markup = 'chtml';
					break;
				default:
					$this->cache_size = 6000;
					$this->preferred_markup = 'chtml';
					$this->is_postcapable = false;
					$this->screen['jpg_support'] = false;
			}
		} else {
			$this->cache_size = 300000;
		}
		// If the server variables are not available,
		// we return the minimum spec of the known color device.
		if (isset($_SERVER['HTTP_X_JPHONE_DISPLAY'])) {
			list($this->screen['width']) = explode('*', $_SERVER['HTTP_X_JPHONE_DISPLAY']);
		} else {
			$this->screen['width'] = 96;
		}
		if (isset($_SERVER['HTTP_X_JPHONE_COLOR'])) {
			$this->screen['color'] = (int) preg_replace('/^[C|G](\d+)$/',"$1",$_SERVER['HTTP_X_JPHONE_COLOR']);
		} else {
			$this->screen['color'] = 256;
		}
	}

	function _detectUPBrowser()
	{
		$this->dev_type = 'au';
		if (preg_match('/\bUP\.Browser\/(\d+)/s', $this->user_agent, $matches)
			and (int) $matches[1] < 6) {
			$this->preferred_markup = 'chtml';
		}
		if (isset($_SERVER['HTTP_X_UP_SUBNO'])) {
			$this->serial_id = $_SERVER['HTTP_X_UP_SUBNO'];
		}
		if (isset($_SERVER['HTTP_X_UP_DEVCAP_MAX_PDU'])) {
			$this->cache_size = $_SERVER['HTTP_X_UP_DEVCAP_MAX_PDU'];
		}
		// If the server variables are not available,
		// we return the minimum spec of the known color device.
		if (isset($_SERVER['HTTP_X_UP_DEVCAP_SCREENPIXELS'])) {
			list($this->screen['width']) = explode(',', $_SERVER['HTTP_X_UP_DEVCAP_SCREENPIXELS']);
		} else {
			$this->screen['width'] = 96;
		}
		if (isset($_SERVER['HTTP_X_UP_DEVCAP_SCREENDEPTH'])) {
			list($this->screen['color']) = explode(',', $_SERVER['HTTP_X_UP_DEVCAP_SCREENDEPTH']);
			$this->screen['color'] = pow(2, (int) $this->screen['color']);
		} else {
			$this->screen['color'] = 256;
		}
		if ($this->screen['color'] < 256) {
			if ($this->cache_size > 1200) {
				$this->cache_size = 1200;
			}
		} else {
			if ($this->preferred_markup == 'chtml' and $this->cache_size > 7500) {
				$this->cache_size = 7500;
			}
		}
		$this->screen['gif_support'] = false;
		$this->screen['jpg_support'] = false;
		$this->screen['png_support'] = true;
	}

	function _detectWillcomPCSV()
	{
		$this->dev_type = 'willcom_pcsv';
		if (preg_match('/\(([^\)]+)\)/', $this->user_agent, $matches)) {
			$array = array();
			foreach (explode(';', $matches[1]) as $value) {
				$array = array_merge($array, explode('/', $value));
			}
			array_walk($array, array(&$this, '_trimExploded'));
			if (isset($array[2])) {
				$this->_detectWillcomScreenWidth($array[2]);
			}
			foreach (array_reverse($array) as $value) {
				if (preg_match('/^C(\d+)$/', $value, $matches)) {
					$this->cache_size = $matches[1] * 1000;
					break;
				}
			}
		}
	}

	function _detectWillcomScreenWidth($name)
	{
		switch ($name) {
			case 'AH-J3001V':
			case 'AH-J3002V':
			case 'AH-J3003S':
				$this->screen['width'] = 128;
				break;
			case 'AH-K3001V':
			case 'AH-K3002V':
			case 'WX300K':
			case 'WX310K':
				$this->screen['width'] = 237;
				break;
			case 'WX310SA':
			case 'WX310J':
			default:
				$this->screen['width'] = 240;
		}
	}

	function _detectMisc()
	{
		$detected = false;
		if (preg_match('/\b(?:Windows\s?CE|Palm\s?OS|Opera|NetFront|jig\s?browser)\b/s', $this->user_agent)) {
			$this->dev_type = 'misc_mobile';
			$detected = true;
		}
		if (preg_match('/^Mozilla\/(\d)[\d\.]+\s?\(([^\)]+)\)(.*)$/', $this->user_agent, $matches)) {
			$detected = true;
			if ((int) $matches[1] < 4) {
				$this->preferred_markup = 'chtml';
			}
			$array = explode(';', $matches[2]);
			if ('' !== $matches[3]) {
				array_push($array, $matches[3]);
			}
			array_walk($array, array(&$this, '_trimExploded'));
			// DoCoMo PCSV
			if (isset($array[1]) and $array[1] == 'FOMA') {
				$this->dev_type = 'docomo_pcsv';
				$this->_retrieveDevInfoDoCoMo($array[0]);
				$this->screen['png_support'] = true;
				// Cache Size:
				$this->cache_size = (isset($array[2])
					and preg_match('/^c(\d{1,})$/', $array[2], $matches))?
					$matches[1] * 1000:
					100000;
			// Willcom PCSV (Browser's native mode)
			} elseif (isset($array[2]) and preg_match('/^(?:KYOCERA|JRC|SANYO)\//', $array[2])) {
				$this->dev_type = 'willcom_pcsv';
				list($vendor, $name) = explode('/', $array[2]);
				$this->_detectWillcomScreenWidth($name);
			}
		}
		if (!$detected) {
			$this->_detectMiscMobile();
		}
	}

	function _detectMiscMobile()
	{
		$this->dev_type = 'misc_mobile';
		$this->cache_size = 5000;
		$this->screen = array(
			'width' => 96,
			'color' => 256,
			'gif_support' => true,
			'jpg_support' => true,
			'png_support' => true
		);
		$this->is_postcapable = false;
		$this->preferred_markup = 'chtml';
		$this->_detectByWurfl();
	}

	function _detectByWurfl()
	{
		if (defined('CLIENTDETECT_PATH_TO_WURFL_CONFIG_FILE')
			and file_exists(CLIENTDETECT_PATH_TO_WURFL_CONFIG_FILE)) {
			require_once CLIENTDETECT_PATH_TO_WURFL_CONFIG_FILE;
			require_once WURFL_CLASS_FILE;
			$wurfl = new wurfl_class;
			$wurfl->GetDeviceCapabilitiesFromAgent($this->user_agent);
			/*
			echo '<pre>';
			print_r($wurfl->capabilities['product_info']);
			echo '</pre>';
			*/
			if (false !== $x = $wurfl->GetDeviceCapability('preferred_markup')) {
				if (false !== strpos($x, 'xhtml')) {
					$this->preferred_markup = 'xhtml';
				} else {
					$this->preferred_markup = 'chtml';
				}
			}
			if (false !== $x = $wurfl->GetDeviceCapability('max_deck_size')) {
				$this->cache_size = ($x < 5000)? 5000: $x;
			}
			if (false !== $x = $wurfl->GetDeviceCapability('max_image_width')) {
				$this->screen['width'] = $x;
			}
			if (false !== $x = $wurfl->GetDeviceCapability('colors')) {
				$this->screen['color'] = $x;
			}
			$this->screen['gif_support'] = $wurfl->GetDeviceCapability('gif');
			$this->screen['jpg_support'] = $wurfl->GetDeviceCapability('jpg');
			$this->screen['png_support'] = $wurfl->GetDeviceCapability('png');
			$this->is_postcapable = $wurfl->GetDeviceCapability('post_method_support');
		}
	}

	function _detectAccept()
	{
		if (isset($_SERVER['HTTP_ACCEPT'])) {
			$this->accept = explode(',', $_SERVER['HTTP_ACCEPT']);
			array_walk($this->accept, array(&$this, '_trimHttpValue'));
		}
	}

	function _detectAcceptCharset()
	{
		if (isset($_SERVER['HTTP_X_UP_DEVCAP_CHARSET'])) {
			$this->accept_charset = explode(',', $_SERVER['HTTP_X_UP_DEVCAP_CHARSET']);
			array_walk($this->accept_charset, array(&$this, '_trimHttpValue'));
		} elseif (isset($_SERVER['HTTP_ACCEPT_CHARSET'])) {
			$this->accept_charset = explode(',', $_SERVER['HTTP_ACCEPT_CHARSET']);
			array_walk($this->accept_charset, array(&$this, '_trimHttpValue'));
		}
	}

	function _trimHttpValue(&$item) {
		$array = explode(';', $item);
		$item = strtolower(trim($array[0]));
	}

	function _trimExploded(&$item) {
		$item = trim($item);
	}

	function _overrideScreenSize()
	{
		if (preg_match('/\b(\d{2,4})x\d{2,4}\b/', $this->user_agent, $matches)) {
			$this->screen['width'] = $matches[1];
		}
	}

	function _overrideGraphicSupport()
	{
		if (in_array('image/gif', $this->accept)) {
			$this->screen['gif_support'] = true;
		}
		if (in_array('image/jpeg', $this->accept)) {
			$this->screen['jpg_support'] = true;
		}
		if (in_array('image/png', $this->accept)) {
			$this->screen['png_support'] = true;
		}
	}

	function _overridePreferredMarkup()
	{
		if (in_array('application/vnd.wap.xhtml+xml', $this->accept)) {
			$this->preferred_markup = 'xhtml';
		} elseif (in_array('application/xhtml+xml', $this->accept)) {
			$this->preferred_markup = 'xhtml';
		} elseif (in_array('text/html', $this->accept)) {
			$this->preferred_markup = 'chtml';
		}
	}

	/**
	 * This method checks if the client is from mobile operators' network
	 * and returns domain suffix for valid mobile clients.
	 * REMOTE_HOST or REMOTE_ADDR is returned as it is for the others.
	 * There are two ways to detect if the client is from valid network.
	 * One is to check if the hostname contains valid domain suffix.
	 * Anotehr is to check if the IP address in the valid network range.
	 * This method takes hostname prior to IP address, and calls getHostByAddr
	 * at first to resolve the client's hostname when REMOTE_HOST is not available.
	 * If your DNS resolver does not work for the function
	 * or if you do not like the DNS resolution overhead,
	 * you may comment out the line including getHostByAddr.
	 *
	 * @return string
	 * @access public
	 */
	function _detectHost()
	{
		$remote_host = '';
		if (isset($_SERVER['REMOTE_HOST']) and $_SERVER['REMOTE_HOST']) {
			$remote_host = $_SERVER['REMOTE_HOST'];
		} elseif (isset($_SERVER['REMOTE_ADDR']) and $_SERVER['REMOTE_ADDR']) {
			$remote_host = (defined('CLIENTDETECT_USE_DNS_TO_RESOLVE_HOSTNAME')
				and CLIENTDETECT_USE_DNS_TO_RESOLVE_HOSTNAME === true)?
				getHostByAddr($_SERVER['REMOTE_ADDR']):
				$_SERVER['REMOTE_ADDR'];
		}
		if (preg_match('/^.+(\.(?:docomo|ezweb|brew|jp-[ckqt]|prin|mopera)\.ne\.jp)$/', $remote_host, $matches)) {
			$this->hostname = $matches[1];
			$this->is_mobile = true;

		// Check if the IP address is in the network address range
		// if DNS resolver does not work for getHostByAddr.
		// Mopera is not detected in this case.
		} elseif (preg_match('/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/', $remote_host)
			and false !== $x = $this->checkIpRange($remote_host)) {
			$this->is_mobile = true;
			switch ($x) {
				case 'docomo':
					$this->hostname = '.docomo.ne.jp';
					break;
				case 'au':
					$this->hostname = '.ezweb.ne.jp';
					break;
				case 'vodafone':
					$this->hostname = '.jp-t.ne.jp';
					break;
				case 'willcom':
					$this->hostname = '.prin.ne.jp';
					break;
			}
		} else {
			$this->hostname = $remote_host;
		}
	}

	/**
	 * This method returns an abbreviated name of the mobile operator
	 * when the IP address is in their network address range.
	 * Otherwise returns false.
	 * This method can be called statically,
	 * but is usually called from getHostname method
	 * only when DNS resolver does not work for getHostByAddr.
	 *
	 * @param string  $address
	 * @return string (docomo|au|vodafone|willcom)
	 * @access public
	 */
	function checkIpRange($address)
	{
		if (false !== @include 'IPRange.inc.php') {
			$address = $this->_dumpAddress($address);
			if ($this->_compareIp($address, $_iprange_docomo)) {
				return 'docomo';
			} elseif ($this->_compareIp($address, $_iprange_au)) {
				return 'au';
			} elseif ($this->_compareIp($address, $_iprange_vodafone)) {
				return 'vodafone';
			} elseif ($this->_compareIp($address, $_iprange_willcom)) {
				return 'willcom';
			}
		}
		return false;
	}

	/**
	 * Private method used by checkIpRange method
	 *
	 * @param  string  $address
	 * @param  array   $iprange
	 * @return bool
	 * @access private
	 */
	function _compareIp($address, $iprange)
	{
		foreach ($iprange as $value) {
			list($network, $mask) = explode('/', $value);
			$network = $this->_dumpAddress($network);
			$mask = $this->_dumpNetmask($mask);
			if (($address & $mask) == ($network & $mask)) {
				return true;
				break;
			}
		}
		return false;
	}

	/**
	 * Private method used by checkIpRange method
	 *
	 * @access private
	 * @param  string  $mask
	 * @return string
	 */
	function _dumpNetmask($mask)
	{
		$i = 0;
		$x = '';
		while ($i < $mask) {
			$x .= '1';
			$i++;
		}
		while ($i < 32) {
			$x .= '0';
			$i++;
		}
		$array = array();
		$array[] = bindec(substr($x, 0, 8));
		$array[] = bindec(substr($x, 8, 8));
		$array[] = bindec(substr($x, 16, 8));
		$array[] = bindec(substr($x, 24, 8));
		return ($array[0] << 24) | ($array[1] << 16) | ($array[2] << 8) | $array[3];
	}

	/**
	 * Private method used by checkIpRange method
	 *
	 * @param  string  $address
	 * @return string
	 * @access private
	 */
	function _dumpAddress($address)
	{
		$array = explode('.', $address);
		return ($array[0] << 24) | ($array[1] << 16) | ($array[2] << 8) | $array[3];
	}
}

?>
