package Win32::Dokan;

use strict;
use warnings;

use threads;

use Carp;

use Win32::Dokan::DokanFileInfo;
use Win32::Dokan::DokanOptions;
use Win32::Dokan::FileInfo;
use Win32::Dokan::FindDataW;

use Encode;
use Fcntl qw(:DEFAULT);
use Errno;

require Exporter;
# use AutoLoader;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Win32::Dokan ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
	DOKAN_DRIVER_INSTALL_ERROR
	DOKAN_DRIVE_LETTER_ERROR
	DOKAN_ERROR
	DOKAN_MOUNT_ERROR
	DOKAN_START_ERROR
	DOKAN_SUCCESS
	FILE_ATTRIBUTE_READONLY
	FILE_ATTRIBUTE_HIDDEN
	FILE_ATTRIBUTE_SYSTEM
	FILE_ATTRIBUTE_DIRECTORY
	FILE_ATTRIBUTE_ARCHIVE
	FILE_ATTRIBUTE_DEVICE
	FILE_ATTRIBUTE_NORMAL
	FILE_ATTRIBUTE_TEMPORARY
	FILE_ATTRIBUTE_SPARSE_FILE
	FILE_ATTRIBUTE_REPARSE_POINT
	FILE_ATTRIBUTE_COMPRESSED
	FILE_ATTRIBUTE_OFFLINE
	FILE_ATTRIBUTE_NOT_CONTENT_INDEXED
	FS_CASE_IS_PRESERVED
	FS_CASE_SENSITIVE
	FS_UNICODE_STORED_ON_DISK
	FS_PERSISTENT_ACLS
	FS_FILE_COMPRESSION
	FS_VOL_IS_COMPRESSED
	FILE_NAMED_STREAMS
	FILE_SUPPORTS_ENCRYPTION
	FILE_SUPPORTS_OBJECT_IDS
	FILE_SUPPORTS_REPARSE_POINTS
	FILE_SUPPORTS_SPARSE_FILES
	FILE_VOLUME_QUOTAS
) ] );


our @EXPORT_OK = qw(
	DOKAN_DRIVER_INSTALL_ERROR
	DOKAN_DRIVE_LETTER_ERROR
	DOKAN_ERROR
	DOKAN_MOUNT_ERROR
	DOKAN_START_ERROR
	DOKAN_SUCCESS
	FILE_ATTRIBUTE_READONLY
	FILE_ATTRIBUTE_HIDDEN
	FILE_ATTRIBUTE_SYSTEM
	FILE_ATTRIBUTE_DIRECTORY
	FILE_ATTRIBUTE_ARCHIVE
	FILE_ATTRIBUTE_DEVICE
	FILE_ATTRIBUTE_NORMAL
	FILE_ATTRIBUTE_TEMPORARY
	FILE_ATTRIBUTE_SPARSE_FILE
	FILE_ATTRIBUTE_REPARSE_POINT
	FILE_ATTRIBUTE_COMPRESSED
	FILE_ATTRIBUTE_OFFLINE
	FILE_ATTRIBUTE_NOT_CONTENT_INDEXED
	FS_CASE_IS_PRESERVED
	FS_CASE_SENSITIVE
	FS_UNICODE_STORED_ON_DISK
	FS_PERSISTENT_ACLS
	FS_FILE_COMPRESSION
	FS_VOL_IS_COMPRESSED
	FILE_NAMED_STREAMS
	FILE_SUPPORTS_ENCRYPTION
	FILE_SUPPORTS_OBJECT_IDS
	FILE_SUPPORTS_REPARSE_POINTS
	FILE_SUPPORTS_SPARSE_FILES
	FILE_VOLUME_QUOTAS
);

our $VERSION = '0.01_1';

sub AUTOLOAD {
    # This AUTOLOAD is used to 'autoload' constants from the constant()
    # XS function.

    my $constname;
    our $AUTOLOAD;
    ($constname = $AUTOLOAD) =~ s/.*:://;
    croak "&Win32::Dokan::constant not defined" if $constname eq 'constant';
    my ($error, $val) = constant($constname);
    if ($error) { croak $error; }
    {
	no strict 'refs';
	# Fixed between 5.005_53 and 5.005_61
#XXX	if ($] >= 5.00561) {
#XXX	    *$AUTOLOAD = sub () { $val };
#XXX	}
#XXX	else {
	    *$AUTOLOAD = sub { $val };
#XXX	}
    }
    goto &$AUTOLOAD;
}

require XSLoader;
XSLoader::load('Win32::Dokan', $VERSION);

our $Fs;

# CreationDisposition
use constant {
    CREATE_NEW => 1,
    CREATE_ALWAYS => 2,
    OPEN_EXISTING => 3,
    OPEN_ALWAYS => 4,
    TRUNCATE_EXISTING => 5};

# DesiredAccess
use constant {
    GENERIC_READ => 0x80000000,
    GENERIC_WRITE => 0x40000000,
    GENERIC_EXECUTE => 0x20000000};

# ShareMode
use constant {
    FILE_SHARE_READ => 0x00000001,
    FILE_SHARE_WRITE => 0x00000002,
    FILE_SHARE_DELETE => 0x00000004};

# open error
use constant {
    ERROR_DIR_NOT_EMPTY => 145,
    ERROR_ALREADY_EXISTS => 183,
};

sub _fsname_to_unicode {
    my $self = shift;
    my $name = shift;

    my $encoding = $Fs->encoding;
    if (!defined($encoding)) {
	return encode('UTF-16LE', $name);
    }
    elsif ($encoding eq 'X-MBCS') {
	return $self->convert_native_to_unicode($name);
    }
    else {
	return encode('UTF-16LE', decode($encoding, $name));
    }
}

sub _unicode_to_fsname {
    my $self = shift;
    my $name = shift;

    my $ret = decode('UTF-16LE', $name);
    $ret =~ s/\\/\//g;

    my $encoding = $Fs->encoding;
    if (!defined($encoding)) {
	return $ret;
    }
    elsif ($encoding eq 'X-MBCS') {
	return $self->convert_unicode_to_native(encode('UTF-16LE', $ret));
    }
    else {
	return encode($encoding, $ret);
    }
}

sub _call_cb_simple {
    my $self = shift;
    my $method = shift;
    my $fileName = shift;
    # rest is in @_

    $fileName = $self->_unicode_to_fsname($fileName);

    if (wantarray()) {
	my @ret;
	eval {
	    @ret = $self->$method($fileName, @_);
	};
	print STDERR "************** exception: $@" if ($@);
	return $@ ? -1 : @ret;
    }
    else {
	my $ret;
	eval {
	    $ret = $self->$method($fileName, @_);
	};
	print STDERR "************** exception: $@" if ($@);
	return $@ ? -1 : $ret;
    }
}

sub _cb_create_file {
    shift->_call_cb_simple("cb_create_file", @_);
}

sub cb_create_file {
    my $self = shift;
    my ($fileName, $desiredAccess, $shareMode, $createDisposition, $flagsAndAttributes, $DFileInfo) = @_;

    my $ret;
    my $mode = 0;

    if ($DFileInfo->is_directory) {
	return 0;
    }

    # "/*" style access ?
    $fileName = $1 if ($fileName =~ /(.*)\/\*?$/);

    my $is_dir = $Fs->opendir($fileName);
    if (defined($is_dir) && $is_dir >= 0) {
	$DFileInfo->is_directory(1);

	if ($createDisposition == CREATE_ALWAYS
	    || $createDisposition == OPEN_ALWAYS) {
	    return ERROR_ALREADY_EXISTS;
	}
	return 0;
    }

    if ($shareMode & FILE_SHARE_READ && $shareMode & FILE_SHARE_WRITE) {
	$mode = O_RDWR;
    }
    elsif ($shareMode & FILE_SHARE_READ) {
	$mode = O_RDONLY;
    }
    elsif ($shareMode & FILE_SHARE_WRITE) {
	$mode = O_WRONLY;
    }
    else {
	$mode = O_RDONLY;
    }

    if ($createDisposition == CREATE_NEW) {
	$mode |= (O_CREAT | O_EXCL);
	$ret = $Fs->create($fileName, $mode, $DFileInfo);
	$ret = 0 if (defined($ret) && $ret >= 0);
    }
    elsif ($createDisposition == CREATE_ALWAYS
	   || $createDisposition == OPEN_ALWAYS) {

	my $stat = $Fs->stat("$fileName", $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    $ret = $Fs->open($fileName, $mode, $DFileInfo);
	    $ret = ERROR_ALREADY_EXISTS if (defined($ret) && $ret >= 0);
	}
	else {
	    $mode |= O_WRONLY unless ($mode & O_RDWR || $mode & O_WRONLY);
	    $mode |= O_CREAT;
	    $ret = $Fs->create($fileName, $mode, $DFileInfo);
	    $ret = 0 if (defined($ret) && $ret >= 0);
	}
    }
    elsif ($createDisposition == OPEN_EXISTING) {
	$ret = $Fs->open($fileName, $mode, $DFileInfo);
	$ret = 0 if (defined($ret) && $ret >= 0);
    }
    elsif ($createDisposition == TRUNCATE_EXISTING) {
	my $stat = $Fs->stat("$fileName", $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    $mode |= O_WRONLY unless ($mode & O_RDWR || $mode & O_WRONLY);
	    $mode |= O_TRUNC;
	    $ret = $Fs->create($fileName, $mode, $DFileInfo);
	    $ret = 0 if (defined($ret) && $ret >= 0);
	}
	else {
	    $ret = -2;
	}
    }

    $ret = -2 unless (defined($ret));

    return $ret;
}

sub _cb_open_directory {
    shift->_call_cb_simple("cb_open_directory", @_);
}

sub cb_open_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    my $ret = $Fs->opendir($pathName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_create_directory {
    shift->_call_cb_simple("cb_create_directory", @_);
}

sub cb_create_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;
 
    my $ret = $Fs->mkdir($pathName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_cleanup {
    shift->_call_cb_simple("cb_cleanup", @_);
}

sub cb_cleanup {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    my $ret = $Fs->cleanup($fileName, $DFileInfo);
    return -1 unless (defined($ret));
    return $ret if ($ret < 0);

    if ($DFileInfo->delete_on_close) {
	if ($DFileInfo->is_directory) {
	    $ret = $Fs->rmdir($fileName, $DFileInfo);
	}
	else {
	    $ret = $Fs->remove($fileName, $DFileInfo);
	}
    }

    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_close_file {
    shift->_call_cb_simple("cb_close_file", @_);
}

sub cb_close_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    my $ret = $Fs->close($fileName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_read_file {
    shift->_call_cb_simple("cb_read_file", @_);
}

sub cb_read_file {
    my $self = shift;
    my ($fileName, $dummy, $offset, $length, $DFileInfo) = @_;

    use Data::Dumper;
    my ($data, $err) = $Fs->read($fileName, $offset, $length, $DFileInfo);
    unless (defined($data)) {
	return $err if (defined($err) && $err < 0);
	return -1;
    }

    $_[1] = $data;

    return 0;
}

sub _cb_write_file {
    shift->_call_cb_simple("cb_write_file", @_);
}

sub cb_write_file {
    my $self = shift;
    my ($fileName, $data, $offset, $dummy, $DFileInfo) = @_;
 
    my ($len, $err) = $Fs->write($fileName, $offset, $data, $DFileInfo);
    $_[3] = defined($len) ? $len : 0;


    return $err if (defined($err) && $err < 0);
    return defined($len) ? 0 : -1;
}

sub _cb_flush_file_buffers {
    shift->_call_cb_simple("cb_flush_file_buffers", @_);
}

sub cb_flush_file_buffers {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    return $Fs->flush($fileName, $DFileInfo);
}

sub _cb_get_file_information {
    shift->_call_cb_simple("cb_get_file_information", @_);
}

sub cb_get_file_information {
    my $self = shift;
    my ($fileName, $fileInfo, $DFileInfo) = @_;

    # "/*" style access ?
    $fileName = $1 if ($fileName =~ /(.*)\/\*?$/);
    my $stat = $Fs->stat($fileName, $DFileInfo);
    if (ref($stat) && ref($stat) eq 'ARRAY') {
	$fileInfo->file_size(int($stat->[0] || 0));
	$fileInfo->file_attributes(int($stat->[1]));
	$fileInfo->creation_time($stat->[2]);
	$fileInfo->last_access_time($stat->[3]);
	$fileInfo->last_write_time($stat->[4]);

	return 0;
    }

    return (defined($stat) && $stat < 0) ? $stat : -1;
}

sub _cb_find_files {
    reverse shift->_call_cb_simple("cb_find_files", @_);
}

sub cb_find_files {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    $pathName = $1 if ($pathName =~ /(.*)\/\*?$/);

    my $elements = $Fs->readdir($pathName, $DFileInfo);
    unless (ref($elements) && ref($elements) eq 'ARRAY') {
	return (defined($$elements) && $elements < 0) ? $elements : -1;
    }

    my @ret;
    for my $e (@{$elements}) {
	my $find_data = Win32::Dokan::FindDataW->new();

	my $stat = $Fs->stat("$pathName/$e", $DFileInfo);
	if (ref($stat) && ref($stat) eq 'ARRAY') {
	    $find_data->file_name($self->_fsname_to_unicode($e));
	    $find_data->file_size(int($stat->[0] || 0));
	    $find_data->file_attributes($stat->[1]);
	    $find_data->creation_time($stat->[2]);
	    $find_data->last_access_time($stat->[3]);
	    $find_data->last_write_time($stat->[4]);
	    $find_data->number_of_links(1);

	    push(@ret, $find_data);
	}
    }

    return (@ret, 0);
}

sub _cb_delete_file {
    shift->_call_cb_simple("cb_delete_file", @_);
}

sub cb_delete_file {
    my $self = shift;
    my ($fileName, $DFileInfo) = @_;

    my $stat = $Fs->stat("$fileName", $DFileInfo);
    if (ref($stat) && ref($stat) eq 'ARRAY') {
	return 0;
    }
    else {
	return -2 unless (defined($stat));
    }

    return $stat >= 0 ? 0 : $stat;
}

sub _cb_delete_directory {
    shift->_call_cb_simple("cb_delete_directory", @_);
}

sub cb_delete_directory {
    my $self = shift;
    my ($pathName, $DFileInfo) = @_;

    my $stat = $Fs->stat("$pathName", $DFileInfo);
    if (ref($stat) && ref($stat) eq 'ARRAY') {
	if ($stat->[1] & Win32::Dokan::FILE_ATTRIBUTE_DIRECTORY()) {
	    if (grep { $_ ne '.' && $_ ne '..' } @{$Fs->readdir($pathName)}) {
		return -ERROR_DIR_NOT_EMPTY();
	    }
	    return 0;
	}
	return 0;
    }
    else {
	return -2 unless (defined($stat));
    }

    return $stat >= 0 ? 0 : $stat;
}

sub _cb_set_file_attributes {
    shift->_call_cb_simple("cb_set_file_attributes", @_);
}

sub cb_set_file_attributes {
    my $self = shift;
    my ($pathName, $attributes, $DFileInfo) = @_;

    my $ret = $Fs->setattr($pathName, $attributes, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_set_file_time {
    shift->_call_cb_simple("cb_set_file_time", @_);
}

sub cb_set_file_time {
    my $self = shift;
    my ($pathName, $ctime, $atime, $mtime, $DFileInfo) = @_;

    my $ret = $Fs->utime($pathName, $ctime, $atime, $mtime, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_move_file {
    my $self = shift;
    my $name1 = shift;
    my $name2 = shift;
    # rest is in @_

    $name1 = $self->_unicode_to_fsname($name1);
    $name2 = $self->_unicode_to_fsname($name2);

    my $ret;
    eval {
	$ret = $self->cb_move_file($name1, $name2, @_);
    };
    return $@ ? -1 : $ret;
}

sub cb_move_file {
    my $self = shift;
    my ($existingName, $newFileName, $repaceExisting, $DFileInfo) = @_;

    my $ret = $Fs->rename($existingName, $newFileName, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_set_end_of_file {
    shift->_call_cb_simple("cb_set_end_of_file", @_);
}

sub cb_set_end_of_file {
    my $self = shift;
    my ($fileName, $length, $DFileInfo) = @_;

    my $ret = $Fs->truncate($fileName, $length, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_lock_file {
    shift->_call_cb_simple("cb_lock_file", @_);
}

sub cb_lock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    my $ret = $Fs->lock($fileName, $offset, $length, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _cb_unlock_file {
    shift->_call_cb_simple("cb_unlock_file", @_);
}

sub cb_unlock_file {
    my $self = shift;
    my ($fileName, $offset, $length, $DFileInfo) = @_;

    my $ret = $Fs->lock($fileName, $offset, $length, $DFileInfo);
    return -1 unless (defined($ret));

    return $ret >= 0 ? 0 : $ret;
}

sub _call_cb_noname {
    my $self = shift;
    my $method = shift;
    # rest is in @_
    if (wantarray()) {
	my @ret;
	eval {
	    @ret = $self->$method(@_);
	};
	return $@ ? -1 : @ret;
    }
    else {
	my $ret;
	eval {
	    $ret = $self->$method(@_);
	};
	return $@ ? -1 : $ret;
    }
}

sub _cb_get_disk_free_space {
    shift->_call_cb_noname("cb_get_disk_free_space", @_);
}

sub cb_get_disk_free_space {
    my $self = shift;
    my ($DFileInfo) = @_;

    # avail (for current user), total (used bytes), free (of disk)
    return (123.0*1E15, 456.0*1E15, 123.0*1E15);
}

sub _cb_get_volume_information {
    shift->_call_cb_noname("cb_get_volume_information", @_);
}

sub cb_get_volume_information {
    my $self = shift;
    my ($DFileInfo) = @_;
    my ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);

    $volume_name = "test";
    $serial = 0x123;
    $component_length = 255;
    $file_system_flags = 0;

    $volume_name = $self->_fsname_to_unicode($volume_name) if (defined($volume_name));
    $file_system_name = $self->_fsname_to_unicode($file_system_name) if (defined($file_system_name));

    return ($volume_name, $serial, $component_length, $file_system_flags, $file_system_name);
}

sub _cb_unmount {
    shift->_call_cb_noname("cb_unmount", @_);
}

sub cb_unmount {
    my $self = shift;
    my ($DFileInfo) = @_;

    return 0;
}

our $_DLL = Win32::Dokan::_DLL->new;

sub prepare_main {
    my $self = shift;
    my $opt = shift;

    $self->start_main_thread($_DLL->{handle}, $opt)
	|| croak("cannot start Dokan main thread");
}

sub mount {
    my $self = shift;
    my ($drive, $fs, $opt) = @_;

    if (defined($opt) && ref($opt) eq 'HASH') {
	$opt = Win32::Dokan::DokanOptions->new(@{$opt});
    }
    unless ($opt) {
	$opt = $self->{options} || Win32::Dokan::DokanOptions->new();
    }

    croak("Bad drive letter: $drive") unless ($drive =~ /^[A-Z]$/i);

    $opt->drive_letter(uc $drive);
    $self->prepare_main($opt);

    local($Fs) = $fs;
    $self->main;
} 

sub new {
    my $class = shift;
    my $opt = shift;
    if (defined($opt) && ref($opt) eq 'HASH') {
	$opt = Win32::Dokan::DokanOptions->new($opt);
    }

    my $self = {
	options => $opt,
    };

    bless $self, $class;
}

sub DESTROY {
}


package Win32::Dokan::_DLL;

use Carp;

sub new {
    my $class = shift;

    my @path = split(';', $ENV{PATH});
    unshift(@path, "$ENV{windir}\\System32");

    my $handle;

    for my $p (@path) {
	my $dll_path = "$p\\Dokan.dll";
	if (-f $dll_path) {
	    last if ($handle = Win32::Dokan->load_library($dll_path));
	}
    }
    croak "Cannot load Dokan.dll" unless ($handle);

    # print STDERR "dll loaded\n";

    bless {
	handle => $handle
    }, $class;
}

sub DESTROY {
    my $self = shift;

    # print STDERR "dll unloaded\n";

    Win32::Dokan->free_library($self->{handle}) if ($self->{handle});
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

Win32::Dokan - Inteface for Dokan library (user mode file system for windows)

=head1 SYNOPSIS

  ###############################################
  # in your filesystem
  #
  package Your::File::System;
  use Win32::Dokan::FS;

  use base qw(Win32::Dokan::FS);
  # override methods below...


  ###############################################
  # in your main script
  #
  use Win32::Dokan;
  use Your::File::System;

  my $dokan = Win32::Dokan->new({debug_mode => 1,
			         use_std_err => 1});
  $dokan->mount('W', Your::File::System->new());


=head1 DESCRIPTION

Win32::Dokan itself is a very low level inteface for Dokan.dll.
Don't use this module directly.

See Win32::Dokan::FS to implement filesystem.

=head2 EXPORT

None by default.

=head2 Exportable constants

  DOKAN_DRIVER_INSTALL_ERROR
  DOKAN_DRIVE_LETTER_ERROR
  DOKAN_ERROR
  DOKAN_MOUNT_ERROR
  DOKAN_START_ERROR
  DOKAN_SUCCESS

  FILE_ATTRIBUTE_READONLY
  FILE_ATTRIBUTE_HIDDEN
  FILE_ATTRIBUTE_SYSTEM
  FILE_ATTRIBUTE_DIRECTORY
  FILE_ATTRIBUTE_ARCHIVE
  FILE_ATTRIBUTE_DEVICE
  FILE_ATTRIBUTE_NORMAL
  FILE_ATTRIBUTE_TEMPORARY
  FILE_ATTRIBUTE_SPARSE_FILE
  FILE_ATTRIBUTE_REPARSE_POINT
  FILE_ATTRIBUTE_COMPRESSED
  FILE_ATTRIBUTE_OFFLINE
  FILE_ATTRIBUTE_NOT_CONTENT_INDEXED

  FS_CASE_IS_PRESERVED
  FS_CASE_SENSITIVE
  FS_UNICODE_STORED_ON_DISK
  FS_PERSISTENT_ACLS
  FS_FILE_COMPRESSION
  FS_VOL_IS_COMPRESSED

  FILE_NAMED_STREAMS
  FILE_SUPPORTS_ENCRYPTION
  FILE_SUPPORTS_OBJECT_IDS
  FILE_SUPPORTS_REPARSE_POINTS
  FILE_SUPPORTS_SPARSE_FILES
  FILE_VOLUME_QUOTAS

=head1 SEE ALSO

Win32::Dokan::FS

Dokan related documents.

=head1 AUTHOR

Toshimitsu FUJIWARA, C<< <tttfjw at gmail.com> >>

=head1 BUGS

Threading is not supported.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Win32::Dokan::FS

=head1 ACKNOWLEDGEMENTS


=head1 COPYRIGHT & LICENSE

Copyright 2009 Toshimitsu FUJIWARA, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.
