<?php
/* vim: set tabstop=4 shiftwidth=4: */

/*
 * Regular Expression Utilities
 *
 *
 * PHP version 5
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   PHP
 * @package    Commons
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    SVN: $Id: RegExpUtils.php 376 2008-09-08 14:34:37Z whitestar $
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      File available since Release 1.0.3
 */

//namespace Commons::Util;

require_once 'Commons/Lang/Object.php';
require_once 'Commons/Util/Regex/PatternSyntaxException.php';
/*
use Commons::Lang::Object;
use Commons::Util::Regex::PatternSyntaxException';
*/

/**
 * RegexUtils
 *
 *
 * @category   PHP
 * @package    Commons.Util
 * @author     Yomei Komiya
 * @copyright  2008 the original author or authors.
 * @license    http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @version    Release: 1.0.3
 * @link       http://phpcommons.sourceforge.jp/
 * @see        
 * @since      Class available since Release 1.0.3
 */
class Commons_Util_RegexUtils extends Commons_Lang_Object {

	private static $_instance = null;
	
	
	public function Commons_Util_RegexUtils() {
		parent::__construct();
	}
	
	
	public function __destruct() {
	}
	
	
	public static function getInstance() {
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}
		
		return self::$_instance;
	}
	
	
	/**
	 * Yet Another preg_match
	 * 
	 * (from Jeffrey Friedl, Mastering Regular Expressions, 3rd edition, ISBN:0-596-52812-4)
	 * 
	 * @param string $regex
	 * @param string $subject
	 * @param array $matches
	 * @param int $offset
	 * @return int result
	 * @author Jeffrey Friedl (the original author)
	 * @link http://regex.info/
	 */
	public static function yaPregMatch($regex, $subject, &$matches, $offset = 0) {
		$result = preg_match($regex, $subject, $matches, PREG_OFFSET_CAPTURE, $offset);
		if ($result) {
			$func = create_function('&$X', '$X = $X[1] < 0 ?  NULL : $X[0];');
			array_walk($matches, $func);
		}
		
		return $result;
	}
	
	
	/**
	 * Given a raw regex in a string (and, optionally, a pattern-modifiers string), return a string suitable
	 * for use as a preg pattern. The regex is wrapped in delimiters, with the modifiers (if any) appended.
	 * 
	 * (from Jeffrey Friedl, Mastering Regular Expressions, 3rd edition, ISBN:0-596-52812-4)
	 * 
	 * @param string $rawRegex
	 * @param string $modifiers
	 * @return string
	 * @author Jeffrey Friedl (the original author)
	 * @link http://regex.info/
	 */
	public static function pregRegex2Pattern($rawRegex, $modifiers = '') {
		/*
		 * To convert a regex to a pattern, we must wrap the pattern in delimiters (we'll use a pair of
		 * forward slashes) and append the modifiers. We must also be sure to escape any unescaped
		 * occurrences of the delimiter within the regex, and to escape a regex-ending escape
		 * (which, if left alone, would end up escaping the delimiter we append).
		 *
		 * We can't just blindly escape embedded delimiters, because it would break a regex containing
		 * an already-escaped delimiter. For example, if the regex is '\/', a blind escape results
		 * in '\\/' which would not work when eventually wrapped with delimiters: '/\\//'.
		 *
		 * Rather, we'll break down the regex into sections: escaped characters, unescaped forward
		 * slashes (which we'll need to escape), and everything else. As a special case, we also look out
		 * for, and escape, a regex-ending escape.
		 */
		if (! preg_match('{\\\\(?:/|$)}', $rawRegex)) { /* '/' followed by '\' or EOS */
			/* There are no already-escaped forward slashes, and no escape at the end, so it's
			 * safe to blindly escape forward slashes. */
			$cooked = preg_replace('!/!', '\/', $rawRegex);
		}
		else {
			/* This is the pattern we'll use to parse $rawRegex.
			 * The two parts whose matches we'll need to escape are within capturing parens. */
			$pattern = '{  [^\\\\/]+  |  \\\\.  |  (  /  |  \\\\$  )  }sx';
			
			/* Our callback function is called upon each successful match of $pattern in $raw-regex.
			 * If $matches[1] is not empty, we return an escaped version of it.
			 * Otherwise, we simply return what was matched unmodified. */
			$func = create_function('$matches', '   // This long
				if (empty($matches[1]))          // singlequoted
					return $matches[0];          // string becomes
				else                             // our function
					return "\\\\" . $matches[1]; // code.
			');
			
			/* Actually apply $pattern to $rawRegex, yielding $cooked */
			$cooked = preg_replace_callback($pattern, $func, $rawRegex);
		}
		
		/* $cooked is now safe to wrap -- do so, append the modifiers, and return */
		return "/$cooked/$modifiers";
	}
	
	
	/**
	 * Validate the given pattern.
	 *
	 * (from Jeffrey Friedl, Mastering Regular Expressions, 3rd edition, ISBN:0-596-52812-4)
	 * 
	 * @param string $pattern
	 * @return bool true if the given pattern argument or its underlying regular expression
	 *		are syntactically valid.
	 * @throws Commons_Util_Regex_PatternSyntaxException
	 * 		if the given pattern argument or its underlying regular expression
	 *		are not syntactically valid.
	 * @author Jeffrey Friedl (the original author)
	 * @link http://regex.info/
	 */
	public static function validatePregPattern($pattern) {
		/*
		 * To tell if the pattern has errors, we simply try to use it.
		 * To detect and capture the error is not so simple, especially if we want to be sociable and not
		 * tramp on global state (e.g., the value of $php_errormsg). So, if 'track_errors' is on, we preserve
		 * the $php_errormsg value and restore it later. If 'track_errors' is not on, we turn it on (because
		 * we need it) but turn it off when we're done.
		 */
		if ($oldTrack = ini_get('track_errors')) {
			$oldMessage = isset($php_errormsg) ? $php_errormsg : false; 
		}
		else {
			ini_set('track_errors', 1);
		}
		/* We're now sure that track_errors is on. */
		
		unset($php_errormsg);
		$result = @preg_match($pattern, ''); /* actually give the pattern a try! */
		//$returnValue = isset($php_errormsg) ? $php_errormsg : true;
		$errorMessage = '';
		if (isset($php_errormsg)) {
			$errorMessage = $php_errormsg;
		}
		
		/* We've now captured what we need; restore global state to what it was. */
		if ($oldTrack) {
			$php_errormsg = isset($oldMessage) ? $oldMessage : false;
		}
		else {
			ini_set('track_errors', 0);
		}

		if ($result === false) {	// error.
			throw new Commons_Util_Regex_PatternSyntaxException($errorMessage, $pattern);
		}
		else {
			return true;
		}
	}
	
	
	/**
	 * Return whether the given pattern is valid or not.
	 *
	 * @param string $pattern
	 * @return bool true if the given pattern argument or its underlying regular expression
	 *		are syntactically valid, oherwise false.
	 */
	public static function isValidPregPattern($pattern) {
		try {
			return self::validatePregPattern($pattern);
		}
		catch (Commons_Util_Regex_PatternSyntaxException $e) {
			return false;
		}
	}
	
	
	/**
	 * Validate the given raw regular expression.
	 *
	 * (from Jeffrey Friedl, Mastering Regular Expressions, 3rd edition, ISBN:0-596-52812-4)
	 * 
	 * @param string $regex
	 * @return bool If the given regular expression is valid, true is returned.
	 * @throws Commons_Util_Regex_PatternSyntaxException
	 * 		If the given regular expression is invalid.
	 * @author Jeffrey Friedl (the original author)
	 * @link http://regex.info/
	 */
	public static function validatePregRegex($rawRegex) {
		return self::validatePregPattern(self::pregRegex2Pattern($rawRegex));
	}


	/**
	 * Return whether the given regular expression is valid or not.
	 * 
	 * @param string $regex
	 * @return bool true if the given regular expression is valid, otherwise false.
	 */
	public static function isValidPregRegex($rawRegex) {
		try {
			return self::validatePregPattern(self::pregRegex2Pattern($rawRegex));
		}
		catch (Commons_Util_Regex_PatternSyntaxException $e) {
			return false;
		}
	}
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * c-hanging-comment-ender-p: nil
 * End:
 */
?>