/* ====================================================================
   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for additional information regarding copyright ownership.
   The ASF licenses this file to You under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
==================================================================== */
package org.apache.poi.xwpf.usermodel;

import java.util.HashMap;
import java.util.Map;

/**
 * Specifies the types of patterns which may be used to create the underline
 * applied beneath the text in a run.
 *
 * @author Gisella Bronzetti
 */
public class UnderlinePatterns {

    /**
     * Specifies an underline consisting of a single line beneath all characters
     * in this run.
     */
    public static final UnderlinePatterns SINGLE = new UnderlinePatterns(1);

    /**
     * Specifies an underline consisting of a single line beneath all non-space
     * characters in the run. There shall be no underline beneath any space
     * character (breaking or non-breaking).
     */
    public static final UnderlinePatterns WORDS = new UnderlinePatterns(2);

    /**
     * Specifies an underline consisting of two lines beneath all characters in
     * this run
     */
    public static final UnderlinePatterns DOUBLE = new UnderlinePatterns(3);

    /**
     * Specifies an underline consisting of a single thick line beneath all
     * characters in this run.
     */
    public static final UnderlinePatterns THICK = new UnderlinePatterns(4);

    /**
     * Specifies an underline consisting of a series of dot characters beneath
     * all characters in this run.
     */
    public static final UnderlinePatterns DOTTED = new UnderlinePatterns(5);

    /**
     * Specifies an underline consisting of a series of thick dot characters
     * beneath all characters in this run.
     */
    public static final UnderlinePatterns DOTTED_HEAVY = new UnderlinePatterns(6);

    /**
     * Specifies an underline consisting of a dashed line beneath all characters
     * in this run.
     */
    public static final UnderlinePatterns DASH = new UnderlinePatterns(7);

    /**
     * Specifies an underline consisting of a series of thick dashes beneath all
     * characters in this run.
     */
    public static final UnderlinePatterns DASHED_HEAVY = new UnderlinePatterns(8);

    /**
     * Specifies an underline consisting of long dashed characters beneath all
     * characters in this run.
     */
    public static final UnderlinePatterns DASH_LONG = new UnderlinePatterns(9);

    /**
     * Specifies an underline consisting of thick long dashed characters beneath
     * all characters in this run.
     */
    public static final UnderlinePatterns DASH_LONG_HEAVY = new UnderlinePatterns(10);

    /**
     * Specifies an underline consisting of a series of dash, dot characters
     * beneath all characters in this run.
     */
    public static final UnderlinePatterns DOT_DASH = new UnderlinePatterns(11);

    /**
     * Specifies an underline consisting of a series of thick dash, dot
     * characters beneath all characters in this run.
     */
    public static final UnderlinePatterns DASH_DOT_HEAVY = new UnderlinePatterns(12);

    /**
     * Specifies an underline consisting of a series of dash, dot, dot
     * characters beneath all characters in this run.
     */
    public static final UnderlinePatterns DOT_DOT_DASH = new UnderlinePatterns(13);

    /**
     * Specifies an underline consisting of a series of thick dash, dot, dot
     * characters beneath all characters in this run.
     */
    public static final UnderlinePatterns DASH_DOT_DOT_HEAVY = new UnderlinePatterns(14);

    /**
     * Specifies an underline consisting of a single wavy line beneath all
     * characters in this run.
     */
    public static final UnderlinePatterns WAVE = new UnderlinePatterns(15);

    /**
     * Specifies an underline consisting of a single thick wavy line beneath all
     * characters in this run.
     */
    public static final UnderlinePatterns WAVY_HEAVY = new UnderlinePatterns(16);

    /**
     * Specifies an underline consisting of a pair of wavy lines beneath all
     * characters in this run.
     */
    public static final UnderlinePatterns WAVY_DOUBLE = new UnderlinePatterns(17);

    /**
     * Specifies no underline beneath this run.
     */
    public static final UnderlinePatterns NONE = new UnderlinePatterns(18);

    private final int value;

    private UnderlinePatterns(int val) {
	value = val;
    }

    public int getValue() {
	return value;
    }

	public static UnderlinePatterns[] values() {
		return new UnderlinePatterns[] { SINGLE, WORDS, DOUBLE, THICK, DOTTED,
				DOTTED_HEAVY, DASH, DASHED_HEAVY, DASH_LONG, DASH_LONG_HEAVY,
				DOT_DASH, DASH_DOT_HEAVY, DOT_DOT_DASH, DASH_DOT_DOT_HEAVY,
				WAVE, WAVY_HEAVY, WAVY_DOUBLE, NONE };
	}

    private static Map imap = new HashMap();
    static {
	for (int i = 0; i < values().length; i++) {
		UnderlinePatterns p = values()[i];
	    imap.put(new Integer(p.getValue()), p);
	}
    }

    public static UnderlinePatterns valueOf(int type) {
	UnderlinePatterns align = (UnderlinePatterns) imap.get(new Integer(type));
	if (align == null)
	    throw new IllegalArgumentException("Unknown underline pattern: "
		    + type);
	return align;
    }

}
