/* ====================================================================
   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for additional information regarding copyright ownership.
   The ASF licenses this file to You under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
==================================================================== */

package org.apache.poi.hwpf.model;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.poi.ddf.DefaultEscherRecordFactory;
import org.apache.poi.ddf.EscherContainerRecord;
import org.apache.poi.ddf.EscherRecord;
import org.apache.poi.ddf.EscherRecordFactory;

/**
 * Based on AbstractEscherRecordHolder from HSSF.
 *
 * @author Squeeself
 */

public final class EscherRecordHolder {
	private final ArrayList escherRecords;

	public EscherRecordHolder() {
		escherRecords = new ArrayList();
	}

	public EscherRecordHolder(byte[] data, int offset, int size) {
		this();
		fillEscherRecords(data, offset, size);
	}

	private void fillEscherRecords(byte[] data, int offset, int size)
	{
		EscherRecordFactory recordFactory = new DefaultEscherRecordFactory();
		int pos = offset;
		while ( pos < offset + size)
		{
			EscherRecord r = recordFactory.createRecord(data, pos);
			escherRecords.add(r);
			int bytesRead = r.fillFields(data, pos, recordFactory);
			pos += bytesRead + 1; // There is an empty byte between each top-level record in a Word doc
		}
	}

	public List getEscherRecords() {
		return escherRecords;
	}

	public String toString() {
		StringBuffer buffer = new StringBuffer();

		if (escherRecords.size() == 0) {
			buffer.append("No Escher Records Decoded").append("\n");
		}
		Iterator iterator = escherRecords.iterator();
		while (iterator.hasNext()) {
			EscherRecord r = (EscherRecord) iterator.next();
			buffer.append(r.toString());
		}
		return buffer.toString();
	}

	/**
	 * If we have a EscherContainerRecord as one of our
	 *  children (and most top level escher holders do),
	 *  then return that.
	 */
	public EscherContainerRecord getEscherContainer() {
		for(Iterator it = escherRecords.iterator(); it.hasNext();) {
			Object er = it.next();
			if(er instanceof EscherContainerRecord) {
				return (EscherContainerRecord)er;
			}
		}
		return null;
	}

	/**
	 * Descends into all our children, returning the
	 *  first EscherRecord with the given id, or null
	 *  if none found
	 */
	public EscherRecord findFirstWithId(short id) {
		return findFirstWithId(id, getEscherRecords());
	}
	private static EscherRecord findFirstWithId(short id, List records) {
		// Check at our level
		for(Iterator it = records.iterator(); it.hasNext();) {
			EscherRecord r = (EscherRecord) it.next();
			if(r.getRecordId() == id) {
				return r;
			}
		}

		// Then check our children in turn
		for(Iterator it = records.iterator(); it.hasNext();) {
			EscherRecord r = (EscherRecord) it.next();
			if(r.isContainerRecord()) {
				EscherRecord found = findFirstWithId(id, r.getChildRecords());
				if(found != null) {
					return found;
				}
			}
		}

		// Not found in this lot
		return null;
	}

    public List getDgContainers()
    {
        List dgContainers = new ArrayList(
                1 );
		for (Iterator iterator = getEscherRecords().iterator(); iterator
				.hasNext();) {
			EscherRecord escherRecord = (EscherRecord) iterator.next();
            if ( escherRecord.getRecordId() == (short) 0xF002 )
            {
                dgContainers.add( (EscherContainerRecord) escherRecord );
            }
        }
        return dgContainers;
    }

    public List getDggContainers()
    {
        List dggContainers = new ArrayList(
                1 );
		for (Iterator iterator = getEscherRecords().iterator(); iterator
				.hasNext();) {
			EscherRecord escherRecord = (EscherRecord) iterator.next();
            if ( escherRecord.getRecordId() == (short) 0xF000 )
            {
                dggContainers.add( (EscherContainerRecord) escherRecord );
            }
        }
        return dggContainers;
    }

    public List getBStoreContainers()
    {
        List bStoreContainers = new ArrayList(
                1 );
		for (Iterator iterator = getDggContainers().iterator(); iterator
				.hasNext();) {
			EscherContainerRecord dggContainer = (EscherContainerRecord) iterator
					.next();
			for (Iterator iterator2 = dggContainer.getChildRecords().iterator(); iterator2
					.hasNext();) {
				EscherRecord escherRecord = (EscherRecord) iterator2.next();
                if ( escherRecord.getRecordId() == (short) 0xF001 )
                {
                    bStoreContainers.add( (EscherContainerRecord) escherRecord );
                }
            }
        }
        return bStoreContainers;
    }

    public List getSpgrContainers()
    {
        List spgrContainers = new ArrayList(
                1 );
		for (Iterator iterator = getDgContainers().iterator(); iterator
				.hasNext();) {
			EscherContainerRecord dgContainer = (EscherContainerRecord) iterator
					.next();
			for (Iterator iterator2 = dgContainer.getChildRecords().iterator(); iterator2
					.hasNext();) {
				EscherRecord escherRecord = (EscherRecord) iterator2.next();
                if ( escherRecord.getRecordId() == (short) 0xF003 )
                {
                    spgrContainers.add( (EscherContainerRecord) escherRecord );
                }
            }
        }
        return spgrContainers;
    }

    public List getSpContainers()
    {
        List spContainers = new ArrayList(
                1 );
		for (Iterator iterator = getSpgrContainers().iterator(); iterator
				.hasNext();) {
			EscherContainerRecord spgrContainer = (EscherContainerRecord) iterator
					.next();
			for (Iterator iterator2 = spgrContainer.getChildRecords()
					.iterator(); iterator2.hasNext();) {
				EscherRecord escherRecord = (EscherRecord) iterator2.next();
                if ( escherRecord.getRecordId() == (short) 0xF004 )
                {
                    spContainers.add( (EscherContainerRecord) escherRecord );
                }
            }
        }
        return spContainers;
    }
}
