//
// PdfGenetator.java
// This file is part of PosterDivider.
//
package jp.sourceforge.posterdivider;

import java.io.OutputStream;

import com.itextpdf.text.Document;
import com.itextpdf.text.Image;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.PdfImportedPage;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.PdfWriter;

public class PdfGenerator {
	public static class CreatePosterArgs {
		private float srcLeft = -1;
		private float srcBottom = -1;
		private float srcRight = -1;
		private float srcTop = -1;
		private float posterWidth = -1;
		private float posterHeight = -1;
		private float paperWidth = -1;
		private float paperHeight = -1;
		private float marginLeft = -1;
		private float marginTop = -1;
		private float marginRight = -1;
		private float marginBottom = -1;

		public float getSrcLeft() {
			return srcLeft;
		}

		public void setSrcLeft(float srcLeft) {
			this.srcLeft = srcLeft;
		}

		public float getSrcBottom() {
			return srcBottom;
		}

		public void setSrcBottom(float srcBottom) {
			this.srcBottom = srcBottom;
		}

		public float getSrcRight() {
			return srcRight;
		}

		public void setSrcRight(float srcRight) {
			this.srcRight = srcRight;
		}

		public float getSrcTop() {
			return srcTop;
		}

		public void setSrcTop(float srcTop) {
			this.srcTop = srcTop;
		}

		public float getSrcWidth() {
			return srcRight - srcLeft;
		}

		public float getSrcHeight() {
			return srcTop - srcBottom;
		}

		public float getPosterWidth() {
			return posterWidth;
		}

		public void setPosterWidth(float posterWidth) {
			this.posterWidth = posterWidth;
		}

		public float getPosterHeight() {
			return posterHeight;
		}

		public void setPosterHeight(float posterHeight) {
			this.posterHeight = posterHeight;
		}

		public float getPaperWidth() {
			return paperWidth;
		}

		public void setPaperWidth(float paperWidth) {
			this.paperWidth = paperWidth;
		}

		public float getPaperHeight() {
			return paperHeight;
		}

		public void setPaperHeight(float paperHeight) {
			this.paperHeight = paperHeight;
		}

		public float getMarginLeft() {
			return marginLeft;
		}

		public void setMarginLeft(float marginLeft) {
			this.marginLeft = marginLeft;
		}

		public float getMarginTop() {
			return marginTop;
		}

		public void setMarginTop(float marginTop) {
			this.marginTop = marginTop;
		}

		public float getMarginRight() {
			return marginRight;
		}

		public void setMarginRight(float marginRight) {
			this.marginRight = marginRight;
		}

		public float getMarginBottom() {
			return marginBottom;
		}

		public void setMarginBottom(float marginBottom) {
			this.marginBottom = marginBottom;
		}

		public void setSrcRect(float left, float bottom, float right, float top) {
			this.srcLeft = left;
			this.srcBottom = bottom;
			this.srcRight = right;
			this.srcTop = top;
		}

		public void setSrcRect(Rectangle rect) {
			setSrcRect(rect.getLeft(), rect.getBottom(), rect.getRight(), rect
					.getTop());
		}

		public void setPosterSize(float width, float height) {
			this.posterWidth = width;
			this.posterHeight = height;
		}

		public void setPosterSize(Rectangle rect) {
			setPosterSize(rect.getWidth(), rect.getHeight());
		}

		public void setPaperSize(float width, float height) {
			this.paperWidth = width;
			this.paperHeight = height;
		}

		public void setPaperSize(Rectangle rect) {
			setPaperSize(rect.getWidth(), rect.getHeight());
		}

		public void setMargin(float left, float top, float right, float bottom) {
			this.marginLeft = left;
			this.marginTop = top;
			this.marginRight = right;
			this.marginBottom = bottom;
		}
	}

	private static int calcPaperCount(float posterLength, float paperLength) {
		if (posterLength < 0 || paperLength <= 0) {
			return -1;
		}
		float count = posterLength / paperLength;
		return (int) Math.ceil(count);
	}

	public static int simulateOutputPaperCount(CreatePosterArgs args) {
		int paperCountX = simulateOutputPaperCountX(args);
		int paperCountY = simulateOutputPaperCountY(args);

		if (paperCountX < 0 || paperCountY < 0) {
			return -1;
		}

		return paperCountX * paperCountY;
	}

	public static int simulateOutputPaperCountX(CreatePosterArgs args) {
		float posterWidth = args.getPosterWidth();
		float paperWidth = args.getPaperWidth();
		float marginLeft = args.getMarginLeft();
		float marginRight = args.getMarginRight();

		if (posterWidth < 0) {
			return -1;
		}
		if (paperWidth < 0) {
			return -1;
		}
		if (marginLeft < 0 || marginRight < 0) {
			return -1;
		}

		float paperTextWidth = paperWidth - marginLeft - marginRight;
		int paperCountX = calcPaperCount(posterWidth, paperTextWidth);

		if (paperCountX < 0) {
			return -1;
		}

		return paperCountX;
	}

	public static int simulateOutputPaperCountY(CreatePosterArgs args) {
		float posterHeight = args.getPosterHeight();
		float paperHeight = args.getPaperHeight();
		float marginTop = args.getMarginTop();
		float marginBottom = args.getMarginBottom();

		if (posterHeight < 0) {
			return -1;
		}
		if (paperHeight < 0) {
			return -1;
		}
		if (marginTop < 0 || marginBottom < 0) {
			return -1;
		}

		float paperTextHeight = paperHeight - marginTop - marginBottom;
		int paperCountY = calcPaperCount(posterHeight, paperTextHeight);

		if (paperCountY < 0) {
			return -1;
		}

		return paperCountY;
	}

	public static void createPosterFromPdf(PdfReader reader, int pageNum,
			CreatePosterArgs args, OutputStream os) {
		float srcX = args.getSrcLeft();
		float srcY = args.getSrcBottom();
		float srcWidth = args.getSrcWidth();
		float srcHeight = args.getSrcHeight();
		float posterWidth = args.getPosterWidth();
		float posterHeight = args.getPosterHeight();
		float paperWidth = args.getPaperWidth();
		float paperHeight = args.getPaperHeight();
		float marginLeft = args.getMarginLeft();
		float marginTop = args.getMarginTop();
		float marginRight = args.getMarginRight();
		float marginBottom = args.getMarginBottom();

		if (reader == null) {
			throw new NullPointerException();
		}
		if (pageNum <= 0 || pageNum > reader.getNumberOfPages()) {
			throw new IllegalArgumentException();
		}
		if (os == null) {
			throw new NullPointerException();
		}
		if (srcWidth <= 0 || srcHeight <= 0) {
			throw new IllegalArgumentException();
		}
		if (posterWidth < 0 || posterHeight < 0) {
			throw new IllegalArgumentException();
		}
		if (paperWidth < 0 || paperHeight < 0) {
			throw new IllegalArgumentException();
		}
		if (marginLeft < 0 || marginTop < 0 || marginRight < 0
				|| marginBottom < 0) {
			throw new IllegalArgumentException();
		}

		Document document = new Document(
				new Rectangle(paperWidth, paperHeight), marginLeft,
				marginRight, marginTop, marginBottom);

		// 念のため生成された方の値に合わせる
		paperWidth = document.getPageSize().getWidth();
		paperHeight = document.getPageSize().getHeight();
		marginLeft = document.leftMargin();
		marginTop = document.topMargin();
		marginRight = document.rightMargin();
		marginBottom = document.bottomMargin();

		float paperTextWidth = paperWidth - marginLeft - marginRight;
		float paperTextHeight = paperHeight - marginTop - marginBottom;
		int paperCountX = calcPaperCount(posterWidth, paperTextWidth);
		int paperCountY = calcPaperCount(posterHeight, paperTextHeight);

		if (paperCountX < 0 || paperCountY < 0) {
			throw new IllegalArgumentException();
		}

		PdfWriter writer;
		try {
			writer = PdfWriter.getInstance(document, os);
		} catch (Exception ex) {
			throw new RuntimeException();
		}

		document.open();
		PdfImportedPage page = writer.getImportedPage(reader, pageNum);

		float rateX = posterWidth / srcWidth;
		float rateY = posterHeight / srcHeight;
		float x0 = marginLeft - srcX * rateX;
		float y0 = paperHeight - marginTop - posterHeight - srcY * rateY;
		float y = y0;
		for (int i = 0; i < paperCountY; i++) {
			float x = x0;
			for (int j = 0; j < paperCountX; j++) {
				document.newPage();
				writer.getDirectContent().addTemplate(page, rateX, 0, 0, rateY,
						x, y);
				x -= paperTextWidth;
			}
			y += paperTextHeight;
		}

		document.close();
		writer.close();
	}

	public static void createPosterFromImage(Image image,
			CreatePosterArgs args, OutputStream os) {
		float srcX = args.getSrcLeft();
		float srcY = args.getSrcBottom();
		float srcWidth = args.getSrcWidth();
		float srcHeight = args.getSrcHeight();
		float posterWidth = args.getPosterWidth();
		float posterHeight = args.getPosterHeight();
		float paperWidth = args.getPaperWidth();
		float paperHeight = args.getPaperHeight();
		float marginLeft = args.getMarginLeft();
		float marginTop = args.getMarginTop();
		float marginRight = args.getMarginRight();
		float marginBottom = args.getMarginBottom();

		if (image == null) {
			throw new NullPointerException();
		}
		if (os == null) {
			throw new NullPointerException();
		}
		if (srcWidth <= 0 || srcHeight <= 0) {
			throw new IllegalArgumentException();
		}
		if (posterWidth < 0 || posterHeight < 0) {
			throw new IllegalArgumentException();
		}
		if (paperWidth < 0 || paperHeight < 0) {
			throw new IllegalArgumentException();
		}
		if (marginLeft < 0 || marginTop < 0 || marginRight < 0
				|| marginBottom < 0) {
			throw new IllegalArgumentException();
		}

		Document document = new Document(
				new Rectangle(paperWidth, paperHeight), marginLeft,
				marginRight, marginTop, marginBottom);

		paperWidth = document.getPageSize().getWidth();
		paperHeight = document.getPageSize().getHeight();
		marginLeft = document.leftMargin();
		marginTop = document.topMargin();
		marginRight = document.rightMargin();
		marginBottom = document.bottomMargin();

		float paperTextWidth = paperWidth - marginLeft - marginRight;
		float paperTextHeight = paperHeight - marginTop - marginBottom;
		int paperCountX = calcPaperCount(posterWidth, paperTextWidth);
		int paperCountY = calcPaperCount(posterHeight, paperTextHeight);

		if (paperCountX < 0 || paperCountY < 0) {
			throw new IllegalArgumentException();
		}

		PdfWriter writer;
		try {
			writer = PdfWriter.getInstance(document, os);
		} catch (Exception ex) {
			throw new RuntimeException();
		}

		document.open();

		float rateXP = posterWidth / srcWidth;
		float rateYP = posterHeight / srcHeight;
		// Imageは倍率なしだと1x1ptとして扱われるのでこうする
		float rateX = rateXP * image.getWidth();
		float rateY = rateYP * image.getHeight();
		float x0 = marginLeft - srcX * rateXP;
		float y0 = paperHeight - marginTop - posterHeight - srcY * rateYP;
		float y = y0;
		for (int i = 0; i < paperCountY; i++) {
			float x = x0;
			for (int j = 0; j < paperCountX; j++) {
				document.newPage();
				try {
					writer.getDirectContent().addImage(image, rateX, 0, 0,
							rateY, x, y);
				} catch (Exception ex) {
					throw new RuntimeException();
				}
				x -= paperTextWidth;
			}
			y += paperTextHeight;
		}

		document.close();
		writer.close();
	}

	public static void createPoster(ImageContainer ic, CreatePosterArgs args,
			OutputStream os) {
		switch (ic.getFileType()) {
		case Lib.FT_PDF:
			createPosterFromPdf(ic.getPdf(), ic.getPageNum(), args, os);
			return;
		case Lib.FT_BMP:
		case Lib.FT_GIF:
		case Lib.FT_JBIG2:
		case Lib.FT_JPEG:
		case Lib.FT_JPEG2000:
		case Lib.FT_PNG:
		case Lib.FT_TIFF:
		case Lib.FT_WMF:
		case Lib.FT_OTHERIMAGE:
			createPosterFromImage(ic.getImage(), args, os);
			return;
		default:
			return;
		}
	}
}
