/*!
  \file
  \brief つくばチャレンジ用のロガープログラム

  \author Satofumi KAMIMURA

  $Id: push_logger.cpp 326 2008-11-01 04:36:22Z satofumi $
*/

#include "mDifferentialDrive.h"
#include "mUrgCtrl.h"
#include "mCameraCtrl.h"
#include "mGpsCtrl.h"
#include "NmeaData.h"
#include "UsbJoystick.h"
#include "Thread.h"
#include "getTicks.h"
#include "delay.h"
#include "MathUtils.h"
#include "ScreenCtrl.h"
#include "Lock.h"

using namespace qrk;


namespace
{
  enum {
    UrgFrameInterval = 5,
  };

  static Lock Run_mutex_;
  static Lock Quit_mutex_;
  static bool quit_ = false;


  void connectDevices(int argc, char *argv[],
                      mDifferentialDrive& run,
                      mUrgCtrl& urg, mCameraCtrl& camera,
                      mGpsCtrl& gps)
  {
    if (! run.connect("/dev/usb/ttyUSB0", 115200)) {
      printf("RunCtrl::connect: %s\n", run.what());
      exit(1);
    }

    if (! urg.connect("/dev/ttyACM0")) {
      printf("UrgCtrl::connect: %s\n", urg.what());
      exit(1);
    }

    if (! camera.connect()) {
      fprintf(stderr, "Warnning: CameraCtrl::connect: %s\n", camera.what());
    }

    if (! gps.connect("/dev/usb/ttyUSB1", 4800)) {
      fprintf(stderr, "Warnning: GpsCtrl::connect: %s\n", gps.what());
    }
  }


  int runCapture_thread(void* args)
  {
    mDifferentialDrive* run = static_cast<mDifferentialDrive*>(args);
    FILE* fd = fopen("run_position.txt", "w");
    if (fd == NULL) {
      delay(1000);
      return -1;
    }

    while (true) {
      Quit_mutex_.lock();
      if (quit_) {
        Quit_mutex_.unlock();
        break;
      }
      Quit_mutex_.unlock();

      int timestamp = -1;
      Run_mutex_.lock();
      Position<int> pos = run->position(NULL, &timestamp);
      Run_mutex_.unlock();
      fprintf(fd, "%d, %d, %d, %d\n", timestamp, pos.x, pos.y, pos.to_deg());
      fflush(fd);

      delay(100);
    }
    run->stop();
    fclose(fd);
    fprintf(stderr, "quit Running thread.\n");
    return 0;
  }


  // URG のデータを読み捨て
  int urgCapture_thread(void* args)
  {
    mUrgCtrl* urg = static_cast<mUrgCtrl*>(args);
    static int times = 0;

    while (1) {
      Quit_mutex_.lock();
      if (quit_) {
        Quit_mutex_.unlock();
        break;
      }
      Quit_mutex_.unlock();

      std::vector<long> data;
      int n = urg->capture(data);
      if (n <= 0) {
        delay(urg->scanMsec() * UrgFrameInterval);

      } else {
        printf("%d: n = %d\n", times, n);
        ++times;
      }
    }
    fprintf(stderr, "quit URG thread.\n");
    return 0;
  }


  // ID を増やしながらカメラ画像取得
  int cameraCapture_thread(void* args)
  {
    mCameraCtrl* camera = static_cast<mCameraCtrl*>(args);
    static FILE* fd = fopen("camera_list.txt", "w");
    if (fd == NULL) {
      delay(2000);
      return -1;
    }

    static int capture_id = 0;
    while (1) {
      Quit_mutex_.lock();
      if (quit_) {
        Quit_mutex_.unlock();
        break;
      }
      Quit_mutex_.unlock();

      char buffer[256];
      snprintf(buffer, 256, "camera_%04d.jpg", capture_id);
      fprintf(fd, "%d\t%s\n", getTicks(), buffer);
      camera->capture(buffer);
      ++capture_id;

      delay(1000);
    }
    fclose(fd);
    fprintf(stderr, "quit Camera thread.\n");
    return 0;
  }


  // GPS のデータを読み捨て
  int gpsCapture_thread(void* args)
  {
    mGpsCtrl* gps = static_cast<mGpsCtrl*>(args);

    NmeaData nmea_data;
    while (1) {
      Quit_mutex_.lock();
      if (quit_) {
        Quit_mutex_.unlock();
        break;
      }
      Quit_mutex_.unlock();

      gps->update(&nmea_data);
      delay(1000);
    }
    fprintf(stderr, "quit gps thread.\n");
    return 0;
  }


  enum {
    BODY_TREAD_MM = 330,
  };

  typedef struct {
    int straight;
    qrk::Angle rotate;
    bool jet;
  } robotInput_t;


  // 加速度を考慮した速度計算
  double setVelocity(double now, double ref, double RefAcc, long diff)
  {
    double sign = ref - now;
    double add = RefAcc * diff / 1000.0;
    double value = now + ((sign > 0.0) ? +add : -add);
    if (sign > 0.0) {
      if (value > ref) {
        value = ref;
      }
    } else {
      if (value < ref) {
        value = ref;
      }
    }
    return value;
  }


  // 入力の取得
  robotInput_t inputHandle(UsbJoystick& js)
  {
    static unsigned long pre_ticks = getTicks();
    static double now_straight = 0;
    static double now_rotate = 0.0;
    const double RotateRef = (2.0 * M_PI) / 4.0;
    const double RotateAcc = (2.0 * M_PI) * 3.0 / 4.0;
    const int StraightRef = 300;
    const int StraightAcc = 300;

    unsigned long ticks = getTicks();
    double ref_straight = 0;
    double ref_rotate = 0.0;
    robotInput_t ri;
    ri.jet = false;

    // ジョイスティック入力
    int js_x = -js.axisValue(0);
    ref_rotate += (js_x > 3200) ? RotateRef : (js_x < -3200) ? -RotateRef : 0;

    if (js.isButtonPressed(1) || js.isButtonPressed(2)) {
      ref_straight = -StraightRef;

    } else if (js.isButtonPressed(6) || js.isButtonPressed(7)) {
      ref_straight = -now_straight / 2;

    } else {
      // 並進移動(前進)
      ref_straight = StraightRef;
    }

    // !!! 常に、速度を標準の２倍にする
    ri.jet = true;
    //}

    if (ri.jet) {
      ref_straight *= 2.0;
    }
    long diff = ticks - pre_ticks;
    pre_ticks = ticks;
    now_straight = setVelocity(now_straight, ref_straight, StraightAcc, diff);
    now_rotate = setVelocity(now_rotate, ref_rotate, RotateAcc, diff);

    ri.straight = static_cast<int>(now_straight);
    ri.rotate = rad(now_rotate);
    return ri;
  }


  // ロボットの制御
  void robotCtrl(mDifferentialDrive& run, const robotInput_t& ri)
  {
    enum { RIGHT = 0, LEFT = 1 };

    // 並進速度、回転速度、トレッドから車輪毎の目標回転速度を計算
    int whl_mm_vel[2];
    int rotate_mm_vel =
      static_cast<int>(BODY_TREAD_MM * ri.rotate.to_rad() / 2);
    whl_mm_vel[RIGHT] = ri.straight + rotate_mm_vel;
    whl_mm_vel[LEFT] = ri.straight - rotate_mm_vel;

    for (int i = 0; i < 2; ++i) {
      run.setWheelVel(i, whl_mm_vel[i]);
    }
  }
};


int main(int argc, char *argv[])
{
  bool use_joystick = false;
  for (int i = 1; i < argc; ++i) {
    if (! strcmp("-j", argv[i])) {
      use_joystick = true;
    }
  }

  mDifferentialDrive run(argc, argv);
  mUrgCtrl urg(argc, argv);
  mCameraCtrl camera(argc, argv);
  mGpsCtrl gps(argc, argv);
  UsbJoystick joystick;

  ScreenCtrl video;
  if (use_joystick) {
    if (video.show() == NULL) {
      printf("ScreenCtrl::show(): %s", video.what());
      exit(1);
    }
    joystick.connect();
  } else {
    // サーボフリー
    run.setServo(false);
  }

  connectDevices(argc, argv, run, urg, camera, gps);

  // URG データの取得間隔を調整
  urg.setCaptureMode(AutoCapture);
  urg.setCaptureTimes(UrgCtrl::Infinity);
  urg.setCaptureFrameInterval(UrgFrameInterval);

  // タイムスタンプ合わせ
  run.setTimestamp(getTicks());
  urg.setTimestamp(getTicks());

  // 定期的なデータ記録
  Thread run_thread(runCapture_thread, &run);
  Thread urg_thread(urgCapture_thread, &urg);
  Thread camera_thread(cameraCapture_thread, &camera);
  Thread gps_thread(gpsCapture_thread, &gps);

  run_thread.run();
  urg_thread.run();
  camera_thread.run();
  gps_thread.run();

  SDL_Event event;
  while (true) {
    while (SDL_PollEvent(&event)) {
      joystick.setEvent(event);
    }

    // 6, 7, 8, 9 が全て押されたら、終了
    if (joystick.isButtonPressed(6) &&
        joystick.isButtonPressed(7) &&
        joystick.isButtonPressed(8) &&
        joystick.isButtonPressed(9)) {
      break;
    }

    if (use_joystick) {
      robotInput_t ri = inputHandle(joystick);
      Run_mutex_.lock();
      robotCtrl(run, ri);
      Run_mutex_.unlock();
    }
    delay(100);
  }
  Quit_mutex_.lock();
  quit_ = true;
  Quit_mutex_.unlock();

  run_thread.stop();
  urg_thread.stop();
  gps_thread.stop();
  camera_thread.stop();

  run_thread.wait();
  urg_thread.wait();
  camera_thread.wait();
  gps_thread.wait();
  printf("end.");

  return 0;
}
