/*!
  \file
  \brief URG  ODE pf

  \author Satofumi KAMIMURA

  $Id: UrgModel.cpp 1227 2009-08-16 03:57:31Z satofumi $
*/

#include "UrgModel.h"
#include "EventScheduler.h"
#include "OdeModel.h"
#include "OdeHandler.h"
#include "ModelManager.h"

using namespace qrk;


namespace
{
  enum {
    X = 0,
    Y = 1,
    Z = 2,
  };


  typedef struct
  {
    dBodyID body;
    dGeomID geometry;
  } object_t;


  typedef struct
  {
    object_t base;
  } sensor_t;
}


struct UrgModel::pImpl
{
  OdeHandler ode_;
  dWorldID world_;
  dSpaceID space_;

  enum {
    BaseModel,
  };
  ModelManager::model_t models_[1];

  EventScheduler event_manager_;

  sensor_t sensor_;
  SensorType sensor_type_;
  dReal sensor_length_[3];
  dReal sensor_weight_;

  bool fixed_;
  dJointID fixed_joint_;


  pImpl(void)
    : world_(ode_.world()), space_(ode_.space()), sensor_type_(UTM_30LX),
      fixed_(false)
  {
  }


  void createModel(void)
  {
    if (sensor_type_ == URG_04LX) {
      sensor_length_[X] = 0.050;
      sensor_length_[Y] = 0.050;
      sensor_length_[Z] = 0.055;
      sensor_weight_ = 0.300;

    } else {
      // UTM_30LX
      sensor_length_[X] = 0.060;
      sensor_length_[Y] = 0.060;
      sensor_length_[Z] = 0.055;
      sensor_weight_ = 0.300;
    }

    // ZT
    sensor_.base.body = dBodyCreate(world_);

    dMass mass;
    dMassSetZero(&mass);
    dMassSetBoxTotal(&mass, sensor_weight_,
                     sensor_length_[X], sensor_length_[Y], sensor_length_[Z]);
    dBodySetMass(sensor_.base.body, &mass);
    sensor_.base.geometry =
      dCreateBox(space_,
                 sensor_length_[X], sensor_length_[Y], sensor_length_[Z]);
    dGeomSetBody(sensor_.base.geometry, sensor_.base.body);
    dBodySetPosition(sensor_.base.body, 0.0, 0.0, sensor_length_[Z] / 2.0);

    registerModel();
  }


  void registerModel(void)
  {
    Color base_color(0.3, 0.3, 0.3);

    ModelManager model_manager;

    models_[BaseModel].type = ModelManager::Box;
    models_[BaseModel].body_id = sensor_.base.body;
    models_[BaseModel].color = base_color;
    models_[BaseModel].obstacle = false;
    model_manager.addBox(&models_[BaseModel], sensor_length_);
  }
};


UrgModel::UrgModel(void) : pimpl(new pImpl)
{
}


UrgModel::~UrgModel(void)
{
}


void UrgModel::setSensorType(SensorType type)
{
  pimpl->sensor_type_ = type;
}


void UrgModel::activate(void)
{
  pimpl->createModel();
}


void UrgModel::setPosition(const Position<long>& position,
                           OdeModel* model, bool fixed)
{
  // wpxɉ]
  double radian = position.rad();
  dMatrix3 R;
  dRFromAxisAndAngle(R, 0.0, 0.0, 1.0, radian);
  dGeomSetRotation(pimpl->sensor_.base.geometry, R);

  // !!! f NULL łȂƂɂ̈ʒuA傫擾
  // !!! ̏㕔̑Έʒuɔzu
  dReal x = position.x / 1000.0;
  dReal y = position.y / 1000.0;
  dReal z = (pimpl->sensor_length_[Z] / 2.0) + 0.002;

  // w胂f̑Έʒuɔzu
  dBodyID base_body_id = 0;
  if (model) {
    base_body_id = model->objectId();
    const dReal* base_position = dBodyGetPosition(base_body_id);
    x += base_position[X];
    y += base_position[Y];
    z += base_position[Z];
  }

  dGeomSetPosition(pimpl->sensor_.base.geometry, x, y, z);

  if (fixed) {
    if (pimpl->fixed_) {
      dJointDestroy(pimpl->fixed_joint_);
    }
    pimpl->fixed_joint_ = dJointCreateFixed(pimpl->world_, 0);
    dJointAttach(pimpl->fixed_joint_,
                 pimpl->sensor_.base.body, base_body_id);
    dJointSetFixed(pimpl->fixed_joint_);
    pimpl->fixed_ = true;
  }
}


Position<long> UrgModel::position(void) const
{
  const dReal* position = dBodyGetPosition(pimpl->sensor_.base.body);
  const dReal* rotation = dBodyGetRotation(pimpl->sensor_.base.body);
  double radian = atan2(-rotation[0], rotation[4]) + (M_PI / 2.0);

  return Position<long>(static_cast<long>(1000.0 * position[X]),
                        static_cast<long>(1000.0 * position[Y]), rad(radian));
}


dBodyID UrgModel::objectId(void) const
{
  return pimpl->sensor_.base.body;
}


void UrgModel::setLaser(long* buffer, double radian, long max_length)
{
  dGeomID ray_id = dCreateRay(pimpl->space_, max_length / 1000.0);

  // URG ʒuƂāAw肳ꂽɃ[U[zu
  const dReal* rotation = dBodyGetRotation(pimpl->sensor_.base.body);
  double sensor_radian = atan2(-rotation[0], rotation[4]) + (M_PI / 2.0);

  dReal x = cos(radian + sensor_radian);
  dReal y = sin(radian + sensor_radian);

  // ZTŔzu[U]
  // !!!

  // !!! ZT𐅕ȊǑXŎg܂łɃ[U̎n_ʒuAC邱
  // !!! ݂̎ƁAZT Z ̏Ɏn_zuĂ
  // !!! {́AZT̉]ɂƂȂĎn_ʒuAړ
  const dReal* position = dBodyGetPosition(pimpl->sensor_.base.body);
  dGeomRaySet(ray_id, position[X], position[Y],
              position[Z] + pimpl->sensor_length_[Z] / 2.0 + 0.010,
              x, y, 0.0);
  pimpl->event_manager_.setLaser(buffer, ray_id);
}
