/*
 * Created on 2005/01/28
 *
 * @author KAMO Masahiko (mkamo@users.sourceforge.jp)
 */
package razgriz.news.ui.views;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.PlatformUI;

import razgriz.news.model.Article;
import razgriz.news.model.Favorite;
import razgriz.news.model.FavoriteList;
import razgriz.news.model.Group;
import razgriz.news.model.ModelElement;
import razgriz.news.model.News;
import razgriz.news.model.util.ModelVisitor;
import razgriz.news.model.util.NewsSwitch;
import razgriz.news.ui.SharedImages;
import razgriz.news.visitor.UnreadArticleCountingVisitor;

/**
 * 
 */
public class FavoriteListTreeContentProvider
    extends LabelProvider implements ITreeContentProvider, Adapter {

    TreeViewer viewer;
    FavoriteList favoriteList;
    
    Set adaptedSet;

    public FavoriteListTreeContentProvider() {
        adaptedSet = new HashSet();
    }

    public void addAdaption(EObject adapted) {
        adapted.eAdapters().add(this);
        adaptedSet.add(adapted);
    }
    public void removeAdaption(EObject adapted) {
        adaptedSet.remove(adapted);
        adapted.eAdapters().remove(this);
    }
    public void clearAdaption() {
        for (Iterator ite = adaptedSet.iterator(); ite.hasNext(); ) {
            EObject adapted = (EObject) ite.next();
            adapted.eAdapters().remove(this);
            ite.remove();
        }
    }

    public Object[] getChildren(Object element) {
        if (element instanceof News) {
            return new Object[0];
        }

        EObject eobj = (EObject) element;
        List contents = eobj.eContents();
        return (Object[]) contents.toArray(new Object[contents.size()]);
    }
    public Object getParent(Object element) {
        EObject eobj = (EObject) element;
        return eobj.eContainer();
    }
    public boolean hasChildren(Object element) {
        Object[] children = getChildren(element);
        return children.length > 0;
    }
    public Object[] getElements(Object inputElement) {
        return getChildren(inputElement);
    }
    public void dispose() {
    }
    public void inputChanged(
        Viewer viewer, Object oldInput, Object newInput
    ) {
        this.viewer = (TreeViewer) viewer;
        FavoriteList newFavlist = (FavoriteList) newInput;

        clearAdaption();
        if (newFavlist != null) {
            newFavlist.accept(new AdapterAddingVisitor(this));
        }
        favoriteList = newFavlist;
    }

    public void notifyChanged(Notification notification) {
        if (notification.isTouch()) {
            return;
        }

        final Object obj = notification.getNotifier();
        final NewsSwitch sw = new ViewerUpdateSwitch(this, notification);
        viewer.getControl().getDisplay().asyncExec(
            new Runnable() {
                public void run() {
                    sw.doSwitch((EObject) obj);
                }
            }
        );
    }
    public Notifier getTarget() {
        return null;
    }
    public void setTarget(Notifier newTarget) {

    }
    public boolean isAdapterForType(Object type) {
        return false;
    }
    
    public String getText(Object element) {
        EObject eobj = (EObject) element;
        NewsTreeLabelSwitch sw = new NewsTreeLabelSwitch();
        return (String) sw.doSwitch(eobj);
    }
    public Image getImage(Object obj) {
        EObject eobj = (EObject) obj;
        NewsTreeImageSwitch sw = new NewsTreeImageSwitch();
        return (Image) sw.doSwitch(eobj);
    }

    private class AdapterAddingVisitor extends ModelVisitor {
        private FavoriteListTreeContentProvider provider;
        public AdapterAddingVisitor(FavoriteListTreeContentProvider provider) {
            this.provider = provider;
        }
        public Object caseFavoriteList(FavoriteList favlist) {
            provider.addAdaption(favlist);
            return favlist;
        }
        public Object caseGroup(Group group) {
            provider.addAdaption(group);
            return group;
        }
        public Object caseNews(News news) {
            provider.addAdaption(news);
            return news;
        }
        public Object caseArticle(Article article) {
            provider.addAdaption(article);
            return article;
        }
        public boolean prune(ModelElement elem) {
            return false;
        }
    }

    private class ViewerUpdateSwitch extends NewsSwitch {
        private FavoriteListTreeContentProvider provider;
        private Notification notification;

        public ViewerUpdateSwitch(
            FavoriteListTreeContentProvider provider, Notification notification
        ) {
            this.provider = provider;
            this.notification = notification;
        }

        private void updateAdapter() {
            switch (notification.getEventType()) {
            case Notification.ADD: {
                EObject eobj = (EObject) notification.getNewValue();
                provider.addAdaption(eobj);
                viewer.refresh();
                break;
            }
            case Notification.ADD_MANY: {
                List eobjlist = (List) notification.getNewValue();
                for (int i = 0; i < eobjlist.size(); i++) {
                    EObject eobj = (EObject) eobjlist.get(i);
                    provider.addAdaption(eobj);
                }
                viewer.refresh();
                break;
            }
            case Notification.REMOVE: {
                EObject eobj = (EObject) notification.getOldValue();
                provider.removeAdaption(eobj);
                viewer.refresh();
                break;
            }
            case Notification.REMOVE_MANY: {
                List eobjlist = (List) notification.getOldValue();
                for (int i = 0; i < eobjlist.size(); i++) {
                    EObject eobj = (EObject) eobjlist.get(i);
                    provider.removeAdaption(eobj);
                }
                viewer.refresh();
                break;
            }
            case Notification.SET: {
                viewer.refresh();
                break;
            }
            }
        }

        public Object caseFavoriteList(FavoriteList favlist) {
            updateAdapter();
            return favlist;
        }
        public Object caseFavorite(Favorite fav) {
            updateAdapter();
            return fav;
        }
        public Object caseArticle(Article article) {
            if (notification.getEventType() == Notification.SET) {
                viewer.refresh();
            }
            return article;
        }
    }

    private class NewsTreeLabelSwitch extends NewsSwitch {
        public Object caseFavorite(Favorite favorite) {
            UnreadArticleCountingVisitor visitor =
                new UnreadArticleCountingVisitor();
            favorite.accept(visitor);
            String countstr =
                visitor.getResult() > 0?
                    " (" + visitor.getResult() + ")":
                    "";
            return favorite.getTitle() + countstr;
        }
    }

    private class NewsTreeImageSwitch extends NewsSwitch {
        private Image getImage(String str) {
            return PlatformUI.getWorkbench().getSharedImages().getImage(str);
        }
        public Object caseNews(News news) {
            return getImage(SharedImages.IMG_OBJ_NEWS);
        }
        public Object caseGroup(Group group) {
            return getImage(SharedImages.IMG_OBJ_GROUP);
        }
    }
}
