#ifndef	   COMMANDBUF_H_INCLUDED
#define	   COMMANDBUF_H_INCLUDED

#ifdef HAVE_CONFIG_H
# include  "config.h"
#endif

#ifdef HAVE_STREAMBUF
# include  <streambuf>
#else
# include  <streambuf.h>
#endif
#include  <string>
#include  <vector>
#include  <csignal>
#include  <cstdio>
#include  <cstdlib>
#include  <unistd.h>
#include  <errno.h>
#include  <sys/types.h>
#include  <sys/wait.h>

class  commandbuf : public std::streambuf
{
public:
	enum { BAD_PROCESSS_ID = -1 };
	enum { BAD_FD = -1 };

protected:
	int	to_com_fd;
	int	from_com_fd;
	pid_t	command_pid;

public:
	void	invoke( const char *  file ,
			const char * const * const  argv );

	void	invoke( const std::string &  file ,
			const std::vector<std::string> &  argv );

protected:
	 commandbuf( const commandbuf & );			// not allowed.
	 commandbuf &	operator= ( const commandbuf & );	// not allowed.

public:
	 commandbuf();

	 commandbuf( const std::string &  file ,
		     const std::vector<std::string> &  argv );
	 commandbuf( const std::vector<std::string> &  argv );

	 commandbuf( const char *  file ,  const char * const * const  argv );
	 commandbuf( const char * const * const  argv );

	virtual	~commandbuf();

	virtual	int  in_avail()
		{
			return( -1 );
		}

	virtual	int  overflow( int  ch = EOF )
		{
			if ( ch == EOF || to_com_fd == BAD_FD )
			{
				return( EOF );
			}

			int	n;
			if ( (n = pptr() - pbase()) != 0 )
			{
				if ( ::write( to_com_fd , pptr() ,
					      static_cast<size_t>(n) ) != n )
				{
					return( EOF );
				}
			}

			unsigned char	c = static_cast<unsigned char>(ch);
			if ( ::write( to_com_fd , &c ,
				      static_cast<size_t>(1) ) != 1 )
			{
				return( EOF );
			}
			else
			{
				return( ch );
			}
		}

	virtual	int  underflow()
		{
			return( EOF );
		}

	virtual	int  uflow()
		{
			if ( from_com_fd == BAD_FD )
			{
				return( EOF );
			}

			unsigned char	ch;
			switch( ::read( from_com_fd , &ch ,
					static_cast<size_t>(1) ))
			{
			case -1:
				perror( "read" );
				return( EOF );
				break;

			case  1:
				return( ch );
				break;

			case  0:
			default:
				return( EOF );
				break;
			}
		}

	virtual std::streamsize xsputn( const char *  s ,  std::streamsize  n )
		{
			if ( to_com_fd == BAD_FD )
			{
				return( 0 );
			}

			std::streamsize		total = 0;

			while( total < n )
			{
				int	n_wrote = ::write( to_com_fd ,
							   s + total ,
							   n - total );

				if ( n_wrote == -1 )
				{
					std::perror( "write" );

					return( total );
				}

				total += n_wrote;
			}

			return( total );
		}

	virtual std::streamsize xsgetn( char *  s ,  std::streamsize  n )
		{
			if ( from_com_fd == BAD_FD )
			{
				return( 0 );
			}

			std::streamsize		total = 0;

			while( total < n )
			{
				int	n_read = ::read( from_com_fd ,
							 s + total ,
							 n - total );

				if ( n_read == -1 )
				{
					std::perror( "read" );

					return( total );
				}
				else if ( n_read == 0 )
				{
					return( total );
				}

				total += n_read;
			}

			return( total );
		}

	virtual	int  sync()
		{
			return( 0 );
		}


	virtual	void  close( int  in_out = std::ios::in | std::ios::out )
		{
			if ( in_out & std::ios::in )
			{
				if ( to_com_fd != BAD_FD
				  && ::close( to_com_fd ) == -1 )
				{
					std::perror( "close" );
				}
				to_com_fd = BAD_FD;
			}

			if ( in_out & std::ios::out )
			{
				if ( from_com_fd != BAD_FD
				  && ::close( from_com_fd ) == -1 )
				{
					std::perror( "close" );
				}
				from_com_fd = BAD_FD;
			}
		}

	virtual	int	fd( int  in_out ) const
		{
			if ( in_out == std::ios::in )
			{
				return( to_com_fd );
			}
			else if ( in_out == std::ios::out )
			{
				return( from_com_fd );
			}
			else
			{
				return( BAD_FD );
			}
		}

	virtual	pid_t	getpid() const { return( command_pid ); }

	virtual	int	kill( int  sig = SIGTERM )
		{
			if ( command_pid == BAD_PROCESSS_ID )
			{
				return( -1 );
			}
			else
			{
				int	ret;

				if ( (ret = ::kill( command_pid , sig )) == -1
				  && errno != ESRCH /* No such process */ )
				{
					std::perror( "kill" );
				}

				return( ret );
			}
		}

	virtual	int	poll() const;
};



inline
void   commandbuf::invoke( const char *  file ,
			   const char * const * const  argv )
{
	this -> close( std::ios::in | std::ios::out );
	this -> kill();

	to_com_fd   = BAD_FD;
	from_com_fd = BAD_FD;
	command_pid = BAD_PROCESSS_ID;

	int	to  [2];
	int	from[2];

	if ( ::pipe( to ) == -1 )
	{
		std::perror( "pipe" );

		return;
	}

	if ( ::pipe( from ) == -1 )
	{
		std::perror( "pipe" );

		if ( ::close( to[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( to[1] ) == -1 ) { std::perror( "close" ); }

		return;
	}


	switch( (command_pid = ::fork()) )
	{
	case  -1:
		// error
		std::perror( "fork" );

		if ( ::close(   to[0] ) == -1 )	{ std::perror( "close" ); }
		if ( ::close(   to[1] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[1] ) == -1 ) { std::perror( "close" ); }

		return;
		break;

	case  0:
		// child process
		if ( ::close(   to[1] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[0] ) == -1 ) { std::perror( "close" ); }

		if ( ::dup2(   to[0] , STDIN_FILENO  ) == -1 )
		{
			std::perror( "dup2" );
		}

		if ( ::dup2( from[1] , STDOUT_FILENO ) == -1 )
		{
			std::perror( "dup2" );
		}

		if ( ::close(   to[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[1] ) == -1 ) { std::perror( "close" ); }

		if ( ::execvp( file , const_cast<char * const *>(argv) )
		     == -1 )
		{
			std::perror( "execvp" );
		}

		std::exit( 1 );

		break;

	default:
		// parent process
		if ( ::close(   to[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[1] ) == -1 ) { std::perror( "close" ); }

		to_com_fd   = to  [1];
		from_com_fd = from[0];

		break;
	}
}

inline
void   commandbuf::invoke( const std::string &  file ,
			   const std::vector<std::string> &  argv )
{
	const char **	v = static_cast<const char **>(0);

	try
	{
		v = new const char * [argv.size() + 1];

		size_t  i;
		for ( i = 0  ;  i < argv.size()  ;  i ++ )
		{
			v[i] = argv[i].c_str();
		}

		v[i] = static_cast<char *>(0);

		this -> invoke( file.c_str() , v );
	}
	catch(...)
	{
		throw;
	}

	delete[] v;
}

inline
commandbuf::commandbuf()
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID )
{
	this -> setbuf( 0 , 0 );
}

inline
commandbuf::commandbuf( const std::string &  file ,
			const std::vector<std::string> &  argv )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID )
{
	this -> setbuf( 0 , 0 );

	this -> invoke( file , argv );
}

inline
commandbuf::commandbuf( const std::vector<std::string> &  argv )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID )
{
	this -> setbuf( 0 , 0 );

	this -> invoke( argv[0] , argv );
}

inline
commandbuf::commandbuf( const char *  file ,
			const char * const * const  argv )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID )
{
	this -> setbuf( 0 , 0 );

	this -> invoke( file , argv );
}

inline
commandbuf::commandbuf( const char * const * const  argv )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID )
{
	this -> setbuf( 0 , 0 );

	this -> invoke( argv[0] , argv );
}


inline
commandbuf::~commandbuf()
{
	this -> close( std::ios::in | std::ios::out );

	if ( command_pid != BAD_PROCESSS_ID
	  && command_pid != 0 )
	{
		if ( ::wait( static_cast<int *>(0) ) == -1 )
		{
			std::perror( "wait" );
		}
	}
}



#include  <unistd.h>
#include  <sys/types.h>
#include  <cstring>

#if defined(TIME_WITH_SYS_TIME) && defined(HAVE_SYS_TIME_H)
    #if TIME_WITH_SYS_TIME
	#include  <sys/time.h>
	#include  <time.h>
    #else
	#if HAVE_SYS_TIME_H
	    #include  <sys/time.h>
	#else
	    #include  <time.h>
	#endif
    #endif
#else
    #include  <sys/time.h>
#endif

inline
int    commandbuf::poll() const
{
	fd_set		readfd;
	FD_ZERO( &readfd );
	FD_SET( from_com_fd , &readfd );

	struct timeval	tv;
	tv.tv_sec  = 0;
	tv.tv_usec = 0;

	int	ret;

	if ( (ret = ::select( from_com_fd + 1 , &readfd ,
			      static_cast<fd_set *>(0) ,
			      static_cast<fd_set *>(0) ,
			      &tv ) ) == -1 )
	{
		std::perror( "select" );

		return( 0 );
	}

	return( ret );
}


#endif	/* COMMANDBUF_H_INCLUDED */
