#include  "d2_straight_line_divided_region.h"
#include  <cassert>
#include  <cmath>

using namespace std;

D2_Straight_Line_Divided_Region::D2_Straight_Line_Divided_Region(
    const D2_Straight_Line &  ln ,  bool  posi_side ,  bool  incl_line )
	: staright_line( ln ) ,
	  positive_side( posi_side ) , include_line( incl_line )
{
}

D2_Straight_Line_Divided_Region::~D2_Straight_Line_Divided_Region()
{
}

bool   D2_Straight_Line_Divided_Region::in_region(
					  const D2_Vector &  vec ) const
{
	if ( positive_side )
	{
		if ( include_line )
		{
			return( staright_line( vec ) >= 0.0 );
		}
		else
		{
			return( staright_line( vec ) > 0.0 );
		}
	}
	else
	{
		if ( include_line )
		{
			return( staright_line( vec ) <= 0.0 );
		}
		else
		{
			return( staright_line( vec ) < 0.0 );
		}
	}
}

ref_count_ptr<const D2_Region_Entity>
  D2_Straight_Line_Divided_Region::copy() const
{
	return( ref_count_ptr<const D2_Region_Entity>(
		      new D2_Straight_Line_Divided_Region( *this ) ) );
}

const D2_Straight_Line &  D2_Straight_Line_Divided_Region::line() const
{
	return( staright_line );
}


D2_Vector  D2_Straight_Line_Divided_Region::cross_point(
			const D2_Straight_Line_Divided_Region &  reg ) const
{
	return( staright_line.cross_point( reg.staright_line ) );
}

bool   D2_Straight_Line_Divided_Region::parallel(
			const D2_Straight_Line_Divided_Region &  reg ) const
{
	return( staright_line.parallel( reg.staright_line ) );
}




//
// D2_Composite_Straight_Line_Divided_Region
//
D2_Composite_Straight_Line_Divided_Region::
	 D2_Composite_Straight_Line_Divided_Region()
	: closed( false )
{
}

D2_Composite_Straight_Line_Divided_Region::
	~D2_Composite_Straight_Line_Divided_Region()
{
#ifdef LIB_MEMORY
	line_divided_region_set.clear();
	cross_point_set.clear();
#endif
}


ref_count_ptr<const D2_Region_Entity>
  D2_Composite_Straight_Line_Divided_Region::copy() const
{
	return( ref_count_ptr<const D2_Region_Entity>(
	      new D2_Composite_Straight_Line_Divided_Region( *this ) ) );
}

bool   D2_Composite_Straight_Line_Divided_Region::in_region(
					    const D2_Vector &  vec ) const
{
	if ( line_divided_region_set.size() == 0 )
	{
		return( false );
	}

	for ( size_t  i = 0  ;  i < line_divided_region_set.size()  ;  i ++ )
	{
		if ( line_divided_region_set[i] -> in_region( vec ) == false )
		{
			return( false );
		}
	}

	return( true );
}

D2_Vector  D2_Composite_Straight_Line_Divided_Region::barycenter()
		  const throw( D2_Region::Cannot_Calculate )
{
	if ( (! closed)	|| (cross_point_set.size() == 0) )
	{
		throw D2_Region::Cannot_Calculate();
	}

	double		s_sum = 0.0;
	D2_Vector	g_sum( D2_Vector::XY , 0.0 , 0.0 );

	const D2_Vector &	p0 = cross_point_set[0] -> point;
	for ( size_t  i = 1  ;  i < cross_point_set.size() - 1  ;  i ++ )
	{
		const D2_Vector &	p1
			    = cross_point_set[i] -> point;
		const D2_Vector &	p2
			    = cross_point_set[i] -> line[0].c_point -> point;

		double	y_offset = 0.0;

		y_offset = (( p0.y() < y_offset ) ? p0.y() : y_offset);
		y_offset = (( p1.y() < y_offset ) ? p1.y() : y_offset);
		y_offset = (( p2.y() < y_offset ) ? p2.y() : y_offset);
		y_offset *= (-1);

		D2_Vector	g = (p0 + p1 + p2) / 3;

		double	s = fabs( (+ ( (p0.x() - p1.x())
				       * (+ (p0.y() + y_offset)
					  + (p1.y() + y_offset) ) )
				   + ( (p1.x() - p2.x())
				       * (+ (p1.y() + y_offset)
					  + (p2.y() + y_offset) ) )
				   + ( (p2.x() - p0.x())
				       * (+ (p2.y() + y_offset)
					  + (p0.y() + y_offset) ) ) )
				  / 2.0 );

		s_sum += s;

		g_sum += g * s;
	}

	if ( s_sum <= D2_Vector::eps() * 2.0 )
	{
		D2_Vector	vec( D2_Vector::XY , 0.0 , 0.0 );
		for ( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
		{
			vec += cross_point_set[i] -> point;
		}

		return( vec / cross_point_set.size() );
	}

	return( g_sum / s_sum );
}


double D2_Composite_Straight_Line_Divided_Region::area()
	 const throw( D2_Region::Cannot_Calculate )
{
	if ( ! closed )
	{
		throw D2_Region::Cannot_Calculate();
	}

	double	y_offset = 0.0;

	for ( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		double	y = cross_point_set[i] -> point.y();

		if ( y < y_offset )
		{
			y_offset = y;
		}
	}

	y_offset *= (-1);

	double	sum = 0.0;

	for ( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		const D2_Vector &	a
			= cross_point_set[i] -> point;
		const D2_Vector &	b
			= cross_point_set[i] -> line[0].c_point -> point;

		sum += (b.x() - a.x())
			* (( (a.y() + y_offset) + (b.y()+ y_offset) ) / 2.0);
	}

	return( fabs(sum) );
}


bool   D2_Composite_Straight_Line_Divided_Region::closed_region() const
{
	return( closed );
}





ref_count_ptr<const D2_Composite_Straight_Line_Divided_Region>
	D2_Composite_Straight_Line_Divided_Region::add(
	  const ref_count_ptr<const D2_Straight_Line_Divided_Region > & reg )
									const
{
	ref_count_ptr<const D2_Composite_Straight_Line_Divided_Region>
	  ret( new D2_Composite_Straight_Line_Divided_Region( *this , reg ) );
	return( ret );
}


// protected constructor
D2_Composite_Straight_Line_Divided_Region::
  D2_Composite_Straight_Line_Divided_Region(
	const D2_Composite_Straight_Line_Divided_Region &  com ,
	const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
	  : closed( com.closed ) ,
	    cross_point_set( com.cross_point_set ) ,
	    line_divided_region_set( com.line_divided_region_set )
{
	if ( closed )
	{
		add_constructor_closed( reg );
		return;
	}

	if ( line_divided_region_set.size() == 0 )
	{
		add_constructor_line_0( reg );
	}
	else if ( line_divided_region_set.size() == 1 )
	{
		add_constructor_line_1( reg );
	}
	else if ( line_divided_region_set.size() == 2 )
	{
		add_constructor_line_2( reg );
	}
	else
	{
		add_constructor_line_3_or_more( reg );
	}
}


void   D2_Composite_Straight_Line_Divided_Region::add_constructor_line_0(
  const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
{
	//                     /
	//                    /
	//            ->     /
	//                  /
	//                 /
	//                /
	line_divided_region_set.push_back( reg );
}


void   D2_Composite_Straight_Line_Divided_Region::add_constructor_line_1(
  const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
{
	// if two lines share a point.
	try
	{
		//
		//       /          \    /    //
		//      /            \  /     //
		//     /      ->      \/      //
		//    /               /\      //
		//   /               /!!\     //
		//  /               /!!!!\    //
		//

		ref_count_ptr<Cross_Point>	cross( new Cross_Point() );
		// if no cross point, throw D2_Region::No_Region_Error
		cross -> point
			= line_divided_region_set[0] -> cross_point( *reg );


		line_divided_region_set.push_back( reg );
		cross -> line[0].line = line_divided_region_set[0];
		cross -> line[0].c_point = static_cast<Cross_Point *>(0);
		cross -> line[1].line = line_divided_region_set[1];
		cross -> line[1].c_point = static_cast<Cross_Point *>(0);
		cross_point_set.push_back( cross );
	}
	// if two lines are parallel
	catch( D2_Region::No_Region_Error )
	{
		if ( line_divided_region_set[0]
		     -> in_region( reg -> line().sample_point() ) )
		{
			if ( reg -> in_region( line_divided_region_set[0]
					         -> line().sample_point() ) )
			{
				//
				//       /!!!!!!!               /!!!!/
				//      /!!!!!!!               /!!!!/
				//     /!!!!!!!      ->       /!!!!/
				//    /!!!!!!!               /!!!!/
				//   /!!!!!!!               /!!!!/
				//  /!!!!!!!               /!!!!/
				//
				line_divided_region_set.push_back( reg );
			}
			else
			{
				//        a             a  b           b
				//       /!!!!!!       /!!/!!!!       /!!!!!!
				//      /!!!!!!       /!!/!!!!       /!!!!!!
				//     /!!!!!!  ->   /!!/!!!!  ->   /!!!!!!
				//    /!!!!!!       /!!/!!!!       /!!!!!!
				//   /!!!!!!       /!!/!!!!       /!!!!!!
				//  /!!!!!!       /!!/!!!!       /!!!!!!
				//

				// erase line_divided_region_set[0],
				//   and register new region.
				line_divided_region_set[0] = reg;
			}
		}
		else
		{
			if ( reg -> in_region( line_divided_region_set[0]
						 -> line().sample_point() ) )
			{
				//        a             b  a           a
				//       /!!!!!!       /!!/!!!!       /!!!!!!
				//      /!!!!!!       /!!/!!!!       /!!!!!!
				//     /!!!!!!  ->   /!!/!!!!  ->   /!!!!!!
				//    /!!!!!!       /!!/!!!!       /!!!!!!
				//   /!!!!!!       /!!/!!!!       /!!!!!!
				//  /!!!!!!       /!!/!!!!       /!!!!!!
				//

				// no processing.
			}
			else
			{
				//
				//       /!!!!!!    !!!!!/     /
				//      /!!!!!!     !!!!/     /!
				//     /!!!!!!  ->  !!!/  &  /!!
				//    /!!!!!!       !!/     /!!!
				//   /!!!!!!        !/     /!!!!
				//  /!!!!!!         /     /!!!!!
				//
				// -> empty region

				line_divided_region_set.clear();
				cross_point_set.clear();
			}
		}

		return;
	}
}

void   D2_Composite_Straight_Line_Divided_Region::add_constructor_line_2(
  const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
{
	if ( cross_point_set.size() == 1 )
	// two lines are crossed.
	try
	{
		D2_Vector	p1;
		D2_Vector	p2;
		D2_Vector	g;

		p1 = reg -> cross_point( *(line_divided_region_set[0]) );
		p2 = reg -> cross_point( *(line_divided_region_set[1]) );
		g = (p1 + p2 + cross_point_set[0] -> point) / 3.0;

		// XXX : if ( 3 cross point is same point )

		if ( line_divided_region_set[0] -> in_region(g)
		  && line_divided_region_set[1] -> in_region(g) )
		{
			if ( reg -> in_region(g) )
			{
				//    \  /             \  /     //
				//     \/               \/      //
				//     /\      ->       /\      //
				//    /!!\             /!!\     //
				//   /!!!!\          --------   //
				//  /!!!!!!\         /      \   //
				line_divided_region_set.insert(
				  line_divided_region_set.begin() + 1 , reg );

				ref_count_ptr<Cross_Point>
					cross_1( new Cross_Point() );

				ref_count_ptr<Cross_Point>
					cross_2( new Cross_Point() );

				cross_point_set.push_back( cross_1 );
				cross_point_set.push_back( cross_2 );

				cross_point_set[1] -> point = p1;
				cross_point_set[2] -> point = p2;

				cross_point_set[0] -> line[0].c_point
					= cross_point_set[1].get();
				cross_point_set[0] -> line[1].c_point
					= cross_point_set[2].get();

				cross_point_set[1] -> line[0].c_point
					= cross_point_set[2].get();
				cross_point_set[1] -> line[0].line
					= line_divided_region_set[1];
				cross_point_set[1] -> line[1].c_point
					= cross_point_set[0].get();
				cross_point_set[1] -> line[1].line
					= line_divided_region_set[0];

				cross_point_set[2] -> line[0].c_point
					= cross_point_set[0].get();
				cross_point_set[2] -> line[0].line
					= line_divided_region_set[2];
				cross_point_set[2] -> line[1].c_point
					= cross_point_set[1].get();
				cross_point_set[2] -> line[1].line
					= line_divided_region_set[1];

				closed = true;
			}
			else
			{
				//    \  /             \  /     //
				//     \/               \/      //
				//     /\      ->       /\      //
				//    /!!\             /  \     //
				//   /!!!!\          --------   //
				//  /!!!!!!\         /!!!!!!\   //

				// XXX
#ifndef NO_IMPLEMENT_WARNING
				cerr << "Not implemented.(open)" << endl;
#endif
			}
		}
		else
		{
			// BUG:
			//
			//  \      /   \!!!!| /    \    | /    //
			//   \    /    !\!!!|/      \   |/     //
			//    \  /     !!\!!|        \  |      //
			//     \/      !!!\/|         \/|      //
			//     /\   -> !!!/\|   ->    /\|      //
			//    /!!\     !!/!!|        /!!|      //
			//   /!!!!\    !/!!!|\      /!!!|\     //
			//  /!!!!!!\   /!!!!|!\    /!!!!| \    //
			//                                     //
			// or                                  //
			//                                     //
			//  \      /   \    |!/    \    | /    //
			//   \    /     \   |/!     \   |/     //
			//    \  /       \  |!!      \  |      //
			//     \/         \/|!!       \/|      //
			//     /\   ->    /\|!! ->    /\|      //
			//    /!!\       /!!|!!      /  |      //
			//   /!!!!\     /!!!|\!     /   |\     //
			//  /!!!!!!\   /!!!!|!\    /    |!\    //
			//                                     //

			if ( reg -> in_region(g) )
			{
				//
				//  \      /   \      /    \      /    //
				//   \    /    _\____/_     \    /     //
				//    \  /     !!\!!/!!      \  /      //
				//     \/      !!!\/!!!       \/       //
				//     /\   -> !!!/\!!! ->    /\       //
				//    /!!\     !!/!!\!!      /!!\      //
				//   /!!!!\    !/!!!!\!     /!!!!\     //
				//  /!!!!!!\   /!!!!!!\    /!!!!!!\    //
				//

				// reject new line devided region.
				// no processing
			}
			else
			{
				//
				//  \      /   \!!!!!!/                   //
				//   \    /    _\____/_                   //
				//    \  /       \  /                     //
				//     \/         \/                      //
				//     /\   ->    /\    ->  empty region. //
				//    /!!\       /!!\                     //
				//   /!!!!\     /!!!!\                    //
				//  /!!!!!!\   /!!!!!!\                   //
				//

				line_divided_region_set.clear();
				cross_point_set.clear();
			}
		}
	}
	catch(...)
	{
		//    \  /        \  /        //
		//     \/          \/         //
		//     /\   ->  \  /\         //
		//    /!!\       \/!!\        //
		//   /!!!!\      /\!!!\       //
		//  /!!!!!!\    /!!\!!!\      //
		//

		// XXX
#ifndef NO_IMPLEMENT_WARNING
		cerr << "Not implemented.(3 parallel)" << endl;
#endif
	}
	// two lines are parallel
	else
	{
		//
		//   |!!!|
		//   |!!!|
		//   |!!!|
		//   |!!!|
		//

#ifndef NO_IMPLEMENT_WARNING
		// XXX
		cerr << "Not implemented.(2 parallel)" << endl;
#endif
	}
}


void   D2_Composite_Straight_Line_Divided_Region::
 add_constructor_line_3_or_more(
  const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
{
	// XXX
	(void)reg;
#ifndef NO_IMPLEMENT_WARNING
	cerr << "Not implemented.(3 or more)" << endl;
#endif
}


void   D2_Composite_Straight_Line_Divided_Region::add_constructor_closed(
  const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
{
	bool	in  = false;
	bool	out = false;

	for( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		if ( reg -> in_region( cross_point_set[i] -> point ) )
		{
			in = true;
		}
		else
		{
			out = true;
		}

		// for optimize only
		if ( in == true  &&  out == true )
		{
			break;
		}
	}

	if ( in == true  &&  out == true )
	{
		add_constructor_closed_cut_off( reg );
	}
	else
	{
		if ( in == true )
		{
			// no processing.
		}
		else
		{
			closed = false;
			line_divided_region_set.clear();
			cross_point_set.clear();
		}
	}
}


void   D2_Composite_Straight_Line_Divided_Region::
	add_constructor_closed_cut_off(
  const ref_count_ptr< const D2_Straight_Line_Divided_Region > &  reg )
{
	// XXX: if point is on line.
	// XXX: if parallel

#if 0
	cout << endl;
	cout << endl;
	cout << endl;
	cout << endl;
	cout << "cut off" << endl;
#endif

#if 0
	cout << "start" << endl;
	for ( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		cout << "cross_point[" << i << "] = ["
		     << cross_point_set[i] -> point << "]" << endl;
		cout << "cross_point[" << i << "] -> line[0].c_point = ["
		     << cross_point_set[i] -> line[0].c_point -> point << "]"
		     << endl;
		cout << "cross_point[" << i
		     << "] -> line[0].line.direction() = "
		     << cross_point_set[i]
			-> line[0].line -> line().direction() << endl;

		cout << "cross_point[" << i << "] -> line[1].c_point = ["
		     << cross_point_set[i] -> line[1].c_point -> point << "]"
		     << endl;
		cout << "cross_point[" << i
		     << "] -> line[1].line.direction() = "
		     << cross_point_set[i]
			-> line[1].line -> line().direction() << endl;
	}
#endif

	assert( cross_point_set.size() >= 3 );

	vector<size_t>	cross;

	for( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		bool	crossed = false;

		if ( reg -> in_region( cross_point_set[i] -> point ) )
		{
			if ( ! reg -> in_region(
				cross_point_set[i]
				-> line[0].c_point -> point ) )
			{
				crossed = true;
			}
		}
		else
		{
			if ( reg -> in_region(
			      cross_point_set[i]
			      -> line[0].c_point -> point ) )
			{
				crossed = true;
			}
		}

		if ( crossed )
		{
			cross.push_back( i );
		}
	}

	// XXX
	if ( cross.size() != 2 )
	{
		return;
	}

#if 0
	cout << "cross[0] = " << cross[0] << endl;
	cout << "cross[1] = " << cross[1] << endl;
#endif

	if ( reg -> in_region(cross_point_set[cross[0]] -> point) )
	try {
		Cross_Point &	p1 = *(cross_point_set[cross[0]]);
		Cross_Point &	p2 = *(cross_point_set[cross[1]]
				       -> line[0].c_point);

		ref_count_ptr<Cross_Point>	new_point_1( new Cross_Point );
		ref_count_ptr<Cross_Point>	new_point_2( new Cross_Point );

		new_point_1 -> point
			= reg -> cross_point( *(p1.line[0].line) );
		new_point_1 -> line[0].line = reg;
		new_point_1 -> line[0].c_point = new_point_2.get();
		new_point_1 -> line[1].line
			= cross_point_set[cross[0]] -> line[0].line;
		new_point_1 -> line[1].c_point
			= cross_point_set[cross[0]].get();


		new_point_2 -> point
			= reg -> cross_point( *(p2.line[1].line) );
		new_point_2 -> line[0].line = p2.line[1].line;
		new_point_2 -> line[0].c_point
			= cross_point_set[cross[1]] -> line[0].c_point;
		new_point_2 -> line[1].line = reg;
		new_point_2 -> line[1].c_point
			= new_point_1.get();

		p1.line[0].c_point = new_point_1.get();
		p2.line[1].c_point = new_point_2.get();

		cross_point_set.erase( cross_point_set.begin() + cross[0] + 1,
				       cross_point_set.begin() + cross[1] + 1);

		cross_point_set.insert( cross_point_set.begin() + cross[0] + 1,
					new_point_1 );
		cross_point_set.insert( cross_point_set.begin() + cross[0] + 2,
					new_point_2 );
	} catch(...){}
	else try {
		Cross_Point &	p1 = *(cross_point_set[cross[0]]
				       -> line[0].c_point);
		Cross_Point &	p2 = *(cross_point_set[cross[1]]);

		ref_count_ptr<Cross_Point>	new_point_1( new Cross_Point );
		ref_count_ptr<Cross_Point>	new_point_2( new Cross_Point );

		new_point_1 -> point
			= reg -> cross_point( *(p1.line[1].line) );
		new_point_1 -> line[0].line = p1.line[1].line;
		new_point_1 -> line[0].c_point
			= cross_point_set[cross[0]] -> line[0].c_point;
		new_point_1 -> line[1].line = reg;
		new_point_1 -> line[1].c_point = new_point_2.get();


		new_point_2 -> point
			= reg -> cross_point( *(p2.line[0].line) );

		new_point_2 -> line[0].line = reg;
		new_point_2 -> line[0].c_point = new_point_1.get();
		new_point_2 -> line[1].line = p2.line[0].line;
		new_point_2 -> line[1].c_point
			= cross_point_set[cross[1]].get();

		p1.line[1].c_point = new_point_1.get();
		p2.line[0].c_point = new_point_2.get();

		cross_point_set.erase( cross_point_set.begin() + cross[1] + 1 ,
				       cross_point_set.end() );

		cross_point_set.erase( cross_point_set.begin() ,
				       cross_point_set.begin() + cross[0] + 1);

		cross_point_set.insert( cross_point_set.begin() ,
					new_point_1 );
		cross_point_set.insert( cross_point_set.end() ,
					new_point_2 );
	} catch(...){}

#if 0
	cout << "end" << endl;
	for ( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		cout << "cross_point[" << i << "] = ["
		     << cross_point_set[i] -> point << "]" << endl;
		cout << "cross_point[" << i << "] -> line[0].c_point = ["
		     << cross_point_set[i] -> line[0].c_point -> point << "]"
		     << endl;
		cout << "cross_point[" << i << "] -> line[1].c_point = ["
		     << cross_point_set[i] -> line[1].c_point -> point << "]"
		     << endl;
	}
#endif
}


vector<D2_Vector>  D2_Composite_Straight_Line_Divided_Region::
	point_list() const
{
	vector<D2_Vector>	ret;

	for ( size_t  i = 0  ;  i < cross_point_set.size()  ;  i ++ )
	{
		ret.push_back( cross_point_set[i] -> point );
	}

	return( ret );
}
