#include  "d2_vector.h"
#include  <cmath>
#include  "math_extention.h"
#include  <cassert>

D2_VECTOR_INLINE
double  D2_Vector::deg_to_rad( double  x )
{
	return( x * Math_Extention::PI / 180.0 );
}

D2_VECTOR_INLINE
double  D2_Vector::rad_to_deg( double  x )
{
	return( x * 180.0 / Math_Extention::PI );
}


D2_VECTOR_INLINE
D2_Vector::D2_Vector()
{
	x_value = 0.0;
	y_value = 0.0;
	xy_calculated = true;

	r_value     = 0.0;
	theta_value = 0.0;
	pole_calculated = true;
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( XY_or_Pole  type ,
		      double  x_or_r ,  double  y_or_theta )
{
	this -> set( type , x_or_r , y_or_theta );
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( XY_or_Pole  type ,
		      double  r ,  const Angle &  theta )
{
	this -> set( type , r , theta );
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( double  x ,  double  y )
{
	this -> set( D2_Vector::XY , x , y );
}

D2_VECTOR_INLINE
D2_Vector::D2_Vector( const D2_Vector &  vec )
{
	this -> set( vec );
}

D2_VECTOR_INLINE
D2_Vector::~D2_Vector()
{
}


D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( XY_or_Pole  type ,
			     double  x_or_r ,  double  y_or_theta )
{
	if ( type == XY )
	{
		x_value = x_or_r;
		y_value = y_or_theta;
		xy_calculated   = true;
		pole_calculated = false;
	}
	else
	{
		r_value     = x_or_r;
		theta_value = y_or_theta;
		pole_calculated = true;
		xy_calculated   = false;
	}

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( XY_or_Pole  type ,
			     double  r ,  const Angle &  theta )
{
	assert( type == Pole );

	return( this -> set( D2_Vector::Pole , r , theta.radian() ) );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( double  r ,  const Angle &  theta )
{
	return( this -> set( D2_Vector::Pole , r , theta.radian() ) );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( double  x_val ,  double  y_val )
{
	return( this -> set( D2_Vector::XY , x_val , y_val ) );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set( const D2_Vector &  vec )
{
	if ( vec.xy_calculated )
	{
		this -> xy_calculated = true;
		this -> x_value       = vec.x_value;
		this -> y_value       = vec.y_value;
	}
	else
	{
		this -> xy_calculated = false;
	}

	if ( vec.pole_calculated )
	{
		this -> pole_calculated = true;
		this -> r_value         = vec.r_value;
		this -> theta_value     = vec.theta_value;
	}
	else
	{
		this -> pole_calculated = false;
	}

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_x( double  x )
{
	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	x_value = x;

	pole_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_y( double  y )
{
	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	y_value = y;

	pole_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_r( double  r )
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	r_value = r;

	xy_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::set_theta( const Angle &  theta )
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	theta_value = theta.radian();

	xy_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator = ( const D2_Vector &  vec )
{
	return( this -> set( vec ) );
}


D2_VECTOR_INLINE
double  D2_Vector::x() const
{
	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	return( x_value );
}

D2_VECTOR_INLINE
double  D2_Vector::y() const
{
	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	return( y_value );
}

D2_VECTOR_INLINE
double  D2_Vector::r() const
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	return( r_value );
}

D2_VECTOR_INLINE
Angle  D2_Vector::theta() const
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	return( Angle( Angle::Radian , theta_value ) );
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::normalize() const
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	D2_Vector	ret = (*this);

	ret.r_value = 1.0;

	if ( ret.r_value <= eps() )
	{
		ret.xy_calculated = false;
	}
	else
	{
		ret.x_value /= this -> r_value;
		ret.y_value /= this -> r_value;
	}

	return( ret );
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::normalize_theta() const
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	D2_Vector	ret = (*this);

	// [-PI , +PI)
	if ( ret.theta_value >= (+ Math_Extention::PI) )
	{
		do
		{
			ret.theta_value -= 2.0 * Math_Extention::PI;

		} while( ret.theta_value >= Math_Extention::PI );
	}
	else if ( ret.theta_value < (- Math_Extention::PI) )
	{
		do
		{
			ret.theta_value += 2.0 * Math_Extention::PI;

		} while( ret.theta_value < (- Math_Extention::PI) );
	}

	if ( ret.theta_value >= (+ Math_Extention::PI)
	  || ret.theta_value <  (- Math_Extention::PI) )
	{
		ret.theta_value = (- Math_Extention::PI);
	}

	assert( (- Math_Extention::PI) - eps() <= ret.theta_value
			&& ret.theta_value < (+ Math_Extention::PI) + eps() );

	return( ret );
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::rotate( const Angle &  d ) const
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	D2_Vector	ret = (*this);

	ret.theta_value += d.radian();
	ret.xy_calculated = false;

	return( ret );
}


D2_VECTOR_INLINE
void  D2_Vector::calculate_xy() const
{
	assert( pole_calculated );

	x_value = r_value * std::cos( theta_value );
	y_value = r_value * std::sin( theta_value );

	xy_calculated = true;
}

D2_VECTOR_INLINE
void  D2_Vector::calculate_pole() const
{
	assert( xy_calculated );

	double	r2 = x_value * x_value + y_value * y_value;

	if ( r2 < 0.0 )
	{
		r2 = 0.0;
	}

	r_value = sqrt( r2 );

	if ( r_value == 0.0 )
	{
		theta_value = 0.0;
	}
	else
	{
		theta_value = std::atan2( y_value , x_value );
	}

	pole_calculated = true;
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator + () const
{
	return( *this );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator - () const
{
	return( D2_Vector::origin() - (*this) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator + ( const D2_Vector &  vec ) const
{
	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	return( D2_Vector( D2_Vector::XY ,
			   this -> x_value + vec.x() ,
			   this -> y_value + vec.y() ) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator - ( const D2_Vector &  vec ) const
{
	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	return( D2_Vector( D2_Vector::XY ,
			   this -> x_value - vec.x() ,
			   this -> y_value - vec.y() ) );
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator * ( double  d ) const
{
	if ( pole_calculated )
	{
		return( D2_Vector( D2_Vector::Pole ,
				   this -> r_value * d ,
				   this -> theta_value ) );
	}
	else
	{
		assert( xy_calculated );

		return( D2_Vector( D2_Vector::XY ,
				   this -> x_value * d ,
				   this -> y_value * d ) );
	}
}

D2_VECTOR_INLINE
D2_Vector  D2_Vector::operator / ( double  d ) const
{
	if ( d <= eps() )
	{
		std::cerr << "Divided by zero." << std::endl;

		return( D2_Vector( D2_Vector::XY , 0.0 , 0.0 ) );
	}

	if ( pole_calculated )
	{
		return( D2_Vector( D2_Vector::Pole ,
				   this -> r_value / d ,
				   this -> theta_value ) );
	}
	else
	{
		assert( xy_calculated );

		return( D2_Vector( D2_Vector::XY ,
				   this -> x_value / d ,
				   this -> y_value / d ) );
	}
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator += ( const D2_Vector &  vec )
{
	// checked: OK. if (this == &vec)

	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	x_value += vec.x();
	y_value += vec.y();

	pole_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator -= ( const D2_Vector &  vec )
{
	// checked: OK. if (this == &vec)

	if ( ! xy_calculated )
	{
		calculate_xy();
	}

	x_value -= vec.x();
	y_value -= vec.y();

	pole_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator *= ( double  d )
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	r_value *= d;

	xy_calculated = false;

	return( *this );
}

D2_VECTOR_INLINE
D2_Vector &  D2_Vector::operator /= ( double  d )
{
	if ( ! pole_calculated )
	{
		calculate_pole();
	}

	if ( d <= eps() )
	{
		std::cerr << "Divided by zero." << std::endl;

		return( *this );
	}

	r_value /= d;

	xy_calculated = false;

	return( *this );
}


D2_VECTOR_INLINE
D2_Vector  operator * ( double  d ,  const D2_Vector &  vec )
{
	return( vec * d );
}


D2_VECTOR_INLINE
bool   D2_Vector::equal( const D2_Vector &  vec , double  epsilon ) const
{
	return( (vec - *(this)).r() <= epsilon );
}


D2_VECTOR_INLINE
bool   D2_Vector::operator == ( const D2_Vector &  vec ) const
{
	return( (vec - *(this)).r() == 0.0 );
}

D2_VECTOR_INLINE
bool   D2_Vector::operator != ( const D2_Vector &  vec ) const
{
	return( (vec - *(this)).r() != 0.0 );
}

D2_VECTOR_INLINE
double  D2_Vector::eps()
{
	return( EPSILON );
}


D2_VECTOR_INLINE
D2_Vector  D2_Vector::origin()
{
	return( D2_Vector( D2_Vector::XY , 0.0 , 0.0 ) );
}


D2_VECTOR_INLINE
std::ostream &  operator << ( std::ostream &  ostr , const D2_Vector &  vec )
{
	ostr << vec.x() << "," << vec.y();

	return( ostr );
}
