#ifndef	   RANGE_H_INCLUDED
#define	   RANGE_H_INCLUDED

#ifdef HAVE_CONFIG_H
# include  "config.h"
#endif

#include  <algorithm>

template<class T>
class  Basic_Range
{
protected:
	T	minimum;
	T	maximum;

public:
	 Basic_Range();
	 Basic_Range( const T &  minimum ,  const T &   maximum )
		 : minimum( minimum ) , maximum( maximum ) {}
	~Basic_Range() {}

	T	min()    const { return( minimum ); }
	T	max()    const { return( maximum ); }
	T	median() const { return( (min() + max()) / 2.0 ); }

	bool	valid() const
		{
			return( minimum <= maximum );
		}

	void	set( const T &  minimum ,  const T &  maximum )
		{
			this -> minimum = minimum;
			this -> maximum = maximum;
		}

	bool	in_range( const T &  p ,  const T & eps ) const
		{
			return( minimum - eps < p  &&  p < maximum + eps );
		}

	T	width() const
		{
			return( maximum - minimum );
		}

	Basic_Range	clip( const Basic_Range<T> &  r ) const
		{
			T	low;
			T	high;

			if ( r.min() > this -> max() )
			{
				low = this -> max();
			}
			else if ( r.min() < this -> min() )
			{
				low = this -> min();
			}
			else
			{
				low = r.min();
			}

			if ( r.max() < this -> min() )
			{
				high = this -> min();
			}
			else if ( r.max() > this -> max() )
			{
				high = this -> max();
			}
			else
			{
				high = r.max();
			}

			return( Basic_Range( low , high ) );
		}


	Basic_Range	operator & ( const Basic_Range<T> &  r ) const
		{
			return( Basic_Range( std::max( this -> minimum ,
						       r.      minimum ) ,
					     std::min( this -> maximum ,
						       r.      maximum ) ) );
		}

	Basic_Range	operator | ( const Basic_Range<T> &  r ) const
		{
			return( Basic_Range( std::min( this -> minimum ,
						       r.      minimum ) ,
					     std::max( this -> maximum ,
						       r.      maximum ) ) );
		}

	Basic_Range	operator + ( const T &  base ) const
		{
			return( Basic_Range( this -> minimum + base ,
					     this -> maximum + base ) );
		}

	Basic_Range	operator - ( const T &  base ) const
		{
			return( Basic_Range( this -> minimum - base ,
					     this -> maximum - base ) );
		}

	Basic_Range	operator + ( const Basic_Range<T> &  r ) const
		{
			return( Basic_Range( this -> minimum + r.minimum ,
					     this -> maximum + r.maximum ) );
		}

	Basic_Range	operator - ( const Basic_Range<T> &  r ) const
		{
			return( Basic_Range( this -> minimum - r.maximum ,
					     this -> maximum - r.minimum ) );
		}

	Basic_Range &	operator &= ( const Basic_Range<T> &  r )
		{
			this -> minimum = std::max( this -> minimum ,
						    r.      minimum );
			this -> maximum = std::min( this -> maximum ,
						    r.      maximum );
			return( *this );
		}

	Basic_Range &	operator |= ( const Basic_Range<T> &  r )
		{
			this -> minimum = std::min( this -> minimum ,
						    r.      minimum );
			this -> maximum = std::max( this -> maximum ,
						    r.      maximum );
			return( *this );
		}

	Basic_Range &	operator += ( const T &  base )
		{
			this -> minimum += base;
			this -> maximum += base;

			return( *this );
		}

	Basic_Range &	operator -= ( const T &  base )
		{
			this -> minimum -= base;
			this -> maximum -= base;

			return( *this );
		}

	Basic_Range &	operator += ( const Basic_Range<T> &  r )
		{
			this -> minimum += r.minimum;
			this -> maximum += r.maximum;

			return( *this );
		}

	Basic_Range &	operator -= ( const Basic_Range<T> &  r )
		{
			this -> minimum -= r.maximum;
			this -> maximum -= r.minimum;

			return( *this );
		}
};

typedef Basic_Range<double>		Double_Range;
typedef Basic_Range<int>		Int_Range;
typedef Basic_Range<long>		Long_Range;
typedef Basic_Range<unsigned int>	Unsigned_Int_Range;
typedef Basic_Range<unsigned long>	Unsigned_Long_Range;

#include  "angle.h"
typedef Basic_Range<Angle> Angle_Range;


template<> inline
  Basic_Range<double   >::Basic_Range() : minimum( 0.0 ) , maximum( 0.0 ) {}
template<> inline
  Basic_Range<int          >::Basic_Range() : minimum( 0 ) , maximum( 0 ) {}
template<> inline
  Basic_Range<long         >::Basic_Range() : minimum( 0 ) , maximum( 0 ) {}
template<> inline
  Basic_Range<unsigned int >::Basic_Range() : minimum( 0 ) , maximum( 0 ) {}
template<> inline
  Basic_Range<unsigned long>::Basic_Range() : minimum( 0 ) , maximum( 0 ) {}
template<> inline
  Basic_Range<Angle>::Basic_Range()
	  : minimum( Angle::origin() ) , maximum( Angle::origin() ) {}


#endif	/* RANGE_H_INCLUDED */
