#include  "udp_connection.h"
#include  "ref_count_ptr.h"
#include  "debugstream.h"
#include  <cstring>
#include  <iostream>
#include  <cstdio>

extern "C"
{
	#include  <stdio.h>
	#include  <string.h>
	#include  <errno.h>
	#include  <sys/types.h>
	#include  <sys/socket.h>
	#include  <netinet/in.h>
	#include  <netdb.h>

	#ifdef HAVE_CONFIG_H
	    #if HAVE_UNISTD_H
		#include  <unistd.h>
	    #endif
	#else
	    #include  <unistd.h>
	#endif

	#ifdef HAVE_SYS_IOCTL_H
	    #if HAVE_SYS_IOCTL_H
		#include  <sys/ioctl.h>
	    #endif
	#else
	    #include  <sys/ioctl.h>
	#endif

	#ifdef HAVE_FCNTL_H
	    #if HAVE_FCNTL_H
		#include  <fcntl.h>
	    #endif
	#else
	    #include  <fcntl.h>
	#endif

	#if defined(TIME_WITH_SYS_TIME) && defined(HAVE_SYS_TIME_H)
	    #if TIME_WITH_SYS_TIME
		#include  <sys/time.h>
		#include  <time.h>
	    #else
		#if HAVE_SYS_TIME_H
		    #include  <sys/time.h>
		#else
		    #include  <time.h>
		#endif
	    #endif
	#else
	    #include  <sys/time.h>
	#endif

	#ifdef sgi
	    #include <bstring.h>
	#endif
}


#ifndef HAVE_SOCKLEN_T
  #ifdef __linux
	typedef	socklen_t	SOCKLEN_T;
  #else
    #ifdef __FreeBSD__
      #if __FreeBSD__ >= 4
	typedef	socklen_t	SOCKLEN_T;
      #else
	typedef	int		SOCKLEN_T;
      #endif
    #else
	typedef	int		SOCKLEN_T;
    #endif
  #endif
#endif

using namespace std;

UDP_Connection::UDP_Connection( const string &  host ,
				ip_port_number_t  port_num ,
				bool  debug ,
				size_t  buf_size ,
				long  max_resp )
	: recv_time( 0 , 0 ) , received( false ) ,
	  buffer_size( buf_size ) , max_response( max_resp ) ,
	  debug_flag( debug )
{
	memset( &(this -> addr) , 0 , sizeof( this -> addr ) );

	if ( (this -> sfd = ::socket( AF_INET , SOCK_DGRAM  , 0 ))  ==  -1 )
	{
		this -> sfd = UDP_Connection::BAD_FD;

		perror( "socket" );
		return;
	}

	struct sockaddr_in	my_addr;
	memset( &my_addr , 0 , sizeof( my_addr ) );
	my_addr.sin_family      = AF_INET;
	my_addr.sin_addr.s_addr = htonl( INADDR_ANY );
	my_addr.sin_port        = htons( 0 );

	if ( ::bind( this -> sfd , (const struct sockaddr *)( &my_addr ) ,
				   sizeof( my_addr ) )  ==  -1 )
	{
		perror( "bind" );
		this -> sfd = UDP_Connection::BAD_FD;
		this -> close();
		return;
	}

	struct hostent *	hp;
	if ( (hp = ::gethostbyname( host.c_str() ))
	     ==  static_cast<struct hostent *>(0) )
	{
		perror( "gethostbyname" );
		this -> sfd = UDP_Connection::BAD_FD;
		this -> close();
		return;
	}

	this -> addr.sin_family = AF_INET;
	memcpy( &(this -> addr.sin_addr.s_addr) ,
		hp -> h_addr , hp -> h_length );
	this -> addr.sin_port = htons( port_num );

	recv_time.set_current_time();
}


UDP_Connection::UDP_Connection( ip_port_number_t  port ,
				bool  debug ,
				size_t  buf_size ,  long  max_resp )
	: recv_time( 0 , 0 ) , received( false ) ,
	  buffer_size( buf_size ) , max_response( max_resp ) ,
	  debug_flag( debug )
{
	memset( &(this -> addr) , 0 , sizeof( this -> addr ) );

	if ( (this -> sfd = ::socket( AF_INET , SOCK_DGRAM  , 0 ))  ==  -1 )
	{
		perror( "socket" );
		this -> sfd = UDP_Connection::BAD_FD;
		return;
	}

	struct sockaddr_in	my_addr;
	memset( &my_addr , 0 , sizeof( my_addr ) );
	my_addr.sin_family      = AF_INET;
	my_addr.sin_addr.s_addr = htonl( INADDR_ANY );
	my_addr.sin_port        = htons( port );

	if ( ::bind( this -> sfd , (const struct sockaddr *)( &my_addr ) ,
				   sizeof( my_addr ) )  ==  -1 )
	{
		perror( "bind" );
		this -> sfd = UDP_Connection::BAD_FD;
		this -> close();
		return;
	}

	this -> addr.sin_family = AF_INET;

	recv_time.set_current_time();
}


UDP_Connection::~UDP_Connection()
{
	this -> close();
}



UDP_Connection::operator bool() const
{
	return( this -> fd() != UDP_Connection::BAD_FD );
}

bool   UDP_Connection::responsive() const
{
	if ( ! (this -> operator bool()) )
	{
		return( false );
	}

	if ( ! received )
	{
		return( false );
	}

	if ( this -> max_response == 0 )
	{
		return( true );
	}
	else
	{
		Time_Stamp	t;
		t.set_current_time();

		return( (t - recv_time).usec() < (this -> max_response) );
	}
}


int    UDP_Connection::fd() const
{
	return( this -> sfd );
}


int    UDP_Connection::close()
{
	// if already closed or not opened
	if ( this -> sfd == UDP_Connection::BAD_FD )
	{
		// return with no error
		return( 0 );
	}
	else
	{
		// close the file descriptor
		int	ret;

		if ( (ret = ::close( this -> sfd ))  ==  -1 )
		{
			perror( "close" );
		}

		this -> sfd = UDP_Connection::BAD_FD;

		return( ret );
	}
}


int    UDP_Connection::send( const string &  str )
{
	if ( this -> sfd == UDP_Connection::BAD_FD )
	{
		return( -1 );
	}

	int	n;
	if ( (n = ::sendto( this -> sfd , str.c_str() , str.length() , 0 ,
			    (struct sockaddr *)( &(this -> addr) ) ,
			    sizeof( this -> addr ) ))  ==  -1 )
	{
		perror( "sendto" );
		return( -1 );
	}

	if ( Debug_Stream::dbg )
	{
		Debug_Stream::dbg << "send[" << str << "]" << endl;
	}


	// XXX
	// compare int and size_t
	// It may cause problem.
	//
	if ( static_cast<size_t>(n) == str.length() )
	{
		return( n );
	}
	else
	{
		return( -1 );
	}
}


int    UDP_Connection::recv( ref_count_ptr<string> *  buf ,
			     long  usec ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	ref_count_ptr<char>	char_buf;

	int	ret = this -> recv( &char_buf , /*append_nul*/ false ,
				    usec , reply );

	if ( ret == -1 )
	{
		*buf = new string();
	}
	else
	{
		*buf = new string( char_buf.get() , ret );
	}

	return( ret );
}


int    UDP_Connection::recv( ref_count_ptr<string> *  buf ,
			     bool  block ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	ref_count_ptr<char>	char_buf;

	int	ret = this -> recv( &char_buf , /*append_nul*/ false ,
				    block , reply );

	if ( ret == -1 )
	{
		*buf = new string();
	}
	else
	{
		*buf = new string( char_buf.get() , ret );
	}

	return( ret );
}


int    UDP_Connection::recv( string *  str ,  long  usec ,  bool  reply )
{
	ref_count_ptr<string>	s;
	int			ret;

	if ( (ret = this -> recv( &s , usec , reply ))  ==  -1 )
	{
		*str = "";
	}
	else
	{
		*str = *s;
	}

	return( ret );
}


int    UDP_Connection::recv( string *  str ,  bool  block ,  bool  reply )
{
	ref_count_ptr<string>	s;
	int			ret;

	if ( (ret = this -> recv( &s , block , reply ))  ==  -1 )
	{
		*str = "";
	}
	else
	{
		*str = *s;
	}

	return( ret );
}


int    UDP_Connection::recv( ref_count_ptr<char> *  buf ,  bool  append_nul ,
			     long  usec ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	size_t	len = this -> buffer_size + (append_nul ? 1 : 0);

	*buf = new char[ len ];

	int	ret = this -> recv( buf -> get() , len , append_nul ,
				    usec , reply );

	if ( ret == -1 )
	{
		*buf = new char[0];
	}

	return( ret );
}


int    UDP_Connection::recv( ref_count_ptr<char> *  buf ,  bool  append_nul ,
			     bool  block ,  bool  reply )
{
	if ( ! buf )
	{
		return( -1 );
	}

	size_t	len = this -> buffer_size + (append_nul ? 1 : 0);

	*buf = new char[ len ];

	int	ret = this -> recv( buf -> get() , len , append_nul ,
				    block , reply );

	if ( ret == -1 )
	{
		*buf = new char[0];
	}

	return( ret );
}


int    UDP_Connection::recv( char *  buf ,  size_t  len ,  bool  append_nul ,
			     long  usec ,  bool  reply )
{
	if ( this -> sfd == UDP_Connection::BAD_FD )
	{
		return( -1 );
	}


	struct sockaddr_in	from;
	SOCKLEN_T		from_size;
	int			n = 0;
	int			ret = 0;

	from_size = sizeof( from );

	struct timeval	tv;
	fd_set		readfd;
	FD_ZERO( &readfd );
	FD_SET( this -> sfd , &readfd );

	tv.tv_sec  = usec / (1000 * 1000);
	tv.tv_usec = usec % (1000 * 1000);

	if ( ::select( this -> sfd + 1 , &readfd ,
		       static_cast<fd_set *>(0) , static_cast<fd_set *>(0) ,
		       &tv )  ==  1 )
	{
		if ( (ret = ::recvfrom( this -> sfd , buf , len , 0 ,
					(struct sockaddr *)( &from ) ,
					&from_size ))  ==  -1 )
		{
			perror( "recvfrom" );
			n = 0;
		}
		else
		{
			recv_time.set_current_time();
			received = true;
			n = ret;
		}
	}
	else
	{
		ret = -1;
		n = 0;
	}

	if ( append_nul )
	{
		buf[n] = '\0';
	}

	if ( Debug_Stream::dbg )
	{
		if ( ret != -1 )
		{
			Debug_Stream::dbg << "recv[";

			for ( int  i = 0  ;  i < n  ;  i ++ )
			{
				Debug_Stream::dbg << buf[i];
			}

			Debug_Stream::dbg << "]" << endl;
		}
	}

	if ( reply )
	{
		if ( from.sin_port != static_cast<ip_port_number_t>(0) )
		{
			this -> addr.sin_port = from.sin_port;
		}
	}

	return( ret );
}


int    UDP_Connection::recv( char *  buf ,  size_t  len ,  bool  append_nul ,
			     bool  block ,  bool  reply )
{
	if ( this -> sfd == UDP_Connection::BAD_FD )
	{
		return( -1 );
	}

	if ( this -> max_response != 0L )
	{
		return( this -> recv( buf , len , append_nul ,
				      (block ? (this -> max_response) : 0L) ,
				      reply ) );
	}


	struct sockaddr_in	from;
	SOCKLEN_T		from_size;
	int			n = 0;
	int			ret = 0;

	from_size = sizeof( from );

	if ( block )
	{
		if ( (ret = ::recvfrom( this -> sfd , buf , len , 0 ,
					(struct sockaddr *)(&from) ,
					&from_size ))  ==  -1 )
		{
			if ( errno != EWOULDBLOCK )
			{
				perror( "recvfrom" );
			}
			n = 0;
		}
		else
		{
			recv_time.set_current_time();
			received = true;
			n = ret;
		}
	}

	if ( append_nul )
	{
		buf[n] = '\0';
	}

	if ( reply )
	{
		if ( from.sin_port != static_cast<ip_port_number_t>(0) )
		{
			this -> addr.sin_port = from.sin_port;
		}
	}

	return( ret );
}
