#include  "command_queue.h"

using namespace std;

//
// Command_Queue
//
Command_Queue::Command_Queue()
	: Soccer_Action( Field_Recog_Interface() )
{
}

Command_Queue::Command_Queue( const Command_Queue &  org )
	: Soccer_Action( Field_Recog_Interface() )
{
	this -> overwrite_this( org );
}

Command_Queue::Command_Queue( const Soccer_Composite_Command &  com )
	: Soccer_Action( Field_Recog_Interface() )
{
	this -> push( com );
}

Command_Queue::Command_Queue( const ref_count_ptr<Soccer_Action> &  act )
	: Soccer_Action( Field_Recog_Interface() )
{
	this -> push( act );
}

Command_Queue::Command_Queue( Soccer_Action *  act )
	: Soccer_Action( Field_Recog_Interface() )
{
	this -> push( act );
}

Command_Queue::~Command_Queue()
{
}

Command_Queue &  Command_Queue::operator = ( const Command_Queue &  org )
{
	this -> overwrite_this( org );

	return( *this );
}

// protected
void   Command_Queue::overwrite_this( const Command_Queue &  org )
{
	if ( &org == this )
	{
		return;
	}

	this -> clear();


	Internal_Queue_Class	org_q = org.q;

	while( ! org_q.empty() )
	{
		if ( org_q.front().is_command )
		{
			// command
			this -> q.push( org_q.front() );
		}
		else
		{
			// action
			Command_or_Action		act;
			act.is_command = false;
			act.action = org_q.front().action -> copy();

			this -> q.push( act );
		}

		(void)(org_q.pop());
	}
}


Command_Queue&  Command_Queue::push( const Soccer_Composite_Command &  com )
{
	Command_or_Action	com_or_act;

	com_or_act.is_command = true;
	com_or_act.command    = com;

	q.push( com_or_act );

	return( *this );
}

Command_Queue&  Command_Queue::push( const Soccer_Composite_Command &  com_1 ,
				     const Soccer_Composite_Command &  com_2 )
{
	try
	{
		this -> push( com_1 + com_2 );

	} catch(...){}

	return( *this );
}


Command_Queue&  Command_Queue::push
				(const ref_count_ptr<Soccer_Action> &  act )
{
	if ( act )
	{
		Command_or_Action	com_or_act;

		com_or_act.is_command = false;
		com_or_act.action     = act;

		q.push( com_or_act );
	}

	return( *this );
}


Command_Queue&  Command_Queue::push( Soccer_Action *  act )
{
	ref_count_ptr<Soccer_Action>	act_ref( act );

	this -> push( act_ref );

	return( *this );
}


Soccer_Composite_Command  Command_Queue::pop()
	throw( Empty_Queue_Pop_Exception )
{
	this -> remove_finished_action();

	if ( q.empty() )
	{
		throw Empty_Queue_Pop_Exception();
	}

	Soccer_Composite_Command	ret;

	if ( q.front().is_command )
	{
		// command
		ret = q.front().command;
		(void)(q.pop());
	}
	else
	{
		// action
		ret = (q.front().action) -> get_command();

		if ( (q.front().action) -> finished() )
		{
			(void)(q.pop());

			if ( ret.base_type()
			     == Soccer_Command::No_Command_Command )
			{
				if ( ! this -> empty() )
				{
					return( this -> pop() );
				}
			}
		}
	}

	return( ret );
}


bool   Command_Queue::empty() const
{
	this -> remove_finished_action();

	return( q.empty() );
}


// protected
void   Command_Queue::remove_finished_action() const
{
	for(;;)
	{
		if ( q.empty()
		  || (q.front().is_command
		     && q.front().command.base_type()
			  != Soccer_Command::Illegal_Command
		     && q.front().command.base_type()
			  != Soccer_Command::No_Command_Command)
		  || (! q.front().is_command
		     && ! (q.front().action -> finished())) )
		{
			return;
		}
		else
		{
			(void)(q.pop());
		}
	}
}


void   Command_Queue::clear()
{
	while( ! q.empty() )
	{
		(void)(q.pop());
	}
}


Command_Queue::operator bool() const
{
	return( ! (this -> empty()) );
}


Soccer_Composite_Command  Command_Queue::action()
{
	return( this -> pop() );
}

ref_count_ptr<Soccer_Action>  Command_Queue::copy() const
{
	return( new Command_Queue( *this ) );
}

bool   Command_Queue::finished() const
{
	return( this -> empty() );
}

bool   Command_Queue::succeeded() const
{
	return( true );
}


//
// Time_Limit_Queue
//
Time_Limit_Queue::Time_Limit_Queue( const Field_Recog_Interface &  f ,
				    long  lim )
	: f( f ) , time_limit( f.current_time().main_step() + lim )
{
}

Time_Limit_Queue::Time_Limit_Queue( const Field_Recog_Interface &  f ,
				    long  lim ,
				    const Soccer_Composite_Command &  com )
	: f( f ) , time_limit( f.current_time().main_step() + lim )
{
	this -> push( com );
}

Time_Limit_Queue::Time_Limit_Queue( const Field_Recog_Interface &  f ,
				    long  lim ,
				    const ref_count_ptr<Soccer_Action> &  act )
	: f( f ) , time_limit( f.current_time().main_step() + lim )
{
	this -> push( act );
}

Time_Limit_Queue::Time_Limit_Queue( const Field_Recog_Interface &  f ,
				    long  lim ,
				    Soccer_Action *  act )
	: f( f ) , time_limit( f.current_time().main_step() + lim )
{
	this -> push( act );
}

Time_Limit_Queue::~Time_Limit_Queue()
{
}

Soccer_Composite_Command  Time_Limit_Queue::pop()
	throw( Empty_Queue_Pop_Exception )
{
	if ( f.current_time().main_step() < time_limit )
	{
		return( Command_Queue::pop() );
	}
	else
	{
		throw Empty_Queue_Pop_Exception();
	}
}

bool   Time_Limit_Queue::empty() const
{
	if ( f.current_time().main_step() < time_limit )
	{
		return( Command_Queue::empty() );
	}
	else
	{
		return( true );
	}
}

void   Time_Limit_Queue::set_limit( long  lim )
{
	this -> time_limit = f.current_time().main_step() + lim;
}

long   Time_Limit_Queue::rest() const
{
	long	r = time_limit - f.current_time().main_step();

	if ( r < 0 )
	{
		return( 0 );
	}
	else
	{
		return( r );
	}
}


//
// Condition_Queue
//
Condition_Queue::Condition_Queue
		    ( const Field_Recog_Interface &  f ,
		      const ref_count_ptr<const Field_Predicate> &  predicate )
	: f( f ) , predicate( predicate )
{
}

Condition_Queue::Condition_Queue
		    ( const Field_Recog_Interface &  f ,
		      const ref_count_ptr<const Field_Predicate> &  predicate ,
		      const Soccer_Composite_Command &  com )
	: f( f ) , predicate( predicate )
{
	this -> push( com );
}

Condition_Queue::Condition_Queue
		    ( const Field_Recog_Interface &  f ,
		      const ref_count_ptr<const Field_Predicate> &  predicate ,
		      const ref_count_ptr<Soccer_Action> &  act )
	: f( f ) , predicate( predicate )
{
	this -> push( act );
}

Condition_Queue::Condition_Queue
		    ( const Field_Recog_Interface &  f ,
		      const ref_count_ptr<const Field_Predicate> &  predicate ,
		      Soccer_Action *  act )
	: f( f ) , predicate( predicate )
{
	this -> push( act );
}

Condition_Queue::~Condition_Queue()
{
}

bool   Condition_Queue::operator() ( const Field_Recog_Interface &  f ) const
{
	if ( this -> predicate )
	{
		if ( this -> predicate -> operator()(f) )
		{
			return( true );
		}
		else
		{
			this -> q.clear();

			return( false );
		}
	}
	else
	{
		return( false );
	}
}

bool   Condition_Queue::empty() const
{
	return( ! (this -> operator()(f)) || Command_Queue::empty() );
}




//
// Exclusive_Queue_Controler
//
#include  <iostream>

Exclusive_Queue_Controler::Exclusive_Queue_Controler()
{
}

Exclusive_Queue_Controler::~Exclusive_Queue_Controler()
{
}

Command_Queue *  Exclusive_Queue_Controler::register_queue
						( Command_Queue *  q )
{
	if ( queue_set.find( q ) != queue_set.end() )
	{
		cerr << "Exclusive queue registered" << endl;
	}

	queue_set.insert( q );

	return( q );
}


Command_Queue &  Exclusive_Queue_Controler::push
				( Command_Queue &  q ,
				  const Soccer_Composite_Command & com )
{
	this -> clear_other_queues( q );

	q.push( com );

	return( q );
}


Command_Queue &  Exclusive_Queue_Controler::push(
				 Command_Queue &  q ,
				 const Soccer_Composite_Command & com_1 ,
				 const Soccer_Composite_Command & com_2 )
{
	this -> clear_other_queues( q );

	q.push( com_1 , com_2 );

	return( q );
}


Command_Queue &  Exclusive_Queue_Controler::push
			( Command_Queue &  q ,
			  const ref_count_ptr<Soccer_Action> &  act )
{
	this -> clear_other_queues( q );

	q.push( act );

	return( q );
}

Command_Queue &  Exclusive_Queue_Controler::push( Command_Queue &  q ,
						  Soccer_Action *  act )
{
	this -> clear_other_queues( q );

	q.push( act );

	return( q );
}


const  Command_Queue &  Exclusive_Queue_Controler::clear_other_queues
						( const Command_Queue &  q )
{
	bool	found = false;

	for( set<Command_Queue *>::iterator  i = queue_set.begin()  ;
	     i != queue_set.end()  ;  i ++ )
	{
		if ( const_cast<const Command_Queue *>( *i ) == &q )
		{
			found = true;
		}
		else
		{
			(*i) -> clear();
		}
	}

	if ( ! found )
	{
		cerr << "internal error:" << endl;
		cerr << "  Exclusive_Queue_Controler: no such queue" << endl;
	}

	return( q );
}


void   Exclusive_Queue_Controler::clear_all_queues()
{
	for( set<Command_Queue *>::iterator  i = queue_set.begin()  ;
	     i != queue_set.end()  ;  i ++ )
	{
		(*i) -> clear();
	}
}
