#include  "sserver_param.h"
#include  "quantize.h"
#include  <cmath>
#include  "math_extention.h"
#include  <cassert>
#include  <algorithm>

using namespace std;

const	ip_port_number_t	SServer_Param::DEFAULT_PORT = 6000;

const	double	SServer_Param::DEFAULT_BALL_SIZE = 0.085;
const	double	SServer_Param::DEFAULT_PLAYER_SIZE = 0.3;

const	double	SServer_Param::DEFAULT_BALL_DECAY = 0.94;
const	double	SServer_Param::DEFAULT_PLAYER_DECAY = 0.4;

const	double	SServer_Param::DEFAULT_PLAYER_SPEED_MAX = 1.0;
const	double	SServer_Param::DEFAULT_BALL_SPEED_MAX = 2.7;

const	double	SServer_Param::DEFAULT_BALL_RAND = 0.05;
const	double	SServer_Param::DEFAULT_PLAYER_RAND = 0.1;

const	double	SServer_Param::DEFAULT_DASH_POWER_RATE = 0.006;
/*XXX*/	double	SServer_Param::DEFAULT_KICK_POWER_RATE = 0.016;

const	double	SServer_Param::DEFAULT_CATCHABLE_AREA_L = 2.0;
const	double	SServer_Param::DEFAULT_CATCHABLE_AREA_W = 1.0;
const	int	SServer_Param::DEFAULT_CATCH_BAN_CYCLE = 5;

const	double	SServer_Param::DEFAULT_KICKABLE_MARGIN = 0.7;

/*XXX*/	double	SServer_Param::DEFAULT_STAMINA_MAX = 3500.0;
/*XXX*/	double	SServer_Param::DEFAULT_STAMINA_INC_MAX = 35.0;

const	double	SServer_Param::DEFAULT_EFFORT_DEC_THR = 0.3;
const	double	SServer_Param::DEFAULT_EFFORT_DEC     = 0.005;
const	double	SServer_Param::DEFAULT_EFFORT_INC_THR = 0.6;
const	double	SServer_Param::DEFAULT_EFFORT_INC     = 0.01;
const	double	SServer_Param::DEFAULT_EFFORT_MIN     = 0.6;

const	double	SServer_Param::DEFAULT_RECOVER_DEC_THR = 0.3;
const	double	SServer_Param::DEFAULT_RECOVER_DEC     = 0.002;
const	double	SServer_Param::DEFAULT_RECOVER_MIN     = 0.5;

const	size_t	SServer_Param::DEFAULT_SAY_MESSAGE_LENGTH = 2048;

const	double	SServer_Param::DEFAULT_MAX_POWER = (+ 100.0);
const	double	SServer_Param::DEFAULT_MIN_POWER = (- 100.0);

const	double	SServer_Param::DEFAULT_MAX_MOMENT = (+ 180.0);
const	double	SServer_Param::DEFAULT_MIN_MOMENT = (- 180.0);

const	double	SServer_Param::DEFAULT_INERTIA_MOMENT = 5.0;

const	int	SServer_Param::DEFAULT_RECV_STEP = 10; // msec

const	double	SServer_Param::DEFAULT_SHORT_SENSOR_LENGTH = 3.0;

const	Angle	SServer_Param::DEFAULT_MAX_NECK_MOMENT = Degree(  180.0 );
const	Angle	SServer_Param::DEFAULT_MIN_NECK_MOMENT = Degree( -180.0 );
const	Angle	SServer_Param::DEFAULT_MAX_NECK_ANGLE  = Degree(   90.0 );
const	Angle	SServer_Param::DEFAULT_MIN_NECK_ANGLE  = Degree(  -90.0 );

const	int	SServer_Param::DEFAULT_HALF_TIME = 300; // sec
const	int	SServer_Param::DEFAULT_SIMULATOR_STEP = 100; // msec

const	double	SServer_Param::DEFAULT_QUANTIZE_STEP = 0.1;
const	double	SServer_Param::DEFAULT_QUANTIZE_STEP_L = 0.01;
const	double	SServer_Param::DEFAULT_GOAL_WIDTH = 14.02;

const	double	SServer_Param::DEFAULT_OFFSIDE_ACTIVE_AREA_SIZE = 5.0;
const	double	SServer_Param::DEFAULT_OFFSIDE_KICK_MARGIN = 9.15;


const	size_t	SServer_Param::SSERVER_MAX_MESSAGE_LENGTH = 4096;


const	size_t	SServer_Param::N_STEP_TO_REACH_CACHE_SIZE = 300;

SServer_Param::SServer_Param() throw()
	: n_step_to_reach_cache( N_STEP_TO_REACH_CACHE_SIZE )
{
	this -> make_cache();
}

SServer_Param::~SServer_Param() throw()
{
}

void   SServer_Param::overwrite_server_param
			( const Server_Raw_Server_Param_Info &  sp )
{
	// XXX: others are ignored
	SServer_Param::DEFAULT_KICK_POWER_RATE = sp.kick_power_rate;
	SServer_Param::DEFAULT_STAMINA_MAX     = sp.stamina_max;
	SServer_Param::DEFAULT_STAMINA_INC_MAX = sp.stamina_inc_max;
}

void   SServer_Param::overwrite_player_param
			( const Server_Raw_Player_Param_Info & )
{
	// XXX
}

// XXX
void   SServer_Param::overwrite_server_param( double  kick_power_rate ,
					      double  stamina_max ,
					      double  stamina_inc_max )
{
	SServer_Param::DEFAULT_KICK_POWER_RATE = kick_power_rate;
	SServer_Param::DEFAULT_STAMINA_MAX     = stamina_max;
	SServer_Param::DEFAULT_STAMINA_INC_MAX = stamina_inc_max;
}


void   SServer_Param::make_cache() const
{
	assert( n_step_to_reach_cache.size() >= 1 );

	double	speed = 0.0;
	double	sum_dist = 0.0;

	for ( size_t  t = 0  ;  t < n_step_to_reach_cache.size()  ;  t ++ )
	{
		n_step_to_reach_cache[t] = sum_dist;

		speed = speed * this -> PLAYER_DECAY()
			+ this -> dash_power_to_accel( this -> MAX_POWER() );

		sum_dist += speed;
	}
}

ip_port_number_t  SServer_Param::PORT() const throw()
{
	return( DEFAULT_PORT );
}

double  SServer_Param::BALL_SIZE() const throw()
{
	return( DEFAULT_BALL_SIZE );
}

double  SServer_Param::PLAYER_SIZE() const throw()
{
	return( DEFAULT_PLAYER_SIZE );
}

double  SServer_Param::BALL_DECAY() const throw()
{
	return( DEFAULT_BALL_DECAY );
}

double  SServer_Param::PLAYER_DECAY() const throw()
{
	return( DEFAULT_PLAYER_DECAY );
}

double  SServer_Param::BALL_SPEED_MAX() const throw()
{
	return( DEFAULT_BALL_SPEED_MAX );
}

double  SServer_Param::PLAYER_SPEED_MAX() const throw()
{
	return( DEFAULT_PLAYER_SPEED_MAX );
}

double  SServer_Param::BALL_RAND() const throw()
{
	return( DEFAULT_BALL_RAND );
}

double  SServer_Param::PLAYER_RAND() const throw()
{
	return( DEFAULT_PLAYER_RAND );
}

double  SServer_Param::DASH_POWER_RATE() const throw()
{
	return( DEFAULT_DASH_POWER_RATE );
}

double  SServer_Param::KICK_POWER_RATE() const throw()
{
	return( DEFAULT_KICK_POWER_RATE );
}

double  SServer_Param::CATCHABLE_AREA_L() const throw()
{
	return( DEFAULT_CATCHABLE_AREA_L );
}

double  SServer_Param::CATCHABLE_AREA_W() const throw()
{
	return( DEFAULT_CATCHABLE_AREA_W );
}

int     SServer_Param::CATCH_BAN_CYCLE() const throw()
{
	return( DEFAULT_CATCH_BAN_CYCLE );
}

double  SServer_Param::KICKABLE_MARGIN() const throw()
{
	return( DEFAULT_KICKABLE_MARGIN );
}

double  SServer_Param::STAMINA_MAX() const throw()
{
	return( DEFAULT_STAMINA_MAX );
}

double  SServer_Param::STAMINA_INC_MAX() const throw()
{
	return( DEFAULT_STAMINA_INC_MAX );
}

double  SServer_Param::EFFORT_DEC_THR() const throw()
{
	return( DEFAULT_EFFORT_DEC_THR );
}

double  SServer_Param::EFFORT_DEC() const throw()
{
	return( DEFAULT_EFFORT_DEC );
}

double  SServer_Param::EFFORT_INC_THR() const throw()
{
	return( DEFAULT_EFFORT_INC_THR );
}

double  SServer_Param::EFFORT_INC() const throw()
{
	return( DEFAULT_EFFORT_INC );
}

double  SServer_Param::EFFORT_MIN() const throw()
{
	return( DEFAULT_EFFORT_MIN );
}


double  SServer_Param::RECOVER_DEC_THR() const throw()
{
	return( DEFAULT_RECOVER_DEC_THR );
}

double  SServer_Param::RECOVER_DEC() const throw()
{
	return( DEFAULT_RECOVER_DEC );
}

double  SServer_Param::RECOVER_MIN() const throw()
{
	return( DEFAULT_RECOVER_MIN );
}

size_t  SServer_Param::SAY_MESSAGE_LENGTH() const throw()
{
	return( DEFAULT_SAY_MESSAGE_LENGTH );
}

double  SServer_Param::MAX_POWER() const throw()
{
	return( DEFAULT_MAX_POWER );
}

double  SServer_Param::MIN_POWER() const throw()
{
	return( DEFAULT_MIN_POWER );
}

double  SServer_Param::MAX_MOMENT() const throw()
{
	return( DEFAULT_MAX_MOMENT );
}

double  SServer_Param::MIN_MOMENT() const throw()
{
	return( DEFAULT_MIN_MOMENT );
}

double  SServer_Param::INERTIA_MOMENT() const throw()
{
	return( DEFAULT_INERTIA_MOMENT );
}

int     SServer_Param::RECV_STEP() const throw()
{
	return( DEFAULT_RECV_STEP );
}


double  SServer_Param::SHORT_SENSOR_LENGTH() const throw()
{
	return( DEFAULT_SHORT_SENSOR_LENGTH );
}

Angle  SServer_Param::MAX_NECK_MOMENT() const throw()
{
	return( DEFAULT_MAX_NECK_MOMENT );
}

Angle  SServer_Param::MIN_NECK_MOMENT() const throw()
{
	return( DEFAULT_MIN_NECK_MOMENT );
}

Angle  SServer_Param::MAX_NECK_ANGLE() const throw()
{
	return( DEFAULT_MAX_NECK_ANGLE );
}

Angle  SServer_Param::MIN_NECK_ANGLE() const throw()
{
	return( DEFAULT_MIN_NECK_ANGLE );
}

int     SServer_Param::HALF_TIME() const throw()
{
	return( DEFAULT_HALF_TIME );
}

int     SServer_Param::SIMULATOR_STEP() const throw()
{
	return( DEFAULT_SIMULATOR_STEP );
}

double  SServer_Param::QUANTIZE_STEP() const throw()
{
	return( DEFAULT_QUANTIZE_STEP );
}

double  SServer_Param::QUANTIZE_STEP_L() const throw()
{
	return( DEFAULT_QUANTIZE_STEP_L );
}

double  SServer_Param::GOAL_WIDTH() const throw()
{
	return( DEFAULT_GOAL_WIDTH );
}

double	SServer_Param::OFFSIDE_ACTIVE_AREA_SIZE() const throw()
{
	return( DEFAULT_OFFSIDE_ACTIVE_AREA_SIZE );
}

double	SServer_Param::OFFSIDE_KICK_MARGIN() const throw()
{
	return( DEFAULT_OFFSIDE_KICK_MARGIN );
}


double  SServer_Param::half_time_step() const throw()
{
	if ( this -> SIMULATOR_STEP() == 0 )
	{
		return( 0 );
	}

	return( this -> HALF_TIME() * 1000
		/ this -> SIMULATOR_STEP() );
}


double  SServer_Param::field_half_length() const throw()
{
	return( 52.5 );
}

double  SServer_Param::field_length() const throw()
{
	return( this -> field_half_length() * 2.0 );
}

double  SServer_Param::field_half_width() const throw()
{
	return( 34.0 );
}

double  SServer_Param::field_width() const throw()
{
	return( this -> field_half_width() * 2.0 );
}

double  SServer_Param::field_diagonal_length() const throw()
{
	return( sqrt(   pow( this -> field_length() , 2.0 )
		      + pow( this -> field_width()  , 2.0 ) ) );
}

double  SServer_Param::center_circle_r() const throw()
{
	return( 9.15 );
}

double  SServer_Param::penalty_area_length() const throw()
{
	return( 16.5 );
}

double  SServer_Param::penalty_area_half_width() const throw()
{
	return( 20.16 );
}

double	SServer_Param::our_penalty_area_line_x() const throw()
{
	return( - (  this -> field_half_length()
		   - this -> penalty_area_length() ) );
}

double	SServer_Param::opponent_penalty_area_line_x() const throw()
{
	return( + (  this -> field_half_length()
		   - this -> penalty_area_length() ) );
}

double	SServer_Param::goal_area_length() const throw()
{
	return( 5.5 );
}

double  SServer_Param::goal_area_half_width() const throw()
{
	return( 9.16 );
}

double	SServer_Param::our_goal_area_line_x() const throw()
{
	return( - (  this -> field_half_length()
		   - this -> goal_area_length() ) );
}

double	SServer_Param::opponent_goal_area_line_x() const throw()
{
	return( + (  this -> field_half_length()
		   - this -> goal_area_length() ) );
}

double  SServer_Param::goal_width() const throw()
{
	return( this -> GOAL_WIDTH() );
}

double  SServer_Param::goal_half_width() const throw()
{
	return( this -> GOAL_WIDTH() / 2.0 );
}

double  SServer_Param::kick_off_clear_distance() const throw()
{
	return( this -> center_circle_r() );
}

double  SServer_Param::free_kick_clear_distance() const throw()
{
	return( 9.15 );
}

double  SServer_Param::player_speed_max_max() const throw()
{
	// XXX
	return( 1.2 );
}

Double_Range  SServer_Param::server_stamina_info_to_real_range(
						double  stamina ) const throw()
{
	return( Double_Range( std::max( stamina - 0.005 , 0.0 ) ,
			      std::min( stamina + 0.005 , STAMINA_MAX() ) ) );
}

Double_Range  SServer_Param::server_effort_info_to_real_range(
						double  effort ) const throw()
{
	return( Double_Range( effort , effort ) );
}

Angle_Range  SServer_Param::server_neck_angle_info_to_real_range(
				     const Angle &  neck_angle ) const throw()
{
	const Angle_Range	valid_range( MIN_NECK_ANGLE() ,
					     MAX_NECK_ANGLE() );

	Angle	min_angle;
	Angle	max_angle;

#ifdef OLD_ANGLE_ROUNDING_POLICY
	if ( neck_angle > Degree(0.5) )
	{
		min_angle = neck_angle;
		max_angle = neck_angle + Degree( 1.0 );
	}
	else if ( neck_angle < Degree( - 0.5 ) )
	{
		min_angle = neck_angle - Degree( 1.0 );
		max_angle = neck_angle;
	}
	else
	{
		min_angle = neck_angle - Degree( 1.0 );
		max_angle = neck_angle + Degree( 1.0 );
	}
#else
	min_angle = neck_angle - Degree( 0.5 );
	max_angle = neck_angle + Degree( 0.5 );
#endif

	Angle_Range	range( min_angle , max_angle );

	return( valid_range.clip( range ) );
}

Double_Range  SServer_Param::server_speed_info_to_real_range( double  speed )
	const throw()
{
	double	speed_min;
	double	speed_max;

	unquantize( speed , 0.01 , &speed_min , &speed_max );

	return( Double_Range( speed_min , speed_max ) );
}


double  SServer_Param::dash_power_to_stamina_dec( double  dash_power )
	const throw()
{
	double	d_pow = dash_power;

	if ( d_pow > (this -> MAX_POWER()) )
	{
		d_pow = (this -> MAX_POWER());
	}
	else if( d_pow < (this -> MIN_POWER()) )
	{
		d_pow = (this -> MIN_POWER());
	}

	return( ((d_pow >= 0) ? d_pow : - d_pow * 2.0 ) );
}

double  SServer_Param::stamina_safe_threshold( bool  use_back_dash )
	const throw()
{
	return( this -> STAMINA_MAX()
		* std::max( this -> RECOVER_DEC_THR() ,
			    this -> EFFORT_DEC_THR() )
		+ (use_back_dash ?
		     this -> MAX_POWER() * 2.0
		   : this -> MAX_POWER() ) );
}

double  SServer_Param::stamina_safe_dash_power( double  stamina ,
						bool  use_back_dash )
	const throw()
{
	if ( stamina >= this -> stamina_safe_threshold( use_back_dash ) )
	{
		return( this -> MAX_POWER() );
	}

	double	threshold = this -> STAMINA_MAX()
			    * std::max( this -> RECOVER_DEC_THR() ,
					this -> EFFORT_DEC_THR() );
	if ( stamina < threshold )
	{
		return( 0.0 );
	}
	else
	{
		return( (stamina - threshold) / (use_back_dash ? 2.0 : 1.0) );
	}
}

double  SServer_Param::stamina_safe_dash_power_filter( double  stamina ,
						       double  dash_power )
	const throw()
{
	if ( dash_power >= 0.0 )
	{
		return( std::min( + this -> stamina_safe_dash_power( stamina ,
								     false ) ,
				  dash_power ) );
	}
	else
	{
		return( std::max( - this -> stamina_safe_dash_power( stamina ,
								     true ) ,
				  dash_power ) );
	}
}


D2_Vector  SServer_Param::initial_position( S_Side_LR  side ,
					    int  player_number )
	const throw()
{
	double	x = - player_number * 3.0;
	double	y = ((side == S_Side_LR::Left_Side) ? 1.0 : -1.0)
		    * (- this -> field_half_width() - 3.0 );

	return( D2_Vector( x , y ) );
}


double  SServer_Param::kickable_distance() const throw()
{
	return( BALL_SIZE() + KICKABLE_MARGIN() + PLAYER_SIZE() );
}

double  SServer_Param::catchable_length() const throw()
{
	return( this -> CATCHABLE_AREA_L() );
}

double  SServer_Param::catchable_width() const throw()
{
	return( this -> CATCHABLE_AREA_W() );
}

double  SServer_Param::catchable_half_width() const throw()
{
	return( this -> CATCHABLE_AREA_W() / 2.0 );
}

double  SServer_Param::catchable_diagona_distance() const throw()
{
	double	len = this -> catchable_length();
	double	wid = this -> catchable_half_width();

	double	length_2 = pow( len , 2.0 ) + pow( wid , 2.0 );
	if ( length_2 < 0.0 )
	{
		length_2 = 0.0;
	}

	double	length = sqrt( length_2 );

	return( length );
}




D2_Vector  SServer_Param::ball_basic_point_n_step_later
				( const D2_Vector &  current_point ,
				  const D2_Vector &  vec ,
				  int  n ) const throw()
{
	return( current_point
		+ vec * ( (1.0 - pow( this -> BALL_DECAY() , n ))
		          / (1.0 - this -> BALL_DECAY()) ) );
}


double  SServer_Param::n_step_for_ball_move( double  distance ,
					     double  ball_speed ) const throw()
{
	static	double	old_ball_decay = DEFAULT_BALL_DECAY;
	static	double	log_ball_decay_cache = std::log( DEFAULT_BALL_DECAY );

	if ( distance < 0.0 || ball_speed < 1.0e-10 )
	{
		return( -1.0 );
	}

	double	numerator_tmp
		= 1.0 - (distance * (1.0 - this -> BALL_DECAY()))
			/ ball_speed;

	if ( numerator_tmp <= 1.0e-5 )
	{
		return( -1.0 );
	}

	if ( old_ball_decay != this -> BALL_DECAY() )
	{
		old_ball_decay       = this -> BALL_DECAY();
		log_ball_decay_cache = std::log( this -> BALL_DECAY() );
	}

	return( std::log( numerator_tmp ) / log_ball_decay_cache );
}

D2_Vector  SServer_Param::ball_reach_point( const D2_Vector &  current_point ,
					    const D2_Vector &  vec )
	const throw()
{
	return( current_point
		+ D2_Vector( D2_Vector::Pole ,
			     this -> ball_reach_distance( vec.r() ) ,
			     vec.theta() ) );
}

double  SServer_Param::ball_reach_distance( double  ball_speed ) const throw()
{
	return( ball_speed / (1.0 - this -> BALL_DECAY()) );
}

double  SServer_Param::ball_reach_distance_to_ball_speed( double  dist )
	const throw()
{
	return( dist * (1.0 - this -> BALL_DECAY()) );
}

D2_Vector  SServer_Param::player_basic_point_n_step_later
					( const D2_Vector &  current_point ,
					  const D2_Vector &  vec ,
					  int  n ) const throw()
{
	return( current_point
		+ vec * ( (1.0 - pow( this -> PLAYER_DECAY() , n ))
		          / (1.0 - this -> PLAYER_DECAY()) ) );
}


double  SServer_Param::required_dash_power_to_reach_next_step( double  dist )
	const throw()
{
	double	needed_power = dist / this -> DASH_POWER_RATE();

	if ( needed_power >= this -> MAX_POWER() )
	{
		return( this -> MAX_POWER() );
	}
	else if ( needed_power <= this -> MIN_POWER() )
	{
		return( this -> MIN_POWER() );
	}
	else
	{
		return( needed_power );
	}
}

double  SServer_Param::player_stop_power( double  current_speed ) const throw()
{
	double	power = - current_speed / (this -> DASH_POWER_RATE());

	if ( power < (this -> MIN_POWER()) )
	{
		power = this -> MIN_POWER();
	}

	if ( power > (this -> MAX_POWER()) )
	{
		power = this -> MAX_POWER();
	}

	return( power );
}

double  SServer_Param::dash_power_to_accel( double  dash_power ) const throw()
{
	return( dash_power * (this -> DASH_POWER_RATE()) );
}

Angle  SServer_Param::actual_moment( const Angle &  dir ,
				     double  self_speed ,
				     bool *  succeeded ) const throw()
{
	try
	{
		Angle	moment
			= dir * (1.0 + this -> INERTIA_MOMENT() * self_speed);

		if ( succeeded )
		{
			*succeeded = true;
		}

		if ( moment > Degree( this -> MAX_MOMENT() ) )
		{
			if ( succeeded )
			{
				*succeeded = false;
			}

			moment = Degree( this -> MAX_MOMENT() );
		}
		else if ( moment < Degree( this -> MIN_MOMENT() ) )
		{
			if ( succeeded )
			{
				*succeeded = false;
			}

			moment = Degree( this -> MIN_MOMENT() );
		}

		return( moment );
	}
	catch(...)
	{
		if ( succeeded )
		{
			*succeeded = false;
		}

		return( Radian(0.0) );
	}
}

Angle  SServer_Param::turn_moment_to_angle( const Angle &  moment ,
					    double  self_speed ) const throw()
{
	Angle	m = moment;

	if ( m > Degree( this -> MAX_MOMENT() ) )
	{
		m = Degree( this -> MAX_MOMENT() );
	}
	else if ( m < Degree( this -> MIN_MOMENT() ) )
	{
		m = Degree( this -> MIN_MOMENT() );
	}

	return( m / ( 1.0 + this -> INERTIA_MOMENT() * self_speed ) );
}


double  SServer_Param::kick_accel_to_power( double  accel ,
					    double  ball_dist ,
					    const Angle &  ball_dir ,
					    bool *  succeeded )
	const throw()
{
	if ( ball_dist < (this -> PLAYER_SIZE() + this -> BALL_SIZE()) )
	{
		ball_dist = this -> PLAYER_SIZE() + this -> BALL_SIZE();
	}

	double	power = accel
			/ this -> KICK_POWER_RATE()
			/ (1.0 - 0.25 * ( ball_dir.normalize().abs().radian()
					  / Math_Extention::PI )
			       - 0.25 * ( (ball_dist - this -> PLAYER_SIZE()
						     - this -> BALL_SIZE())
					  / this -> KICKABLE_MARGIN() ));

	if ( succeeded )
	{
		*succeeded = true;
	}

	if ( power > this -> MAX_POWER() )
	{
		power = this -> MAX_POWER();

		if( succeeded )
		{
			*succeeded = false;
		}
	}
	else if ( power < this -> MIN_POWER() )
	{
		power = this -> MIN_POWER();

		if( succeeded )
		{
			*succeeded = false;
		}
	}

	return( power );
}

double  SServer_Param::kick_power_to_accel( double  power ,
					    double  ball_dist ,
					    const Angle &  ball_dir )
								const throw()
{
	if ( ball_dist < (this -> PLAYER_SIZE() + this -> BALL_SIZE()) )
	{
		ball_dist = this -> PLAYER_SIZE() + this -> BALL_SIZE();
	}

	double	accel = power
			* this -> KICK_POWER_RATE()
			* (1.0 - 0.25 * ( ball_dir.normalize().abs().radian()
					  / Math_Extention::PI )
			       - 0.25 * ( (ball_dist - this -> PLAYER_SIZE()
						     - this -> BALL_SIZE())
					  / this -> KICKABLE_MARGIN() ));

	return( accel );
}

double  SServer_Param::kick_power_to_ball_reach_distance
						( double  power ,
						  double  ball_dist ,
						  const Angle &  ball_dir )
	const throw()
{
	return( this -> ball_reach_distance
			( this -> kick_power_to_accel( power ,
						       ball_dist ,
						       ball_dir ) ) );
}

long   SServer_Param::n_step_to_turn_body( const Angle &  angle ,
					   double  speed )
	const throw()
{
	Angle	target_angle = angle.abs();

	Angle	turnable_angle( Angle::Radian , 0.0 );
	double	sp = speed;

	for ( long  t = 0  ;  ;  t ++ )
	{
		if ( turnable_angle >= target_angle )
		{
			return( t );
		}

		turnable_angle += this -> turn_moment_to_angle(
					Degree( this -> MAX_MOMENT() ) , sp );

		sp *= PLAYER_DECAY();
	}

	/* NOTREACHED */
}

long   SServer_Param::n_step_to_reach( double  dist ) const throw()
{
	assert( dist > 0.0 );

	// cache
	if ( dist < *(n_step_to_reach_cache.rbegin() + 1) )
	{
		return( static_cast<long>
			( std::lower_bound( n_step_to_reach_cache.begin() ,
					    n_step_to_reach_cache.end() ,
					    dist )
			  - n_step_to_reach_cache.begin() ) );
	}


	double	speed = 0.0;
	double	rest_dist = dist;

	for ( long  t = 0  ;  ;  t ++ )
	{
		if ( rest_dist <= 0.0 )
		{
			return( t );
		}

		speed = speed * this -> PLAYER_DECAY()
			+ this -> dash_power_to_accel( this -> MAX_POWER() );

		if ( speed >= (this -> PLAYER_SPEED_MAX()) - 1.0e-5 )
		{
			return( t + static_cast<long>
				    (rest_dist / (this -> PLAYER_SPEED_MAX()))
				+ 1 );
		}

		rest_dist -= speed;
	}

	/* NOTREACHED */
}
