#include  "field_recog.h"
#include  "sight_info_analyzer.h"
#include  "debugstream.h"
using Debug_Stream::dbg;

#include  <cstddef>
#include  <algorithm>
#include  <set>
#include  <cassert>
using namespace std;

Field_Recog::Field_Recog( const SServer_Param *  param ,
			  const Game_Info &  g_info ,
			  bool  self_goalie_flag ,
			  bool  true_view )
	: param( param ) , self_goalie_flag( self_goalie_flag ) ,
	  true_view_flag( true_view ) ,
	  current_state( new Field_Recog_State( param ) ) ,
	  next_state( static_cast<Field_Recog_State *>(0) )
{
	current_state -> game_state.game_info = g_info;
	current_state -> game_state.play_mode_start_time = 0;

	if ( self_goalie_flag )
	{
		current_state -> our_team_goalie_number
					= g_info.self_player_number;
	}
	else
	{
		current_state -> our_team_goalie_number = -1;
	}
	current_state -> opponent_team_goalie_number = -1;


	// initialize body info
	current_state -> body.view_width.set( View_Width::Normal );
	current_state -> body.view_quality.set( View_Quality::High );
	current_state -> body.stamina.set( param -> STAMINA_MAX() ,
					   param -> STAMINA_MAX() ,
					   param -> STAMINA_MAX() );
	current_state -> body.effort.set( 1.0 , 1.0 , 1.0 );
	current_state -> body.speed.set( 0.0 , 0.0 , 0.0 );
	current_state -> body.neck_angle.set( Radian(0.0) ,
					      Radian(0.0) ,
					      Radian(0.0) );

	// XXX
	current_state -> body.my_body_angle.set( Radian(0.0) ,
						 Radian(0.0) ,
						 Radian(0.0) );

	D2_Vector	init_pos = param -> initial_position
						( g_info.our_side ,
						  g_info.self_player_number );

	current_state -> body.my_coordinate.set( init_pos , init_pos );

	current_state -> body.my_velocity.set( D2_Vector( 0.0, 0.0 ) ,
					       D2_Vector( 0.0, 0.0 ) );

	current_state -> ball.coordinate.set( D2_Vector( 0.0, 0.0 ) ,
					      D2_Vector( 0.0, 0.0 ) );
	current_state -> ball.velocity.set( D2_Vector( 0.0, 0.0 ) ,
					    D2_Vector( 0.0, 0.0 ) );

	next_state = new Field_Recog_State( *current_state );
}

Field_Recog::~Field_Recog()
{
}


//
// protocol between Soccer_Scheduler and Field_Recog
//
void   Field_Recog::register_info( const Server_Raw_Info &  info ,
				   const Time_Stamp &  time_stamp )
{
	switch( info.type() )
	{
	case Server_Raw_Info::Type_Sight:
		next_state -> buffer.sight_info = info.get_sight_info();
		next_state -> buffer.sight_info_time = info.time();
		next_state -> buffer.sight_info_time_stamp = time_stamp;
		if ( next_state -> buffer.info_time < info.time() )
		{
			next_state -> buffer.info_time = info.time();
		}
		break;

	case Server_Raw_Info::Type_Body:
		next_state -> buffer.body_info = info.get_body_info();
		next_state -> buffer.body_info_time = info.time();
		next_state -> buffer.body_info_time_stamp = time_stamp;
		if ( next_state -> buffer.info_time < info.time() )
		{
			next_state -> buffer.info_time = info.time();
		}
		break;

	case Server_Raw_Info::Type_Whistle:
		this -> register_whistle_info( next_state ,
					       info.get_whistle_info() ,
					       info.time() );

		if ( next_state -> buffer.info_time < info.time() )
		{
			next_state -> buffer.info_time = info.time();
		}
		break;

	case Server_Raw_Info::Type_Audio:
		// XXX
		if ( next_state -> buffer.info_time < info.time() )
		{
			next_state -> buffer.info_time = info.time();
		}
		break;

	case Server_Raw_Info::Type_Server_Param:
		current_state -> game_state.player_type
		  -> register_server_param( info.get_server_param_info() );
		break;

	case Server_Raw_Info::Type_Player_Param:
		current_state -> game_state.player_type
		  -> register_player_param( info.get_player_param_info() );
		break;

	case Server_Raw_Info::Type_Player_Type:
		current_state -> game_state.player_type
		  -> register_player_type( info.get_player_type_info() );
		break;

	case Server_Raw_Info::Type_Debug:
		next_state -> buffer.debug_info = info.get_debug_info();
		next_state -> buffer.debug_info_time = info.time();
		next_state -> buffer.debug_info_time_stamp = time_stamp;
		if ( next_state -> buffer.info_time < info.time() )
		{
			next_state -> buffer.info_time = info.time();
		}
		break;

	case Server_Raw_Info::Type_Unknown:
	default:
		// ignore this infomation
		break;
	}
}


void   Field_Recog::turn_neck_sent( const Angle &  moment ,
				    const Time_Stamp &  time_stamp )
{
	Angle	m = moment;

	if ( moment < param -> MIN_NECK_MOMENT() )
	{
		m = param -> MIN_NECK_MOMENT();
	}
	else if ( moment > param -> MAX_NECK_MOMENT() )
	{
		m = param -> MAX_NECK_MOMENT();
	}

	next_state -> buffer.turn_neck_moment     = m;
	next_state -> buffer.turn_neck_time_stamp = time_stamp;
}


void   Field_Recog::base_command_sent( const Soccer_Command &  command ,
				       const Time_Stamp & )
{
	next_state -> buffer.command = command;
}

void   Field_Recog::update()
{
	next_state -> update_state( current_state ,
				    param ,
				    next_state -> buffer.command ,
				    true_view_flag );

	current_state = next_state;

	next_state = new Field_Recog_State( *current_state );
}


//
// protocol between User and Field_Recog
//
const SServer_Param &  Field_Recog::sserver_param() const
{
	return( *param );
}

ref_count_ptr<const Field_Recog_Abstract::Field_Recog_Snapshot>
						Field_Recog::snapshot() const
{
	return( current_state );
}

Soccer_Composite_Command  Field_Recog::previous_command() const
{
	Soccer_Composite_Command	com( current_state -> buffer.command );

	if ( current_state -> buffer.turn_neck_sent )
	{
		com.add_turn_neck( current_state -> buffer.turn_neck_moment );
	}

	return( com );
}

const Field_Recog_Abstract::Ball_Info &  Field_Recog::ball_info() const
{
	return( current_state -> ball );
}

const vector<Field_Recog_Abstract::Player_Info> &  Field_Recog::teammate_info()
	const
{
	return( current_state -> teammate );
}

const vector<Field_Recog_Abstract::Player_Info> &  Field_Recog::opponent_info()
	const
{
	return( current_state -> opponent );
}

const vector<Field_Recog_Abstract::Player_Info> &
	Field_Recog::unknown_player_info() const
{
	return( current_state -> unknown_player );
}

const Field_Recog_Abstract::Body_Info &  Field_Recog::body_info() const
{
	return( current_state -> body );
}

const Field_Recog_Abstract::Sight_Info &  Field_Recog::sight_info() const
{
	return( current_state -> sight );
}

const Field_Recog_Abstract::Game_State_Info &  Field_Recog::game_state_info()
	const
{
	return( current_state -> game_state );
}

const Field_Recog_Abstract::Debug_Info &  Field_Recog::debug_info() const
{
	return( current_state -> debug );
}


// protected
void   Field_Recog::register_whistle_info(
		const ref_count_ptr<Field_Recog_State> &  state ,
		const ref_count_ptr<const Server_Raw_Whistle_Info > &  w ,
		long  info_time )
{
	// game over
	if ( w -> type == Judgement_Type::Time_Up
	  || w -> type == Judgement_Type::Time_Up_Without_A_Team
	  || (w -> type == Judgement_Type::One_of_Play_Modes
	      && w -> play_mode == Play_Mode::Time_Over) )
	{
		state -> game_state.game_over_flag = true;
	}

	if ( w -> type == Judgement_Type::One_of_Play_Modes )
	{
		state -> game_state.game_info.play_mode = w -> play_mode;
		state -> game_state.play_mode_start_time = info_time;
	}
	else
	{
		state -> game_state.last_judgement = *w;
		state -> game_state.last_judgement_time = info_time;

		// set score
		if ( w -> type == Judgement_Type::Goal )
		{
			if ( w -> side == S_Side::Our_Side )
			{
				state -> game_state.game_info.our_score
								= w -> score;
			}
			else
			{
				state -> game_state.game_info.opponent_score
								= w -> score;
			}

			state -> game_state.game_info.play_mode
					= Play_Mode::Before_Kick_Off;
			state -> game_state.play_mode_start_time = info_time;
		}
		else if ( w -> type == Judgement_Type::Offside )
		{
			switch( w -> side )
			{
			case S_Side::Our_Side:
				state -> game_state.game_info.play_mode
					   = Play_Mode::Opponent_Free_Kick;
				state -> game_state.play_mode_start_time
					   = info_time;
				break;

			case S_Side::Opponent_Side:
				state -> game_state.game_info.play_mode
					   = Play_Mode::Our_Free_Kick;
				state -> game_state.play_mode_start_time
					   = info_time;
				break;

			case S_Side::Unknown:
				break;
			}
		}
	}
}




//
// Field_Recog_State
//
const double	Field_Recog::Field_Recog_State::EPS = 1.0e-6;

Field_Recog::Field_Recog_State::Field_Recog_State( const SServer_Param *  p )
	: Field_Recog_Snapshot( p )
{
}

Field_Recog::Field_Recog_State::Field_Recog_State(
					  const Field_Recog_State &  old )
	: Field_Recog_Snapshot( old ) , buffer( /* new buffer */ )
{
}

Field_Recog::Field_Recog_State::~Field_Recog_State()
{
}

void   Field_Recog::Field_Recog_State::update_state(
	      const ref_count_ptr<const Field_Recog_State> &  old_state ,
	      const SServer_Param *  param ,
	      const Soccer_Composite_Command &  prev_com ,
	      bool  true_view )
{
	//*******************************************************************//
	//  set current time
	//*******************************************************************//
	if ( buffer.info_time
	     == old_state -> game_state.current_time.main_step() )
	{
		game_state.current_time
			.set( game_state.current_time.main_step() ,
			      game_state.current_time.sub_step() + 1 );
	}
	else
	{
		game_state.current_time.set( buffer.info_time , 0 );
	}


	//*******************************************************************//
	//  check have current * info
	//*******************************************************************//
	bool	have_current_body_info  = false;
	bool	have_current_sight_info = false;
	bool	have_current_debug_info = false;

	// XXX
	if ( buffer.body_info_time == game_state.current_time.main_step()
	  && buffer.body_info )
	{
		have_current_body_info = true;
	}

	if ( buffer.sight_info_time == game_state.current_time.main_step()
	  && buffer.sight_info )
	{
		have_current_sight_info = true;
	}

	if ( buffer.debug_info_time == game_state.current_time.main_step()
	  && buffer.debug_info )
	{
		have_current_debug_info = true;
	}


	//*******************************************************************//
	//  last command accepted check
	//*******************************************************************//
	bool	last_command_accepted;

	// XXX
	(void)prev_com;
	last_command_accepted = true;


	bool	last_turn_neck_accepted = true;

	if ( have_current_body_info && old_state -> buffer.body_info
	  && buffer.body_info -> n_turn_neck
	     <= old_state -> buffer.body_info -> n_turn_neck )
	{
		last_turn_neck_accepted = false;
	}


	//*******************************************************************//
	//  set debug info
	//*******************************************************************//
	if ( have_current_debug_info )
	{
		debug.latest_debug = buffer.debug_info;
	}


	//*******************************************************************//
	//  set neck_angle
	//*******************************************************************//
	this -> update_neck_angle( old_state , param ,
				   have_current_body_info ,
				   last_turn_neck_accepted );


	//*******************************************************************//
	//  set sight info
	//*******************************************************************//
	bool	sight_info_with_current_neck_angle;
	// XXX: not used yet
	sight_info_with_current_neck_angle = true;

	ref_count_ptr<Sight_Info_Analyzer>	sight_analyzer(0);

	if ( have_current_sight_info )
	{
		sight_analyzer = new Sight_Info_Analyzer
					  ( buffer.sight_info ,
					    param ,
					    body.neck_angle.range() );
	}


	//*******************************************************************//
	//  set goalie flag
	//*******************************************************************//
	this -> our_team_goalie_number
		= old_state -> our_team_goalie_number;
	this -> opponent_team_goalie_number
		= old_state -> opponent_team_goalie_number;

	if ( sight_analyzer
	     && ( this -> our_team_goalie_number == -1
		  || this -> opponent_team_goalie_number == -1 ) )
	{
		int	our_team;
		int	opponent_team;

		sight_analyzer -> get_goalie_number( &our_team ,
						     &opponent_team );

		if ( this -> our_team_goalie_number == -1
		  && our_team != -1 )
		{
			this -> our_team_goalie_number = our_team;
		}

		if ( this -> opponent_team_goalie_number == -1
		  && opponent_team != -1 )
		{
			this -> opponent_team_goalie_number = opponent_team;
		}
	}


	//*******************************************************************//
	//  set body info
	//*******************************************************************//

	// XXX
	// body.body_info.view_width   = ...;
	// body.body_info.view_quality = ...;

	//
	// set stamina and effort
	//
	this -> update_stamina_effort( old_state ,
				       param ,
				       prev_com ,
				       last_command_accepted ,
				       have_current_body_info );

	//
	// set speed
	//
	this -> update_speed( old_state ,
			      param ,
			      prev_com ,
			      last_command_accepted ,
			      have_current_body_info );


	//
	// set body_angle
	//
	this -> update_body_angle( sight_analyzer , old_state , param ,
				   prev_com , last_command_accepted );


	//
	// set self velocity
	//
	this -> update_self_velocity( old_state , param ,
				      have_current_body_info );


	//
	// set self coordinate
	//
	this -> update_self_coordinate( sight_analyzer , old_state ,
					prev_com , last_command_accepted );


	//
	// set ball coordinate and velocity
	//

	// XXX
	if ( sight_analyzer )
	{
		sight_analyzer -> advice_self_velocity( body.my_velocity );
	}

	this -> update_ball_coordinate_and_velocity( old_state ,
						     param , sight_analyzer ,
						     prev_com );

	//
	// set player coordinate
	//
	this -> update_player_coordinate( old_state , param , sight_analyzer );


	//*******************************************************************//
	//  check calculated result
	//*******************************************************************//
	if ( have_current_debug_info && have_current_body_info
	  && ! true_view )
	{
		if ( game_state.game_info.play_mode == Play_Mode::Play_On )
		{
			if ( dbg )
			{
				dbg << "&&& neck_angle_error "
				    << (body.neck_angle.point()
					- buffer.debug_info
					  -> self.true_neck_angle)
					.normalize().degree()
				    << endl;

				dbg << "&&& body_angle_error "
				    << (body.my_body_angle.point()
					- buffer.debug_info
					  -> self.true_body_angle)
					.normalize().degree()
				    << endl;

				dbg << "&&& coordinate_error "
				    << (body.my_coordinate.point()
					- buffer.debug_info
					  -> self.true_coordinate).r() << endl;

				dbg << "&&& coordinate_x_error "
				    << body.my_coordinate.point().x()
					- buffer.debug_info
					  -> self.true_coordinate.x() << endl;

				dbg << "&&& coordinate_y_error "
				    << body.my_coordinate.point().y()
					- buffer.debug_info
					  -> self.true_coordinate.y() << endl;
			}

			if ( (body.my_body_angle.range().valid()
			      && (body.my_body_angle.range().median()
				  - buffer.debug_info -> self.true_body_angle)
				  .normalize().abs()
				> body.my_body_angle.range().width() / 2.0
				   + Degree( 0.05 ) ) )
			{
				cerr << "body angle error!!" << endl;

				if ( dbg )
				{
					dbg  << "body angle error!!" << endl;

					dbg  << "body angle range = "
					     << body.my_body_angle.range()
						.min().degree() << ","
					     << body.my_body_angle.range()
						.max().degree() << endl;

					dbg  << "true body angle = "
					     << buffer.debug_info
						-> self.true_body_angle
								    .degree()
					     << endl;
				}
			}

			if ( body.my_coordinate.valid()
			  && ! body.my_coordinate.region().in_region
			       ( buffer.debug_info -> self.true_coordinate) )
			{
				cerr << "coordinate error!!" << endl;

				if ( dbg )
				{
					dbg  << "coordinate error!!" << endl;
				}
			}
		}

#if 0
		cout << "self velocity error = "
		     << (body.my_velocity.point()
			 - buffer.debug_info -> self.true_velocity).r()
		     << endl;
#endif
#if 0
		if ( ball.velocity_accuracy == 0 )
		{
			D2_Vector	ball_diff
			  = (buffer.debug_info -> ball.true_coordinate
			     - buffer.debug_info -> self.true_coordinate);

			if ( 2.0 <= ball_diff.r()  &&  ball_diff.r() <= 10.0 )
			{
				cerr << "ball velocity error = "
				     << (ball.velocity.point()
					 - buffer.debug_info
					   -> ball_true_velocity).r()
				     << endl;

				cerr << "ball velocity dir error = "
				     << (ball.velocity.point().theta()
					 - buffer.debug_info
					   -> ball_true_velocity.theta())
					.normalize().degree()
				     << endl;
			}
		}
#endif
#if 0
		if ( ball.velocity_accuracy == 0 )
		{
			cout << "ball velocity r error = "
			     << ball.velocity.point().r()
				- buffer.debug_info -> ball_true_velocity.r()
			     << endl;
		}
#endif
	}


	//
	// overwrite with true_view
	//
	if ( true_view && have_current_debug_info )
	{
		// self
		body.my_coordinate
			.set( buffer.debug_info -> self.true_coordinate ,
			      buffer.debug_info -> self.true_coordinate );

		body.my_velocity
			.set( buffer.debug_info -> self.true_velocity ,
			      buffer.debug_info -> self.true_velocity );

		body.last_info_time = game_state.current_time;


		// ball
		ball.coordinate
			.set( buffer.debug_info -> ball_true_coordinate ,
			      buffer.debug_info -> ball_true_coordinate );
		ball.coordinate_accuracy = 0;
		ball.last_coordinate_info_time = game_state.current_time;

		ball.velocity.set( buffer.debug_info -> ball_true_velocity ,
				   buffer.debug_info -> ball_true_velocity );
		ball.velocity_accuracy = 0;
		ball.last_velocity_info_time = game_state.current_time;

		// teammate
		for ( int i = 0  ;  i < MAX_PLAYER  ;  i ++ )
		{
			if ( ! (buffer.debug_info -> teammate[i].have_info) )
			{
				continue;
			}

			this -> teammate[i].coordinate
				.set( buffer.debug_info
				      -> teammate[i].true_coordinate ,
				      buffer.debug_info
				      -> teammate[i].true_coordinate );
			this -> teammate[i].coordinate_accuracy = 0;

			this -> teammate[i].velocity
				.set( buffer.debug_info
				      -> teammate[i].true_velocity ,
				      buffer.debug_info
				      -> teammate[i].true_velocity );
			this -> teammate[i].velocity_accuracy = 0;

			this -> teammate[i].body_angle
				.set( buffer.debug_info
				      -> teammate[i].true_body_angle ,
				      buffer.debug_info
				      -> teammate[i].true_body_angle ,
				      buffer.debug_info
				      -> teammate[i].true_body_angle );

			this -> teammate[i].face_angle
				.set( buffer.debug_info
				      -> teammate[i].true_neck_angle ,
				      buffer.debug_info
				      -> teammate[i].true_neck_angle ,
				      buffer.debug_info
				      -> teammate[i].true_neck_angle );

			this -> teammate[i].last_info_time
				= game_state.current_time;
		}

		// opponent
		for ( int i = 0  ;  i < MAX_PLAYER  ;  i ++ )
		{
			if ( ! (buffer.debug_info -> opponent[i].have_info) )
			{
				continue;
			}

			this -> opponent[i].coordinate
				.set( buffer.debug_info
				      -> opponent[i].true_coordinate ,
				      buffer.debug_info
				      -> opponent[i].true_coordinate );
			this -> opponent[i].coordinate_accuracy = 0;

			this -> opponent[i].velocity
				.set( buffer.debug_info
				      -> opponent[i].true_velocity ,
				      buffer.debug_info
				      -> opponent[i].true_velocity );
			this -> opponent[i].velocity_accuracy = 0;

			this -> opponent[i].body_angle
				.set( buffer.debug_info
				      -> opponent[i].true_body_angle ,
				      buffer.debug_info
				      -> opponent[i].true_body_angle ,
				      buffer.debug_info
				      -> opponent[i].true_body_angle );

			this -> opponent[i].face_angle
				.set( buffer.debug_info
				      -> opponent[i].true_neck_angle ,
				      buffer.debug_info
				      -> opponent[i].true_neck_angle ,
				      buffer.debug_info
				      -> opponent[i].true_neck_angle );

			this -> opponent[i].last_info_time
				= game_state.current_time;
		}

		this -> unknown_teammate.clear();
		this -> unknown_opponent.clear();
		this -> unknown_player.clear();
	}


	//
	// set offside line
	//
	this -> update_offside();
}

void   Field_Recog::Field_Recog_State::update_stamina_effort(
	      const ref_count_ptr<const Field_Recog_State> &  old_state ,
	      const SServer_Param *  param ,
	      const Soccer_Composite_Command &  prev_com ,
	      bool  last_command_accepted ,
	      bool  have_current_body_info )
{
	if ( ! have_current_body_info )
	{
		return;
	}


	//
	// set effort
	//
	Double_Range	ef = param -> server_effort_info_to_real_range
			       ( buffer.body_info -> effort );
	body.effort.set( ef , buffer.body_info -> effort );


	//
	// set stamina
	//
	Double_Range	st = old_state -> body.stamina.range();

	if ( last_command_accepted
	  && prev_com.base_type() == Soccer_Command::Dash_Command )
	{
		st -= param -> dash_power_to_stamina_dec
				 ( prev_com.base_command().dash_power() );

		st.set( std::max( st.min() , 0.0 ) ,
			std::max( st.max() , 0.0 ) );
	}

	// XXX: when no effort,
	//        stamina does not increment STAMINA_INC_MAX()
	st += param -> STAMINA_INC_MAX();

	st.set( std::min( st.min() , param -> STAMINA_MAX() ) ,
		std::min( st.max() , param -> STAMINA_MAX() ) );

	// XXX: check if culculated stamina is range from body info.
	body.stamina.set( st , buffer.body_info -> stamina );
	Double_Range	strict_stamina_range
			  = param -> server_stamina_info_to_real_range(
				     buffer.body_info -> stamina );

	if ( strict_stamina_range.in_range( st.median() , EPS ) )
	{
		body.stamina.set( st , st.median() );
	}
	else
	{
		body.stamina.set( strict_stamina_range ,
				  strict_stamina_range.median() );
	}
}

void   Field_Recog::Field_Recog_State::update_speed(
	      const ref_count_ptr<const Field_Recog_State> &  old_state ,
	      const SServer_Param *  param ,
	      const Soccer_Composite_Command &  prev_com ,
	      bool  last_command_accepted ,
	      bool  have_current_body_info )
{
	if ( have_current_body_info )
	{
		Double_Range	sp = param -> server_speed_info_to_real_range(
						   buffer.body_info -> speed );
		body.speed.set( sp , buffer.body_info -> speed );
	}
	else
	{
		// XXX
		(void)old_state;
		(void)prev_com;
		(void)last_command_accepted;

		Double_Range	sp( - param -> PLAYER_SPEED_MAX() ,
				    + param -> PLAYER_SPEED_MAX() );

		body.speed.set( sp , 0.0 );
	}
}

void   Field_Recog::Field_Recog_State::update_neck_angle(
	      const ref_count_ptr<const Field_Recog_State> &  old_state ,
	      const SServer_Param *  param ,
	      bool  have_current_body_info ,
	      bool  last_turn_neck_accepted )
{
	const Angle_Range	valid_range( param -> MIN_NECK_ANGLE() ,
					     param -> MAX_NECK_ANGLE() );

	//
	// set neck_angle from
	//    old neck_angle & turn_neck_angle_moment
	//
	Angle_Range	neck_angle_range
			    = old_state -> body.neck_angle.range();

	if ( last_turn_neck_accepted )
	{
		neck_angle_range += buffer.turn_neck_moment;
	}

	neck_angle_range = valid_range.clip( neck_angle_range );


	//
	// check if calculated neck_angle is in range from current body_info
	//
	if ( have_current_body_info )
	{
		Angle_Range	strict_neck_angle_range
				= param -> server_neck_angle_info_to_real_range
					   ( buffer.body_info -> neck_angle );

		if ( ! strict_neck_angle_range
			 .in_range( neck_angle_range.median() , Degree(EPS) ) )
		{
			// set neck_angle info from new body info
			neck_angle_range = strict_neck_angle_range;
		}
	}

	// set calculated neck_angle
	body.neck_angle.set( neck_angle_range , neck_angle_range.median() );
}

void   Field_Recog::Field_Recog_State::update_body_angle
		( const ref_count_ptr<Sight_Info_Analyzer> &  sight_analyzer ,
		  const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const SServer_Param *  param ,
		  const Soccer_Composite_Command &  prev_com ,
		  bool  last_command_accepted )
{
	// XXX
	if ( sight_analyzer )
	{
		sight_analyzer -> get_body_angle_range
					( &(body.my_body_angle) );
	}
	else
	{
		Soccer_Command	com = prev_com.base_command();

		if ( com == Soccer_Command::Turn_Command
		     && last_command_accepted )
		{
			Angle	angle_min;
			Angle	angle_max;
			Angle	angle_pt;

			angle_min = param -> turn_moment_to_angle
				     ( com.turn_moment() ,
				       old_state -> body.speed.range().min() );

			angle_min = param -> turn_moment_to_angle
				     ( com.turn_moment() ,
				       old_state -> body.speed.range().min() );

			angle_pt = param -> turn_moment_to_angle
				   ( com.turn_moment() ,
				     old_state -> body.speed.point() );

			if ( angle_min > angle_max )
			{
				std::swap( angle_min , angle_max );
			}

			Angle	angle_min_error = angle_min - angle_pt;
			Angle	angle_max_error = angle_max - angle_pt;

			Angle_Info	old = old_state -> body.my_body_angle;

			body.my_body_angle.set( old.range().min()
						  + angle_min_error ,
						old.range().max()
						  + angle_max_error ,
						angle_pt );
		}
		else
		{
			body.my_body_angle = old_state -> body.my_body_angle;
		}
	}
}

void   Field_Recog::Field_Recog_State::update_self_coordinate
		( const ref_count_ptr<Sight_Info_Analyzer> &  sight_analyzer ,
		  const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const Soccer_Composite_Command &  prev_com ,
		  bool  last_command_accepted )
{
	if ( prev_com.base_command() == Soccer_Command::Move_Command
	  && prev_com.base_command().move_x() < 0.0
	  && last_command_accepted )
	{
		D2_Vector	p( prev_com.base_command().move_x() ,
				   prev_com.base_command().move_y() );

		body.my_coordinate.set( p , p );
		return;
	}

	if ( game_state.game_info.play_mode == Play_Mode::Before_Kick_Off
	  || game_state.game_info.play_mode == Play_Mode::Our_Goal
	  || game_state.game_info.play_mode == Play_Mode::Opponent_Goal )
	{
		if ( (! (prev_com.base_command()
			 == Soccer_Command::Move_Command
			 && prev_com.base_command().move_x() >= 0.0 )
		  && old_state -> body.my_coordinate.point().x() < 0.0 ) )
		{
			body.my_coordinate = old_state -> body.my_coordinate;
			return;
		}
	}

	if ( sight_analyzer
	  && (sight_analyzer -> get_self_coordinate( &(body.my_coordinate) )) )
	{
#if 1
		sight_analyzer
			-> get_body_angle_range( &(body.my_body_angle) );
#endif

		return;
	}

	// XXX
	body.my_coordinate = old_state -> body.my_coordinate;
}

void   Field_Recog::Field_Recog_State::update_self_velocity
		( const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const SServer_Param *  param ,
		  bool  have_current_body_info )
{
	(void)old_state;
	(void)param;

	if ( have_current_body_info )
	{
		// XXX
		body.my_velocity.set( D2_Region::universal_region() ,
				      D2_Vector
				      ( D2_Vector::Pole ,
					body.speed.point() ,
					(buffer.body_info -> speed_angle
					 + body.my_body_angle.point()
					 + body.neck_angle.point())
					.normalize() ) );
	}
	else
	{
		// XXX
		body.my_velocity.set( D2_Region::universal_region() ,
				      D2_Vector
				      ( D2_Vector::Pole ,
					body.speed.point() ,
					body.my_body_angle.point() ) );
	}
}

#if 0
void   Field_Recog::Field_Recog_State::update_ball_coordinate_and_velocity
		( const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const SServer_Param *  param ,
		  const ref_count_ptr<Sight_Info_Analyzer> &  sight_analyzer ,
		  bool  ball_fixed ,
		  const Soccer_Composite_Command &  prev_com )
{
	bool	have_ball_sight_info = false;

	if ( sight_analyzer )
	{
		sight_analyzer -> get_ball_info( &ball );

		if ( ball.coordinate.valid() )
		{
			have_ball_sight_info = true;

			ball.last_coordinate_info_time
				= game_state.current_time;
			ball.coordinate_accuracy = 0;
		}

		if ( ball.velocity.valid() )
		{
			ball.last_velocity_info_time
				= game_state.current_time;
			ball.velocity_accuracy = 0;
		}
	}
	else
	{
		ball.coordinate.reset();
		ball.velocity.reset();

		ball.coordinate_accuracy = Ball_Info::INVALID_ACCURACY;
		ball.velocity_accuracy   = Ball_Info::INVALID_ACCURACY;
	}

	if ( have_ball_sight_info )
	{
		this -> buffer.ball_seen_count
			= old_state -> buffer.ball_seen_count + 1;
	}
	else
	{
		this -> buffer.ball_seen_count = 0;
	}


	if ( ball_fixed )
	{
		ball.velocity.set( D2_Vector::origin() ,
				   D2_Vector::origin() );

		ball.velocity_accuracy = 0;
	}


	bool	use_calculated_ball_coordinate = false;
	bool	use_calculated_ball_velocity   = false;

	//
	// set original coordinate, velocity
	//
	D2_Vector	ball_coordinate_org;

	if ( ball.coordinate.valid() )
	{
		ball_coordinate_org = ball.coordinate.point();
	}
	else
	{
		use_calculated_ball_coordinate = true;
	}

	// if ball is in short sensor, not trust.
	if ( prev_com.base_type() == Soccer_Command::Kick_Command
	  && old_state -> ball.coordinate.valid()
	  && ball.coordinate.valid() )
	{
		use_calculated_ball_coordinate = true;
	}


	//
	// calculate ball velocity
	//
	D2_Vector	old_ball_velocity;

	if ( old_state -> ball.velocity.valid() )
	{
		old_ball_velocity = old_state -> ball.velocity.point();
	}
	else
	{
		old_ball_velocity.set( 0.0 , 0.0 );
	}

	if ( prev_com.base_command().type() == Soccer_Command::Kick_Command
	   && (old_state -> ball.coordinate.point()
	       - old_state -> body.my_coordinate.point()).r()
	     <= param -> kickable_distance() )
	{
		D2_Vector	old_ball_relative
		= (old_state -> ball.coordinate.point()
		   - old_state -> body.my_coordinate.point())
		  .rotate( - old_state -> body.my_body_angle.point() );

		double	accel = param -> kick_power_to_accel
				  ( prev_com.base_command().kick_power() ,
				    old_ball_relative.r() ,
				    old_ball_relative.theta().normalize() );

		D2_Vector	kick_vec( D2_Vector::Pole ,
					  accel ,
					  old_state
					    -> body.my_body_angle.point()
					  + prev_com.base_command()
						       .kick_direction() );

		old_ball_velocity += kick_vec;
	}

	D2_Vector	calculated_ball_velocity
			  = old_ball_velocity * sserver_param -> BALL_DECAY();

	if ( use_calculated_ball_velocity )
	{
		if ( old_state -> ball.velocity_accuracy
		     != Ball_Info::INVALID_ACCURACY )
		{
			ball.velocity_accuracy
				= old_state -> ball.velocity_accuracy - 1;
		}

		ball.velocity.set( D2_Region::universal_region() /* XXX */ ,
				   old_ball_velocity
				     * sserver_param -> BALL_DECAY() );
	}


	//
	// coordinate update from old state
	//
	bool		have_culculated_ball_coordinate = false;
	D2_Vector	culculated_ball_coordinate;

	if ( use_calculated_ball_coordinate )
	{
		culculated_ball_coordinate
			= old_state -> ball.coordinate.point()
			  + old_ball_velocity;

		have_culculated_ball_coordinate = true;

		if ( old_state -> ball.coordinate_accuracy
		     != Ball_Info::INVALID_ACCURACY )
		{
			ball.coordinate_accuracy
				= old_state -> ball.coordinate_accuracy - 1;
		}

		ball.coordinate.set( D2_Region::universal_region() /* XXX */ ,
				     old_state -> ball.coordinate.point()
				     + old_ball_velocity );
	}


	if ( this -> buffer.ball_seen_count >= 2
	  && old_state -> ball.velocity.valid()
	  && have_culculated_ball_coordinate
	  && (ball.coordinate.point() - body.my_coordinate.point()).r()
	     >= 20.0 )
	{
		ball.coordinate.set( ball.coordinate.region() ,
				     ball_coordinate_org
				      / this -> buffer.ball_seen_count
				     + culculated_ball_coordinate
				      / (this -> buffer.ball_seen_count - 1) );
	}


	// if calculated ball coordinate is too far from original,
	// reset it
	if ( use_calculated_ball_coordinate && have_ball_sight_info )
	{
		if ( (ball.coordinate.point() - body.my_coordinate.point()).r()
		     <= param -> SHORT_SENSOR_LENGTH()
		  && (ball.coordinate.point() - ball_coordinate_org).r()
		     >= 0.15 )
		{
			ball.coordinate.set( ball.coordinate.region() ,
					     ball_coordinate_org );
		}

		ball.coordinate_accuracy = 0;
	}


	// if calculated ball coordinate is in short senser
	//    and ball is not found in short senser,
	// decrease ball accuracy
	if ( ! have_ball_sight_info
	     && (ball.coordinate.point() - body.my_coordinate.point()).r()
		<= param -> SHORT_SENSOR_LENGTH() - 0.5 )
	{
#if 0
		cerr << "ball coordinate canceled!!" << endl;
#endif
		ball.coordinate_accuracy = Ball_Info::INVALID_ACCURACY;
	}
}
#else
void   Field_Recog::Field_Recog_State::update_ball_coordinate_and_velocity
		( const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const SServer_Param *  param ,
		  const ref_count_ptr<Sight_Info_Analyzer> &  sight_analyzer ,
		  const Soccer_Composite_Command &  prev_com )
{
	switch( game_state.game_info.play_mode )
	{
	case Play_Mode::Before_Kick_Off:

	case Play_Mode::Opponent_Kick_Off:
	case Play_Mode::Our_Kick_Off:
		ball.coordinate.set( D2_Vector::origin() ,
				     D2_Vector::origin() );
		ball.coordinate_accuracy = 0;

		ball.velocity.set( D2_Vector::origin() ,
				   D2_Vector::origin() );
		ball.velocity_accuracy = 0;
		return;
		break;

	default:
		break;
	}


	bool	ball_fixed = false;

	switch( game_state.game_info.play_mode )
	{
	case Play_Mode::Before_Kick_Off:

	case Play_Mode::Opponent_Kick_Off:
	case Play_Mode::Opponent_Kick_In:
	case Play_Mode::Opponent_Free_Kick:
	case Play_Mode::Opponent_Corner_Kick:
	case Play_Mode::Opponent_Goal:
	case Play_Mode::Opponent_Catch:

	case Play_Mode::Our_Kick_Off:
	case Play_Mode::Our_Kick_In:
	case Play_Mode::Our_Free_Kick:
	case Play_Mode::Our_Corner_Kick:
	case Play_Mode::Our_Goal:
	case Play_Mode::Our_Catch:
		ball_fixed = true;
		break;

	case Play_Mode::Play_On:
	case Play_Mode::Our_Goal_Kick:
	case Play_Mode::Opponent_Goal_Kick:
		ball_fixed = false;
		break;
	}


	bool	ball_not_found = true;

	if ( sight_analyzer )
	{
		sight_analyzer -> get_ball_info( &ball );

		if ( ball.coordinate.valid() )
		{
			ball_not_found = false;

			ball.last_coordinate_info_time
				= game_state.current_time;
			ball.coordinate_accuracy = 0;
		}

		if ( ball.velocity.valid() )
		{
			ball.last_velocity_info_time
				= game_state.current_time;
			ball.velocity_accuracy = 0;
		}
	}
	else
	{
		ball.coordinate.reset();
		ball.velocity.reset();

		ball.coordinate_accuracy = Ball_Info::INVALID_ACCURACY;
		ball.velocity_accuracy   = Ball_Info::INVALID_ACCURACY;
	}

	if ( ball_fixed )
	{
		ball.velocity.set( D2_Vector::origin() ,
				   D2_Vector::origin() );

		ball.velocity_accuracy = 0;
	}

	// if ball is in short sensor, not trust.
	D2_Vector	ball_coordinate_backup;
	bool		use_calculated_ball_coordinate = false;

	// XXX
	if ( prev_com.base_type() == Soccer_Command::Kick_Command
	  && old_state -> ball.coordinate.valid()
	  && ball.coordinate.valid() )
	{
		if ( dbg )
		{
			dbg << "## use_calculated_ball_coordinate!!" << endl;
		}

		ball_coordinate_backup = ball.coordinate.point();

		use_calculated_ball_coordinate = true;
	}

	if ( ball.coordinate.valid()
	  && ball.velocity.valid()
	  && ! use_calculated_ball_coordinate )
	{
		return;
	}


	//
	// update ball velocity
	//
	D2_Vector	old_ball_velocity;

	if ( old_state -> ball.velocity.valid() )
	{
		old_ball_velocity = old_state -> ball.velocity.point();
	}
	else
	{
		old_ball_velocity.set( 0.0 , 0.0 );
	}

	// calculate kick effect
	if ( prev_com.base_command().type() == Soccer_Command::Kick_Command
	   && (old_state -> ball.coordinate.point()
	       - old_state -> body.my_coordinate.point()).r()
	     <= param -> kickable_distance() )
	{
		D2_Vector	old_ball_relative
		= (old_state -> ball.coordinate.point()
		   - old_state -> body.my_coordinate.point())
		  .rotate( - old_state -> body.my_body_angle.point() );

		if ( old_ball_relative.r() < param -> kickable_distance() )
		{
			double	accel
				= param -> kick_power_to_accel
				  ( prev_com.base_command().kick_power() ,
				    old_ball_relative.r() ,
				    old_ball_relative.theta().normalize() );

			D2_Vector	kick_vec( D2_Vector::Pole ,
						  accel ,
						  old_state
						  -> body.my_body_angle.point()
						  + prev_com.base_command()
						       .kick_direction() );

			old_ball_velocity += kick_vec;

			if ( old_ball_velocity.r()
			     > param -> BALL_SPEED_MAX() )
			{
				old_ball_velocity
					.set_r( param -> BALL_SPEED_MAX() );
			}
		}
	}

	if ( ! ball_fixed )
	{
		if ( old_state -> ball.velocity_accuracy
		     != Ball_Info::INVALID_ACCURACY )
		{
			ball.velocity_accuracy
				= old_state -> ball.velocity_accuracy - 1;
		}

		ball.velocity.set( D2_Region::universal_region() /* XXX */ ,
				   old_ball_velocity
				     * sserver_param -> BALL_DECAY() );
	}


	//
	// coordinate update from old state
	//
	if ( (! ball.coordinate.valid() || use_calculated_ball_coordinate)
	  && old_state -> ball.coordinate.valid() )
	{
		ball.coordinate.set
			( D2_Region::universal_region() /* XXX */ ,
			  old_state -> ball.coordinate.point()
			  + old_ball_velocity );

		if ( old_state -> ball.coordinate_accuracy
		     != Ball_Info::INVALID_ACCURACY )
		{
			ball.coordinate_accuracy
				= old_state -> ball.coordinate_accuracy - 1;
		}
	}


	if ( use_calculated_ball_coordinate )
	{
		if ( (ball.coordinate.point() - ball_coordinate_backup).r()
		     >= 0.15 )
		{
			if ( dbg )
			{
				dbg << "## bad ball update!!" << endl;
				dbg << "## fail back to sight info" << endl;
			}

			ball.coordinate.set( ball.coordinate.region() ,
					     ball_coordinate_backup );
		}

		ball.coordinate_accuracy = 0;
	}


	// if calculated ball coordinate is in short senser
	//    and ball is not found in short senser,
	// decrease ball accuracy
	if ( ball_not_found )
	{
		D2_Vector	ball_relative
				= ball.coordinate.point()
				   - body.my_coordinate.point();

		if ( ball_relative.r() <= param -> SHORT_SENSOR_LENGTH() - 0.5
		  ||
		     ((ball_relative.theta()
		       - (body.my_body_angle.point()
			  + body.neck_angle.point())).normalize().abs()
			  <= View_Width( View_Width::Narrow ).half_angle()
			      - Degree( 5.0 )
		      // XXX: magic number 40.0
		      && ball_relative.r() < 40.0 - 1.0) )
		{
			if ( dbg )
			{
				dbg << "## ball coordinate canceled!!" << endl;
			}

			if ( ball.coordinate_accuracy
			     != Ball_Info::INVALID_ACCURACY )
			{
				ball.coordinate_accuracy -= 100;
			}

			if ( ball.velocity_accuracy
			     != Ball_Info::INVALID_ACCURACY )
			{
				ball.velocity_accuracy -= 100;
			}
		}
	}
}
#endif

void   Field_Recog::Field_Recog_State::update_player_coordinate
		( const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const SServer_Param *  param ,
		  const ref_count_ptr<Sight_Info_Analyzer> &  sight_analyzer )
{
	if ( sight_analyzer )
	{
		sight_analyzer -> get_player_info
			( &teammate , &opponent ,
			  &unknown_teammate , &unknown_opponent ,
			  &unknown_player );
	}
	else
	{
		// XXX

		for ( size_t  i = 0  ;  i < teammate.size()  ;  i ++ )
		{
			teammate[i].coordinate.reset();
			teammate[i].velocity.reset();
			teammate[i].body_angle.reset();
			teammate[i].face_angle.reset();
			teammate[i].coordinate_accuracy
				= Player_Info::INVALID_ACCURACY;
			teammate[i].coordinate_accuracy
				= Player_Info::INVALID_ACCURACY;
		}

		for ( size_t  i = 0  ;  i < opponent.size()  ;  i ++ )
		{
			opponent[i].coordinate.reset();
			opponent[i].velocity.reset();
			opponent[i].body_angle.reset();
			opponent[i].face_angle.reset();
			opponent[i].coordinate_accuracy
				= Player_Info::INVALID_ACCURACY;
			opponent[i].coordinate_accuracy
				= Player_Info::INVALID_ACCURACY;
		}

		unknown_teammate.clear();
		unknown_opponent.clear();
		unknown_player.clear();
	}


	int	self_player_number = game_state.game_info.self_player_number;

	if ( 1 <= self_player_number
	  && static_cast<size_t>(self_player_number) <= teammate.size() )
	{
		teammate[ self_player_number - 1 ].coordinate
			= body.my_coordinate;
	}


	//
	// unknown player matching
	//
	this -> match_unknown_player( old_state , param );


	//
	// update accuracy
	//
	for ( size_t  i = 0  ;  i < teammate.size()  ;  i ++ )
	{
		if ( teammate[i].coordinate.valid() )
		{
			teammate[i].coordinate_accuracy = 0;
		}
		else
		{
			teammate[i].coordinate
				= old_state -> teammate[i].coordinate;

			if ( old_state -> teammate[i].coordinate_accuracy
			     == Player_Info::INVALID_ACCURACY )
			{
				teammate[i].coordinate_accuracy
					= Player_Info::INVALID_ACCURACY;
			}
			else
			{
				D2_Vector	teammate_relative;
				teammate_relative
					= teammate[i].coordinate.point()
					   - body.my_coordinate.point();

				Angle	absolute_view_angle
					= body.my_body_angle.point()
					   + body.neck_angle.point();

				if ( (teammate_relative.theta()
				      - absolute_view_angle).normalize().abs()
				     <= View_Width( View_Width::Narrow )
								.half_angle()
					 - Degree( 5.0 )
				     // XXX: magic number 40.0
				     && teammate_relative.r() < 40.0 - 5.0)
				{
					teammate[i].coordinate_accuracy
					  = old_state -> teammate[i]
						.coordinate_accuracy - 50;
				}
				else
				{
					teammate[i].coordinate_accuracy
					  = old_state -> teammate[i]
						.coordinate_accuracy - 1;
				}
			}
		}
	}

	for ( size_t  i = 0  ;  i < opponent.size()  ;  i ++ )
	{
		if ( opponent[i].coordinate.valid() )
		{
			opponent[i].coordinate_accuracy = 0;
		}
		else
		{
			opponent[i].coordinate
				= old_state -> opponent[i].coordinate;

			if ( old_state -> opponent[i].coordinate_accuracy
			     == Player_Info::INVALID_ACCURACY )
			{
				opponent[i].coordinate_accuracy
					= Player_Info::INVALID_ACCURACY;
			}
			else
			{
				opponent[i].coordinate_accuracy
				 = old_state -> opponent[i].coordinate_accuracy
					- 1;
			}
		}
	}


	if ( 1 <= self_player_number
	  && static_cast<size_t>(self_player_number) <= teammate.size() )
	{
		teammate[ self_player_number - 1 ].coordinate
			= body.my_coordinate;
		teammate[ self_player_number - 1 ].velocity
			= body.my_velocity;
		teammate[ self_player_number - 1 ].body_angle
			= body.my_body_angle;
		teammate[ self_player_number - 1 ].face_angle
			= body.neck_angle;
		teammate[ self_player_number - 1 ].coordinate_accuracy
			= (body.my_coordinate.valid() ?
			   0 : Player_Info::INVALID_ACCURACY);
		teammate[ self_player_number - 1 ].velocity_accuracy
			= (body.my_velocity.valid() ?
			   0 : Player_Info::INVALID_ACCURACY);
	}
}


void   Field_Recog::Field_Recog_State::match_unknown_player
		( const ref_count_ptr<const Field_Recog_State> &  old_state ,
		  const SServer_Param *  param )
{
	const	long	VALID_MATCH_ACCURACY = - 100;


	bool	matching_occured = false;

	// match unknown_teammate and teammate
	do
	{
		matching_occured = false;

		std::vector<Player_Info>	new_unknown_teammate;

		for ( size_t  un = 0  ;
		      un < unknown_teammate.size()  ;  un ++ )
		{
			int	matched_index = -1;
			int	matched_count = 0;

			for ( size_t  pl = 0  ;
			      pl < teammate.size()  ;  pl ++ )
			{
				if ( teammate[pl].coordinate.valid()
				     || old_state
					  -> teammate[pl].coordinate_accuracy
				     < VALID_MATCH_ACCURACY )
				{
					continue;
				}

				if ((old_state -> teammate[pl]
				     .coordinate.point()
				     - unknown_teammate[un]
				     .coordinate.point()).r()
				    < (- old_state
				       -> teammate[pl].coordinate_accuracy + 1)
				    * param -> player_speed_max_max() )
				{
					matched_index = pl;

					matched_count ++;
					if ( matched_count >= 2 )
					{
						break;
					}
				}
			}

			if ( matched_count == 1 )
			{
				teammate[matched_index] = unknown_teammate[un];
				matching_occured = true;
			}
			else
			{
				new_unknown_teammate
					.push_back( unknown_teammate[un] );
			}
		}

		unknown_teammate = new_unknown_teammate;

	} while( matching_occured );


	// match unknown_opponent and opponent
	do
	{
		matching_occured = false;

		std::vector<Player_Info>	new_unknown_opponent;

		for ( size_t  un = 0  ;
		      un < unknown_opponent.size()  ;  un ++ )
		{
			int	matched_index = -1;
			int	matched_count = 0;

			for ( size_t  pl = 0  ;
			      pl < opponent.size()  ;  pl ++ )
			{
				if ( opponent[pl].coordinate.valid()
				     || old_state
					  -> opponent[pl].coordinate_accuracy
				     < VALID_MATCH_ACCURACY )
				{
					continue;
				}

				if ((old_state -> opponent[pl]
				     .coordinate.point()
				     - unknown_opponent[un]
				     .coordinate.point()).r()
				    < (- old_state
				       -> opponent[pl].coordinate_accuracy + 1)
				    * param -> player_speed_max_max() )
				{
					matched_index = pl;

					matched_count ++;
					if ( matched_count >= 2 )
					{
						break;
					}
				}
			}

			if ( matched_count == 1 )
			{
				opponent[matched_index] = unknown_opponent[un];
				matching_occured = true;
			}
			else
			{
				new_unknown_opponent
					.push_back( unknown_opponent[un] );
			}
		}

		unknown_opponent = new_unknown_opponent;

	} while( matching_occured );



	// match unknown player and player

	std::vector<Player_Info>	new_unknown_player;

	for ( size_t  un = 0  ;  un < unknown_player.size()  ;  un ++ )
	{
		std::vector<Player_Info>::iterator
			matched_iterator = unknown_player.end();

		int	matched_count = 0;

		for ( size_t  pl = 0  ;  pl < teammate.size()  ;  pl ++ )
		{
			if ( matched_count >= 2 )
			{
				break;
			}

			if ( teammate[pl].coordinate.valid()
			  || old_state -> teammate[pl].coordinate_accuracy
			     < VALID_MATCH_ACCURACY )
			{
				continue;
			}

			if ( (old_state -> teammate[pl].coordinate.point()
			      - unknown_player[un].coordinate.point()).r()
			     < (- old_state
				  -> teammate[pl].coordinate_accuracy + 1)
			       * param -> player_speed_max_max() )
			{
				matched_iterator = teammate.begin() + pl;

				matched_count ++;
			}
		}

		for ( size_t  pl = 0  ;  pl < opponent.size()  ;  pl ++ )
		{
			if ( matched_count >= 2 )
			{
				break;
			}

			if ( opponent[pl].coordinate.valid()
			  || old_state -> opponent[pl].coordinate_accuracy
			     < VALID_MATCH_ACCURACY )
			{
				continue;
			}

			if ( (old_state -> opponent[pl].coordinate.point()
			      - unknown_player[un].coordinate.point()).r()
			     < (- old_state
				  -> opponent[pl].coordinate_accuracy + 1)
			       * param -> player_speed_max_max() )
			{
				matched_iterator = opponent.begin() + pl;

				matched_count ++;
			}
		}

		if ( matched_count == 1 )
		{
			*matched_iterator = unknown_player[un];
		}
		else
		{
			new_unknown_player.push_back( unknown_player[un] );
		}
	}

	unknown_player = new_unknown_player;
}


void   Field_Recog::Field_Recog_State::update_offside()
{
	static	const	long	OFFSIDE_LINE_ACCURACY_LEVEL = -40;

	// culculate defence_line
	{
		std::multiset<double>	x_set;

		for ( size_t  i = 0  ;  i < teammate.size()  ;  i ++ )
		{
			if ( teammate[i].coordinate.valid()
			  && teammate[i].coordinate_accuracy
			     >= OFFSIDE_LINE_ACCURACY_LEVEL )
			{
				x_set.insert( teammate[i]
					       .coordinate.point().x() );
			}
		}

		for ( size_t  i = 0  ;  i < unknown_teammate.size()  ;  i ++ )
		{
			if ( unknown_teammate[i].coordinate.valid()
			  && unknown_teammate[i].coordinate_accuracy
			     >= OFFSIDE_LINE_ACCURACY_LEVEL )
			{
				x_set.insert( unknown_teammate[i]
					       .coordinate.point().x() );
			}
		}

		double	line_x = 0.0;

		if ( x_set.size() >= 2 )
		{
			std::multiset<double>::iterator	it;
			it = x_set.begin();
			it ++;

			if ( (*it) < line_x )
			{
				line_x = (*it);
			}
		}

		if ( ball.coordinate.valid()
		  && ball.coordinate.point().x() < line_x )
		{
			line_x = ball.coordinate.point().x();
		}

		this -> defence_line = line_x;
	}

	// culculate offence_line
	{
		std::multiset<double>	x_set;

		for ( size_t  i = 0  ;  i < opponent.size()  ;  i ++ )
		{
			if ( opponent[i].coordinate.valid()
			  && opponent[i].coordinate_accuracy
			     >= OFFSIDE_LINE_ACCURACY_LEVEL )
			{
				x_set.insert( opponent[i]
					       .coordinate.point().x() );
			}
		}

		for ( size_t  i = 0  ;  i < unknown_opponent.size()  ;  i ++ )
		{
			if ( unknown_opponent[i].coordinate.valid()
			  && unknown_opponent[i].coordinate_accuracy
			     >= OFFSIDE_LINE_ACCURACY_LEVEL )
			{
				x_set.insert( unknown_opponent[i]
					       .coordinate.point().x() );
			}
		}

		double	line_x = 0.0;

		if ( x_set.size() >= 2 )
		{
			std::multiset<double>::iterator	it;
			it = x_set.end();
			it --;
			it --;

			if ( (*it) > line_x )
			{
				line_x = (*it);
			}
		}

		if ( ball.coordinate.valid()
		  && ball.coordinate.point().x() > line_x )
		{
			line_x = ball.coordinate.point().x();
		}

		this -> offence_line = line_x;
	}
}
