#include  "field_recog_interface.h"
#include  <cassert>
#include  <set>

using namespace std;

//
// Field_Recog_Interface
//
Field_Recog_Interface::Field_Recog_Interface
				( const Field_Recog_Abstract *  field )
	: field( field ) ,
	  snap() ,
	  have_target_teammate_flag( false ) , target_teammate( -1 ) ,
	  have_target_point_flag( false ) , target_point() ,
	  have_message_flag( false ) , message() ,
	  ball_reference_cache() ,
	  teammate_reference_cache() ,
	  opponent_reference_cache() ,
	  self_reference_cache( static_cast<Player_Reference *>(0) )
{
	this -> make_cache();
}

Field_Recog_Interface::Field_Recog_Interface
		( const ref_count_ptr<const Field_Recog_Abstract
					    ::Field_Recog_Snapshot> &  snap )
	: field( static_cast<const Field_Recog_Abstract *>(0) ) ,
	  snap( snap ) ,
	  have_target_teammate_flag( false ) , target_teammate( -1 ) ,
	  have_target_point_flag( false ) , target_point() ,
	  have_message_flag( false ) , message() ,
	  ball_reference_cache() ,
	  teammate_reference_cache() ,
	  opponent_reference_cache() ,
	  self_reference_cache( static_cast<Player_Reference *>(0) )
{
	this -> make_cache();
}

Field_Recog_Interface::Field_Recog_Interface()
	: field( static_cast<Field_Recog_Abstract *>(0) ) ,
	  snap() ,
	  have_target_teammate_flag( false ) , target_teammate( -1 ) ,
	  have_target_point_flag( false ) , target_point() ,
	  have_message_flag( false ) , message() ,
	  ball_reference_cache() ,
	  teammate_reference_cache() ,
	  opponent_reference_cache() ,
	  self_reference_cache( static_cast<Player_Reference *>(0) )
{
}


Field_Recog_Interface::~Field_Recog_Interface()
{
}

// private
void   Field_Recog_Interface::make_cache() const
{
	// ball cache
	ball_reference_cache = new Ball_Reference( *this );


	// teammate, opponent cache
	teammate_reference_cache.clear();
	opponent_reference_cache.clear();

	teammate_reference_cache.push_back( Player_Reference( *this ) );
	opponent_reference_cache.push_back( Player_Reference( *this ) );

	for ( int  i = 1  ;  i <= MAX_PLAYER  ;  i ++ )
	{
		teammate_reference_cache.push_back
			( Player_Reference( *this ,
					    S_Side::Our_Side , i ) );

		opponent_reference_cache.push_back
			( Player_Reference( *this ,
					    S_Side::Opponent_Side , i ) );
	}

	assert( this -> snapshot().game_state.game_info.self_player_number
		>= 0 /* coach */
	     && this -> snapshot().game_state.game_info.self_player_number
		<= MAX_PLAYER );

	// self cache
	self_reference_cache
	    = &(teammate_reference_cache
		[this -> snapshot().game_state.game_info.self_player_number]);
}

const Field_Recog_Abstract::Field_Recog_Snapshot &
					Field_Recog_Interface::snapshot() const
{
	if ( field )
	{
		return( *(this -> field -> snapshot()) );
	}
	else
	{
		return( *(this -> snap) );
	}
}

void   Field_Recog_Interface::update()
{
	// XXX
	this -> have_target_teammate_flag = false;
	this -> have_target_point_flag    = false;
	this -> have_message_flag         = false;

	if ( this -> ball_reference_cache )
	{
		this -> ball_reference_cache -> update();
	}

	for ( size_t  i = 0  ;  i < teammate_reference_cache.size()  ;  i ++ )
	{
		this -> teammate_reference_cache[i].update();
	}

	for ( size_t  i = 0  ;  i < opponent_reference_cache.size()  ;  i ++ )
	{
		this -> opponent_reference_cache[i].update();
	}
}


const Field_Recog_Abstract::Debug_Info  Field_Recog_Interface::debug() const
{
	return( this -> snapshot().debug );
}

const SServer_Param &  Field_Recog_Interface::sserver_param() const
{
	return( *(this -> snapshot().sserver_param) );
}

bool   Field_Recog_Interface::game_over() const
{
	return( this -> snapshot().game_state.game_over_flag );
}

SServer_Time  Field_Recog_Interface::current_time() const
{
	return( this -> snapshot().game_state.current_time );
}

Play_Mode_Reference  Field_Recog_Interface::play_mode() const
{
	return( Play_Mode_Reference( *this ) );
}

const Ball_Reference &  Field_Recog_Interface::ball() const
{
	return( *ball_reference_cache );
}

const Player_Reference &  Field_Recog_Interface::self() const
{
	return( *self_reference_cache );
}

const Player_Reference &  Field_Recog_Interface::teammate( int  player_number )
	const
{
	return( teammate_reference_cache[ player_number ] );
}

const Player_Reference &  Field_Recog_Interface::opponent( int  player_number )
	const
{
	return( opponent_reference_cache[ player_number ] );
}

Player_Reference  Field_Recog_Interface::player( const S_Side &  side ,
						 int  player_number ) const
{
	return( Player_Reference( *this , side , player_number ) );
}

size_t  Field_Recog_Interface::n_unknown_teammate() const
{
	return( this -> snapshot().unknown_teammate.size() );
}

Player_Reference  Field_Recog_Interface::unknown_teammate( int  indx ) const
{
	return( Player_Reference( *this ,
				  S_Side::Our_Side , -1 , indx ) );
}

size_t  Field_Recog_Interface::n_unknown_opponent() const
{
	return( this -> snapshot().unknown_opponent.size() );
}

Player_Reference  Field_Recog_Interface::unknown_opponent( int  indx ) const
{
	return( Player_Reference( *this ,
				  S_Side::Opponent_Side , -1 , indx ) );
}

size_t  Field_Recog_Interface::n_unknown_player() const
{
	return( this -> snapshot().unknown_player.size() );
}

Player_Reference  Field_Recog_Interface::unknown_player( int  indx ) const
{
	return( Player_Reference( *this , S_Side::Unknown , -1 , indx ) );
}

Team_Reference  Field_Recog_Interface::our_team() const
{
	return( Team_Reference( *this , S_Side::Our_Side ) );
}

Team_Reference  Field_Recog_Interface::opponent_team() const
{
	return( Team_Reference( *this , S_Side::Opponent_Side ) );
}

Team_Reference  Field_Recog_Interface::team( const S_Side &  side ) const
{
	return( Team_Reference( *this , side ) );
}

bool   Field_Recog_Interface::offside_position( const D2_Vector &  p ) const
{
	return( p.x() >= this -> opponent_team().defence_line() );
}


const Player_Type_Manager &  Field_Recog_Interface::player_type_manager()
	const
{
	return( *(this -> snapshot().game_state.player_type) );
}


Player_Set  Field_Recog_Interface::player_set( const Player_Predicate &  p )
	const
{
	return( Player_Set( *this , p ) );
}


Soccer_Composite_Command  Field_Recog_Interface::previous_command() const
{
	if ( field )
	{
		return( field -> previous_command() );
	}
	else
	{
		return( Soccer_Command::No_Command() );
	}
}

#include  <vector>
#include  <utility>

// XXX: temporary interface
D2_Vector  Field_Recog_Interface::ball_predict_point( long  t ) const
{
	const long	PREDICT_MAX_STEP = 50;

	if ( t == LONG_MIN )
	{
		long	self_access_step;
		long	teammate_access_step;
		long	opponent_access_step;
		long	unknown_access_step;

		this -> ball_access_step( &self_access_step ,
					  &teammate_access_step ,
					  &opponent_access_step ,
					  &unknown_access_step );

		long	predict_time
			= this -> strict_ball_lose_step( PREDICT_MAX_STEP );

		if ( self_access_step != -1 )
		{
			predict_time = std::max( predict_time ,
						 self_access_step );
		}

		if ( teammate_access_step != -1 )
		{
			predict_time = std::min( predict_time ,
						 teammate_access_step );
		}

		if ( opponent_access_step != -1 )
		{
			predict_time = std::min( predict_time ,
						 opponent_access_step );
		}

		if ( unknown_access_step != -1 )
		{
			predict_time = std::min( predict_time ,
						 unknown_access_step );
		}

		return( this -> ball_predict_point( predict_time ) );
	}


	static	std::vector< pair<SServer_Time,D2_Vector> >	cache( 200 );

	assert( t >= 0 );

	if ( static_cast<size_t>(t) >= cache.size() )
	{
		return( this -> sserver_param().ball_basic_point_n_step_later
			( this -> ball().coordinate() ,
			  this -> ball().velocity() ,
			  t ) );
	}

	if ( cache[t].first == this -> current_time() )
	{
		return( cache[t].second );
	}
	else
	{
		cache[t].first = this -> current_time();

		cache[t].second
			= this -> sserver_param().ball_basic_point_n_step_later
			  ( this -> ball().coordinate() ,
			    this -> ball().velocity() ,
			    t );

		return( cache[t].second );
	}
}


// XXX
#include  <list>
long   Field_Recog_Interface::strict_ball_lose_step( long  max_step ,
						     long  acc_level ) const
{
	//
	// cache
	//
	static	SServer_Time	cache_time( -1 );
	static	long		cache_acc_level = 0;
	static	long		cache_value = 0;

	if ( this -> current_time() == cache_time
	  && cache_acc_level == acc_level )
	{
		return( cache_value );
	}

	cache_time      = this -> current_time();
	cache_acc_level = acc_level;


	// XXX
	if ( ! this -> ball().coordinate().accuracy_check( 0 )
	  || ! this -> ball().velocity().accuracy_check( 0 )
	  || ! this -> self().body_angle().accuracy_check( 0 ) )
	{
		cache_value = 0;

		return( cache_value );
	}


	//
	// make list of player to be checked.
	//
	list<Player_Reference>	check_list;

	for ( int  i = 1  ;  i <= MAX_PLAYER  ;  i ++ )
	{
		if ( i != this -> self().player_number() )
		{
			if ( this -> teammate(i).coordinate()
			     .accuracy_check( acc_level ) )
			{
				check_list.push_back( this -> teammate(i) );
			}
		}
	}

	for ( int  i = 1  ;  i <= MAX_PLAYER  ;  i ++ )
	{
		if ( this -> opponent(i).coordinate()
		     .accuracy_check( acc_level ) )
		{
			check_list.push_back( this -> opponent(i) );
		}
	}

	for ( size_t  i = 0  ;  i < this -> n_unknown_player()  ;  i ++ )
	{
		check_list.push_back( this -> unknown_player(i) );
	}


	for ( long  n = 0  ;  n <= max_step  ;  n ++ )
	{
		double	kickable_max_dist
				= sserver_param().PLAYER_SPEED_MAX()
				    * static_cast<double>(n);

		list<Player_Reference>::iterator  it = check_list.begin();

		D2_Vector	ball_coord = this -> ball_predict_point( n );

		while ( it != check_list.end() )
		{
			try
			{
				double	d;
				d = ((*it).coordinate() - ball_coord).r();

				if ( d - (sserver_param().kickable_distance())
				     <= kickable_max_dist + 0.1 )
				{
					return( n );
				}
				else
				{
					double	k = sserver_param()
						      .PLAYER_SPEED_MAX()
							* static_cast<double>
							  ( max_step - n );

					double	b = sserver_param()
						      .BALL_SPEED_MAX()
							* static_cast<double>
							  ( max_step - n );

					if ( d >= k + b + 5.0 )
					{
						throw exception();
					}
				}

				it ++;
			}
			catch(...)
			{
				list<Player_Reference>::iterator  er;
				er = it;
				it ++;

				check_list.erase(er);
			}
		}
	}

	cache_value = max_step + 1;

	return( cache_value );
}


static
long   player_ball_accessable_step( const Field_Recog_Interface &  f ,
				    const Player_Reference &  pl ,
				    long  step )
{
	assert( step >= 0 );

	const SServer_Param &	param = f.sserver_param();

	D2_Vector	rel = (f.ball_predict_point( step )
			       - param.player_basic_point_n_step_later
				 ( pl.coordinate() , pl.velocity() , step ))
			       .rotate( - pl.body_angle() );

	long	spend_time = 0;

	Angle	diff_angle = rel.theta().normalize().abs();
	Angle	cone_angle = Angle::atan2( param.kickable_distance() ,
					   rel.r() ).normalize().abs();

	if ( cone_angle < Degree( 2.0 ) )
	{
		cone_angle = Degree( 2.0 );
	}

	if ( pl.body_angle().accuracy_check( 0 )
	  && diff_angle >= cone_angle )
	{
		spend_time += param.n_step_to_turn_body
			       ( diff_angle , pl.velocity().r() );
	}


	double	dist = rel.r() - param.kickable_distance();

	if ( dist > 0.0 )
	{
		spend_time += param.n_step_to_reach( dist );
	}

	return( spend_time );
}


#include  "d2_straight_line.h"
#include  <algorithm>
#include  <climits>

static
long   player_ball_access_step( const Field_Recog_Interface &  f ,
				const Player_Reference &  pl ,
				long  max_step )
{
	const	long	strict_calculate_step_threshold = 5;

	//
	// if can't access in max_step, return max_step + 1
	//
	if ( max_step != -1
	  && (f.ball().coordinate() - pl.coordinate()).r()
	     > (f.sserver_param().PLAYER_SPEED_MAX()
		+ f.sserver_param().BALL_SPEED_MAX()) * max_step
	       + f.sserver_param().kickable_distance() )
	{
		return( max_step + 1 );
	}


	long  t;

	//
	// for a few step, clucluate strictly
	//
	for ( t = 0  ;  (max_step == -1 || t <= max_step)
			&&  t <= strict_calculate_step_threshold  ;  t ++ )
	{
		if ( player_ball_accessable_step( f , pl , t ) <= t )
		{
			return( t );
		}
	}


	//
	// calculate step count or set t_min & max
	//
	long	t_min;
	long	t_max;

	// if ball is getting far away
	if ( (f.ball().velocity().theta()
	      - (f.self().coordinate() - f.ball().coordinate()).theta())
	     .normalize().abs() <= Degree( 90.0 ) )
	{
		t_min = std::max( static_cast<long>
				  ((f.ball().coordinate()
				    - f.self().coordinate()).r()
				   / f.sserver_param().PLAYER_SPEED_MAX()
				   - 1 ) ,
				  strict_calculate_step_threshold + 1 );
		// XXX
		t_max = std::max( static_cast<long>( 200 ) , t_min );
	}
	else
	{
		D2_Straight_Line  ball_line( f.ball().coordinate() ,
					     f.ball().velocity().theta() );


		double	ball_project_point_dist
			= (f.ball().coordinate()
			   - ball_line.project_point( pl.coordinate() )).r();

		double	project_ball_reach_step
			= f.sserver_param().n_step_for_ball_move
			  ( ball_project_point_dist ,
			    f.ball().velocity().r() );

		if ( project_ball_reach_step
		     > static_cast<double>(LONG_MAX - 1) )
		{
			for ( ;  (max_step == -1) || (t <= max_step)  ;  t ++ )
			{
				if ( player_ball_accessable_step( f , pl , t )
				     <= t )
				{
					return( t );
				}
			}

			return( t );
		}

		if ( project_ball_reach_step < 0.0 )
		{
			t_min
			= std::max( static_cast<long>
				    ( (f.ball().coordinate()
				      - f.self().coordinate()).r()
				     / f.sserver_param().PLAYER_SPEED_MAX()
				      - 1 ) ,
				    strict_calculate_step_threshold + 1 );
			// XXX
			t_max = std::max( static_cast<long>( 200 ) , t_min );
		}
		else
		{
			long	proj_ball_reach_step_min
				  = static_cast<long>(project_ball_reach_step);
			long	proj_ball_reach_step_max
				  = proj_ball_reach_step_min + 1;

			long	proj_access_step_min
				  = player_ball_accessable_step
				    ( f , pl , proj_ball_reach_step_min );
			long	proj_access_step_max
				  = player_ball_accessable_step
				    ( f , pl , proj_ball_reach_step_max );

			if ( proj_access_step_min <= proj_ball_reach_step_min )
			{
				t_min = strict_calculate_step_threshold + 1;
				t_max = std::max( proj_access_step_min ,
						  t_min );
			}
			else
			{
				if ( proj_access_step_max
				     <= proj_ball_reach_step_max )
				{
					return( proj_ball_reach_step_max );
				}
				else
				{
					t_min
					  = std::max
					    ( proj_ball_reach_step_max + 1 ,
					      strict_calculate_step_threshold
									 + 1 );

					if ( max_step == -1 )
					{
						t_max
						  = proj_ball_reach_step_max
							+ 50;

						while
						  ( player_ball_accessable_step
						    ( f , pl , t_max )
						    > t_max )
						{
							t_max += 50;
						}
					}
					else
					{
						t_max = max_step + 1;
					}

					t_max
					  = std::max
					    ( strict_calculate_step_threshold
									 + 1 ,
					      t_max );

					t_max = std::max( t_max , t_min );
				}
			}
		}
	}


	assert( t_min >= strict_calculate_step_threshold + 1 );
	assert( t_max >= t_min );

	for(;;)
	{
		long	medium = (t_min + t_max) / 2;

		if ( medium == t_min )
		{
			if ( player_ball_accessable_step
			     ( f , pl , medium ) <= medium )
			{
				return( t_min );
			}
			else
			{
				return( t_max );
			}
		}

		if ( player_ball_accessable_step( f , pl , medium ) <= medium )
		{
			t_max = medium;
		}
		else
		{
			t_min = medium;
		}
	}
}


void   Field_Recog_Interface::ball_access_step( const Player_Set &  players ,
						long *  self_access_step ,
						long *  teammate_access_step ,
						long *  opponent_access_step ,
						long *  unknown_access_step ,
						long  max_step ,
						long  acc_level ) const
{
	if ( teammate_access_step )
	{
		*teammate_access_step = max_step;
	}

	if ( opponent_access_step )
	{
		*opponent_access_step = max_step;
	}

	if ( unknown_access_step )
	{
		*unknown_access_step = max_step;
	}


	for ( Player_Set::iterator  it = players.begin()  ;
	      it != players.end()  ;  it ++ )
	{
		if ( it -> self_player() )
		{
			if ( ! self_access_step )
			{
				continue;
			}

			*self_access_step = player_ball_access_step
						    ( *this , *it , max_step );

		}
		else if ( it -> side() == S_Side::Our_Side )
		{
			if ( ! teammate_access_step
			  || *teammate_access_step == 0
			  || ! (it -> coordinate()
					.accuracy_check( acc_level ) ) )
			{
				continue;
			}


			long	teammate_max_step = *teammate_access_step;
			if ( teammate_max_step != -1 )
			{
				teammate_max_step -= -1;
			}


			long	step = player_ball_access_step
				       ( *this , *it , teammate_max_step );

			if ( *teammate_access_step == -1
			     || step < *teammate_access_step )
			{
				*teammate_access_step = step;
			}
		}
		else if ( it -> side() == S_Side::Opponent_Side )
		{
			if ( ! opponent_access_step
			  || *opponent_access_step == 0
			  || ! (it -> coordinate()
					.accuracy_check( acc_level ) ) )
			{
				continue;
			}


			long	opponent_max_step = *opponent_access_step;
			if ( opponent_max_step != -1 )
			{
				opponent_max_step -= -1;
			}


			long	step = player_ball_access_step
				       ( *this , *it , opponent_max_step );

			if ( *opponent_access_step == -1
			     || step < *opponent_access_step )
			{
				*opponent_access_step = step;
			}
		}
		else // side == Unknown
		{
			if ( ! unknown_access_step
			  || *unknown_access_step == 0
			  || ! (it -> coordinate()
					.accuracy_check( acc_level ) ) )
			{
				continue;
			}


			long	unknown_max_step = *unknown_access_step;
			if ( unknown_max_step != -1 )
			{
				unknown_max_step -= -1;
			}


			long	step = player_ball_access_step
				       ( *this , *it , unknown_max_step );

			if ( *unknown_access_step == -1
			     || step < *unknown_access_step )
			{
				*unknown_access_step = step;
			}
		}
	}
}


void   Field_Recog_Interface::ball_access_step( long *  self_access_step ,
						long *  teammate_access_step ,
						long *  opponent_access_step ,
						long *  unknown_access_step ,
						long  max_step ,
						long  acc_level ) const
{
	//
	// cache
	//
	static	SServer_Time	cache_time( -1 );
	static	long		cache_acc_level = 0;
	static	long		cache_max_step = 0;

	static	bool		cache_have_self     = false;
	static	bool		cache_have_teammate = false;
	static	bool		cache_have_opponent = false;
	static	bool		cache_have_unknown  = false;

	static	long		cache_self_access_step = 0;
	static	long		cache_teammate_access_step = 0;
	static	long		cache_opponent_access_step = 0;
	static	long		cache_unknown_access_step = 0;


	// XXX
	bool	use_cache = (this -> current_time() == cache_time
			     && cache_acc_level == acc_level
			     && cache_max_step == max_step);

	if ( ! use_cache )
	{
		cache_have_self     = false;
		cache_have_teammate = false;
		cache_have_opponent = false;
		cache_have_unknown  = false;
	}



	// XXX
	if ( self_access_step )
	{
		if ( ! cache_have_self )
		{
			cache_self_access_step
				= player_ball_access_step( *this ,
							   this -> self() ,
							   max_step );

			cache_have_self = true;
		}

		(*self_access_step) = cache_self_access_step;
	}


	long	search_max;

	if ( teammate_access_step )
	{
		if ( ! cache_have_teammate )
		{
			cache_teammate_access_step = -1;
			search_max = max_step;
			for ( int  i = 1  ;  i <= MAX_PLAYER  ;  i ++ )
			{
				if ( i == this -> self().player_number() )
				{
					continue;
				}

				if ( this -> teammate(i).coordinate()
					     .accuracy_check( acc_level ) )
				{
					long	step = player_ball_access_step
						       ( *this ,
							 this -> teammate(i) ,
							 search_max );


					if ( cache_teammate_access_step == -1
					  || step < cache_teammate_access_step)
					{
						cache_teammate_access_step
							= step;

						if ( step == 0 )
						{
							break;
						}

						search_max = step - 1;
					}
				}
			}


			for ( size_t  i = 0  ;
			      i < this -> n_unknown_teammate()  ;
			      i ++ )
			{
				if ( this -> unknown_teammate(i).coordinate()
					     .accuracy_check( acc_level ) )
				{
					long	step = player_ball_access_step
						( *this ,
						  this -> unknown_teammate(i) ,
						  search_max );

					if ( cache_teammate_access_step == -1
					  || step
					     < cache_teammate_access_step )
					{
						cache_teammate_access_step
							= step;

						if ( step == 0 )
						{
							break;
						}

						search_max = step - 1;
					}
				}
			}

			cache_have_teammate = true;
		}

		*teammate_access_step = cache_teammate_access_step;
	}


	if ( opponent_access_step )
	{
		if ( ! cache_have_opponent )
		{
			cache_opponent_access_step = -1;
			search_max = max_step;
			for ( int  i = 1  ;  i <= MAX_PLAYER  ;  i ++ )
			{
				if ( this -> opponent(i).coordinate()
					     .accuracy_check( acc_level ) )
				{
					long	step = player_ball_access_step
						       ( *this ,
							 this -> opponent(i) ,
							 search_max );


					if ( cache_opponent_access_step == -1
					  || step < cache_opponent_access_step)
					{
						cache_opponent_access_step
							= step;

						if ( step == 0 )
						{
							break;
						}

						search_max = step - 1;
					}
				}
			}


			for ( size_t  i = 0  ;
			      i < this -> n_unknown_opponent()  ;
			      i ++ )
			{
				if ( this -> unknown_opponent(i).coordinate()
					     .accuracy_check( acc_level ) )
				{
					long	step = player_ball_access_step
						( *this ,
						  this -> unknown_opponent(i) ,
						  search_max );

					if ( cache_opponent_access_step == -1
					  || step
					     < cache_opponent_access_step )
					{
						cache_opponent_access_step
							= step;

						if ( step == 0 )
						{
							break;
						}

						search_max = step - 1;
					}
				}
			}

			cache_have_opponent = true;
		}

		*opponent_access_step = cache_opponent_access_step;
	}


	if ( unknown_access_step )
	{
		if ( ! cache_have_unknown )
		{
			cache_unknown_access_step = -1;
			search_max = max_step;
			for ( size_t  i = 0  ;
			      i < this -> n_unknown_player()  ;  i ++ )
			{
				if ( this -> unknown_player(i).coordinate()
					     .accuracy_check( acc_level ) )
				{
					long	step
						= player_ball_access_step
						  ( *this ,
						    this -> unknown_player(i) ,
						    search_max );

					if ( cache_unknown_access_step == -1
					  || step < cache_unknown_access_step )
					{
						cache_unknown_access_step
							= step;

						if ( step == 0 )
						{
							break;
						}

						search_max = step - 1;
					}
				}
			}

			cache_have_unknown = true;
		}

		*unknown_access_step = cache_unknown_access_step;
	}

	cache_time      = this -> current_time();
	cache_acc_level = acc_level;
	cache_max_step  = max_step;
}

// XXX
void   Field_Recog_Interface::set_target_teammate( int  num ) const
{
	this -> target_teammate = num;
	this -> have_target_teammate_flag = true;
}

// XXX
bool   Field_Recog_Interface::have_target_teammate() const
{
	return( this -> have_target_teammate_flag );
}

// XXX
int   Field_Recog_Interface::get_target_teammate() const
{
	return( this -> target_teammate );
}


// XXX
void   Field_Recog_Interface::set_target_point( const D2_Vector &  point )
									const
{
	this -> target_point = point;
	this -> have_target_point_flag = true;
}

// XXX
bool   Field_Recog_Interface::have_target_point() const
{
	return( this -> have_target_point_flag );
}

// XXX
D2_Vector  Field_Recog_Interface::get_target_point() const
{
	return( this -> target_point );
}


// XXX
void   Field_Recog_Interface::set_message( const std::string &  mes ) const
{
	this -> message = mes;
	this -> have_message_flag = true;
}

// XXX
bool   Field_Recog_Interface::have_message() const
{
	return( this -> have_message_flag );
}

// XXX
std::string  Field_Recog_Interface::get_message() const
{
	return( this -> message );
}
