#include  "sight_info_analyzer.h"
#include  "d2_universal_region.h"
#include  "d2_composite_straight_line_divided_region_interface.h"
#include  "fixed_sobject_location.h"
#include  "quantize.h"
#include  "field_recog_abstract.h"
#include  "math_extention.h"
#include  <vector>
#include  <utility>
#include  <algorithm>

using namespace std;

//
// external interface
//
Sight_Info_Analyzer::Sight_Info_Analyzer
	   ( const ref_count_ptr<const Server_Raw_Sight_Info> &  sight ,
	     const SServer_Param *  param ,
	     const Angle_Range &  neck_angle )
	: sight( sight ) ,
	  param( param ) ,
	  neck_angle( neck_angle ) ,
	  view_angle_cached( false ) , view_angle() ,
	  body_angle_cached( false ) , body_angle() ,
	  self_coordinate_cached( false ) , self_coordinate() ,
	  self_velocity_cached( false ) , self_velocity() ,
	  ball_info_cached( false ) , ball_info_valid( false ) , ball_info() ,
	  player_info_cached( false ) , player_info_valid( false ) ,
	  teammate_info( MAX_PLAYER ) , opponent_info( MAX_PLAYER ) ,
	  unknown_teammate_info() , unknown_opponent_info() ,
	  unknown_player_info() ,
	  relative_field_marker_info_cached( false ) ,
	  relative_field_marker_info() ,
	  unknown_marker_info()
{
}

Sight_Info_Analyzer::~Sight_Info_Analyzer()
{
}


bool   Sight_Info_Analyzer::get_body_angle_range( Angle_Info *  result )
	const
{
	if ( ! (this -> require_body_angle()) )
	{
		result -> set( Radian(0.0) , Radian(0.0) , Radian(0.0) );
		return( false );
	}
	else
	{
		result -> set( this -> body_angle );
		return( true );
	}
}


bool   Sight_Info_Analyzer::get_self_coordinate( Region_Info *  result )
	const
{
	if ( ! (this -> require_self_coordinate()) )
	{
		result -> set( D2_Region::empty_region() ,
			       D2_Vector::origin() );
		return( false );
	}
	else
	{
		result -> set( this -> self_coordinate );
		return( true );
	}
}


#if 1
bool   Sight_Info_Analyzer::get_self_velocity( Region_Info *  result )
	const
{
	if ( ! (this -> require_self_velocity()) )
	{
		result -> set( D2_Region::empty_region() ,
			       D2_Vector::origin() );
		return( false );
	}
	else
	{
		result -> set( this -> self_velocity );
		return( true );
	}
}
#endif


void   Sight_Info_Analyzer::get_ball_info
				( Field_Recog_Abstract::Ball_Info *  result )
	const
{
	if ( ! (this -> require_ball_info()) )
	{
		result -> coordinate.reset();
		result -> velocity.reset();
		result -> in_sight = false;
	}
	else
	{
		*result = ball_info;
	}
}

bool   Sight_Info_Analyzer::get_player_info(
	  std::vector<Field_Recog_Abstract::Player_Info> *  teammate ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  opponent ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  unknown_teammate ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  unknown_opponent ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  unknown_player )
	const
{
	if ( ! (this -> require_player_info()) )
	{
		return( false );
	}
	else
	{
		*teammate         = teammate_info;
		*opponent         = opponent_info;
		*unknown_teammate = unknown_teammate_info;
		*unknown_opponent = unknown_opponent_info;
		*unknown_player   = unknown_player_info;

		return( true );
	}
}

void   Sight_Info_Analyzer::get_goalie_number( int *  our_team ,
					       int *  opponent_team )
	const
{
	if ( our_team )
	{
		*our_team = -1;
	}

	if ( opponent_team )
	{
		*opponent_team = -1;
	}

	for ( size_t  i = 0  ;  i < sight -> player.size()  ;  i ++ )
	{
		const Raw_Player_Info &	pl = sight -> player[i];

		if ( pl.have_goalie_info && pl.goalie )
		{
			if ( pl.entity.side == S_Side::Our_Side && our_team )
			{
				*our_team = pl.entity.player_number;
			}
			else if ( pl.entity.side == S_Side::Opponent_Side
			       && opponent_team )
			{
				*opponent_team = pl.entity.player_number;
			}
		}
	}
}



//
// internal method
//

//
// view angle
//
bool   Sight_Info_Analyzer::require_view_angle()
	const
{
	if ( ! this -> view_angle_cached )
	{
		this -> calc_view_angle_from_line( &(this -> view_angle) );

		this -> view_angle_cached = true;
	}

	return( this -> view_angle.valid() );
}

bool   Sight_Info_Analyzer::calc_view_angle_from_line( Angle_Info *  result )
	const
{
	// return self angle [- PI , + PI)

	if ( sight -> line.size() == 0 )
	{
		result -> reset();

		return( false );
	}

	const Angle	line_theta = sight -> line[0].relative.direction;

	Angle	angle;
	Angle	err_min;
	Angle	err_max;

	if ( line_theta < Degree( 0.0 ) )
	{
		angle = Degree( - 90.0 ) - line_theta;
	}
	else
	{
		angle = Degree( + 90.0 ) - line_theta;
	}


#ifdef OLD_ANGLE_ROUNDING_POLICY
	if ( line_theta < Degree( - 0.5 ) )
	{
		err_min = Degree(   0.0 );
		err_max = Degree( + 1.0 );
	}
	else if ( line_theta > Degree( + 0.5 ) )
	{
		err_min = Degree( - 1.0 );
		err_max = Degree(   0.0 );
	}
	else
	{
		err_min = Degree( - 1.0 );
		err_max = Degree( + 1.0 );
	}
#else
	err_min = Degree( - 0.5 );
	err_max = Degree( + 0.5 );
#endif


	switch( sight -> line[0].entity.which_line )
	{
	case SObject_Line_Identifier::Opponent_Goal_Line:
		break;

	case SObject_Line_Identifier::Left_Wing_Line:
		angle += Degree( 90.0 );
		break;

	case SObject_Line_Identifier::Our_Goal_Line:
		angle += Degree( 180.0 );
		break;

	case SObject_Line_Identifier::Right_Wing_Line:
		angle += Degree( 270.0 );
		break;
	}

	if ( sight -> line.size() >= 2 )
	{
		angle += Degree( 180.0 );
	}

	angle = angle.normalize();

	result -> set( angle + err_min , angle + err_max ,
		       angle + (err_max + err_min) / 2.0 );

	return( true );
}


//
// body angle
//
bool   Sight_Info_Analyzer::require_body_angle()
	const
{
	if ( ! this -> body_angle_cached )
	{
		this -> calc_body_angle( &(this -> body_angle) );

		this -> body_angle_cached = true;
	}


	return( this -> body_angle.valid() );
}

bool   Sight_Info_Analyzer::calc_body_angle( Angle_Info *  result )
	const
{
	// return self absolute body angle [- PI , + PI)

	if ( ! (require_view_angle()) )
	{
		result -> reset();

		return( false );
	}

	Angle	neck_angle_error
		= ((neck_angle.max() - neck_angle.min()) / 2.0)
		  .normalize().abs();

	Angle	view_angle_error
		= ((view_angle.range().max() - view_angle.range().min()) / 2.0)
		  .normalize().abs();

	Angle	err = view_angle_error + neck_angle_error;

	Angle	med = (view_angle.range().median() - neck_angle.median())
		      .normalize();

	result -> set( med - err ,  med + err ,  med );

	return( true );
}


//
// self_coordinate
//
bool   Sight_Info_Analyzer::require_self_coordinate()
	const
{
	if ( ! this -> self_coordinate_cached )
	{
		this -> calc_self_coordinate_by_composite
					( &(this -> self_coordinate) );

		this -> self_coordinate_cached = true;
	}

	return( this -> self_coordinate.valid() );
}


bool   Sight_Info_Analyzer::calc_self_coordinate_by_composite
					( Region_Info *  result )
	const
{
#if PRECISION_LEVEL >= 1
	D2_Region	region;

	// region cut
	if ( ! calc_self_coordinate_by_region_cut( &region ) )
	{
		return( false );
	}

	result -> set( region , region.barycenter() );


	// more region cuttings
	for ( int  i = 1  ;  i <= PRECISION_LEVEL - 1  ;  i ++ )
	{
		if ( ! calc_self_coordinate_by_region_cut( &region ) )
		{
			return( false );
		}

		result -> set( region , region.barycenter() );
	}


	if ( unknown_marker_info.size() >= 1 )
	{
		static	Fixed_SObject_Location_Translator	loc;

		D2_Vector	closest_object_point;
		loc.get_closest_object_location
			( &closest_object_point ,
			  region.barycenter() ,
			  unknown_marker_info[0].ident.entity.marker_flag );

		Relative_Field_Marker_Info	closest_marker;
		closest_marker = unknown_marker_info[0];
		closest_marker.absolute_point = closest_object_point;

		D2_Vector	point
				= object_coordinate_info_to_self_coordinate
				  ( closest_marker , view_angle );

		result -> set( region , point );
	}
#else
	D2_Vector	sum_point;

	// region cut
	if ( ! calc_self_coordinate_by_sum( &sum_point ) )
	{
		return( false );
	}

	result -> set( D2_Region::universal_region() , sum_point );


	if ( unknown_marker_info.size() >= 1 )
	{
		static	Fixed_SObject_Location_Translator	loc;

		D2_Vector	closest_object_point;
		loc.get_closest_object_location
			( &closest_object_point ,
			  sum_point ,
			  unknown_marker_info[0].ident.entity.marker_flag );

		Relative_Field_Marker_Info	closest_marker;
		closest_marker = unknown_marker_info[0];
		closest_marker.absolute_point = closest_object_point;

		D2_Vector	point
				= object_coordinate_info_to_self_coordinate
				  ( closest_marker , view_angle );

		result -> set( D2_Region::universal_region() , point );
	}
#endif

	return( true );
}

bool   Sight_Info_Analyzer::calc_self_coordinate_by_sum( D2_Vector *  res_vec )
	const
{
	this -> require_relative_field_marker_info();

	if ( relative_field_marker_info.size() == 0
	  || ! require_view_angle() )
	{
		res_vec -> set( 0.0 , 0.0 );

		return( false );
	}

	// these values are generated by GA
	const	double	one_flag_threshold = 30.3;
	const	double	n_flag_decay       = 0.781245;
	const	double	dist_decay         = 0.0114752;

	if ( relative_field_marker_info[0].relative_dist_range.median()
	     < one_flag_threshold )
	{
		(*res_vec) = object_coordinate_info_to_self_coordinate
				 ( relative_field_marker_info[0] ,
				   view_angle );

		return( true );
	}

	D2_Vector	vec( 0.0 , 0.0 );
	double		n = 0.0;
	double		c = 1.0;

	for ( size_t  i = 0  ;  i < relative_field_marker_info.size()  ;
	      i ++ )
	{
		D2_Vector	v = object_coordinate_info_to_self_coordinate
				    ( relative_field_marker_info[i] ,
				      view_angle );

		double	e = std::exp( - dist_decay
				      * relative_field_marker_info[i]
					.relative_dist_range.median() );
		vec += e * c * v;

		n += e * c;

		c *= n_flag_decay;
	}

	(*res_vec) = vec / n;

	return( true );
}

bool   Sight_Info_Analyzer::calc_self_coordinate_by_region_cut
				( D2_Region *  res_reg )
	const
{
	const	Angle	EPS = Degree( 1.0e-6 );

	this -> require_relative_field_marker_info();

	const std::vector<Sight_Info_Analyzer::Relative_Field_Marker_Info> &
		objects = relative_field_marker_info;

	if ( ! this -> require_view_angle() || objects.size() == 0 )
	{
		*res_reg = D2_Region::empty_region();

		return( false );
	}

	D2_Composite_Straight_Line_Divided_Region_Interface	region;

	for ( size_t  i = 0  ;  i < objects.size()  ;  i ++ )
	{
		const D2_Vector	&	coord = objects[i].absolute_point;

		Angle_Range	point_abs_angle
				= objects[i].relative_angle_range
				  + view_angle.range();

		region.add( coord ,
			    point_abs_angle.max() + EPS ,
			    coord + D2_Vector( D2_Vector::Pole , 100.0 ,
					       point_abs_angle.median()
						 + Degree( 90.0 ) ) );

		region.add( coord ,
			    point_abs_angle.min() - EPS ,
			    coord + D2_Vector( D2_Vector::Pole , 100.0 ,
					       point_abs_angle.median()
						 - Degree( 90.0 ) ) );

		double	r_min = objects[i].relative_dist_range.min() - 0.01;
		double	r_max = objects[i].relative_dist_range.max() + 0.01;

		D2_Vector	max_point_vector( D2_Vector::Pole ,
						  r_max ,
						  point_abs_angle.median()
						    + Degree( 180.0 ) );

		D2_Vector	max_point = coord + max_point_vector;

		region.add( max_point ,
			    point_abs_angle.median() + Degree( 90.0 ) ,
			    coord );


		D2_Vector	min_point_vector_1( D2_Vector::Pole ,
						    r_min ,
						    point_abs_angle.median()
						     + Degree( 180.0 )
						     + Degree( 2.0 + 0.01 ) );

		D2_Vector	min_point_vector_2( D2_Vector::Pole ,
						    r_min ,
						    point_abs_angle.median()
						      + Degree( 180.0 )
						      - Degree( 2.0 + 0.01 ) );

		D2_Vector	min_point_1 = coord + min_point_vector_1;
		D2_Vector	min_point_2 = coord + min_point_vector_2;

		region.add( min_point_1 , min_point_2 , max_point );
	}

	if ( ! (region.closed()) )
	{
		*res_reg = D2_Region::empty_region();

		return( false );
	}

	*res_reg = region.get_region();

#if 1
	//
	// calculate new body angle
	//
	std::vector<D2_Vector>	point_list = region.point_list();

	if ( point_list.size() == 0 )
	{
		return( true );
	}

	Angle_Range	new_view_angle_range = this -> view_angle.range();

# undef	VIEW_ANGLE_DEBUG

# ifdef	VIEW_ANGLE_DEBUG
	cout << "view_angle: "
	     << this -> view_angle.range().min().degree() << ", "
	     << this -> view_angle.range().max().degree() << endl;
# endif

	for ( size_t  obj = 0  ;  obj < objects.size()  ;  obj ++ )
	try
	{
		Angle	absolute_dir_base
			= (objects[obj].absolute_point - region.barycenter())
			  .theta().normalize();

		Angle	min_error = Angle::origin();
		Angle	max_error = Angle::origin();

		for ( size_t  i = 0  ;  i < point_list.size()  ;  i ++ )
		{
			Angle	dir = ((objects[obj].absolute_point
					- point_list[i]).theta()
				       - absolute_dir_base).normalize();

			if ( dir < min_error )
			{
				min_error = dir;
			}
			else if ( dir > max_error )
			{
				max_error = dir;
			}
		}


		const Angle_Range	absolute_dir_error_range( min_error ,
								  max_error );

		const Angle		relative_dir_base
					= objects[obj].relative_angle_range
					  .median();

# ifdef	VIEW_ANGLE_DEBUG
		cout << "objects[" << obj << "].relative_angle_range = "
		     << objects[obj].relative_angle_range.min().degree()
		     << ", "
		     << objects[obj].relative_angle_range.max().degree()
		     << endl;

		cout << "relative_dir_base = "
		     << relative_dir_base.degree() << endl;
# endif

		const Angle_Range	relative_dir_error_range
					= objects[obj].relative_angle_range
					  - relative_dir_base;

		const Angle		restrict_base
					= (absolute_dir_base
					   - relative_dir_base
					   - new_view_angle_range.median())
					   .normalize()
					  + new_view_angle_range.median();

# ifdef	VIEW_ANGLE_DEBUG
		cout << "restrict_base = " << restrict_base.degree() << endl;

		cout << "absolute_dir_error_range = "
		     << absolute_dir_error_range.min().degree() << ", "
		     << absolute_dir_error_range.max().degree() << endl;

		cout << "relative_dir_error_range = "
		     << relative_dir_error_range.min().degree() << ", "
		     << relative_dir_error_range.max().degree() << endl;

		cout << "(absolute_dir_error_range "
			"- relative_dir_error_range) = "
		     << (absolute_dir_error_range
			 - relative_dir_error_range).min().degree() << ", "
		     << (absolute_dir_error_range
			 - relative_dir_error_range).max().degree() << endl;
# endif

		const Angle_Range	restrict_range
					= (absolute_dir_error_range
					   - relative_dir_error_range)
					  + restrict_base;

# ifdef	VIEW_ANGLE_DEBUG
		cout << "restrict_range[" << obj << "] = "
		     << restrict_range.min().degree() << ","
		     << restrict_range.max().degree() << endl;
# endif

		// XXX
		new_view_angle_range &= restrict_range;

# ifdef	VIEW_ANGLE_DEBUG
		cout << "new_view_angle_range = "
		     << new_view_angle_range.min().degree() << ", "
		     << new_view_angle_range.max().degree() << endl;
		cout << endl;
# endif

	} catch(...){}


	if ( new_view_angle_range.median() >= Degree( 180.0 ) )
	{
		new_view_angle_range -= Degree( 360.0 );
	}
	else if ( new_view_angle_range.median() < Degree( - 180.0 ) )
	{
		new_view_angle_range += Degree( 360.0 );
	}

	this -> view_angle.set( new_view_angle_range ,
				new_view_angle_range.median() );

	this -> calc_body_angle( &(this -> body_angle) );

# ifdef	VIEW_ANGLE_DEBUG
	cout << "!view_angle: "
	     << this -> view_angle.range().min().degree() << ", "
	     << this -> view_angle.range().max().degree() << endl
	     << endl;
# endif
#endif

	return( true );
}

// protected static
D2_Vector  Sight_Info_Analyzer::object_coordinate_info_to_self_coordinate
				( const Relative_Field_Marker_Info &  marker ,
				  const Angle_Info &  view_ang )
{
	D2_Vector	vec( D2_Vector::Pole ,
			     marker.relative_dist_range.median() ,
			     marker.relative_angle_range.median()
			     + view_ang.point() + Degree( 180.0 ) );

	return( marker.absolute_point + vec );
}


//
// self_velocity
//
bool   Sight_Info_Analyzer::require_self_velocity()
	const
{
	// XXX
	return( this -> self_velocity_cached );
}

void   Sight_Info_Analyzer::advice_self_velocity( const Region_Info &  v )
{
	// XXX
	this -> self_velocity_cached = true;

	self_velocity = v;
}


//
// relative field marker info
//
void   Sight_Info_Analyzer::require_relative_field_marker_info()
	const
{
	if ( relative_field_marker_info_cached )
	{
		return;
	}

	this -> relative_field_marker_info_cached = true;

	//
	// sort markers by distance
	//
	std::vector< std::pair<int, double> >	marker_indx;
	std::vector< std::pair<int, double> >	unknown_marker_indx;

	// set distance information for sorting
	for ( size_t  i = 0  ;  i < (sight -> field_marker.size())  ;  i ++ )
	{
		// alias
		const Raw_Field_Marker_Info &
			marker = sight -> field_marker[i];

		if ( marker.relative.have_distance
		  && marker.relative.have_direction )
		{
			if ( marker.entity.location_known )
			{
				marker_indx.push_back
				 ( std::pair<int,double>
				   ( i , marker.relative.distance ) );
			}
			else
			{
				unknown_marker_indx.push_back
				 ( std::pair<int,double>
				   ( i , marker.relative.distance ) );
			}
		}
	}

	// sort function
	class  Compare_by_Distance
	{
	public:
		static	bool	compare( const std::pair<int, double> &  a ,
					 const std::pair<int, double> &  b )
		{
			return( a.second < b.second );
		}
	};

	// sort by distance
	std::sort( marker_indx.begin() , marker_indx.end() ,
		   Compare_by_Distance::compare );

	std::sort( unknown_marker_indx.begin() , unknown_marker_indx.end() ,
		   Compare_by_Distance::compare );


	//
	// push to relative_field_marker_info
	//
	for ( size_t  i = 0  ;  i < marker_indx.size()  ;  i ++ )
	{
		// alias
		const Raw_Field_Marker_Info &
			marker = sight -> field_marker[marker_indx[i].first];

		Relative_Field_Marker_Info	info;

		info.absolute_point = marker.entity.location;
		info.ident = marker;

		if ( marker.relative.have_direction
		  && marker.relative.have_distance )
		{
			// angle
			server_angle_to_angle_range
				( &(info.relative_angle_range) ,
				  marker.relative.direction );

			// distance
			double	r_min;
			double	r_max;

			far_object_unquantized_distance
				( *param ,
				  marker.relative.distance , false ,
				  &r_min , &r_max );

			info.relative_dist_range.set( r_min , r_max );


			// push
			relative_field_marker_info.push_back( info );
		}
	}


	//
	// push to unknown_marker_info
	//
	for ( size_t  i = 0  ;  i < unknown_marker_indx.size()  ;  i ++ )
	{
		const Raw_Field_Marker_Info &	marker
			= sight -> field_marker[unknown_marker_indx[i].first];

		Relative_Field_Marker_Info	info;

		info.absolute_point.set( 0.0 , 0.0 );
		info.ident = marker;

		if ( marker.relative.have_direction
		  && marker.relative.have_distance )
		{
			// angle
			server_angle_to_angle_range
				( &(info.relative_angle_range) ,
				  marker.relative.direction );

			// distance
			double	r_min;
			double	r_max;

			far_object_unquantized_distance
				( *param ,
				  marker.relative.distance , false ,
				  &r_min , &r_max );

			info.relative_dist_range.set( r_min , r_max );


			// push
			unknown_marker_info.push_back( info );
		}
	}
}


// protected static
bool   Sight_Info_Analyzer::relative_sight_to_relative_point
		    ( D2_Vector *  point ,
		      const Raw_SObject_Locational_Info &  relative )
{
	if ( relative.have_direction && relative.have_distance )
	{
		Angle_Range	direction_range( Radian(0.0) , Radian(0.0) );
		server_angle_to_angle_range( &direction_range ,
					     relative.direction );

		// XXX
		point -> set( D2_Vector::Pole ,
			      relative.distance , direction_range.median() );

		return( true );
	}
	else
	{
		return( false );
	}
}

// protected static
bool   Sight_Info_Analyzer::relative_sight_to_relative_velocity
		    ( D2_Vector *  vel ,
		      const Raw_SObject_Locational_Info &  relative )
{
	D2_Vector	relative_point;

	if ( ! (relative.have_d_direction && relative.have_d_distance)
	 ||  ! relative_sight_to_relative_point( &relative_point , relative ) )
	{
		return( false );
	}
	else
	{
		Angle_Range	d_direction_range( Radian(0.0) , Radian(0.0) );
		server_angle_to_angle_range( &d_direction_range ,
					     relative.d_direction ,
					     false );

		vel -> set( D2_Vector::XY ,
			    relative.d_distance ,
			    d_direction_range.median().radian()
			    * relative_point.r() );

		*vel = vel -> rotate( relative.direction /* XXX */ );

		return( true );
	}
}

// protected static
bool   Sight_Info_Analyzer::relative_sight_to_absolute_point_and_velocity
		    ( Region_Info *  coord ,  Region_Info *  vel ,
		      const Raw_SObject_Locational_Info &  relative ,
		      const Region_Info &  self_coord ,
		      const Angle_Info &  view_ang ,
#if 1
		      const Region_Info &  self_vel
#endif
		      )
{
	D2_Vector	relative_point;
	if ( relative_sight_to_relative_point( &relative_point , relative ) )
	{
		coord -> set( D2_Region::universal_region() /*XXX*/ ,
			      self_coord.point()
				+ relative_point.rotate( view_ang.point() ) );
	}
	else
	{
		coord -> reset();
		vel   -> reset();

		return( false );
	}


	D2_Vector	relative_vector;
	if ( relative_sight_to_relative_velocity( &relative_vector ,
						  relative ) )
	{
		vel -> set( D2_Region::universal_region() /*XXX*/ ,
			    relative_vector.rotate( view_ang.point() )
			    + self_vel.point()
			  );
	}
	else
	{
		vel -> reset();
	}

	return( true );
}

// protected static
void   Sight_Info_Analyzer::relative_sight_to_player_info
		    ( Region_Info *  coord ,  Region_Info *  vel ,
		      Angle_Info *  body_angle ,  Angle_Info *  face_angle ,
		      const Raw_SObject_Locational_Info &  relative ,
		      const Region_Info &  self_coord ,
		      const Angle_Info &  view_ang
#if 1
		      ,
		      const Region_Info &  self_vel )
#endif
{
#if 0
	relative_sight_to_absolute_point_and_velocity
	      ( coord , vel , relative , self_coord , view_ang );
#else
	relative_sight_to_absolute_point_and_velocity
	      ( coord , vel , relative , self_coord , view_ang , self_vel );
#endif

	if ( relative.have_body_direction )
	{
		Angle_Range	body_direction_range( Radian(0.0) ,
						      Radian(0.0) );
		server_angle_to_angle_range( &body_direction_range ,
					     relative.body_direction );

		Angle	min_err =   (body_direction_range.min()
				     - body_direction_range.median())
				  + (view_ang.range().min()
				     - view_ang.point());

		Angle	max_err =   (body_direction_range.max()
				     - body_direction_range.median())
				  + (view_ang.range().max()
				     - view_ang.point());

		Angle	point = (body_direction_range.median()
				 + view_ang.point()).normalize();

		body_angle -> set( point - min_err , point + max_err , point );
	}
	else
	{
		body_angle -> reset();
		face_angle -> reset();
		return;
	}

	if ( relative.have_face_direction )
	{
		Angle_Range	face_direction_range( Radian(0.0) ,
						      Radian(0.0) );
		server_angle_to_angle_range( &face_direction_range ,
					     relative.face_direction );

		Angle	min_err =   (face_direction_range.min()
				     - face_direction_range.median())
				  + (view_ang.range().min()
				     - view_ang.point())
				  + (body_angle -> range().min()
				     - body_angle -> point() );

		Angle	max_err =   (face_direction_range.max()
				     - face_direction_range.median())
				  + (view_ang.range().max()
				     - view_ang.point())
				  + (body_angle -> range().max()
				     - body_angle -> point() );

		Angle	point = (face_direction_range.median()
				 - body_angle -> point()
				 + view_ang.point()).normalize();

		face_angle -> set( point - min_err , point + max_err , point );
	}
	else
	{
		face_angle -> reset();
	}
}


// protected static
void   Sight_Info_Analyzer::server_angle_to_angle_range
	( Angle_Range *  range ,  const Angle &  angle ,  bool  do_normalize )
{
	Angle	err_min;
	Angle	err_max;

#ifdef OLD_ANGLE_ROUNDING_POLICY
	if ( angle < Degree( - 0.5 ) )
	{
		err_min = Degree( - 1.0 );
		err_max = Degree(   0.0 );
	}
	else if ( angle > Degree( + 0.5 ) )
	{
		err_min = Degree(   0.0 );
		err_max = Degree( + 1.0 );
	}
	else
	{
		err_min = Degree( - 1.0 );
		err_max = Degree( + 1.0 );
	}
#else
	err_min = Degree( - 0.5 );
	err_max = Degree( + 0.5 );
#endif

	Angle	point = angle;

	if ( do_normalize )
	{
		point = point.normalize();
	}

	range -> set( point + err_min , point + err_max );
}


//
// ball info
//
bool   Sight_Info_Analyzer::require_ball_info()
	const
{
	if ( ! this -> ball_info_cached )
	{
		this -> ball_info_valid
			= calc_ball_info( &(this -> ball_info) );

		this -> ball_info_cached = true;
	}

	return( this -> ball_info_valid );
}

bool   Sight_Info_Analyzer::calc_ball_info
				( Field_Recog_Abstract::Ball_Info *  result )
	const
{
	if ( sight -> ball.size() == 0
	  || (! this -> require_view_angle())
	  || (! this -> require_self_coordinate()) )
	{
		result -> coordinate.reset();
		result -> velocity.reset();
		result -> in_sight = false;

		return( false );
	}

	result -> in_sight = sight -> ball[0].relative.in_sight;

#if 1
	if ( this -> require_self_velocity() )
	{
		return( relative_sight_to_absolute_point_and_velocity
			( &(result -> coordinate) ,
			  &(result -> velocity) ,
			  sight -> ball[0].relative ,
			  self_coordinate , view_angle ,
			  self_velocity ) );
	}
	else
	{
		Region_Info	r( D2_Region::universal_region() ,
				   D2_Vector::origin() );

		return( relative_sight_to_absolute_point_and_velocity
			( &(result -> coordinate) ,
			  &(result -> velocity) ,
			  sight -> ball[0].relative ,
			  self_coordinate , view_angle ,
			  r ) );
	}
#else
	return( relative_sight_to_absolute_point_and_velocity
		( &(result -> coordinate) ,
		  &(result -> velocity) ,
		  sight -> ball[0].relative ,
		  self_coordinate , view_angle ) );
#endif
}


//
// player info
//
bool   Sight_Info_Analyzer::require_player_info()
	const
{
	if ( ! this -> player_info_cached )
	{
		player_info_valid
			= calc_player_info( &(this -> teammate_info) ,
					    &(this -> opponent_info) ,
					    &(this -> unknown_teammate_info) ,
					    &(this -> unknown_opponent_info) ,
					    &(this -> unknown_player_info) );

		this -> player_info_cached = true;
	}

	return( player_info_cached );
}

bool   Sight_Info_Analyzer::calc_player_info(
	  std::vector<Field_Recog_Abstract::Player_Info> *  teammate ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  opponent ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  unknown_teammate ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  unknown_opponent ,
	  std::vector<Field_Recog_Abstract::Player_Info> *  unknown_player )
	const
{
	for ( size_t  i = 0  ;  i < teammate -> size()  ;  i ++ )
	{
		(*teammate)[i].coordinate.reset();
		(*teammate)[i].velocity.reset();
		(*teammate)[i].body_angle.reset();
		(*teammate)[i].face_angle.reset();
		(*teammate)[i].coordinate_accuracy
			= Field_Recog_Abstract::Player_Info::INVALID_ACCURACY;
		(*teammate)[i].velocity_accuracy
			= Field_Recog_Abstract::Player_Info::INVALID_ACCURACY;
	}

	for ( size_t  i = 0  ;  i < opponent -> size()  ;  i ++ )
	{
		(*opponent)[i].coordinate.reset();
		(*opponent)[i].velocity.reset();
		(*opponent)[i].body_angle.reset();
		(*opponent)[i].face_angle.reset();
		(*opponent)[i].coordinate_accuracy
			= Field_Recog_Abstract::Player_Info::INVALID_ACCURACY;
		(*opponent)[i].velocity_accuracy
			= Field_Recog_Abstract::Player_Info::INVALID_ACCURACY;
	}

	if ( (! this -> require_view_angle())
	  || (! this -> require_self_coordinate()) )
	{
		return( false );
	}

#if 1
	Region_Info	self_vel;

	if ( this -> require_self_velocity() )
	{
		self_vel = self_velocity;
	}
	else
	{
		self_vel.set( D2_Region::universal_region() ,
			      D2_Vector::origin() );
	}
#endif

	for ( size_t  i = 0  ;  i < sight -> player.size()  ;  i ++ )
	{
		const Raw_Player_Info &	pl = sight -> player[i];

		Region_Info	coord;
		Region_Info	vel;
		Angle_Info	b_angle;
		Angle_Info	f_angle;

#if 1
		relative_sight_to_player_info
		    ( &coord , &vel , &b_angle , &f_angle ,
		      pl.relative , self_coordinate , view_angle ,
		      self_vel );
#else
		relative_sight_to_player_info
		    ( &coord , &vel , &b_angle , &f_angle ,
		      pl.relative , self_coordinate , view_angle );
#endif

		if ( pl.entity.side == S_Side::Unknown
		  || ! pl.entity.player_number_valid() )
		{
			std::vector<Field_Recog_Abstract::Player_Info> *
				player_array;

			if ( pl.entity.side == S_Side::Our_Side )
			{
				player_array = unknown_teammate;
			}
			else if ( pl.entity.side == S_Side::Our_Side )
			{
				player_array = unknown_opponent;
			}
			else
			{
				player_array = unknown_player;
			}

			Field_Recog_Abstract::Player_Info	u_player;
			u_player.coordinate = coord;
			u_player.velocity   = vel;
			u_player.body_angle = b_angle;
			u_player.face_angle = f_angle;
			u_player.coordinate_accuracy = 0;
			u_player.velocity_accuracy
			 = Field_Recog_Abstract::Player_Info::INVALID_ACCURACY;

			(*player_array).push_back( u_player );
		}
		else if ( pl.entity.side == S_Side::Our_Side )
		{
			(*teammate)[ pl.entity.player_number - 1 ].coordinate
				= coord;
			(*teammate)[ pl.entity.player_number - 1 ].velocity
				= vel;
			(*teammate)[ pl.entity.player_number - 1 ].body_angle
				= b_angle;
			(*teammate)[ pl.entity.player_number - 1 ].face_angle
				= f_angle;

			if ( coord.valid() )
			{
				(*teammate)[ pl.entity.player_number - 1 ]
					.coordinate_accuracy = 0;
			}

			if ( vel.valid() )
			{
				(*teammate)[ pl.entity.player_number - 1 ]
					.velocity_accuracy = 0;
			}
		}
		else // if ( pl.entity.side == S_Side::Opponent_Side )
		{
			(*opponent)[ pl.entity.player_number - 1 ].coordinate
				= coord;
			(*opponent)[ pl.entity.player_number - 1 ].velocity
				= vel;
			(*opponent)[ pl.entity.player_number - 1 ].body_angle
				= b_angle;
			(*opponent)[ pl.entity.player_number - 1 ].face_angle
				= f_angle;

			if ( coord.valid() )
			{
				(*opponent)[ pl.entity.player_number - 1 ]
					.coordinate_accuracy = 0;
			}

			if ( vel.valid() )
			{
				(*opponent)[ pl.entity.player_number - 1 ]
					.velocity_accuracy = 0;
			}
		}
	}

	return( true );
}
