#include  "soccer_manager.h"
#include  "debugstream.h"
#include  "time_stamp.h"
#include  "adjust_sleep.h"
#include  <iostream>
#include  <string>

using namespace std;


//
// Time_Manager
//
Soccer_Manager::Time_Manager::Time_Manager( bool  true_view )
	: true_view( true_view ) ,
	  have_debug_server( false ) ,
	  view_width( View_Width::Normal ) ,
	  view_quality( View_Quality::High ) ,
	  latest_body_info_time( -1 ) ,
	  latest_sight_info_time( -1 ) ,
	  latest_debug_info_time( -1 ) ,
	  this_step_sight_info( false ) ,
	  latest_body_info_time_stamp() ,
	  old_step_count( -1 )
{
}

void   Soccer_Manager::Time_Manager::register_info
					 ( const Server_Raw_Info &  info ,
					   const Time_Stamp &  stamp ,
					   bool *  time_manager_internal_use ,
					   Soccer_Command *  adjust_command )
{
	//
	// set information
	//
	if ( info.type() == Server_Raw_Info::Type_Body )
	{
		this -> latest_body_info_time = info.time();
		this -> latest_body_info_time_stamp = stamp;

		if ( this -> latest_sight_info_time != -1
		  && info.time() != old_step_count )
		{
			this -> this_step_sight_info = true;
		}
	}
	else if ( info.type() == Server_Raw_Info::Type_Sight )
	{
		if ( this -> view_width   == View_Width  ::Narrow
		  && this -> view_quality == View_Quality::Low )
		{
			// ignore
		}
		else
		{
			this -> latest_sight_info_time = info.time();

			if ( this -> internal_step_started() )
			{
				this -> this_step_sight_info = true;
			}
		}
	}
	else if ( info.type() == Server_Raw_Info::Type_Debug )
	{
		this -> have_debug_server = true;

		this -> latest_debug_info_time = info.time();
	}


	//
	// adjust sight information arrival timing
	//
	View_Width	w = this -> view_width;
	View_Quality	q = this -> view_quality;

	if ( info.type() == Server_Raw_Info::Type_Sight )
	{
		Time_Stamp	current;
		current.set_current_time();

		Time_Stamp::Time_Diff	diff
			= (current - this -> latest_body_info_time_stamp);

		const long	thinking_time_min = 25; // msec

		if ( diff.msec() + 150 + thinking_time_min < 200 - 10 )
		{
			w = View_Width  ::Normal;
			q = View_Quality::High;
		}
		else if ( diff.msec() + 75 < 100 - 5 )
		{
			w = View_Width  ::Narrow;
			q = View_Quality::Low;
		}
		else
		{
			w = View_Width  ::Narrow;
			q = View_Quality::High;
		}
	}

	if ( info.type() == Server_Raw_Info::Type_Sight
	  && this -> view_width   == View_Width  ::Narrow
	  && this -> view_quality == View_Quality::Low )
	{
		*time_manager_internal_use = true;
	}
	else
	{
		*time_manager_internal_use = false;
	}

	if ( this -> view_width != w  ||  this -> view_quality != q )
	{
		*adjust_command = Soccer_Command::Change_View( w , q );
	}
	else
	{
		*adjust_command = Soccer_Command::No_Command();
	}
}

bool   Soccer_Manager::Time_Manager::internal_step_started() const
{
	return( this -> latest_body_info_time != -1 );
}

Time_Stamp  Soccer_Manager::Time_Manager::internal_step_start_time_stamp()
									const
{
	return( this -> latest_body_info_time_stamp );
}

long   Soccer_Manager::Time_Manager::internal_step_count() const
{
	if ( ! this -> internal_step_started() )
	{
		return( -1 );
	}

	return( this -> latest_body_info_time );
}

bool   Soccer_Manager::Time_Manager::next_step_ready() const
{
	if ( ! this -> internal_step_started() )
	{
		return( false );
	}

	if ( this -> view_width   == View_Width  ::Narrow
	  && this -> view_quality == View_Quality::Low )
	{
		return( false );
	}

	if ( this -> true_view && this -> have_debug_server
	  && this -> latest_debug_info_time == this -> internal_step_count() )
	{
		return( true );
	}

	if ( this -> latest_sight_info_time != -1
	  && this -> latest_sight_info_time == this -> latest_body_info_time
	  && this -> this_step_sight_info
	  && ( (! this -> have_debug_server)
	       || this -> latest_debug_info_time
				== this -> latest_body_info_time) )
	{
		return( true );
	}
	else
	{
		return( false );
	}
}

void   Soccer_Manager::Time_Manager::start_step_notify()
{
	if ( Debug_Stream::dbg )
	{
		Time_Stamp	current;
		current.set_current_time();

		Time_Stamp::Time_Diff	diff
			= (current - this -> latest_body_info_time_stamp);


		Debug_Stream::dbg << "thinking time = "
				  << 100 - diff.msec() << endl;
	}

	this -> latest_body_info_time  = -1;
	this -> latest_sight_info_time = -1;
	this -> latest_debug_info_time = -1;
	this -> this_step_sight_info = false;
	this -> old_step_count = internal_step_count();
}

void   Soccer_Manager::Time_Manager::change_view_notify( View_Width  w ,
							 View_Quality  q )
{
	this -> view_width   = w;
	this -> view_quality = q;
}



//
// Soccer_Manager
//
Soccer_Manager::Soccer_Manager( const ref_count_ptr<SServer_Param> &  p ,
				SServer_Player_Connection *  s ,
				bool  true_view )
	: param( p ) ,
	  server_connection( s ) ,
	  field_recog_list() ,
	  main_field_recog( static_cast<Field_Recog_Abstract *>(0) ) ,
	  field_recog_interface_list() ,
	  debug_connection( static_cast<Debug_Client_Connection *>(0) ) ,
	  debug_view( new Client_Debug_View( p.get() ) ) ,
	  time_manager( true_view ) ,
	  have_target_teammate( false ) , target_teammate( -1 ) ,
	  have_target_point( false ) , target_point() ,
	  have_message( false ) , message()
{
}

Soccer_Manager::~Soccer_Manager()
{
}

void   Soccer_Manager::register_field_recog( Field_Recog_Abstract *  fr ,
					     bool  main_field_recog_flag )
{
	field_recog_list.push_back( fr );

	if ( main_field_recog_flag )
	{
		main_field_recog = fr;
	}
}

void   Soccer_Manager::register_field_recog_interface
					( Field_Recog_Interface *  f )
{
	field_recog_interface_list.push_back( f );
}

void   Soccer_Manager::register_debug_connection( Debug_Client_Connection * d )
{
	debug_connection = d;
}

bool   Soccer_Manager::game_over() const throw()
{
	if ( main_field_recog && main_field_recog -> snapshot() )
	{
		return( main_field_recog -> snapshot()
			 -> game_state.game_over_flag );
	}
	else
	{
		return( false );
	}
}

bool   Soccer_Manager::server_no_response() const throw()
{
	return( ! (server_connection -> responsive()) );
}

// XXX
void   Soccer_Manager::set_target_teammate( int  num )
{
	this -> have_target_teammate = true;
	this -> target_teammate = num;
}

// XXX
void   Soccer_Manager::set_target_point( const D2_Vector &  point )
{
	this -> have_target_point = true;
	this -> target_point = point;
}

// XXX
void   Soccer_Manager::set_message( const std::string &  mes )
{
	this -> have_message = true;
	this -> message = mes;
}

void   Soccer_Manager::next_step()
{
	if ( Debug_Stream::dbg )
	{
		Debug_Stream::dbg << endl;
		Debug_Stream::dbg << endl;
	}

	//
	// send debug info to debug monitor
	//
	this -> send_debug();

	this -> have_target_teammate = false;
	this -> have_target_point    = false;
	this -> have_message         = false;


	//
	// update information & wait for time_manager's permision
	//
	while( server_connection -> responsive() )
	{
		if ( this -> next_step_ready() )
		{
			this -> time_manager.start_step_notify();
			break;
		}

		this -> update( true );
	}

	for ( size_t  i = 0  ;  i < field_recog_list.size()  ;  i ++ )
	{
		field_recog_list[i] -> update();
	}

	for ( size_t  i = 0  ;  i < field_recog_interface_list.size()  ;
	      i ++ )
	{
		field_recog_interface_list[i] -> update();
	}


	//
	// make debug view
	//
	if ( debug_connection )
	{
		debug_view = new Client_Debug_View( this -> param.get() );

		if ( main_field_recog )
		{
			debug_view -> set_field_snapshot( main_field_recog
							  -> snapshot() );
		}
	}
}


bool   Soccer_Manager::next_step_ready() const
{
	return( this -> time_manager.next_step_ready() );
}


void   Soccer_Manager::update( bool  block )
{
	Server_Raw_Info	info;

	if ( server_connection -> responsive()
	  && server_connection -> recv_info( &info , block ) )
	{
		Time_Stamp	stamp;
		stamp.set_current_time();


		bool		time_manager_internal_use;
		Soccer_Command	time_adjust_command;

		time_manager.register_info( info , stamp ,
					    &time_manager_internal_use ,
					    &time_adjust_command );

		if ( time_adjust_command != Soccer_Command::No_Command() )
		{
			this -> send_command( time_adjust_command );
		}

		if ( ! time_manager_internal_use )
		{
			this -> register_info( info , stamp );
		}

		if ( info.type() == Server_Raw_Info::Type_Server_Param )
		{
			this -> param -> overwrite_server_param
				( *(info.get_server_param_info()) );
		}
		else if ( info.type() == Server_Raw_Info::Type_Player_Param )
		{
			this -> param -> overwrite_player_param
				( *(info.get_player_param_info()) );
		}
	}
}


void   Soccer_Manager::send_debug()
{
	if ( debug_connection )
	{
		// XXX
		if ( have_target_teammate )
		{
			debug_view -> set_target_teammate( target_teammate );
		}

		// XXX
		if ( have_target_point )
		{
			debug_view -> set_target_point( target_point );
		}

		// XXX
		if ( have_message )
		{
			debug_view -> set_message( message );
		}

		std::string	encoded_string;

		if ( debug_view -> encode( &encoded_string ) )
		{
			debug_connection -> send( encoded_string );
		}
	}
}



// protected:
void   Soccer_Manager::register_info( const Server_Raw_Info &  info ,
				      const Time_Stamp &  stamp )
{
	for ( size_t  i = 0  ;  i < field_recog_list.size()  ;  i ++ )
	{
		field_recog_list[i] -> register_info( info , stamp );
	}
}


int    Soccer_Manager::init_connection( const std::string &  team_name ,
					bool  goalie ,
					Game_Info *  g_info ,
					int  major_version ,
					int  minor_version ,
					int  n_trial )
	throw()
{
	if ( ! goalie )
	{
		Time_Stamp	stamp;
		stamp.set_current_time();

		Adjust_Sleep::usleep( (stamp.usec() % 300) * 1000 );
	}

	for ( int  n = 1  ;  n <= n_trial  ;  n ++ )
	{
		Game_Info	game_info;

		int	ret = server_connection -> send_init( team_name ,
							      goalie ,
							      &game_info ,
							      major_version ,
							      minor_version );

		if ( g_info )
		{
			*g_info = game_info;
		}

		if ( ret != -1
		  && ((1 <= game_info.self_player_number
		       &&   game_info.self_player_number <= MAX_PLAYER)
		     || game_info.self_player_number == 0 /* coach */) )
		{
			// succeeded
			return( ret );
		}

		if ( n != n_trial )
		{
			std::cerr << "not registered. (Retrying)" << endl;

			Adjust_Sleep::sleep( 1 );
		}
		else
		{
			std::cerr << "not registered." << endl;
		}
	}

	return( -1 );
}

int    Soccer_Manager::reconnect_connection( const std::string &  team_name ,
					     int  player_number ,
					     Game_Info *  g_info ,
					     int  n_trial ) throw()
{
	Time_Stamp	stamp;
	stamp.set_current_time();

	Adjust_Sleep::usleep( (stamp.usec() % 300) * 1000 );


	for ( int  n = 1  ;  n <= n_trial  ;  n ++ )
	{
		Game_Info	game_info;

		int	ret = server_connection
				-> send_reconnect( team_name ,
						   player_number ,
						   &game_info ,
						   true );

		if ( g_info )
		{
			*g_info = game_info;
		}

		if ( ret != -1
		  && ((1 <= game_info.self_player_number
		       &&   game_info.self_player_number <= MAX_PLAYER)
		     || game_info.self_player_number == 0 /* coach */) )
		{
			return( ret );
		}

		if ( n != n_trial )
		{
			std::cerr << "not registered. (Retrying)" << endl;

			Adjust_Sleep::sleep( 1 );
		}
		else
		{
			std::cerr << "not registered." << endl;
		}
	}

	return( -1 );
}


int    Soccer_Manager::send_bye( long  interval_usec ) throw()
{
	return( server_connection -> send_bye( interval_usec ) );
}

int    Soccer_Manager::send_command( const Soccer_Command &  command ,
				     long  interval_usec ) throw()
{
	int	ret = server_connection -> send_command( command ,
							 interval_usec );

	Time_Stamp	stamp;
	stamp.set_current_time();

	switch( command.type() )
	{
	case Soccer_Command::Move_Command:
	case Soccer_Command::Dash_Command:
	case Soccer_Command::Kick_Command:
	case Soccer_Command::Turn_Command:
	case Soccer_Command::Catch_Ball_Command:
	case Soccer_Command::Wait_Command:
	case Soccer_Command::Change_Player_Type_Command:
	case Soccer_Command::No_Command_Command:
	case Soccer_Command::Illegal_Command:
	case Soccer_Command::Bye_Command:
		for ( size_t  i = 0  ;  i < field_recog_list.size()  ;  i ++ )
		{
			field_recog_list[i]
			    -> base_command_sent( command , stamp );
		}
		break;

	case Soccer_Command::Turn_Neck_Command:
		for ( size_t  i = 0  ;  i < field_recog_list.size()  ;  i ++ )
		{
			field_recog_list[i]
			    -> turn_neck_sent( command.turn_neck_direction() ,
					       stamp );
		}
		break;

	case Soccer_Command::Change_View_Command:
		this -> time_manager.change_view_notify
			( command.change_view_view_width() ,
			  command.change_view_view_quality() );
		break;

	case Soccer_Command::Sense_Body_Command:
	case Soccer_Command::Say_Command:
		break;
	}

	return( ret );
}


int    Soccer_Manager::send_composite_command
		( const Soccer_Composite_Command &  composite_command ,
		  long  interval_usec ) throw()
{
	int	ret = -1;

	//
	// Send Main Command
	//
	switch( composite_command.base_type() )
	{
	case Soccer_Command::Move_Command:
	case Soccer_Command::Dash_Command:
	case Soccer_Command::Kick_Command:
	case Soccer_Command::Turn_Command:
	case Soccer_Command::Catch_Ball_Command:
	case Soccer_Command::Wait_Command:
	case Soccer_Command::Change_Player_Type_Command:
	case Soccer_Command::No_Command_Command:
	case Soccer_Command::Illegal_Command:
	case Soccer_Command::Bye_Command:
	      {
		ret = server_connection -> send_command
			( composite_command.base_command() , interval_usec );

		Time_Stamp	stamp;
		stamp.set_current_time();

		for ( size_t  i = 0  ;  i < field_recog_list.size()  ;  i ++ )
		{
			field_recog_list[i]
			    -> base_command_sent
				( composite_command.base_command() , stamp );
		}
	      }
		break;

	case Soccer_Command::Turn_Neck_Command:
	case Soccer_Command::Change_View_Command:
	case Soccer_Command::Sense_Body_Command:
	case Soccer_Command::Say_Command:
		ret = -1;
		break;
	}


	//
	// Clear Stream
	//
	Server_Raw_Info		info;
	while( server_connection -> recv_info( &info ) )
	{
		if ( ! server_connection -> responsive() )
		{
			break;
		}

		Time_Stamp	stamp;
		stamp.set_current_time();

		this -> register_info( info , stamp );
	}


	//
	// Return If Illegal Command
	//
	if ( composite_command.base_type() == Soccer_Command::Illegal_Command )
	{
		return( ret );
	}


	//
	// Send Sub Commands
	//

	// turn_neck
	if ( composite_command.turn_neck() )
	{
		Time_Stamp	stamp;
		stamp.set_current_time();

		if ( server_connection -> send_command
					   ( *(composite_command.turn_neck()) ,
					     interval_usec ) == -1 )
		{
			ret = -1;
		}

		for ( size_t  i = 0  ;  i < field_recog_list.size()  ;  i ++ )
		{
			field_recog_list[i]
			  -> turn_neck_sent( composite_command.turn_neck()
						  -> turn_neck_direction() ,
					     stamp );
		}
	}

	// change_view
	if ( composite_command.change_view() )
	{
		if ( server_connection -> send_command
		     ( *(composite_command.change_view()) ,
		       interval_usec ) == -1 )
		{
			ret = -1;
		}

		this -> time_manager.change_view_notify
			( composite_command.change_view()
			    -> change_view_view_width() ,
			  composite_command.change_view()
			    -> change_view_view_quality() );
	}

	// sense_body
	if ( composite_command.sense_body() )
	{
		if ( server_connection -> send_command
		     ( *(composite_command.sense_body()) ,
		       interval_usec ) == -1 )
		{
			ret = -1;
		}
	}

	// say
	if ( composite_command.say() )
	{
		if ( server_connection -> send_command
		     ( *(composite_command.say()) ,
		       interval_usec ) == -1 )
		{
			ret = -1;
		}
	}

	return( ret );
}

int    Soccer_Manager::send_raw_message( const std::string &  str ,
					 long  interval_usec ) throw()
{
	return( server_connection -> send_with_interval( str ,
							 interval_usec ) );
}




int    Soccer_Manager::send_bye( bool  default_wait ) throw()
{
	return( server_connection -> send_bye( default_wait ) );
}

int    Soccer_Manager::send_command( const Soccer_Command &  command ,
				     bool  default_wait ) throw()
{
	return( server_connection -> send_command( command , default_wait ) );
}

int    Soccer_Manager::send_composite_command
		( const Soccer_Composite_Command &  composite_command ,
		  bool  default_wait ) throw()
{
	return( this -> send_composite_command
		( composite_command ,
		  (default_wait ?
		   (server_connection -> default_wait()) : 0) ) );
}

int    Soccer_Manager::send_raw_message( const std::string &  str ,
					 bool  default_wait ) throw()
{
	return( server_connection -> send_with_interval( str ,
							 default_wait ) );
}
