#include  "soccer_object_reference.h"
#include  "field_recog_interface.h"

//
// Soccer_Object_Reference
//
Soccer_Object_Reference::Cache	Soccer_Object_Reference::cache;

Soccer_Object_Reference::Soccer_Object_Reference
			( const Field_Recog_Interface &  f )
	: field( &f )
{
}

Soccer_Object_Reference::~Soccer_Object_Reference()
{
}

const Field_Recog_Abstract::Field_Recog_Snapshot &
	Soccer_Object_Reference::snapshot() const
{
	return( this -> field -> snapshot() );
}

const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Soccer_Object_Reference::coordinate() const
{
	return( cache.coordinate );
}

const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Soccer_Object_Reference::coordinate_region() const
{
	return( cache.coordinate_region );
}

const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Soccer_Object_Reference::velocity() const
{
	return( cache.velocity );
}

const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Soccer_Object_Reference::velocity_region() const
{
	return( cache.velocity_region );
}


//
// Ball_Reference
//
Ball_Reference::Ball_Reference( const Field_Recog_Interface &  f )
	: Soccer_Object_Reference( f )
{
	this -> update();
}

Ball_Reference::~Ball_Reference()
{
}

void   Ball_Reference::update()
{
	if ( snapshot().ball.coordinate.valid() )
	{
		cache_coordinate
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( snapshot().ball.coordinate.point() ,
			    snapshot().ball.coordinate_accuracy );

		cache_coordinate_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( snapshot().ball.coordinate.region() ,
			    snapshot().ball.coordinate_accuracy );
	}
	else
	{
		cache_coordinate
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( snapshot().ball.coordinate.point() ,
			    Soccer_Object_Reference::D2_Region_with_Accuracy
			      ::INVALID_ACCURACY );
		cache_coordinate_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( snapshot().ball.coordinate.region() ,
			    Soccer_Object_Reference::D2_Region_with_Accuracy
			      ::INVALID_ACCURACY );
	}

	if ( snapshot().ball.velocity.valid() )
	{
		cache_velocity
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( snapshot().ball.velocity.point() ,
			    snapshot().ball.velocity_accuracy );

		cache_velocity_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( snapshot().ball.velocity.region() ,
			    snapshot().ball.velocity_accuracy );
	}
	else
	{
		cache_velocity
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( snapshot().ball.velocity.point() ,
			    Soccer_Object_Reference::D2_Vector_with_Accuracy
			      ::INVALID_ACCURACY );

		cache_velocity_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( snapshot().ball.velocity.region() ,
			    Soccer_Object_Reference::D2_Region_with_Accuracy
			      ::INVALID_ACCURACY );
	}
}

#if 0
// moved to soccer_object_reference.h
const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Ball_Reference::coordinate() const
{
	return( this -> cache_coordinate );
}

const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Ball_Reference::coordinate_region() const
{
	return( this -> cache_coordinate_region );
}

const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Ball_Reference::velocity() const
{
	return( this -> cache_velocity );
}

const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Ball_Reference::velocity_region() const
{
	return( this -> cache_velocity_region );
}
#endif


//
// Player_Reference
//
const Field_Recog_Abstract::Player_Info  Player_Reference::unknown_player;

Player_Reference::Player_Reference( const Field_Recog_Interface &  f )
	: Soccer_Object_Reference( f )
{
	this -> valid_flag          = false;
	this -> unknown_flag        = true;
	this -> ident.side          = S_Side::Unknown;
	this -> ident.player_number = -1;
	this -> unknown_index       = 0;
	this -> created_time        = f.current_time();

	this -> update();
}

Player_Reference::Player_Reference( const Field_Recog_Interface &  f ,
				    const S_Side &  side ,  int  num )
	: Soccer_Object_Reference( f )
{
	assert( (1 <= num && num <= MAX_PLAYER)  &&  side != S_Side::Unknown );

	this -> valid_flag          = true;
	this -> unknown_flag        = false;
	this -> ident.side          = side;
	this -> ident.player_number = num;
	this -> unknown_index       = 0;
	this -> created_time        = f.current_time();

	this -> update();
}

Player_Reference::Player_Reference( const Field_Recog_Interface &  f ,
				    const S_Side &  side ,  int  num ,
				    int  indx )
	: Soccer_Object_Reference( f )
{
	assert( num == -1 );

	this -> valid_flag          = true;
	this -> unknown_flag        = true;
	this -> ident.side          = side;
	this -> ident.player_number = -1;
	this -> unknown_index       = indx;
	this -> created_time        = f.current_time();

	this -> update();
}

Player_Reference::~Player_Reference()
{
}

void   Player_Reference::update()
{
	this -> pl_cache = &(this -> player_info_no_cache());

	// coordinate
	if ( this -> player_info().coordinate.valid() )
	{
		cache_coordinate
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( this -> player_info().coordinate.point() ,
			    this -> player_info().coordinate_accuracy );

		cache_coordinate_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( this -> player_info().coordinate.region() ,
			    this -> player_info().coordinate_accuracy );
	}
	else
	{
		cache_coordinate
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( this -> player_info().coordinate.point() ,
			    Soccer_Object_Reference::D2_Vector_with_Accuracy
			      ::INVALID_ACCURACY );

		cache_coordinate_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( this -> player_info().coordinate.region() ,
			    Soccer_Object_Reference::D2_Region_with_Accuracy
			      ::INVALID_ACCURACY );
	}

	// velocity
	if ( this -> player_info().velocity.valid() )
	{
		cache_velocity
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( this -> player_info().velocity.point() ,
			    this -> player_info().velocity_accuracy );

		cache_velocity_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( this -> player_info().velocity.region() ,
			    this -> player_info().velocity_accuracy );
	}
	else
	{
		cache_velocity
			= Soccer_Object_Reference::D2_Vector_with_Accuracy
			  ( this -> player_info().velocity.point() ,
			    Soccer_Object_Reference::D2_Vector_with_Accuracy
			      ::INVALID_ACCURACY );

		cache_velocity_region
			= Soccer_Object_Reference::D2_Region_with_Accuracy
			  ( this -> player_info().velocity.region() ,
			    Soccer_Object_Reference::D2_Region_with_Accuracy
			      ::INVALID_ACCURACY );
	}

	// body_angle
	if ( this -> player_info().body_angle.valid() )
	{
		cache_body_angle
			= Soccer_Object_Reference::Angle_with_Validity
			  ( this -> player_info().body_angle.point() , true );

		cache_body_angle_range
			= Soccer_Object_Reference::Angle_Range_with_Validity
			  ( this -> player_info().body_angle.range() , true );
	}
	else
	{
		cache_body_angle
			= Soccer_Object_Reference::Angle_with_Validity
			  ( this -> player_info().body_angle.point() , false );
		cache_body_angle_range
			= Soccer_Object_Reference::Angle_Range_with_Validity
			  ( this -> player_info().body_angle.range() , false );
	}

	// face angle
	if ( this -> player_info().face_angle.valid() )
	{
		cache_face_angle
			= Soccer_Object_Reference::Angle_with_Validity
			  ( this -> player_info().face_angle.point() , true );

		cache_face_angle_range
			= Soccer_Object_Reference::Angle_Range_with_Validity
			  ( this -> player_info().face_angle.range() , true );
	}
	else
	{
		cache_face_angle
			= Soccer_Object_Reference::Angle_with_Validity
			  ( this -> player_info().face_angle.point() , false );

		cache_face_angle_range
			= Soccer_Object_Reference::Angle_Range_with_Validity
			  ( this -> player_info().face_angle.range() , false );
	}
}


// internal
const Field_Recog_Abstract::Player_Info &  Player_Reference::player_info()
	const
{
	if ( this -> created_time == field -> current_time() )
	{
		return( *pl_cache );
	}
	else
	{
		return( this -> player_info_no_cache() );
	}
}

// internal
const Field_Recog_Abstract::Player_Info &
				Player_Reference::player_info_no_cache() const
{
	if ( ! valid() )
	{
		return( unknown_player );
	}

	switch( this -> side() )
	{
	case S_Side::Our_Side:
		if ( this -> player_number() == -1 )
		{
			assert( this -> unknown_flag
			     && 0 <= this -> unknown_index
			     && this -> unknown_index
				< static_cast<int>
				  ( snapshot().unknown_teammate.size() ) );

			return( snapshot()
				.unknown_teammate[ this -> unknown_index ] );
		}
		else
		{
			assert( 1 <= this -> player_number()
				&& this -> player_number() <= MAX_PLAYER );

			return( snapshot()
				.teammate[ this -> player_number() - 1 ] );
		}
		break;

	case S_Side::Opponent_Side:
		if ( this -> player_number() == -1 )
		{
			assert( this -> unknown_flag
				&& 0 <= this -> unknown_index
				&& this -> unknown_index
				   < static_cast<int>
				     ( snapshot().unknown_opponent.size() ) );

			return( snapshot()
				.unknown_opponent[ this -> unknown_index ] );
		}
		else
		{
			assert( 1 <= this -> player_number()
				&& this -> player_number() <= MAX_PLAYER );

			return( snapshot()
				.opponent[ this -> player_number() - 1 ] );
		}
		break;

	default:
	case S_Side::Unknown:
		{
			assert( 0 <= this -> unknown_index
				&& this -> unknown_index
				   < static_cast<int>
				     ( snapshot().unknown_player.size() ) );

			return( snapshot()
				.unknown_player[ this -> unknown_index ] );
		}
		break;
	}
}

Player_Reference &  Player_Reference::operator= ( const Player_Reference &  p )
{
	this -> valid_flag    = p.valid_flag;
	this -> unknown_flag  = p.unknown_flag;
	this -> ident         = p.ident;
	this -> unknown_index = p.unknown_index;
	this -> created_time  = p.created_time;
	this -> pl_cache      = p.pl_cache;
	this -> cache         = p.cache;

	this -> cache_coordinate        = p.cache_coordinate;
	this -> cache_coordinate_region = p.cache_coordinate_region;
	this -> cache_velocity          = p.cache_velocity;
	this -> cache_velocity_region   = p.cache_velocity_region;
	this -> cache_body_angle        = p.cache_body_angle;
	this -> cache_body_angle_range  = p.cache_body_angle_range;
	this -> cache_face_angle        = p.cache_face_angle;
	this -> cache_face_angle_range  = p.cache_face_angle_range;

	return( *this );
}

bool   Player_Reference::valid() const
{
	return( this -> valid_flag
		&& ! (this -> unknown_flag
		      && this -> created_time
		         != this -> field -> current_time()) );
}

S_Side  Player_Reference::side() const
{
	return( this -> ident.side );
}

bool   Player_Reference::teammate() const
{
	return( this -> ident.side == S_Side::Our_Side );
}

bool   Player_Reference::opponent() const
{
	return( this -> ident.side == S_Side::Opponent_Side );
}

bool   Player_Reference::unknown() const
{
	return( this -> ident.side == S_Side::Unknown );
}

S_Side_LR  Player_Reference::side_lr() const
{
	S_Side_LR	our_side = snapshot().game_state.game_info.our_side;

	if ( this -> side() == S_Side::Unknown
	  || our_side == S_Side_LR::Unknown )
	{
		return( S_Side_LR::Unknown );
	}

	if ( this -> side() == S_Side::Our_Side )
	{
		return( our_side );
	}
	else
	{
		if ( our_side == S_Side_LR::Left_Side )
		{
			return( S_Side_LR::Right_Side );
		}
		else
		{
			return( S_Side_LR::Left_Side );
		}
	}
}

Team_Reference  Player_Reference::team() const
{
	return( Team_Reference( *(this -> Soccer_Object_Reference::field) ,
				this -> side() ) );
}

int    Player_Reference::player_number() const
{
	return( this -> ident.player_number );
}

bool   Player_Reference::self_player() const
{
	return( this -> side() == S_Side::Our_Side
	     && this -> player_number() != -1
	     && this -> player_number()
		 == this -> snapshot()
				.game_state.game_info.self_player_number );
}

Soccer_Object_Reference::Bool_with_Validity  Player_Reference::goalie() const
{
	// XXX

	if ( this -> side() == S_Side::Our_Side
	  && this -> snapshot().our_team_goalie_number != -1
	  && this -> snapshot().our_team_goalie_number
	     == this -> player_number() )
	{
		return( Bool_with_Validity( true , true ) );
	}

	if ( this -> side() == S_Side::Opponent_Side
	  && this -> snapshot().opponent_team_goalie_number != -1
	  && this -> snapshot().opponent_team_goalie_number
	     == this -> player_number() )
	{
		return( Bool_with_Validity( true , true ) );
	}

	return( Bool_with_Validity( false , false ) );
}

Soccer_Object_Reference::Bool_with_Validity  Player_Reference::field_player()
	const
{
	return( ! (this -> goalie()) );
}

Soccer_Object_Reference::Bool_with_Validity  Player_Reference::actual_goalie()
	const
{
	// XXX

	if ( ! this -> goalie() )
	{
		return( Bool_with_Validity( false , false ) );
	}

	if ( std::fabs( this -> coordinate().y() )
	     > this -> field -> sserver_param().penalty_area_half_width() )
	{
		return( Bool_with_Validity( false , false ) );
	}

	double	penalty_area_abs_x
		= this -> field -> sserver_param().field_half_length()
		  - this -> field -> sserver_param().penalty_area_length();

	if ( this -> side() == S_Side::Our_Side
	     && this -> coordinate().x() <= - penalty_area_abs_x
	  || this -> side() == S_Side::Opponent_Side
	     && this -> coordinate().x() >= + penalty_area_abs_x )
	{
		return( Bool_with_Validity( true , true ) );
	}
	else
	{
		return( Bool_with_Validity( false , false ) );
	}
}


#if 0
const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Player_Reference::coordinate() const
{
	return( this -> cache_coordinate );
}

const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Player_Reference::coordinate_region() const
{
	return( this -> cache_coordinate_region );
}

const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Player_Reference::velocity() const
{
	return( this -> cache_velocity );
}

const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Player_Reference::velocity_region() const
{
	return( this -> cache_velocity_region );
}

const Soccer_Object_Reference::Angle_with_Validity &
	Player_Reference::body_angle() const
{
	return( this -> cache_body_angle );
}

const Soccer_Object_Reference::Angle_Range_with_Validity &
	Player_Reference::body_angle_range() const
{
	return( this -> cache_body_angle_range );
}

const Soccer_Object_Reference::Angle_with_Validity &
	Player_Reference::face_angle() const
{
	return( this -> cache_face_angle );
}

const Soccer_Object_Reference::Angle_Range_with_Validity &
	Player_Reference::face_angle_range() const
{
	return( this -> cache_face_angle_range );
}
#endif

double  Player_Reference::stamina() const
{
	if ( this -> self_player() )
	{
		return( this -> snapshot().body.stamina.point() );
	}
	else
	{
		return( this -> field -> sserver_param().STAMINA_MAX() );
	}
}

Double_Info  Player_Reference::stamina_range() const
{
	if ( this -> self_player() )
	{
		return( this -> snapshot().body.stamina );
	}
	else
	{
		return( Double_Info
			( 0.0 ,
			  field -> sserver_param().STAMINA_MAX() ,
			  field -> sserver_param().STAMINA_MAX() ) );
	}
}

View_Width  Player_Reference::view_width() const
{
	if ( this -> self_player()
	  && this -> snapshot().body.view_width.valid() )
	{
		return( this -> snapshot().body.view_width.width() );
	}

	return( View_Width::Unknown );
}

double  Player_Reference::ball_kickable_distance() const
{
	// XXX
	return( field -> sserver_param().kickable_distance() - 0.05 );
}

double  Player_Reference::ball_controllable_distance() const
{
	if ( this -> actual_goalie() )
	{
		return( field -> sserver_param().catchable_length() - 0.05 );
	}
	else
	{
		return( this -> ball_kickable_distance() );
	}
}

bool   Player_Reference::ball_kickable() const
{
	return( field -> ball().coordinate().have_info()
		&& this -> coordinate().have_info()
		&& (field -> ball().coordinate() - this -> coordinate()).r()
		    <= this -> ball_controllable_distance() );
}

bool   Player_Reference::ball_controllable() const
{
	return( field -> ball().coordinate().have_info()
		&& this -> coordinate().have_info()
		&& (field -> ball().coordinate() - this -> coordinate()).r()
		    <= this -> ball_kickable_distance() );
}

bool   Player_Reference::offside_position() const
{
	switch( this -> ident.side )
	{
	case S_Side::Our_Side:
		return( this -> coordinate().x()
			> this -> team().offence_line() );

	case S_Side::Opponent_Side:
		return( this -> coordinate().x()
			< this -> team().offence_line() );

	case S_Side::Unknown:
	default:
		return( false );
	}
}


//
// Team_Reference
//
Team_Reference::Team_Reference( const Field_Recog_Interface &  f ,
				const S_Side &  s )
	: field( &f ) , side_ident( s )
{
}

Team_Reference::~Team_Reference()
{
}

S_Side  Team_Reference::side() const
{
	return( side_ident );
}

S_Side_LR  Team_Reference::side_lr() const
{
	S_Side_LR	our_side
			  = field -> snapshot().game_state.game_info.our_side;

	if ( this -> side() == S_Side::Unknown
	  || our_side == S_Side_LR::Unknown )
	{
		return( S_Side_LR::Unknown );
	}

	if ( this -> side() == S_Side::Our_Side )
	{
		return( our_side );
	}
	else
	{
		if ( our_side == S_Side_LR::Left_Side )
		{
			return( S_Side_LR::Right_Side );
		}
		else
		{
			return( S_Side_LR::Left_Side );
		}
	}
}

int   Team_Reference::sign() const
{
	switch( this -> side_ident )
	{
	case S_Side::Our_Side:
		return( -1 );

	case S_Side::Opponent_Side:
		return( + 1 );

	case S_Side::Unknown:
	default:
		return( 0 );
	}
}

int   Team_Reference::score() const
{
	switch( this -> side_ident )
	{
	case S_Side::Our_Side:
		return( field -> snapshot()
				   .game_state.game_info.our_score );

	case S_Side::Opponent_Side:
		return( field -> snapshot()
				   .game_state.game_info.opponent_score );

	case S_Side::Unknown:
	default:
		return( -1 );
	}
}

Team_Reference  Team_Reference::teammate_team() const
{
	return( *this );
}

Team_Reference  Team_Reference::opponent_team() const
{
	switch( this -> side() )
	{
	case S_Side::Our_Side:
		return( field -> opponent_team() );

	case S_Side::Opponent_Side:
		return( field -> our_team() );

	case S_Side::Unknown:
	default:
		return( field -> team( S_Side::Unknown ) );
	}
}

double  Team_Reference::offside_line() const
{
	switch( this -> side() )
	{
	case S_Side::Our_Side:
		return( field -> snapshot().defence_line );

	case S_Side::Opponent_Side:
		return( field -> snapshot().offence_line );

	case S_Side::Unknown:
	default:
		return( 0.0 );
	}
}

double  Team_Reference::defence_line() const
{
	return( this -> offside_line() );
}

double  Team_Reference::offence_line() const
{
	return( this -> opponent_team().offside_line() );
}

D2_Vector  Team_Reference::goal_coordinate() const throw()
{
	switch( this -> side() )
	{
	case S_Side::Our_Side:
		return( D2_Vector( - this -> field -> sserver_param()
						        .field_half_length() ,
				   0.0 ) );

	case S_Side::Opponent_Side:
		return( D2_Vector( + this -> field -> sserver_param()
						        .field_half_length() ,
				   0.0 ) );

	case S_Side::Unknown:
	default:
		return( D2_Vector::origin() );
	}
}

D2_Vector  Team_Reference::positive_goal_post_coordinate() const throw()
{
	switch( this -> side() )
	{
	case S_Side::Our_Side:
		return( D2_Vector( - this -> field -> sserver_param()
						        .field_half_length() ,
				   + this -> field -> sserver_param()
						        .goal_half_width() ) );

	case S_Side::Opponent_Side:
		return( D2_Vector( + this -> field -> sserver_param()
						        .field_half_length() ,
				   + this -> field -> sserver_param()
						        .goal_half_width() ) );

	case S_Side::Unknown:
	default:
		return( D2_Vector::origin() );
	}
}

D2_Vector  Team_Reference::negative_goal_post_coordinate() const throw()
{
	switch( this -> side() )
	{
	case S_Side::Our_Side:
		return( D2_Vector( - this -> field -> sserver_param()
						        .field_half_length() ,
				   - this -> field -> sserver_param()
						        .goal_half_width() ) );

	case S_Side::Opponent_Side:
		return( D2_Vector( + this -> field -> sserver_param()
						        .field_half_length() ,
				   - this -> field -> sserver_param()
						        .goal_half_width() ) );

	case S_Side::Unknown:
	default:
		return( D2_Vector::origin() );
	}
}

Player_Reference  Team_Reference::goalie() const
{
	if ( this -> side() == S_Side::Our_Side
	     && field -> snapshot().our_team_goalie_number != -1 )
	{
		return( field -> teammate
			( field -> snapshot().our_team_goalie_number ) );
	}
	else if ( this -> side() == S_Side::Our_Side
	     && field -> snapshot().opponent_team_goalie_number != -1 )
	{
		return( field -> opponent
			( field -> snapshot().opponent_team_goalie_number ) );
	}
	else
	{
		Player_Reference	unknown( *field );

		return( unknown );
	}
}


//
// Play_Mode_Reference
//
Play_Mode_Reference::Play_Mode_Reference( const Field_Recog_Interface &  f )
	: field( &f )
{
}

Play_Mode_Reference::~Play_Mode_Reference()
{
}

Play_Mode  Play_Mode_Reference::mode() const
{
	return( field -> snapshot().game_state.game_info.play_mode );
}

long   Play_Mode_Reference::start_time() const
{
	return( field -> snapshot().game_state.play_mode_start_time );
}

Server_Raw_Whistle_Info  Play_Mode_Reference::start_reason() const
{
	return( field -> snapshot().game_state.play_mode_start_reason() );
}

bool   Play_Mode_Reference::kickable_mode() const
{
	return(    this -> mode() != Play_Mode::Before_Kick_Off
		&& this -> mode() != Play_Mode::Our_Goal
		&& this -> mode() != Play_Mode::Opponent_Goal
		&& ! ((   this -> mode() == Play_Mode::Our_Free_Kick
		       || this -> mode() == Play_Mode::Opponent_Free_Kick)
		      && this -> start_reason().type
			  == Judgement_Type::Offside )
		&& this -> mode() != Play_Mode::Unknown );
}

bool   Play_Mode_Reference::catchable_mode() const
{
	return( this -> mode() == Play_Mode::Play_On );
}

bool   Play_Mode_Reference::movable_mode() const
{
	return(    this -> mode() == Play_Mode::Before_Kick_Off
		|| this -> mode() == Play_Mode::Our_Goal
		|| this -> mode() == Play_Mode::Opponent_Goal );
}

Play_Mode_Reference::operator Play_Mode() const
{
	return( this -> mode() );
}

bool   Play_Mode_Reference::operator== ( Play_Mode::Mode  m ) const
{
	return( (this -> mode() == m) );
}

bool   Play_Mode_Reference::operator!= ( Play_Mode::Mode  m ) const
{
	return( (this -> mode() != m) );
}


#if 0
//
// Player_Type_Reference
//
Player_Type_Reference::Player_Type_Reference
				( const Field_Recog_Interface &  f )
	: field( &f ) , player_type_id( player_type_id )
{
}

Player_Type_Reference::~Player_Type_Reference()
{
}

int    Player_Type_Reference::id() const
{
	return( this -> player_type_id );
}


const Server_Raw_Player_Type_Info &  Player_Type_Reference::param() const
{
	ref_count_ptr<const Server_Raw_Player_Type_Info>	p_type;

	p_type = field -> snapshot().game_state.player_type
				-> get_type( this -> id() );

	if ( p_type )
	{
		return( *p_type );
	}
	else
	{
		// XXX
		p_type = field -> snapshot().game_state.player_type
			-> get_default_type();
	}
}
#endif
