#ifndef	   SOCCER_OBJECT_REFERENCE_H_INCLUDED
#define	   SOCCER_OBJECT_REFERENCE_H_INCLUDED

// Author:		H. Shimora
// Last-Modified:	Jan 24 2001
// Created:		Jan 24 2001
// Version:		0.00

///-----------------------------------------------
/// Change Log:
///-----------------------------------------------
// version 0.00  Jan 24 2001    base version.
//
//

#include  "field_recog_abstract.h"
#include  "field_momentary_info.h"
#include  "s_basic.h"
#include  <climits>

class  Field_Recog_Interface;


//
// Soccer Object Reference
//
class  Soccer_Object_Reference
{
protected:
	class  D2_Vector_with_Accuracy : public D2_Vector
	{
	public:
		enum{ INVALID_ACCURACY = LONG_MIN };

	private:
		long	acc;

	public:
		D2_Vector_with_Accuracy( const D2_Vector &  vec ,  long  acc )
			: D2_Vector( vec ) ,  acc( acc ) {}

		D2_Vector_with_Accuracy()
			: D2_Vector() ,  acc( INVALID_ACCURACY ) {}

		long	accuracy_level() const
		{
			return( acc );
		}

		bool	accuracy_check( int  level = 0 ) const
		{
			return( acc >= level );
		}

		bool	have_info() const
		{
			return( acc != INVALID_ACCURACY );
		}
	};

	class  D2_Region_with_Accuracy : public D2_Region
	{
	public:
		enum{ INVALID_ACCURACY = LONG_MIN };

	private:
		long	acc;

	public:
		D2_Region_with_Accuracy( const D2_Region &  reg ,  long  acc )
			: D2_Region( reg ) , acc( acc ) {}

		D2_Region_with_Accuracy()
			: D2_Region() ,  acc( INVALID_ACCURACY ) {}

		long	accuracy_level() const
		{
			return( acc );
		}

		bool	accuracy_check( int  level = 0 ) const
		{
			return( acc >= level );
		}

		bool	have_info() const
		{
			return( acc != INVALID_ACCURACY );
		}
	};

	class  Angle_with_Validity : public Angle
	{
	public:
		enum{ INVALID_ACCURACY = LONG_MIN };

	private:
		bool	validity;

	public:
		Angle_with_Validity( const Angle &  ang ,  bool  validity )
			: Angle( ang ) , validity( validity ) {}

		Angle_with_Validity()
			: Angle( Angle::origin() ) ,
			  validity( INVALID_ACCURACY ) {}

		long	accuracy_level() const
		{
			return( validity ? 0 : INVALID_ACCURACY );
		}

		bool	accuracy_check( int = 0 ) const
		{
			return( validity );
		}

		bool	have_info() const
		{
			return( validity );
		}
	};

	class  Angle_Range_with_Validity : public Angle_Range
	{
	public:
		enum{ INVALID_ACCURACY = LONG_MIN };

	private:
		bool	validity;

	public:
		Angle_Range_with_Validity( const Angle_Range &  ang_range ,
					   bool  validity )
			: Angle_Range( ang_range ) , validity( validity ) {}

		Angle_Range_with_Validity()
			: Angle_Range( Degree( -180.0 ) , Degree( +180.0 ) ) ,
			  validity( INVALID_ACCURACY ) {}

		long	accuracy_level() const
		{
			return( validity ? 0 : INVALID_ACCURACY );
		}

		bool	accuracy_check( int = 0 ) const
		{
			return( validity );
		}

		bool	have_info() const
		{
			return( validity );
		}
	};

	class  Bool_with_Validity
	{
	public:
		enum{ INVALID_ACCURACY = LONG_MIN };

	private:
		bool	value;
		bool	validity;

	public:
		Bool_with_Validity( bool  value , bool  validity )
			: value( value ) , validity( validity ) {}

		operator bool() const
		{
			return( value );
		}

		Bool_with_Validity	operator !() const
		{
			return( Bool_with_Validity( ! value , validity ) );
		}

		long	accuracy_level() const
		{
			return( validity ? 0 : INVALID_ACCURACY );
		}

		bool	accuracy_check( int = 0 ) const
		{
			return( validity );
		}

		bool	have_info() const
		{
			return( validity );
		}
	};

protected:
	static	struct  Cache
	{
		D2_Vector_with_Accuracy		coordinate;
		D2_Region_with_Accuracy		coordinate_region;
		D2_Vector_with_Accuracy		velocity;
		D2_Region_with_Accuracy		velocity_region;
	} cache;

protected:
	const Field_Recog_Interface *	field;

protected:
	const Field_Recog_Abstract::Field_Recog_Snapshot&	snapshot()
									const;

public:
		 Soccer_Object_Reference( const Field_Recog_Interface &  f );
	virtual	~Soccer_Object_Reference();

	virtual	const D2_Vector_with_Accuracy &		coordinate() const;
	virtual	const D2_Region_with_Accuracy &		coordinate_region()
									 const;
	virtual	const D2_Vector_with_Accuracy &		velocity() const;
	virtual	const D2_Region_with_Accuracy &		velocity_region()
									 const;
};


//
// Ball
//
class  Ball_Reference : public Soccer_Object_Reference
{
protected:
	D2_Vector_with_Accuracy		cache_coordinate;
	D2_Region_with_Accuracy		cache_coordinate_region;
	D2_Vector_with_Accuracy		cache_velocity;
	D2_Region_with_Accuracy		cache_velocity_region;

public:
	void	update();

public:
		 Ball_Reference( const Field_Recog_Interface &  f );
	virtual	~Ball_Reference();

	virtual	const D2_Vector_with_Accuracy &		coordinate() const;
	virtual	const D2_Region_with_Accuracy &		coordinate_region()
									 const;
	virtual	const D2_Vector_with_Accuracy &		velocity() const;
	virtual	const D2_Region_with_Accuracy &		velocity_region()
									 const;
};


//
// Player
//
class  Team_Reference;

class  Player_Reference : public Soccer_Object_Reference
{
protected:
	bool				valid_flag;
	bool				unknown_flag;
	SObject_Player_Identifier	ident;
	int				unknown_index;
	SServer_Time			created_time;

	const Field_Recog_Abstract::Player_Info *	pl_cache;

protected:
	D2_Vector_with_Accuracy		cache_coordinate;
	D2_Region_with_Accuracy		cache_coordinate_region;
	D2_Vector_with_Accuracy		cache_velocity;
	D2_Region_with_Accuracy		cache_velocity_region;
	Angle_with_Validity		cache_body_angle;
	Angle_Range_with_Validity	cache_body_angle_range;
	Angle_with_Validity		cache_face_angle;
	Angle_Range_with_Validity	cache_face_angle_range;

protected:
	static	const Field_Recog_Abstract::Player_Info	unknown_player;
	const Field_Recog_Abstract::Player_Info &	player_info_no_cache()
									const;
	const Field_Recog_Abstract::Player_Info &	player_info() const;

public:
	void	update();

public:
		 Player_Reference( const Field_Recog_Interface &  f );

		 Player_Reference( const Field_Recog_Interface &  f ,
				   const S_Side &  s ,  int  num );

		 Player_Reference( const Field_Recog_Interface &  f ,
				   const S_Side &  side ,  int  num ,
				   int  indx );

	virtual	~Player_Reference();

	virtual	Player_Reference &	operator= ( const Player_Reference & );

	virtual	bool			valid() const;

	virtual	S_Side			side() const;
	virtual	S_Side_LR		side_lr() const;
	virtual	Team_Reference		team() const;
	virtual	bool			teammate() const;
	virtual	bool			opponent() const;
	virtual	bool			unknown() const;
	virtual	int			player_number() const;
	virtual	bool			self_player() const;
	virtual	Bool_with_Validity	goalie() const;
	virtual	Bool_with_Validity	field_player() const;
	virtual	Bool_with_Validity	actual_goalie() const;


	virtual	const D2_Vector_with_Accuracy &		coordinate() const;
	virtual	const D2_Region_with_Accuracy &		coordinate_region()
									 const;
	virtual	const D2_Vector_with_Accuracy &		velocity() const;
	virtual	const D2_Region_with_Accuracy &		velocity_region()
									 const;
	virtual	const Angle_with_Validity &		body_angle() const;
	virtual	const Angle_Range_with_Validity &	body_angle_range()
									 const;
	virtual	const Angle_with_Validity &		face_angle() const;
	virtual	const Angle_Range_with_Validity &	face_angle_range()
									 const;

	virtual	double			stamina() const;
	virtual	Double_Info		stamina_range() const;

	virtual	View_Width		view_width() const;

	virtual	double			ball_kickable_distance() const;
	virtual	double			ball_controllable_distance() const;

	virtual	bool			ball_kickable() const;
	virtual	bool			ball_controllable() const;

	virtual	bool			offside_position() const;
};


//
// Team
//
class  Team_Reference
{
protected:
	const Field_Recog_Interface *  field;
	S_Side	side_ident;

public:
		 Team_Reference( const Field_Recog_Interface &  f ,
				 const S_Side &  s );
	virtual	~Team_Reference();

	virtual	S_Side		side() const;
	virtual	S_Side_LR	side_lr() const;

	virtual	int		sign() const;	//    return -1 if our team
						// or return +1 opponent team
						// or return  0 unknown team

	virtual	int		score() const;

	virtual	Team_Reference	teammate_team() const;
	virtual	Team_Reference	opponent_team() const;

	virtual	double	offside_line() const;	// alias for defence_line
	virtual	double	defence_line() const;
	virtual	double	offence_line() const;

	virtual	D2_Vector	goal_coordinate() const throw();
	virtual	D2_Vector	positive_goal_post_coordinate() const throw();
	virtual	D2_Vector	negative_goal_post_coordinate() const throw();

	virtual	Player_Reference	goalie() const;
};


//
// Play Mode
//
class  Play_Mode_Reference
{
protected:
	const Field_Recog_Interface *  field;

public:
		 Play_Mode_Reference( const Field_Recog_Interface &  f );
	virtual	~Play_Mode_Reference();

	virtual	Play_Mode			mode() const;
	virtual	long				start_time() const;
	virtual Server_Raw_Whistle_Info		start_reason() const;

	virtual	bool	kickable_mode() const;
	virtual	bool	catchable_mode() const;
	virtual	bool	movable_mode() const;

	virtual	operator Play_Mode() const;
	virtual	bool	operator== ( Play_Mode::Mode  m ) const;
	virtual	bool	operator!= ( Play_Mode::Mode  m ) const;
};


#if 0
//
// Player Type
//
class  Player_Type_Reference
{
protected:
	const Field_Recog_Interface *  field;

private:
	int	player_type_id;

public:
		 Player_Type_Reference( const Field_Recog_Interface &  f );
	virtual	~Player_Type_Reference();

	virtual	int	id() const;
	virtual	const Server_Raw_Player_Type_Info &	param() const;
};
#endif


#if 1
inline
const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Ball_Reference::coordinate() const
{
	return( this -> cache_coordinate );
}

inline
const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Ball_Reference::coordinate_region() const
{
	return( this -> cache_coordinate_region );
}

inline
const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Ball_Reference::velocity() const
{
	return( this -> cache_velocity );
}

inline
const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Ball_Reference::velocity_region() const
{
	return( this -> cache_velocity_region );
}

inline
const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Player_Reference::coordinate() const
{
	return( this -> cache_coordinate );
}

inline
const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Player_Reference::coordinate_region() const
{
	return( this -> cache_coordinate_region );
}

inline
const Soccer_Object_Reference::D2_Vector_with_Accuracy &
	Player_Reference::velocity() const
{
	return( this -> cache_velocity );
}

inline
const Soccer_Object_Reference::D2_Region_with_Accuracy &
	Player_Reference::velocity_region() const
{
	return( this -> cache_velocity_region );
}

inline
const Soccer_Object_Reference::Angle_with_Validity &
	Player_Reference::body_angle() const
{
	return( this -> cache_body_angle );
}

inline
const Soccer_Object_Reference::Angle_Range_with_Validity &
	Player_Reference::body_angle_range() const
{
	return( this -> cache_body_angle_range );
}

inline
const Soccer_Object_Reference::Angle_with_Validity &
	Player_Reference::face_angle() const
{
	return( this -> cache_face_angle );
}

inline
const Soccer_Object_Reference::Angle_Range_with_Validity &
	Player_Reference::face_angle_range() const
{
	return( this -> cache_face_angle_range );
}
#endif


#endif	/* SOCCER_OBJECT_REFERENCE_H_INCLUDED */
