#ifdef    SSERVER_MESSAGE_TRANSLATOR_TEST
#include  "sserver_message_translator.h"
#include  "ref_count_ptr.h"
#include  <string>
#include  <iostream>

#include  "prof_stopwatch.h"

using namespace std;

int    main( void )
{
	Server_Raw_Info	info;

	ref_count_ptr<string>	input;

	input = new string( "(see 0 ((G) 2.5 180) ((g r) 102.5 0) ((f c) 49.9 0) ((f c t) 60.3 -34) ((f c b) 60.3 34) ((f r t) 107.8 -18) ((f r b) 107.8 18) ((f p l c) 14 0 -0 0) ((f p r t) 88.2 -13) ((f p r c) 85.6 0) ((f p r b) 88.2 13) ((f g r t) 102.5 -3) ((f g r b) 102.5 3) ((f t l 10) 55.7 -44) ((f t 0) 63.4 -37) ((f t r 10) 71.5 -33) ((f t r 20) 79.8 -29) ((f t r 30) 89.1 -25) ((f t r 40) 98.5 -23) ((f t r 50) 107.8 -21) ((f b l 10) 55.7 44) ((f b 0) 63.4 37) ((f b r 10) 71.5 33) ((f b r 20) 79.8 29) ((f b r 30) 89.1 25) ((f b r 40) 98.5 23) ((f b r 50) 107.8 21) ((f r t 30) 111.1 -15) ((f r t 20) 108.9 -10) ((f r t 10) 107.8 -5) ((f r 0) 107.8 0) ((f r b 10) 107.8 5) ((f r b 20) 108.9 10) ((f r b 30) 111.1 15) ((b) 49.4 0) ((p \"test\" 2 goalie) 0.6 0 0 0 0 0) ((l r) 102.5 -90))\n" );

#if 1
	SServer_Message_Translator::parse_message( &info ,
						   input ,
						   S_Side_LR::Left_Side ,
						   "test-team" ,
						   1 );
#else
	Prof_Stopwatch	sw;

	sw.start();

	int	LOOP = 10000;

	for ( int i = 0  ;  i < LOOP  ;  i ++ )
	{
		SServer_Message_Translator::parse_message
			( &info , input ,
			  S_Side_LR::Left_Side , "test-team" , 1 ,
			  /* danger_parse */ true );
	}

	sw.stop();

	cout << sw.real().usec() / LOOP << " usec" << endl;
#endif

	if ( info.illegal() )
	{
		cerr << "Unknown format: ["
		     << info.raw_message() << "]" << endl;
	}

	return( 0 );
}
#endif // SSERVER_MESSAGE_TRANSLATOR_TEST




#include  "sserver_message_translator.h"
#include  "ref_count_ptr.h"
#include  <string>
#include  <strstream>
#include  <cstring>
#include  <cfloat>
#include  <cmath>
#include  <cstdio>
#include  <climits>

using namespace std;

#include  "sserver_param.h"
const	size_t	SServer_Message_Translator::SSERVER_MAX_MESSAGE_LENGTH
			= SServer_Param::SSERVER_MAX_MESSAGE_LENGTH;

static	const	double	SHOWINFO_SCALE = 16.0;

string  SServer_Message_Translator::command( const Soccer_Command &  command )
{
	switch( command.type() )
	{
	case Soccer_Command::Move_Command:
		return( move_command( command.move_x() , command.move_y() ) );

	case Soccer_Command::Dash_Command:
		return( dash_command( command.dash_power() ) );

	case Soccer_Command::Kick_Command:
		return( kick_command( command.kick_power() ,
				      command.kick_direction() ) );

	case Soccer_Command::Turn_Command:
		return( turn_command( command.turn_moment() ) );

	case Soccer_Command::Catch_Ball_Command:
		return( catch_command( command.catch_direction() ) );

	case Soccer_Command::Change_Player_Type_Command:
		return( change_player_type_command
			( command.change_player_type_player_number() ,
			  command.change_player_type_player_type() ) );

	case Soccer_Command::Turn_Neck_Command:
		return( turn_neck_command( command.turn_neck_direction() ) );

	case Soccer_Command::Change_View_Command:
		return( change_view_command
			( command.change_view_view_width() ,
			  command.change_view_view_quality() ) );

	case Soccer_Command::Sense_Body_Command:
		return( sense_body_command() );

	case Soccer_Command::Say_Command:
		return( say_command( command.say_message() ) );

	case Soccer_Command::Bye_Command:
		return( bye_command() );

	case Soccer_Command::Wait_Command:
		return( "(wait)" );

	case Soccer_Command::No_Command_Command:
		return( "(no_command)" );

	case Soccer_Command::Illegal_Command:
		return( "(illegal)" );

	default:
		return( "" );
	}
}

string  SServer_Message_Translator::init_command( const string &  team_name ,
						  bool  goalie ,
						  int  major_version ,
						  int  minor_version )
{
	//
	// Send "init" message to server.
	//
	string	send_string = static_cast<string>("(init ") + team_name;

	if ( major_version >= 0 )
	{
		ref_count_ptr<char>	version_string;
		try
		{
			version_string
			    = new char[1024 + SSERVER_MAX_MESSAGE_LENGTH + 1];
		}
		catch( ... )
		{
			cerr << "Can't allocate memory" << endl;
			throw;
		}

		sprintf( version_string.get() , " (version %d.%02d)" ,
						major_version ,
						minor_version );
		send_string += version_string.get();
	}

	if ( goalie )
	{
		send_string += " (goalie)";
	}

	send_string += ")\n";

	return( send_string );
}


string  SServer_Message_Translator::reconnect_command(
					      const string &  team_name ,
					      int  player_number )
{
	bool	bad_player_number = false;

	if ( ! (1 <= player_number && player_number <= MAX_PLAYER) )
	{
		cerr << "bad player number: " << player_number << endl;
		bad_player_number = true;
	}

#if ! defined(LIB_MEMORY)
	strstream	send_string;

	send_string  <<  "(reconnect " << team_name << " "
		     <<  player_number << ")" << endl  << ends;

	return( send_string.str() );
#else
	if ( team_name.length() > SSERVER_MAX_MESSAGE_LENGTH )
	{
		cerr << "teamname too long: \"" << team_name << "\"" << endl;
		return( "" );
	}

	if ( bad_player_number )
	{
		return( "" );
	}

	ref_count_ptr<char>	buf;
	try
	{
		buf = new char[ 1024 + SSERVER_MAX_MESSAGE_LENGTH + 1 ];
	}
	catch( ... )
	{
		cerr << "Can't allocate memory" << endl;
		throw;
	}

	sprintf( buf.get() , "(reconnect %s %d)\n" ,
		 team_name.c_str() , player_number );

	return( buf.get() );
#endif
}


string  SServer_Message_Translator::bye_command()
{
	return( "(bye)\n" );
}


string  SServer_Message_Translator::move_command( double  x ,  double  y )
{
#if ! defined(LIB_MEMORY)
	strstream	send_string;
	send_string.precision( DBL_DIG );

	send_string  <<  "(move " << x << " " << y << ")" << endl  <<  ends;

	return( send_string );
#else
	char	buf[1024 + DBL_DIG * 2];
	sprintf( buf , "(move %.*f %.*f)\n" ,
		 DBL_DIG , x , DBL_DIG , y );

	return( buf );
#endif
}


string  SServer_Message_Translator::dash_command( double  power )
{
#if ! defined(LIB_MEMORY)
	strstream	send_string;
	send_string.precision( DBL_DIG );

	send_string  <<  "(dash " << power << ")" << endl  << ends;

	return( send_string );
#else
	char	buf[1024 + DBL_DIG];
	sprintf( buf , "(dash %.*f)\n" , DBL_DIG , power );

	return( buf );
#endif
}


string  SServer_Message_Translator::kick_command( double  power ,
						  const Angle &  direction )
{
#if ! defined(LIB_MEMORY)
	strstream	send_string;
	send_string.precision( DBL_DIG );

	send_string  <<  "(kick " << power << " " << direction.degree() << ")"
		     <<  endl  << ends;

	return( send_string );
#else
	char	buf[1024 + DBL_DIG * 2];

	sprintf( buf , "(kick %.*f %.*f)\n" ,
		 DBL_DIG , power , DBL_DIG , direction.degree() );

	return( buf );
#endif
}

string  SServer_Message_Translator::turn_command( const Angle &  moment )
{
#if ! defined(LIB_MEMORY)
	strstream	send_string;
	send_string.precision( DBL_DIG );

	send_string  << "(turn " << moment.degree() << ")" << endl  << ends;

	return( send_string );
#else
	char	buf[1024 + DBL_DIG];

	sprintf( buf , "(turn %.*f)\n" , DBL_DIG , moment.degree() );

	return( buf );
#endif
}


string  SServer_Message_Translator::say_command( const string &  message )
{
	// no check for message here.
	// message must be consist of [a-zA-Z().+-*/?<>_].
	string	send_string = static_cast<string>("(say ") + message + ")\n";

	return( send_string );
}


string  SServer_Message_Translator::catch_command( const Angle &  direction )
{
#if ! defined(LIB_MEMORY)
	strstream	send_string;

	send_string  <<  "(catch " << direction.degree() << ")" << endl
		     << ends;

	return( send_string );
#else
	char	buf[1024 + DBL_DIG];

	sprintf( buf , "(catch %.*f)\n" , DBL_DIG , direction.degree() );

	return( buf );
#endif
}

std::string  SServer_Message_Translator::change_player_type_command
				    ( int  player_number ,  int  player_type )
{
	char	buf[1024 + (sizeof(int) * CHAR_BIT + 1) * 2 ];

	sprintf( buf , "(change_player_type %d %d)\n" ,
		 player_number , player_type );

	return( buf );
}


string  SServer_Message_Translator::change_view_command(
						View_Width  width ,
						View_Quality  quality )
{
	string	width_string;
	string	quality_string;

	switch( width )
	{
		case View_Width::Narrow:	width_string = "narrow"; break;
		case View_Width::Normal:	width_string = "normal"; break;
		case View_Width::Wide:		width_string = "wide"  ; break;
	}

	switch( quality )
	{
		case View_Quality::High:	quality_string = "high"; break;
		case View_Quality::Low:		quality_string = "low" ; break;
	}

	string	send_string = static_cast<string>("(change_view ")
				+ width_string + " " + quality_string + ")\n";

	return( send_string );
}


string  SServer_Message_Translator::sense_body_command()
{
	return( "(sense_body)" );
}


string  SServer_Message_Translator::turn_neck_command(
						const Angle &  direction )
{
#if ! defined(LIB_MEMORY)
	strstream	send_string;

	send_string  <<  "(turn_neck "
		     <<  direction.degree() << ")" << endl  << ends;

	return( send_string );
#else
	char	buf[1024 + DBL_DIG];

	sprintf( buf , "(turn_neck %.*f)\n" , DBL_DIG , direction.degree() );

	return( buf );
#endif
}


#if 1
//
// Coach Commands
//
string  SServer_Message_Translator::look_command()
{
	return( "(look)" );
}

string  SServer_Message_Translator::check_ball_command()
{
	return( "(check_ball)" );
}

string  SServer_Message_Translator::ear_command( bool  on_off )
{
	string	send_string = static_cast<string>("(err ")
		+ (on_off ? "on" : "off") + ")\n";

	return( send_string );
}

#if 0
string  SServer_Message_Translator::change_mode_command( Play_Mode  mode )
{
	string	send_string = (string)"(change_mode "
		+ play_mode_to_string( mode , our_side ) + ")\n";

	return( send_string );
}
#endif
#endif

//
// Monitor Commands
//
string  SServer_Message_Translator::dispinit_command( int  version )
{
	char *	buf;

	try
	{
		buf = new char[ 1024 + SSERVER_MAX_MESSAGE_LENGTH + 1 ];
	}
	catch( ... )
	{
		cerr << "Can't allocate memory" << endl;
		throw;
	}


	if ( version >= 2 )
	{
		sprintf( buf , "(dispinit version %d)" , version );
	}
	else
	{
		strcpy( buf , "(dispinit)" );
	}


	string	ret( buf );

	delete[] buf;

	return( ret );
}

string  SServer_Message_Translator::dispstart_command()
{
	return( "(dispstart)" );
}

string  SServer_Message_Translator::dispdiscard_command( S_Side_LR  side ,
							 int  player_number )
{
	int	side_number;

	switch( side )
	{
	case S_Side_LR::Left_Side:
		side_number = 1;
		break;


	case S_Side_LR::Right_Side:
		side_number = -1;
		break;

	case S_Side_LR::Unknown:
	default:
		return( "" );
		break;
	}

	char	buf[ 1024 + (sizeof(int) * CHAR_BIT + 1) * 2 ];

	sprintf( buf , "(dispdiscard %d %d)" , side_number , player_number );

	return( buf );
}

string  SServer_Message_Translator::dispfoul_command( double  x ,
						      double  y ,
						      S_Side_LR  side )
{
	int	side_number;

	switch( side )
	{
	case S_Side_LR::Left_Side:
		side_number = 1;
		break;

	case S_Side_LR::Right_Side:
		side_number = -1;
		break;

	case S_Side_LR::Unknown:
		side_number = 0;
		break;

	default:
		return( "" );
		break;
	}

	char	buf[ 1024 + (sizeof(int) * CHAR_BIT + 1) * 3 ];

	sprintf( buf , "(dispfoul %d %d %d)" ,
		 double_to_nearest_int( x * SHOWINFO_SCALE ) ,
		 double_to_nearest_int( y * SHOWINFO_SCALE ) ,
		 side_number );

	return( buf );
}


//
// Fuss Monitor Command
//
string  SServer_Message_Translator::fuss_monitor_init_command()
{
	return( "init" );
}




void   SServer_Message_Translator::parse_init_message
				 ( Game_Info *  g_info ,
				   const ref_count_ptr<const  string> &  mes )
{
	int			player_number;
	ref_count_ptr<char>	play_mode;
	try
	{
		play_mode = new char[ SSERVER_MAX_MESSAGE_LENGTH + 1 ];
	}
	catch( ... )
	{
		cerr << "Can't allocate memory" << endl;
		throw;
	}

	char	side;
	char	close_paren = '\0';

	// normal player init message
	if ( sscanf( mes -> c_str() , "(init %c %d %[a-z_]%c" ,
		     &side , &player_number , play_mode.get() ,
		     &close_paren ) == 4
	       && (side == 'l' || side == 'r')
	       && close_paren == ')' )
	{
		g_info -> our_side
			= ((side == 'l') ?
			   S_Side_LR::Left_Side : S_Side_LR::Right_Side);
		g_info -> self_player_number = player_number;
		g_info -> play_mode = string_to_play_mode(
						  play_mode.get() ,
						  g_info -> our_side );
	}
	// coach init message
	else if ( sscanf( mes -> c_str() ,
			  "(init %c ok%c" , &side , &close_paren ) == 2
	  && (side == 'l' || side == 'r')
	  && close_paren == ')' )
	{
		g_info -> our_side = ((side == 'l')
				      ? S_Side_LR::Left_Side
				      : S_Side_LR::Right_Side);
		g_info -> self_player_number = 0;
		g_info -> play_mode = Play_Mode::Unknown;
	}
	// error
	else
	{
		cerr << "\"" << (*mes) << "\" : ";
		cerr << "Unexpected initial message from server." << endl;

		g_info -> our_side           = S_Side_LR::Unknown;
		g_info -> self_player_number = -1;
		g_info -> play_mode          = Play_Mode::Unknown;
	}
}


void   SServer_Message_Translator::parse_reconnect_message
				   ( Game_Info *  g_info ,
				     const ref_count_ptr<const string> &  mes ,
				     int  player_number )
{
	char			side;
	ref_count_ptr<char>	play_mode;
	try
	{
		play_mode = new char[ SSERVER_MAX_MESSAGE_LENGTH + 1 ];
	}
	catch( ... )
	{
		cerr << "Can't allocate memory" << endl;
		throw;
	}

	if ( sscanf( mes -> c_str() , "(reconnect %c %[a-z_])" ,
		     &side , play_mode.get() ) != 2
	  || ! (side == 'l' || side == 'r') )
	{
		cerr << "\"" << (*mes) << "\" : ";
		cerr << "Unexpected reconnect message from server." << endl;

		g_info -> our_side           = S_Side_LR::Unknown;
		g_info -> self_player_number = -1;
		g_info -> play_mode          = Play_Mode::Unknown;
	}
	else
	{
		g_info -> our_side
			= ((side == 'l') ?
			   S_Side_LR::Left_Side : S_Side_LR::Right_Side);
		g_info -> self_player_number = player_number;
		g_info -> play_mode = string_to_play_mode(
						  play_mode.get() ,
						  g_info -> our_side );
	}
}


Play_Mode  SServer_Message_Translator::string_to_play_mode
						( const string &  mode ,
						  S_Side_LR  our_side )
{
	if ( mode  ==  "before_kick_off" )
	{
		return( Play_Mode::Before_Kick_Off );
	}
	else if ( mode  ==  "play_on" )
	{
		return( Play_Mode::Play_On );
	}
	else if ( mode  ==  "time_over" )
	{
		return( Play_Mode::Time_Over );
	}


	string	l_or_r;
	string	mode_base;

	if ( mode.length() < 2
	 ||  our_side == S_Side_LR::Unknown )
	{
		return( Play_Mode::Unknown );
	}

	l_or_r    = mode.substr( mode.length() - 2 , 2 );
	mode_base = mode.substr( 0 , mode.length() - 2 );

	if ( ! (l_or_r == "_l" || l_or_r == "_r" ) )
	{
		return( Play_Mode::Unknown );
	}

	S_Side	side;
	if ( (l_or_r == "_l" && our_side == S_Side_LR::Left_Side)
	  || (l_or_r == "_r" && our_side == S_Side_LR::Right_Side) )
	{
		side = S_Side::Our_Side;
	}
	else
	{
		side = S_Side::Opponent_Side;
	}

	if ( mode_base == "kick_off" )
	{
		if ( side == S_Side::Our_Side )
		{
			return( Play_Mode::Our_Kick_Off );
		}
		else
		{
			return( Play_Mode::Opponent_Kick_Off );
		}
	}
	else if ( mode_base == "kick_in" )
	{
		if ( side == S_Side::Our_Side )
		{
			return( Play_Mode::Our_Kick_In );
		}
		else
		{
			return( Play_Mode::Opponent_Kick_In );
		}
	}
	else if ( mode_base == "free_kick" )
	{
		if ( side == S_Side::Our_Side )
		{
			return( Play_Mode::Our_Free_Kick );
		}
		else
		{
			return( Play_Mode::Opponent_Free_Kick );
		}
	}
	else if ( mode_base == "corner_kick" )
	{
		if ( side == S_Side::Our_Side )
		{
			return( Play_Mode::Our_Corner_Kick );
		}
		else
		{
			return( Play_Mode::Opponent_Corner_Kick );
		}
	}
	else if ( mode_base == "goal_kick" )
	{
		if ( side == S_Side::Our_Side )
		{
			return( Play_Mode::Our_Goal_Kick );
		}
		else
		{
			return( Play_Mode::Opponent_Goal_Kick );
		}
	}
	else if ( mode_base == "goal" )
	{
		if ( side == S_Side::Our_Side )
		{
			return( Play_Mode::Our_Goal );
		}
		else
		{
			return( Play_Mode::Opponent_Goal );
		}
	}
	else
	{
		return( Play_Mode::Unknown );
	}
}


void   SServer_Message_Translator::parse_message
				   ( Server_Raw_Info *  info ,
				     const ref_count_ptr<const string> &  mes ,
				     S_Side_LR  our_side ,
				     const string &  team_name ,
				     int  self_player_number ,
				     bool  danger_parse )
{
	info -> reset_info();
	info -> set_time( -1 );
	info -> set_raw_message( mes );


	Server_Raw_Info	illegal_info;
	illegal_info.reset_info();
	illegal_info.set_time( -1 );
	illegal_info.set_raw_message( mes );

	if ( ! mes  ||  mes -> length() == 0 )
	{
		info -> set_no_info();
		return;
	}

	if ( strncmp( mes -> c_str() ,
		      "(hear " , strlen("(hear ") ) == 0 )
	{
		// analyze audio and whistle info.
		if ( ! analyze_audio_info_brief( info , *mes , our_side ) )
		{
			info -> reset_info();
		}

		return;
	}
#if ! defined(LISP_LEX_PARSE)
	else if ( strncmp( mes -> c_str() ,
			   "(sense_body " , strlen("(sense_body ") ) == 0 )
	{
		// analyze body info.
		if ( ! analyze_body_info_brief( info , *mes ) )
		{
			info -> reset_info();
		}

		return;
	}
	else if ( strncmp( mes -> c_str() ,
			   "(see " , strlen("(see ") ) == 0 )
	{
		// analyze sight info.
		if ( danger_parse )
		{
			if ( ! analyze_sight_info_danger( info ,
							  *mes ,
							  our_side ,
							  team_name ) )
			{
				info -> reset_info();
			}
		}
		else
		{
			if ( ! analyze_sight_info_brief( info ,
							 *mes ,
							 our_side ,
							 team_name ) )
			{
				info -> reset_info();
			}
		}

		return;
	}
#endif
	else if ( strncmp( mes -> c_str() ,
			   "(server_param " , strlen("(server_param ") ) == 0 )
	{
		// analyze server_param
		if ( ! analyze_server_param_info_brief( info , *mes ) )
		{
			info -> reset_info();
		}

		return;
	}
	else if ( strncmp( mes -> c_str() ,
			   "(player_param " , strlen("(player_param ") ) == 0 )
	{
		// analyze player_param
		if ( ! analyze_player_param_info_brief( info , *mes ) )
		{
			info -> reset_info();
		}

		return;
	}
	else if ( strncmp( mes -> c_str() ,
			   "(player_type " , strlen("(player_type ") ) == 0 )
	{
		// analyze player_type
		if ( ! analyze_player_type_info_brief( info , *mes ) )
		{
			info -> reset_info();
		}

		return;
	}
	else if ( strncmp( mes -> c_str() ,
			   "(fullstate " , strlen("(fullstate ") ) == 0 )
	{
		// analyze debug info.
		if ( ! analyze_debug_info( info , *mes ,
					   our_side , self_player_number ) )
		{
			info -> reset_info();
		}

		return;
	}


	Lisp_Lexical_Analyser	lisp_lex;
	if ( ! lisp_lex.analyze( *mes ) )
	{
		info -> reset_info();
		info -> set_time( -1 );
		info -> set_raw_message( mes );
		return;
	}
	Lisp_Lexical_Object	input = lisp_lex.result();

	// Make sure each input format is this.
	//	(% % ...)
	// % : symbol
	if ( ! (   input.list.size() == 1
		&& input.list[0].type == Lisp_Lexical_Object::List
		&& input.list[0].has_right_paren == true
		&& input.list[0].list.size() >= 2
		&& input.list[0].list[0].type == Lisp_Lexical_Object::Symbol
		&& input.list[0].list[1].type == Lisp_Lexical_Object::Symbol) )
	{
		return;
	}

	long	info_time;
	if ( input.list[0].list[0].symbol == "error" )
	{
		analyze_error_info( info , input.list[0] );
		return;
	}
	else if ( ! str_to_long( &info_time , input.list[0].list[1].symbol ) )
	{
		info -> reset_info();
		info -> set_time( -1 );
		info -> set_raw_message( mes );
		return;
	}

	info -> set_time( info_time );

	string	info_type = input.list[0].list[0].symbol;
#ifdef LISP_LEX_PARSE
	static_cast<void>( danger_parse );	// unused
	if ( info_type == "see" )
	{
		// analyze sight info.
		analyze_sight_info( info , input.list[0] ,
				    our_side , team_name );
	}
#else
	if ( 0 )
	{
	}
#endif
#ifdef LISP_LEX_PARSE
	else if ( info_type == "sense_body" )
	{
		// analyze body info.
		analyze_body_info( info , input.list[0] );
	}
#endif
	else
	{
		info -> reset_info();
		info -> set_time( -1 );
		info -> set_raw_message( mes );
	}
}


#ifdef LISP_LEX_PARSE
// only for SServer_Message_Translator::analyze_sight_info()
bool   SServer_Message_Translator::analyze_sight_info_check_format(
					   const Lisp_Lexical_Object &  obj )
{
	// Make sure each input entry's format is this.
	//	     ((...) *)
	//	  || ((...) * *)
	//	  || ((...) * * * *)
	//	  || ((...) * * * * *)
	//	  || ((...) * * * * * * *)
	if ( obj.type != Lisp_Lexical_Object::List
	 ||   (    obj.list.size() != 2
	        && obj.list.size() != 3
	        && obj.list.size() != 5
	        && obj.list.size() != 6
	        && obj.list.size() != 7 )
	 ||  obj.list[0].type != Lisp_Lexical_Object::List )
	{
		return( false );
	}

	// Make sure each input entry's format is this.
	//	     ((% ...) %)
	//	  || ((% ...) % %)
	//	  || ((% ...) % % % %)
	//	  || ((% ...) % % % % %)
	//	  || ((% ...) % % % % % % %)
	for( unsigned int  i = 0  ;  i  < obj.list[0].list.size()  ;  i ++ )
	{
		if ( obj.list[0].list[i].type
		     != Lisp_Lexical_Object::Symbol )
		{
			return( false );
		}
	}

	// Make sure each input entry's format is this.
	//	     ((% ...) %)
	//	  || ((% ...) % %)
	//	  || ((% ...) % % % %)
	//	  || ((% ...) % % % % %)
	//	  || ((% ...) % % % % % % %)
	for( unsigned int  i = 1  ;  i  < obj.list.size()  ;  i ++ )
	{
		if ( obj.list[i].type != Lisp_Lexical_Object::Symbol )
		{
			return( false );
		}
	}

	return( true );
}

// only for SServer_Message_Translator::analyze_sight_info()
bool   SServer_Message_Translator::analyze_sight_info_get_relative(
			Raw_SObject_Locational_Info *  relative ,
			const Lisp_Lexical_Object &  obj )
{
	//
	// get locational information
	//
	relative -> have_distance       = false;
	relative -> have_direction      = false;
	relative -> have_d_distance     = false;
	relative -> have_d_direction    = false;
	relative -> have_body_direction = false;
	relative -> have_face_direction = false;

	relative -> distance       = 0.0;
	relative -> direction      = Radian(0.0);
	relative -> d_distance     = 0.0;
	relative -> d_direction    = Radian(0.0);
	relative -> body_direction = Radian(0.0);
	relative -> face_direction = Radian(0.0);


	if ( obj.list.size() >= 8 )
	{
		return( false );
	}

	if ( obj.list.size() >= 7 )
	{
		relative -> have_face_direction = true;
		if ( ! server_str_to_dir( &(relative -> face_direction) ,
					  obj.list[6].symbol ) )
		{
			return( false );
		}
	}

	if ( obj.list.size() >= 6 )
	{
		relative -> have_body_direction = true;
		if ( ! server_str_to_dir( &(relative -> body_direction) ,
					  obj.list[5].symbol ) )
		{
			return( false );
		}
	}

	if ( obj.list.size() >= 5 )
	{
		relative -> have_d_direction = true;
		relative -> have_d_distance  = true;

		if ( ! server_str_to_dir( &(relative -> d_direction) ,
					  obj.list[4].symbol )
		  || ! str_to_real( &(relative -> d_distance) ,
				    obj.list[3].symbol ) )
		{
			return( false );
		}
	}

	if ( obj.list.size() >= 3 )
	{
		relative -> have_direction = true;
		relative -> have_distance  = true;

		if ( ! server_str_to_dir( &(relative -> direction) ,
					  obj.list[2].symbol )
		  || ! str_to_real( &(relative -> distance) ,
				    obj.list[1].symbol ) )
		{
			return( false );
		}
	}
	else // if ( obj.list.size() == 2 )
	{
		relative -> have_direction = true;

		if ( ! server_str_to_dir( &relative -> direction ,
					  obj.list[1].symbol ) )
		{
			return( false );
		}
	}

	return( true );
}


#include  "fixed_sobject_location.h"
void   SServer_Message_Translator::analyze_sight_info(
					    Server_Raw_Info * info ,
					    Lisp_Lexical_Object & input ,
					    S_Side_LR  our_side ,
					    const string &  team_name )
{
	static	Fixed_SObject_Location_Translator	obj_to_location;

	// Make sure input format is this.
	//	(* * ...)
	if ( input.list.size()  <  2 )
	{
		info -> reset_info();
		return;
	}


	ref_count_ptr<Server_Raw_Sight_Info>
		sight_info( new Server_Raw_Sight_Info );

	string	quoted_team_name = string("\"") + team_name + "\"";

	for ( unsigned int  i = 2  ;  i < input.list.size()  ;  i ++ )
	{
		Lisp_Lexical_Object &	obj = input.list[i];

		if ( ! analyze_sight_info_check_format( obj ) )
		{
			info -> reset_info();
			return;
		}

		Raw_SObject_Locational_Info	relative;
		if ( ! analyze_sight_info_get_relative( &relative , obj ) )
		{
			info -> reset_info();
			return;
		}

		//
		// Get object identification and set to Sight_Info_from_Server
		//

		Lisp_Lexical_Object &	obj_ident = obj.list[0];
		string	obj_type = obj_ident.list[0].symbol;

		if ( obj_type == "f"    || obj_type == "F"
		  || obj_type == "flag" || obj_type == "Flag" )
		{
			if ( ! (   (2 <= obj_ident.list.size()
				    &&   obj_ident.list.size() <= 4)
			        || (obj_type[0] == 'F'
				    && obj_ident.list.size() == 1) ) )
			{
				info -> reset_info();
				return;
			}

			Field_Marker_Info_from_Server	flag_info;
			flag_info.relative = relative;
			flag_info.relative.in_sight = (obj_type[0] == 'f');
			flag_info.entity.marker_flag = true;

			flag_info.entity.location_known
				= obj_to_location.get_location(
				      &flag_info.entity.location ,
				      obj_ident , our_side );

			sight_info -> field_marker.push_back( flag_info );
		}

		else if ( obj_type == "p"      || obj_type == "P"
		       || obj_type == "player" || obj_type == "Player" )
		{
			Player_Info_from_Server	player_info;
			player_info.relative = relative;
			player_info.relative.in_sight = (obj_type[0] == 'p');

			if ( obj_type[0] == 'p' )
			{
				if ( ! (obj_ident.list.size() <= 4) )
				{
					info -> reset_info();
					return;
				}

				if ( obj_ident.list.size() < 2 )
				{
					player_info.entity.side
						= S_Side::Unknown;
				}
				else if ( obj_ident.list[1].symbol
						    == quoted_team_name
				       || obj_ident.list[1].symbol
						    == team_name )
				{
					player_info.entity.side
						= S_Side::Our_Side;
				}
				else
				{
					player_info.entity.side
						= S_Side::Opponent_Side;
				}

				if ( obj_ident.list.size() < 3 )
				{
					player_info.entity.player_number = -1;

					player_info.have_goalie_info = false;
					player_info.goalie = false;
				}
				else
				{
					str_to_int(
					   &player_info.entity.player_number ,
					   obj_ident.list[2].symbol );

					if (! (1 <= player_info.entity
						     .player_number
					        &&  player_info.entity
						     .player_number
						<= MAX_PLAYER) )
					{
						info -> reset_info();
						return;
					}

					player_info.have_goalie_info = true;

					if ( obj_ident.list.size() >= 4 )
					{
						if ( obj_ident.list[3].symbol
						     != "goalie" )
						{
							info -> reset_info();
							return;
						}
						else
						{
							player_info.goalie
								= true;
						}
					}
					else
					{
						player_info.goalie = false;
					}
				}
			}
			else // if ( obj_type == "P" || obj_type == "Player" )
			{
				if ( obj_ident.list.size() != 1 )
				{
					info -> reset_info();
					return;
				}

				player_info.entity.side = S_Side::Unknown;
				player_info.entity.player_number = -1;
			}

			sight_info -> player.push_back( player_info );
		}
		else if ( obj_type == "b"     ||  obj_type == "B"
		       || obj_type == "ball"  ||  obj_type == "Ball" )
		{
			if ( obj_ident.list.size() != 1 )
			{
				info -> reset_info();
				return;
			}

			Ball_Info_from_Server	ball_info;
			ball_info.relative = relative;
			ball_info.relative.in_sight = (obj_type[0] == 'b');

			sight_info -> ball.push_back( ball_info );
		}
		else if ( obj_type == "g"    || obj_type == "G"
		       || obj_type == "goal" || obj_type == "Goal" )
		{
			Field_Marker_Info_from_Server	goal_info;
			goal_info.relative = relative;
			goal_info.relative.in_sight = (obj_type[0] == 'g');

			goal_info.entity.marker_flag = false;

			if ( obj_type[0] == 'g' )
			{
				const string	side_str
						  = obj_ident.list[1].symbol;
				if ( ! ( obj_ident.list.size() == 2
				      && (side_str == "l"
				       || side_str == "r" ) ) )
				{
					info -> reset_info();
					return;
				}

				goal_info.entity.location_known
					= obj_to_location.get_location(
						&goal_info.entity.location ,
						obj_ident , our_side );
			}
			else // if ( obj_type == "G" || obj_type == "Goal" )
			{
				if ( ! (obj_ident.list.size() == 1) )
				{
					info -> reset_info();
					return;
				}

				goal_info.entity.location_known = false;
				goal_info.entity.location.set( D2_Vector::XY ,
							       0.0 , 0.0 );
			}

			sight_info -> field_marker.push_back( goal_info );
		}
		else if ( obj_type == "l" || obj_type == "line" )
		{
			if ( obj_ident.list.size() != 2 )
			{
				info -> reset_info();
				return;
			}

			Line_Info_from_Server	line_info;
			line_info.relative = relative;
			line_info.relative.in_sight = true;

			const string	ln_str = obj_ident.list[1].symbol;

			if ( (ln_str == "l"
				  && our_side == S_Side_LR::Left_Side)
			  || (ln_str == "r"
				  && our_side == S_Side_LR::Right_Side) )
			{
				line_info.entity.which_line
				 = SObject_Line_Identifier::Our_Goal_Line;
			}
			else if ( (ln_str == "l"
				       && our_side == S_Side_LR::Right_Side)
			       || (ln_str == "r"
				       && our_side == S_Side_LR::Left_Side) )
			{
				line_info.entity.which_line
				 = SObject_Line_Identifier::Opponent_Goal_Line;
			}
			else if ( (ln_str == "b"
				       && our_side == S_Side_LR::Left_Side)
			       || (ln_str == "t"
				       && our_side == S_Side_LR::Right_Side) )
			{
				line_info.entity.which_line
				 = SObject_Line_Identifier::Left_Wing_Line;
			}
			else if ( (ln_str == "b"
				       && our_side == S_Side_LR::Right_Side)
			       || (ln_str == "t"
				       && our_side == S_Side_LR::Left_Side) )
			{
				line_info.entity.which_line
				 = SObject_Line_Identifier::Right_Wing_Line;
			}
			else
			{
				info -> reset_info();
				return;
			}

			sight_info -> line.push_back( line_info );
		}
		else
		{
			info -> reset_info();
			return;
		}
	}

	info -> set_sight_info( sight_info );
}
#endif

#if ! defined(LISP_LEX_PARSE)
// XXX
//   The follow code is nearly same as analyze_sight_info*,
//   but this code is faster.
//
// only for SServer_Message_Translator::analyze_sight_info_brief()

#include  "fixed_sobject_location.h"
bool   SServer_Message_Translator::analyze_sight_info_brief_get_obj_info(
		  const ref_count_ptr<Server_Raw_Sight_Info> &  sight_info ,
		  int  obj_ident_size ,
		  char **  obj_ident ,
		  const Raw_SObject_Locational_Info &  relative ,
		  S_Side_LR  our_side ,
		  const string &  team_name )
{
	static	Fixed_SObject_Location_Translator	obj_to_location;

	if ( obj_ident_size <= 0 )
	{
		return( false );
	}

	const string	obj_type( obj_ident[0] );

	string	quoted_team_name = string("\"") + team_name + "\"";

	if ( obj_type == "f"    || obj_type == "F"
	  || obj_type == "flag" || obj_type == "Flag" )
	{
		if ( ! (    (2 <= obj_ident_size &&  obj_ident_size <= 4)
			 || (obj_type[0] == 'F' && obj_ident_size == 1)  ) )
		{
			return( false );
		}

		Raw_Field_Marker_Info	marker_info;
		marker_info.relative = relative;
		marker_info.relative.in_sight = (obj_type[0] == 'f');
		marker_info.entity.location_known
		 = obj_to_location.get_location( &marker_info
						   .entity.location ,
						 obj_ident_size , obj_ident ,
						 our_side );

		marker_info.entity.marker_flag = true;
		sight_info -> field_marker.push_back( marker_info );
	}
	else if ( obj_type == "p"      || obj_type == "P"
	       || obj_type == "player" || obj_type == "Player" )
	{
		Raw_Player_Info		player_info;
		player_info.relative = relative;
		player_info.relative.in_sight = (obj_type[0] == 'p');

		if ( obj_type[0] == 'p' )
		{
			if ( ! (obj_ident_size <= 4) )
			{
				return( false );
			}

			if ( obj_ident_size < 2 )
			{
				player_info.entity.side = S_Side::Unknown;
			}
			else if ( obj_ident[1] == quoted_team_name
			       || obj_ident[1] == team_name )
			{
				player_info.entity.side = S_Side::Our_Side;
			}
			else
			{
				player_info.entity.side
					= S_Side::Opponent_Side;
			}

			if ( obj_ident_size < 3 )
			{
				player_info.entity.player_number = -1;

				player_info.have_goalie_info = false;
				player_info.goalie = false;
			}
			else
			{
				str_to_int( &player_info
					      .entity.player_number ,
					    obj_ident[2] );

				if (! (1 <= player_info.entity.player_number
				       &&  player_info.entity.player_number
				       <= MAX_PLAYER) )
				{
					return( false );
				}

				player_info.have_goalie_info = true;

				if ( obj_ident_size >= 4 )
				{
					if ( strcmp( obj_ident[3] ,
						     "goalie" ) != 0 )
					{
						return( false );
					}
					else
					{
						player_info.goalie = true;
					}
				}
				else
				{
					player_info.goalie = false;
				}
			}
		}
		else // if ( obj_type == "P" || obj_type == "Player" )
		{
			if ( obj_ident_size != 1 )
			{
				return( false );
			}

			player_info.entity.side = S_Side::Unknown;
			player_info.entity.player_number = -1;
		}

		sight_info -> player.push_back( player_info );
	}
	else if ( obj_type == "b"     ||  obj_type == "B"
	       || obj_type == "ball"  ||  obj_type == "Ball" )
	{
		if ( obj_ident_size != 1 )
		{
			return( false );
		}

		Raw_Ball_Info	ball_info;
		ball_info.relative = relative;
		ball_info.relative.in_sight = (obj_type[0] == 'b');

		sight_info -> ball.push_back( ball_info );
	}
	else if ( obj_type == "g"    || obj_type == "G"
	       || obj_type == "goal" || obj_type == "Goal" )
	{
		Raw_Field_Marker_Info	marker_info;
		marker_info.relative = relative;
		marker_info.relative.in_sight = (obj_type[0] == 'g');

		if ( obj_type[0] == 'g' )
		{
			const string	side_str = obj_ident[1];
			if ( ! ( obj_ident_size == 2
				 && (side_str == "l"
				     || side_str == "r" ) ) )
			{
				return( false );
			}

			marker_info.entity.location_known
				= obj_to_location.get_location(
					     &marker_info.entity.location ,
					     obj_ident_size , obj_ident ,
					     our_side );
		}
		else // if ( obj_type == "G" || obj_type == "Goal" )
		{
			if ( ! (obj_ident_size == 1) )
			{
				return( false );
			}

			marker_info.entity.location_known = false;
			marker_info.entity.location.set( D2_Vector::XY ,
							 0.0 , 0.0 );
		}
		marker_info.entity.marker_flag = false;
		sight_info -> field_marker.push_back( marker_info );
	}
	else if ( obj_type == "l" || obj_type == "line" )
	{
		if ( obj_ident_size != 2 )
		{
			return( false );
		}

		Raw_Line_Info	line_info;
		line_info.relative = relative;
		line_info.relative.in_sight = true;

		const string	ln_str = obj_ident[1];

		if ( (ln_str == "l"
		      && our_side == S_Side_LR::Left_Side)
		     || (ln_str == "r"
			 && our_side == S_Side_LR::Right_Side) )
		{
			line_info.entity.which_line
				= SObject_Line_Identifier::Our_Goal_Line;
		}
		else if ( (ln_str == "l"
			   && our_side == S_Side_LR::Right_Side)
			  || (ln_str == "r"
			      && our_side == S_Side_LR::Left_Side) )
		{
			line_info.entity.which_line
				= SObject_Line_Identifier::Opponent_Goal_Line;
		}
		else if ( (ln_str == "b"
			   && our_side == S_Side_LR::Left_Side)
			  || (ln_str == "t"
			      && our_side == S_Side_LR::Right_Side) )
		{
			line_info.entity.which_line
				= SObject_Line_Identifier::Left_Wing_Line;
		}
		else if ( (ln_str == "b"
			   && our_side == S_Side_LR::Right_Side)
			  || (ln_str == "t"
			      && our_side == S_Side_LR::Left_Side) )
		{
			line_info.entity.which_line
				= SObject_Line_Identifier::Right_Wing_Line;
		}
		else
		{
			return( false );
		}

		sight_info -> line.push_back( line_info );
	}
	else
	{
		return( false );
	}

	return( true );
}


// XXX
// only for SServer_Message_Translator::analyze_sight_info_brief()
bool   SServer_Message_Translator::analyze_sight_info_brief_get_relative(
		 Raw_SObject_Locational_Info *  relative ,
		 int  n_data ,  double  data[] )
{
	//
	// get locational information
	//
#if 0
	relative -> have_distance       = false;
	relative -> have_direction      = false;
	relative -> have_d_distance     = false;
	relative -> have_d_direction    = false;
	relative -> have_body_direction = false;
	relative -> have_face_direction = false;

	relative -> distance       = 0.0;
	relative -> direction      = Radian(0.0);
	relative -> d_distance     = 0.0;
	relative -> d_direction    = Radian(0.0);
	relative -> body_direction = Radian(0.0);
	relative -> face_direction = Radian(0.0);
#endif


	if ( n_data >= 7 )
	{
		return( false );
	}

	if ( n_data >= 6 )
	{
		relative -> have_face_direction = true;
		relative -> face_direction = Degree( data[5] );
	}

	if ( n_data >= 5 )
	{
		relative -> have_body_direction = true;
		relative -> body_direction     = Degree( data[4] );
	}

	if ( n_data >= 4 )
	{
		relative -> have_d_direction = true;
		relative -> have_d_distance  = true;

		relative -> d_direction = Degree( data[3] );
		relative -> d_distance  = data[2];
	}

	if ( n_data >= 2 )
	{
		relative -> have_direction = true;
		relative -> have_distance  = true;

		relative -> direction = Degree( data[1] );
		relative -> distance = data[0];
	}
	else // if ( n_data == 1 )
	{
		relative -> have_direction = true;

		relative -> direction = Degree( data[0] );
	}

	return( true );
}

bool   SServer_Message_Translator::analyze_sight_info_brief(
				    Server_Raw_Info *  info ,
				    const string &  mes ,
				    S_Side_LR  our_side ,
				    const string &  team_name )
{
	static	Fixed_SObject_Location_Translator	obj_to_location;
	ref_count_ptr<Server_Raw_Sight_Info>
		sight( new Server_Raw_Sight_Info );

	long	info_time;
	char	next_char;
	int	start_point;

	if ( sscanf( mes.c_str() ,
		     "(see %ld %n%c" ,
		     &info_time , &start_point , &next_char ) != 2 )
	{
		return( false );
	}

	info -> set_time( info_time );

	if ( next_char == ')' )
	{
		info -> set_sight_info( sight );

		return( true );
	}

	if ( next_char != '(' )
	{
		return( false );
	}


	const char *	ptr = mes.c_str() + start_point;

	// XXX: magic number 256
	char	obj_id_entity[4][256 + 1];
	char *	obj_id[4];
	int	obj_id_n[4];
	int	obj_id_n_match;

	for ( size_t  i = 0  ;  i < 4  ;  i ++ )
	{
		obj_id[i] = obj_id_entity[i];
	}

	for(;;)
	{
		// XXX: magic number 256
		obj_id_n_match = sscanf( ptr ,
					 " ((%256[^() ] %n%256[^() ] %n"
				            "%256[^() ] %n%256[^() ] %n" ,
					 obj_id[0] , obj_id_n + 0 ,
					 obj_id[1] , obj_id_n + 1 ,
					 obj_id[2] , obj_id_n + 2 ,
					 obj_id[3] , obj_id_n + 3 );

		if ( obj_id_n_match <= 0 )
		{
			return( false );
		}

		if ( *(ptr += obj_id_n[obj_id_n_match - 1]) != ')' )
		{
			return( false );
		}

		ptr ++;


		double	rel[6];
		int	rel_n[6];
		int	rel_n_match;

		rel_n_match = sscanf( ptr ,
				      "%lf%n%lf%n%lf%n%lf%n%lf%n%lf%n" ,
				      &rel[0] , &rel_n[0] ,
				      &rel[1] , &rel_n[1] ,
				      &rel[2] , &rel_n[2] ,
				      &rel[3] , &rel_n[3] ,
				      &rel[4] , &rel_n[4] ,
				      &rel[5] , &rel_n[5] );

		if ( rel_n_match <= 0 )
		{
			return( false );
		}

		Raw_SObject_Locational_Info	relative;

		if ( ! analyze_sight_info_brief_get_relative( &relative ,
							      rel_n_match ,
							      rel ) )
		{
			return( false );
		}


		if ( *(ptr += rel_n[rel_n_match - 1]) != ')' )
		{
			return( false );
		}

		ptr ++;

		if ( ! analyze_sight_info_brief_get_obj_info(
							sight ,
							obj_id_n_match ,
							obj_id ,
							relative ,
							our_side ,
							team_name ) )
		{
			return( false );
		}

		if ( *ptr == ')' )
		{
			ptr ++;

			if ( strcmp( ptr , "\n" ) != 0
			 &&  strcmp( ptr , "" )   != 0 )
			{
				return( false );
			}
			else
			{
				break;
			}
		}
	}

	info -> set_sight_info( sight );

	return( true );
}


bool   SServer_Message_Translator::analyze_sight_info_danger(
					Server_Raw_Info *  info ,
					const std::string &  mes ,
					S_Side_LR  our_side ,
					const std::string &  team_name )
{
	ref_count_ptr<Server_Raw_Sight_Info>
			sight( new Server_Raw_Sight_Info );

	const char *	p = mes.c_str();

	//
	// set information time
	//
	long	info_time = strtol( p + strlen( "see " ) ,
				    const_cast<char **>( &p ) , 10 );

	if ( info_time == LONG_MIN || info_time == LONG_MAX )
	{
		return( false );
	}

	info -> set_time( info_time );


	if ( *p == ')' )
	{
		info -> set_sight_info( sight );

		return( true );
	}

	if ( *p++ != ' ' ) /* skip a white space */
	{
		return( false );
	}

	// XXX: magic number 256
	char	obj_id_entity[4][256 + 1];
	char *	obj_id[4];

	for ( size_t  i = 0  ;  i < 4  ;  i ++ )
	{
		obj_id[i] = obj_id_entity[i];
	}

	//
	// read each object information
	//
	for(;;) /* return if close paren found */
	{
		//
		// if close paren found
		//
		if ( *p == ')' )
		{
			p ++;

			if ( *p == '\n' || *p == ' ' )
			{
				p ++;
			}

			if ( *p != '\0' )
			{
				return( false );
			}

			info -> set_sight_info( sight );

			return( true );
		}


		//
		// object identification
		//

		// skip "(("
		if ( *p++ != '(' || *p++ != '(' )
		{
			return( false );
		}

		int	n_obj_id = 0;
		for (;;)
		{
			if ( *p == ' ' )
			{
				p ++;
			}
			else if ( *p == ')' )
			{
				p ++;
				break;
			}

			if ( n_obj_id
			     >= static_cast<int>( sizeof(obj_id)
						  / sizeof(obj_id[0])) )
			{
				break;
			}

			char *  cp;
			char *	cp_end = obj_id[n_obj_id] + 256 - 1;
			for( cp = obj_id[n_obj_id]  ;
			     (*p != ' ' && *p != ')') && cp < cp_end  ;
			     cp ++ )
			{
				*cp = *p;
				p ++;
			}

			*cp = '\0';
			n_obj_id ++;
		}


		//
		// object relative location
		//
		int	n_data = 0;
		double  data[7];

		for ( n_data = 0  ;
		      n_data < static_cast<int>( sizeof(data)
						 / sizeof(data[0]))  ;
		      n_data ++ )
		{
			if ( *p == ')' )
			{
				p ++;

				if ( *p == ' ' )
				{
					p ++;
				}

				break;
			}

			const char *	old_p = p;
			double	val = strtod( p , const_cast<char **>( &p ) );

			if ( p == old_p )
			{
				return( false );
			}

			if ( val == - HUGE_VAL || val == HUGE_VAL )
			{
				return( false );
			}

			data[n_data] = val;
		}


		Raw_SObject_Locational_Info	relative;

		if ( ! analyze_sight_info_brief_get_relative
		       ( &relative , n_data , data ) )
		{
			return( false );
		}

		if ( ! analyze_sight_info_brief_get_obj_info
		       ( sight , n_obj_id , obj_id , relative ,
			 our_side , team_name ) )
		{
			return( false );
		}
	}

	return( true );
}
#endif



bool   SServer_Message_Translator::analyze_audio_info_brief(
					    Server_Raw_Info * info ,
					    const string &  mes ,
					    S_Side_LR  our_side )
{
	long	info_time;

	ref_count_ptr<char>	from;
	from = new char[SSERVER_MAX_MESSAGE_LENGTH + 1];

	int	n;

	bool	new_hear_format = false;
	static	char	teamname[2048];
	int	player_number;

	{
		static	bool	initialized = false;
		static	char	format[2048
				       + (sizeof(int) * CHAR_BIT + 1) * 2];
		static	char	format2[4096
					+ (sizeof(int) * CHAR_BIT + 1) * 3];

		if ( ! initialized )
		{
			sprintf( format ,
				 "(hear %%ld %%%ds %%n" ,
				 static_cast<int>(SSERVER_MAX_MESSAGE_LENGTH)
				 );

			sprintf( format2 ,
				 "(hear %%ld ( %%%ds %%%ds %%d)%%n" ,
				 static_cast<int>(SSERVER_MAX_MESSAGE_LENGTH) ,
				 static_cast<int>(SSERVER_MAX_MESSAGE_LENGTH)
				 );

			initialized = true;
		}

		if ( sscanf( mes.c_str() , format ,
			     &info_time , from.get() , &n ) != 2
		     || from.get()[0] == '(' )
		{
			if ( sscanf( mes.c_str() , format2 ,
				     &info_time ,
				     from.get() ,
				     teamname , &player_number , &n ) == 4 )
			{
				new_hear_format = true;
			}
			else
			{
				return( false );
			}
		}
	}


	info -> set_time( info_time );

	string	m( mes.c_str() + n );
	size_t	m_len = m.length();

	if ( m_len >= 1  &&  m[ m_len - 1 ] == '\n' )
	{
		m_len --;
		m.resize( m_len );
	}

	if ( m_len < 1  ||  m[ m_len - 1 ] != ')' )
	{
		return( false );
	}

	m_len --;
	m.resize( m_len );

	const string	message( m );

	if ( strcmp( from.get() , "referee" ) == 0 )
	{
		ref_count_ptr<Server_Raw_Whistle_Info>
			wi( new Server_Raw_Whistle_Info );
		wi -> type = Judgement_Type::Unknown;
		wi -> play_mode = Play_Mode::Unknown;
		wi -> score = 0;
		wi -> side = S_Side::Unknown;
		wi -> player_number = -1;

		Play_Mode	play_mode;
		play_mode = SServer_Message_Translator::string_to_play_mode(
							message , our_side );

		if ( play_mode != Play_Mode::Unknown )
		{
			wi -> type = Judgement_Type::One_of_Play_Modes;
			wi -> play_mode = play_mode;
		}
		else if ( message == "half_time" )
		{
			wi -> type = Judgement_Type::Half_Time;
		}
		else if ( message == "time_up" )
		{
			wi -> type = Judgement_Type::Time_Up;
		}
		else if ( message == "time_up_without_a_team" )
		{
			wi -> type = Judgement_Type::
				      Time_Up_Without_A_Team;
		}
		else if ( message == "time_extended" )
		{
			wi -> type = Judgement_Type::Time_Extended;
		}
		else if ( message.length() == strlen("goalie_catch_ball_?")
		       && (   message[strlen("goalie_catch_ball_")] == 'l'
			   || message[strlen("goalie_catch_ball_")] == 'r' ) )
		{
			wi -> type = Judgement_Type::Goalie_Catch_Ball;
			wi -> side
			  = lr_to_side( message[strlen("goalie_catch_ball_")] ,
					our_side );
		}
		else if ( message.length() >= strlen("goal_?_?")
		      &&  message.substr(0 , strlen("goal_")) == "goal_"
		      &&  (   message[strlen("goal_")] == 'l'
			   || message[strlen("goal_")] == 'r' )
		      &&  message[strlen("goal_") + 1] == '_' )
		{
			wi -> type = Judgement_Type::Goal;
			wi -> side = lr_to_side( message[strlen("goal_")] ,
						 our_side );
			if ( !(str_to_int( &(wi -> score) ,
					   message.substr(
						     strlen("goal_?_")) )) )
			{
				info -> reset_info();
			}
		}
		else if ( message.length() >= strlen("foul_?")
		      &&  message.substr(0 , strlen("foul_")) == "foul_"
		      &&  (message[strlen("foul_")] == 'l'
			|| message[strlen("foul_")] == 'r' ) )
		{
			wi -> type = Judgement_Type::Foul;
			wi -> side = lr_to_side( message[strlen("foul_")] ,
						 our_side );

			if ( message.length() >= strlen("foul_?_" ) )
			{
				if ( message[strlen("foul_") + 1] != '_'
				 || !(str_to_int( &(wi -> player_number) ,
					message.substr(strlen("foul_?_")) )) )
				{
					info -> reset_info();
				}
			}
			else
			{
				wi -> player_number = -1;
			}

		}
		else if ( message == "drop_ball" )
		{
			wi -> type = Judgement_Type::Drop_Ball;
		}
		else if ( message.length() >= strlen("offside_?")
		      &&  message.substr(0 , strlen("offside_")) == "offside_"
		      &&  (message[strlen("offside_")] == 'l'
			|| message[strlen("offside_")] == 'r') )
		{
			wi -> type = Judgement_Type::Offside;
			wi -> side = lr_to_side( message[strlen("offside_")] ,
						 our_side );
		}
		else
		{
			return( false );
		}

		info -> set_whistle_info( wi );
	}
	else if ( strcmp( from.get() , "online_coach_left"  ) == 0
	       || strcmp( from.get() , "online_coach_right" ) == 0 )
	{
		ref_count_ptr<Server_Raw_Audio_Info>
			au( new Server_Raw_Audio_Info );

		au -> message = message;

		if ( strcmp( from.get() , "online_coach_left" ) == 0
		     && our_side == S_Side_LR::Left_Side
		  || strcmp( from.get() , "online_coach_right" ) == 0
		     && our_side == S_Side_LR::Right_Side )
		{
			au -> sender
			  = Server_Raw_Audio_Info::ST_Our_Online_Coach;
		}
		else if ( strcmp( from.get() , "online_coach_right" ) == 0
			  && our_side == S_Side_LR::Left_Side
		       || strcmp( from.get() , "online_coach_left" ) == 0
			  && our_side == S_Side_LR::Right_Side )
		{
			au -> sender
			  = Server_Raw_Audio_Info::ST_Opponent_Online_Coach;
		}

		au -> direction = Degree(0.0);

		info -> set_audio_info( au );
	}
	else
	{
		ref_count_ptr<Server_Raw_Audio_Info>
			au( new Server_Raw_Audio_Info );

		au -> message = message;

		if ( strcmp( from.get() , "self" ) == 0 )
		{
			au -> sender    = Server_Raw_Audio_Info::ST_Self;
			au -> direction = Radian(0.0);
		}
		else if ( new_hear_format )
		{
			// XXX
			au -> sender = Server_Raw_Audio_Info::ST_Player;
			au -> direction = Degree(0.0);
		}
		else
		{
			au -> sender = Server_Raw_Audio_Info::ST_Player;
			if ( ! server_str_to_dir( &(au -> direction) ,
						  from.get() ) )
			{
				return( false );
			}
		}

		info -> set_audio_info( au );
	}

	return( true );
}



void   SServer_Message_Translator::analyze_body_info(
					      Server_Raw_Info * info ,
					      Lisp_Lexical_Object & input )
{
	//
	// Format Check
	//
	// Make sure input format is this.
	//	(* * (view_mode % %)
	//	     (stamina % %)
	//	     (speed %)
	//	     (kick %)
	//	     (dash %)
	//	     (turn %)
	//	     (say %))
	// * : list or symbol
	// % : symbol

	// start
	//
	struct  Sense_Body_Format
	{
		string		parameter;
		unsigned int	n_arguments;
	};

	static const  size_t		n_entry_ver4 = 7;
	static const  Sense_Body_Format	format_ver4[n_entry_ver4]
	 = { { "view_mode" , 2 },
	     { "stamina"   , 2 },
	     { "speed"     , 1 },
	     { "kick"      , 1 },
	     { "dash"      , 1 },
	     { "turn"      , 1 },
	     { "say"       , 1 } };

	static const  size_t		n_entry_ver5 = 9;
	static const  Sense_Body_Format	format_ver5[n_entry_ver5]
	 = { { "view_mode" , 2 },
	     { "stamina"   , 2 },
	     { "speed"     , 1 },
	     { "head_angle", 1 },
	     { "kick"      , 1 },
	     { "dash"      , 1 },
	     { "turn"      , 1 },
	     { "say"       , 1 },
	     { "turn_neck" , 1 } };

	static const  size_t		n_entry_ver6 = 9;
	static const  Sense_Body_Format	format_ver6[n_entry_ver6]
	 = { { "view_mode" , 2 },
	     { "stamina"   , 2 },
	     { "speed"     , 2 },
	     { "head_angle", 1 },
	     { "kick"      , 1 },
	     { "dash"      , 1 },
	     { "turn"      , 1 },
	     { "say"       , 1 },
	     { "turn_neck" , 1 } };

	static const  size_t		n_entry_ver7 = 12;
	static const  Sense_Body_Format	format_ver7[n_entry_ver7]
	 = { { "view_mode"  , 2 },
	     { "stamina"    , 2 },
	     { "speed"      , 2 },
	     { "head_angle" , 1 },
	     { "kick"       , 1 },
	     { "dash"       , 1 },
	     { "turn"       , 1 },
	     { "say"        , 1 },
	     { "turn_neck"  , 1 },
	     { "catch"      , 1 },
	     { "move"       , 1 },
	     { "change_view", 1 } };

	// Make sure input format is this.
	//	(* * * * * * * * *) || (* * * * * * * * * * *)
	int	sense_body_version;
	size_t	n_entry;
	const  Sense_Body_Format *	format;
	if ( input.list.size() == (2 + n_entry_ver4) )
	{
		sense_body_version = 4;
		format  = format_ver4;
		n_entry = n_entry_ver4;
	}
	else if ( input.list.size() == (2 + n_entry_ver5)
	       && input.list[4].list.size() == 2 )
	{
		sense_body_version = 5;
		format  = format_ver5;
		n_entry = n_entry_ver5;
	}
	else if ( input.list.size() == (2 + n_entry_ver6)
	       && input.list[4].list.size() == 3 )
	{
		sense_body_version = 6;
		format  = format_ver6;
		n_entry = n_entry_ver6;
	}
	else if ( input.list.size() == (2 + n_entry_ver7) )
	{
		sense_body_version = 7;
		format  = format_ver7;
		n_entry = n_entry_ver7;
	}
	else
	{
		info -> reset_info();
		return;
	}


	// Make sure input format is this.
	//	(* * (view_mode % %)
	//	     (stamina % %)
	//	     (speed %)
	//	     (kick %)
	//	     (dash %)
	//	     (turn %)
	//	     (say %))
	for ( unsigned int  i = 0  ;  i < n_entry  ;  i ++ )
	{
		Lisp_Lexical_Object &	entry = input.list[i + 2];
		Sense_Body_Format	entry_format = format[i];

		// Make sure each input entry's format is this.
		//	     (* *) || (* * *)
		if ( entry.type  !=  Lisp_Lexical_Object::List
		 ||  entry.list.size() != entry_format.n_arguments + 1)
		{
			info -> reset_info();
			return;
		}

		// Make sure each input entry's format is this.
		//	     (% %) || (% % %)
		for ( unsigned int  x = 0  ;  x < entry_format.n_arguments  ;
		      x ++ )
		{
			if ( entry.list[x].type
			     !=  Lisp_Lexical_Object::Symbol )
			{
				info -> reset_info();
				return;
			}
		}

		// Make sure each input entry's format is this.
		//	(view_mode % %) || (stamina % %) || ...
		if ( entry.list[0].symbol != entry_format.parameter )
		{
			info -> reset_info();
			return;
		}
	}
	//
	// Test input format
	//
	// end


	// no errors.
	// now, we know input data format is valid.


	//
	// Set parameter to Body_Info_from_Server instance.
	//
	// start

	ref_count_ptr<Server_Raw_Body_Info>
		body( new Server_Raw_Body_Info );

	string	vq = input.list[2].list[1].symbol;
	if ( vq == "high" )
	{
		body -> v_quality = View_Quality::High;
	}
	else if ( vq == "low" )
	{
		body -> v_quality = View_Quality::Low;
	}
	else
	{
		info -> reset_info();
		return;
	}


	string	va = input.list[2].list[2].symbol;
	if ( va == "wide" )
	{
		body -> v_width = View_Width::Wide;
	}
	else if ( va == "normal" )
	{
		body -> v_width = View_Width::Normal;
	}
	else if ( va == "narrow" )
	{
		body -> v_width = View_Width::Narrow;
	}
	else
	{
		info -> reset_info();
		return;
	}

	if ( sense_body_version == 4 )
	{
		if ( ! str_to_real( &(body -> stamina) ,
				    input.list[3].list[1].symbol )
		  || ! str_to_real( &(body -> effort) ,
				    input.list[3].list[2].symbol )
		  || ! str_to_real( &(body -> speed) ,
				    input.list[4].list[1].symbol )
		  || ! str_to_int ( &(body -> n_kick) ,
				    input.list[5].list[1].symbol )
		  || ! str_to_int ( &(body -> n_dash) ,
				    input.list[6].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn) ,
				    input.list[7].list[1].symbol )
		  || ! str_to_int ( &(body -> n_say) ,
				    input.list[8].list[1].symbol ) )
		{
			info -> reset_info();
			return;
		}

		body -> have_speed_angle = false;
		body -> neck_angle  = Radian(0.0);
		body -> n_turn_neck = 0;
		body -> n_catch = 0;
		body -> n_move = 0;
		body -> n_change_view = 0;
	}
	else if ( sense_body_version == 5 )
	{
		if ( ! str_to_real( &(body -> stamina) ,
				    input.list[3].list[1].symbol )
		  || ! str_to_real( &(body -> effort) ,
				    input.list[3].list[2].symbol )
		  || ! str_to_real( &(body -> speed) ,
				    input.list[4].list[1].symbol )
		  || ! server_str_to_dir( &(body -> neck_angle) ,
					  input.list[5].list[1].symbol )
		  || ! str_to_int ( &(body -> n_kick) ,
				    input.list[6].list[1].symbol )
		  || ! str_to_int ( &(body -> n_dash) ,
				    input.list[7].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn) ,
				    input.list[8].list[1].symbol )
		  || ! str_to_int ( &(body -> n_say) ,
				    input.list[9].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn_neck) ,
				    input.list[10].list[1].symbol ) )
		{
			info -> reset_info();
			return;
		}

		body -> have_speed_angle = false;
		body -> n_turn_neck = 0;
		body -> n_catch = 0;
		body -> n_move = 0;
		body -> n_change_view = 0;
	}
	else if ( sense_body_version == 6 )
	{
		if ( ! str_to_real( &(body -> stamina) ,
				    input.list[3].list[1].symbol )
		  || ! str_to_real( &(body -> effort) ,
				    input.list[3].list[2].symbol )
		  || ! str_to_real( &(body -> speed) ,
				    input.list[4].list[1].symbol )
		  || ! server_str_to_dir( &(body -> speed_angle) ,
				    input.list[4].list[2].symbol )
		  || ! server_str_to_dir( &(body -> neck_angle) ,
					  input.list[5].list[1].symbol )
		  || ! str_to_int ( &(body -> n_kick) ,
				    input.list[6].list[1].symbol )
		  || ! str_to_int ( &(body -> n_dash) ,
				    input.list[7].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn) ,
				    input.list[8].list[1].symbol )
		  || ! str_to_int ( &(body -> n_say) ,
				    input.list[9].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn_neck) ,
				    input.list[10].list[1].symbol ) )
		{
			info -> reset_info();
			return;
		}

		body -> have_speed_angle = true;

		body -> n_catch = 0;
		body -> n_move = 0;
		body -> n_change_view = 0;
	}
	else // if ( sense_body_version == 7 )
	{
		if ( ! str_to_real( &(body -> stamina) ,
				    input.list[3].list[1].symbol )
		  || ! str_to_real( &(body -> effort) ,
				    input.list[3].list[2].symbol )
		  || ! str_to_real( &(body -> speed) ,
				    input.list[4].list[1].symbol )
		  || ! server_str_to_dir( &(body -> speed_angle) ,
					  input.list[4].list[2].symbol )
		  || ! server_str_to_dir( &(body -> neck_angle) ,
					  input.list[5].list[1].symbol )
		  || ! str_to_int ( &(body -> n_kick) ,
				    input.list[6].list[1].symbol )
		  || ! str_to_int ( &(body -> n_dash) ,
				    input.list[7].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn) ,
				    input.list[8].list[1].symbol )
		  || ! str_to_int ( &(body -> n_say) ,
				    input.list[9].list[1].symbol )
		  || ! str_to_int ( &(body -> n_turn_neck) ,
				    input.list[10].list[1].symbol )
		  || ! str_to_int ( &(body -> n_catch) ,
				    input.list[11].list[1].symbol )
		  || ! str_to_int ( &(body -> n_move) ,
				    input.list[12].list[1].symbol )
		  || ! str_to_int ( &(body -> n_change_view) ,
				    input.list[13].list[1].symbol ) )
		{
			info -> reset_info();
			return;
		}

		body -> have_speed_angle = true;
	}


	//
	// Set parameter to Body_Info_from_Server instance.
	//
	// end

	//
	// Rest is to set data that we get only.
	//
	info -> set_body_info( body );
}


bool   SServer_Message_Translator::analyze_body_info_brief(
					      Server_Raw_Info * info ,
					      const string &  mes )
{
	//
	// Format
	//
	//	(sense_body % (view_mode % %)
	//	              (stamina % %)
	//	              (speed %)
	//	              (kick %)
	//	              (dash %)
	//	              (turn %)
	//	              (say %))
	// * : list or symbol
	// % : symbol

	long	info_time;

	ref_count_ptr<Server_Raw_Body_Info>
		body( new Server_Raw_Body_Info );

	// XXX: magic number 256
	char	view_quality[256 + 1];
	char	view_angle[256 + 1];

	double	head_ang;
	double	speed_ang;

	char	close_paren = '\0';

	// soccer server version 7
	if ( sscanf( mes.c_str() ,
		     "(sense_body %ld (view_mode %256s %256s"
		     " (stamina %lf %lf)"
		     " (speed %lf %lf) (head_angle %lf)"
		     " (kick %d) (dash %d) (turn %d)"
		     " (say %d) (turn_neck %d)"
		     " (catch %d) (move %d) (change_view %d)%c" ,
		     &info_time , view_quality , view_angle ,
		     &(body -> stamina) , &(body -> effort) ,
		     &(body -> speed) , &speed_ang , &head_ang ,
		     &(body -> n_kick) , &(body -> n_dash) ,
		     &(body -> n_turn) , &(body -> n_say) ,
		     &(body -> n_turn_neck) , &(body -> n_catch) ,
		     &(body -> n_move) , &(body -> n_change_view) ,
		     &close_paren ) == 17
	     && close_paren == ')' )
	{
		body -> neck_angle = Degree(head_ang);
		body -> speed_angle = Degree(speed_ang);
		body -> have_speed_angle = true;
	}
	// soccer server version 6
	else if ( sscanf( mes.c_str() ,
		     "(sense_body %ld (view_mode %256s %256s"
		     " (stamina %lf %lf)"
		     " (speed %lf %lf) (head_angle %lf)"
		     " (kick %d) (dash %d) (turn %d)"
		     " (say %d) (turn_neck %d)%c" ,
		     &info_time , view_quality , view_angle ,
		     &(body -> stamina) , &(body -> effort) ,
		     &(body -> speed) , &speed_ang , &head_ang ,
		     &(body -> n_kick) , &(body -> n_dash) ,
		     &(body -> n_turn) , &(body -> n_say) ,
		     &(body -> n_turn_neck) ,
		     &close_paren ) == 14
	     && close_paren == ')' )
	{
		body -> neck_angle = Degree(head_ang);
		body -> speed_angle = Degree(speed_ang);
		body -> have_speed_angle = true;

		body -> n_catch = 0;
		body -> n_move = 0;
		body -> n_change_view = 0;
	}
	// soccer server version 5
	else if ( sscanf( mes.c_str() ,
			  "(sense_body %ld (view_mode %256s %256s"
			  " (stamina %lf %lf)"
			  " (speed %lf) (head_angle %lf)"
			  " (kick %d) (dash %d) (turn %d)"
			  " (say %d) (turn_neck %d)%c" ,
			  &info_time , view_quality , view_angle ,
			  &(body -> stamina) , &(body -> effort) ,
			  &(body -> speed) , &head_ang ,
			  &(body -> n_kick) , &(body -> n_dash) ,
			  &(body -> n_turn) , &(body -> n_say) ,
			  &(body -> n_turn_neck) ,
			  &close_paren ) == 13
		  && close_paren == ')' )
	{
		body -> neck_angle = Degree(head_ang);
		body -> n_turn_neck = 0;
		body -> have_speed_angle = true;

		body -> n_catch = 0;
		body -> n_move = 0;
		body -> n_change_view = 0;
	}
	// soccer server version 4
	else if ( sscanf( mes.c_str() ,
			  "(sense_body %ld (view_mode %256s %256s"
			  " (stamina %lf %lf)"
			  " (speed %lf)"
			  " (kick %d) (dash %d) (turn %d)"
			  " (say %d)%c" ,
			  &info_time , view_quality , view_angle ,
			  &(body -> stamina) , &(body -> effort) ,
			  &(body -> speed) ,
			  &(body -> n_kick) , &(body -> n_dash) ,
			  &(body -> n_turn) , &(body -> n_say) ,
			  &close_paren ) == 11
		  && close_paren == ')' )
	{
		body -> neck_angle  = Radian(0.0);
		body -> n_turn_neck = 0;
		body -> have_speed_angle = false;

		body -> n_turn_neck = 0;
		body -> n_catch = 0;
		body -> n_move = 0;
		body -> n_change_view = 0;
	}
	else
	{
		return( false );
	}


	info -> set_time( info_time );


	if ( strcmp( view_quality , "high" ) == 0 )
	{
		body -> v_quality = View_Quality::High;
	}
	else if ( strcmp( view_quality , "low" ) == 0 )
	{
		body -> v_quality = View_Quality::Low;
	}
	else
	{
		return( false );
	}

	if ( strcmp( view_angle , "wide)" ) == 0 )
	{
		body -> v_width = View_Width::Wide;
	}
	else if ( strcmp( view_angle , "normal)" ) == 0 )
	{
		body -> v_width = View_Width::Normal;
	}
	else if ( strcmp( view_angle , "narrow)" ) == 0 )
	{
		body -> v_width = View_Width::Narrow;
	}
	else
	{
		return( false );
	}

	info -> set_body_info( body );

	return( true );
}


void   SServer_Message_Translator::analyze_error_info(
					      Server_Raw_Info *  info ,
					      Lisp_Lexical_Object &  input )
{
	if ( input.list.size() != 2
	  || input.list[1].type != Lisp_Lexical_Object::Symbol )
	{
		info -> reset_info();
		return;
	}

	ref_count_ptr<Server_Raw_Error_Info>
		err( new Server_Raw_Error_Info );
	err -> error_message = input.list[1].symbol;

	info -> set_error_info( err );
}


bool   SServer_Message_Translator::analyze_server_param_info_brief(
					Server_Raw_Info *  info ,
					const std::string &  mes )
{
	info -> set_time( 0 );

	ref_count_ptr<Server_Raw_Server_Param_Info>
		sp( new Server_Raw_Server_Param_Info );

	istrstream	mes_stream( mes.c_str() );
	string		open_paren_plus_server_param;

	mes_stream
		>> open_paren_plus_server_param
		>> (sp -> goal_width)
		>> (sp -> inertia_moment)
		>> (sp -> player_size)
		>> (sp -> player_decay)
		>> (sp -> player_rand)
		>> (sp -> player_weight)
		>> (sp -> player_speed_max)
		>> (sp -> player_accel_max)
		>> (sp -> stamina_max)
		>> (sp -> stamina_inc_max)
		>> (sp -> recover_init)
		>> (sp -> recover_dec_thr)
		>> (sp -> recover_min)
		>> (sp -> recover_dec)
		>> (sp -> effort_init)
		>> (sp -> effort_dec_thr)
		>> (sp -> effort_min)
		>> (sp -> effort_dec)
		>> (sp -> effort_inc_thr)
		>> (sp -> effort_inc)
		>> (sp -> kick_rand)
		>> (sp -> team_actuator_noise)
		>> (sp -> player_rand_factor_l)
		>> (sp -> player_rand_factor_r)
		>> (sp -> kick_rand_factor_l)
		>> (sp -> kick_rand_factor_r)
		>> (sp -> ball_size)
		>> (sp -> ball_decay)
		>> (sp -> ball_rand)
		>> (sp -> ball_weight)
		>> (sp -> ball_speed_max)
		>> (sp -> ball_accel_max)
		>> (sp -> dash_power_rate)
		>> (sp -> kick_power_rate)
		>> (sp -> kickable_margin)
		>> (sp -> control_radius)
		>> (sp -> control_radius_width)
		>> (sp -> max_power)
		>> (sp -> min_power)
		>> (sp -> max_moment)
		>> (sp -> min_moment)
		>> (sp -> max_neck_moment)
		>> (sp -> min_neck_moment)
		>> (sp -> max_neck_angle)
		>> (sp -> min_neck_angle)
		>> (sp -> visible_angle)
		>> (sp -> visible_distance)
		>> (sp -> wind_direction)
		>> (sp -> wind_force)
		>> (sp -> wind_angle)
		>> (sp -> wind_rand)
		>> (sp -> kickable_area)
		>> (sp -> catch_area_l)
		>> (sp -> catch_area_w)
		>> (sp -> catch_probability)
		>> (sp -> goalie_max_moves)
		>> (sp -> corner_kick_margin)
		>> (sp -> offside_active_area)
		>> (sp -> wind_factor_none)
		>> (sp -> wind_random)
		>> (sp -> say_cnt_max)
		>> (sp -> say_coach_message_size)
		>> (sp -> clang_win_size)
		>> (sp -> clang_define_win)
		>> (sp -> clang_meta_win)
		>> (sp -> clang_advice_win)
		>> (sp -> clang_info_win)
		>> (sp -> clang_mess_delay)
		>> (sp -> clang_mess_per_cycle)
		>> (sp -> half_time)
		>> (sp -> simulator_step)
		>> (sp -> send_step)
		>> (sp -> recv_step)
		>> (sp -> sense_body_step)
		>> (sp -> lcm_st)
		>> (sp -> say_message_size)
		>> (sp -> hear_max)
		>> (sp -> hear_inc)
		>> (sp -> hear_decay)
		>> (sp -> catch_ban_cycle)
		>> (sp -> slow_down_factor)
		>> (sp -> use_offside)
		>> (sp -> kick_off_offside)
		>> (sp -> offside_kick_margin)
		>> (sp -> audio_cut_dist)
		>> (sp -> dist_quantize_step)
		>> (sp -> dist_quantize_step_land)
		>> (sp -> dir_quantize_step)
		>> (sp -> dist_quantize_step_left)
		>> (sp -> dist_quantize_step_right)
		>> (sp -> land_quantize_step_left)
		>> (sp -> land_quantize_step_right)
		>> (sp -> dir_quantize_step_left)
		>> (sp -> dir_quantize_step_right)
		>> (sp -> coach_mode)
		>> (sp -> coach_with_referee_mode)
		>> (sp -> old_hear)
		>> (sp -> online_coach_look_step);

	char	close_paren = '0';

	mes_stream >> close_paren;

	if ( ! mes_stream
	     && open_paren_plus_server_param == "(server_param"
	     && close_paren != ')' )
	{
		return( false );
	}

	info -> set_server_param_info( sp );

	return( true );
}

bool   SServer_Message_Translator::analyze_player_param_info_brief(
					Server_Raw_Info *  info ,
					const std::string &  mes )
{
	info -> set_time( 0 );

	ref_count_ptr<Server_Raw_Player_Param_Info>
		pp( new Server_Raw_Player_Param_Info );

	istrstream	mes_stream( mes.c_str() );
	string		open_paren_plus_player_param;

	mes_stream
		>> open_paren_plus_player_param
		>> (pp -> player_types)
		>> (pp -> subs_max)
		>> (pp -> pt_max)
		>> (pp -> player_speed_max_delta_min)
		>> (pp -> player_speed_max_delta_max)
		>> (pp -> stamina_inc_max_delta_factor)
		>> (pp -> player_decay_delta_min)
		>> (pp -> player_decay_delta_max)
		>> (pp -> inertia_moment_delta_factor)
		>> (pp -> dash_power_rate_delta_min)
		>> (pp -> dash_power_rate_delta_max)
		>> (pp -> player_size_delta_factor)
		>> (pp -> kickable_margin_delta_min)
		>> (pp -> kickable_margin_delta_max)
		>> (pp -> kick_rand_delta_factor)
		>> (pp -> extra_stamina_delta_min)
		>> (pp -> extra_stamina_delta_max)
		>> (pp -> effort_max_delta_factor)
		>> (pp -> effort_min_delta_factor);

	char	close_paren = '0';

	mes_stream >> close_paren;

	if ( ! mes_stream
	     && open_paren_plus_player_param == "(player_param"
	     && close_paren != ')' )
	{
		return( false );
	}

	info -> set_player_param_info( pp );

	return( true );
}


bool   SServer_Message_Translator::analyze_player_type_info_brief(
					Server_Raw_Info *  info ,
					const std::string &  mes )
{
	info -> set_time( 0 );

	ref_count_ptr<Server_Raw_Player_Type_Info>
		pt( new Server_Raw_Player_Type_Info );

	istrstream	mes_stream( mes.c_str() );
	string		open_paren_plus_player_type;

	mes_stream
		>> open_paren_plus_player_type
		>> (pt -> id)
		>> (pt -> player_speed_max)
		>> (pt -> stamina_inc_max)
		>> (pt -> player_decay)
		>> (pt -> inertia_moment)
		>> (pt -> dash_power_rate)
		>> (pt -> player_size)
		>> (pt -> kickable_margin)
		>> (pt -> kick_rand)
		>> (pt -> extra_stamina)
		>> (pt -> effort_max)
		>> (pt -> effort_min);

	char	close_paren = '0';

	mes_stream >> close_paren;

	if ( ! mes_stream
	     && open_paren_plus_player_type == "(player_type"
	     && close_paren != ')' )
	{
		return( false );
	}

	info -> set_player_type_info( pt );

	return( true );
}


bool   SServer_Message_Translator::analyze_debug_info
					( Server_Raw_Info *  info ,
					  const std::string &  mes ,
					  S_Side_LR  our_side ,
					  int  self_player_number )
{
	ref_count_ptr<Server_Raw_Debug_Info>
		debug( new Server_Raw_Debug_Info );

	long		info_time;
	const char *	p = mes.c_str();
	int		n;

	if ( sscanf( p , "(fullstate %ld %n" , &info_time , &n ) != 1 )
	{
		return( false );
	}

	info -> set_time( info_time );

	p += n;

	double	sign = (our_side == S_Side_LR::Right_Side ? (-1.0) : (1.0));
	Angle	side_offset = (our_side == S_Side_LR::Right_Side ?
			       Degree(180.0) : Degree(0.0));

	for(;;)
	{
		if ( *p == ')' )
		{
			break;
		}


		// player
		{
			char	side;
			int	num;
			double	coord_x;
			double	coord_y;
			double	vel_x;
			double	vel_y;
			double	body_angle;
			double	neck_angle;
			double	stamina;
			double	effort;
			double	recover;

			if ( sscanf( p , " (%c_%d %lf %lf %lf %lf"
					 " %lf %lf %lf %lf %lf) %n" ,
				     &side , &num ,
				     &coord_x , &coord_y , &vel_x , &vel_y ,
				     &body_angle , &neck_angle ,
				     &stamina , &effort , &recover , &n )
			     == 11 )
			{
				p += n;

				Server_Raw_Debug_Info
					::Player_Debug_Info *	d;

				S_Side	s = lr_to_side( side , our_side );

				switch( s )
				{
				case S_Side::Our_Side:
					d = debug -> teammate;
					break;

				case S_Side::Opponent_Side:
					d = debug -> opponent;
					break;

				case S_Side::Unknown:
				default:
					return( false );
					break;
				}

				if ( num < 1 || MAX_PLAYER < num )
				{
					return( false );
				}

				d[num - 1].have_info = true;

				d[num - 1].true_coordinate
					.set( coord_x * sign ,
					      coord_y * sign );
				d[num - 1].true_velocity.set( vel_x * sign ,
							      vel_y * sign );
				d[num - 1].true_body_angle
					= (Degree( body_angle ) + side_offset)
					  .normalize();
				d[num - 1].true_neck_angle
					= (Degree( neck_angle ) + side_offset)
					  .normalize();
				d[num - 1].true_stamina = stamina;
				d[num - 1].true_effort = effort;

				if ( s == S_Side::Our_Side
				  && num == self_player_number )
				{
					debug -> self = d[num - 1];
				}

				continue;
			}
		}

		// ball
		{
			double	coord_x;
			double	coord_y;
			double	vel_x;
			double	vel_y;
			if ( sscanf( p , " (ball %lf %lf %lf %lf) %n" ,
				     &coord_x , &coord_y , &vel_x , &vel_y ,
				     &n )
			     == 4 )
			{
				p += n;

				debug -> ball_true_coordinate
						   .set( coord_x * sign ,
							 coord_y * sign );
				debug -> ball_true_velocity
						   .set( vel_x * sign ,
							 vel_y * sign );

				continue;
			}
		}

		if ( (p = strchr( p , ')' )) == static_cast<char *>(0) )
		{
			return( false );
		}

		p ++;
	}

	info -> set_debug_info( debug );

	return( true );
}




bool   SServer_Message_Translator::str_to_real( double *  num ,
						const string &  str )
{
	if ( str.find( " "  ) != string::npos
	  || str.find( "\t" ) != string::npos
	  || str.find( "\n" ) != string::npos )
	{
		return( false );
	}

	char *	c;
	double	double_num;
	double_num = strtod( str.c_str() , &c );

	if ( *c != '\0'
	 ||  double_num == HUGE_VAL )
	{
		return( false );
	}
	else
	{
		*num = double_num;

		return( true );
	}
}

bool   SServer_Message_Translator::str_to_int( int *  num ,
					       const string &  str )
{
	if ( str.find( " "  ) != string::npos
	  || str.find( "\t" ) != string::npos
	  || str.find( "\n" ) != string::npos )
	{
		return( false );
	}

	char *	c;
	long	long_num;
	long_num = strtol( str.c_str() , &c , 10 );

	if ( *c != '\0'
	 ||  (long_num == LONG_MIN || long_num == LONG_MAX)
	 ||  (long_num < INT_MIN || INT_MAX < long_num) )
	{
		return( false );
	}
	else
	{
		*num = static_cast<int>(long_num);

		return( true );
	}
}

bool   SServer_Message_Translator::str_to_long( long *  num ,
						const string &  str )
{
	if ( str.find( " "  ) != string::npos
	  || str.find( "\t" ) != string::npos
	  || str.find( "\n" ) != string::npos )
	{
		return( false );
	}

	char *	c;
	*num = strtol( str.c_str() , &c , 10 );

	if ( *c != '\0'
	  || (*num == LONG_MIN || *num == LONG_MAX) )
	{
		return( false );
	}
	else
	{
		return( true );
	}
}


bool   SServer_Message_Translator::server_str_to_dir( Angle *  dir ,
						      const string &  str )
{
	double	d = 0.0;

	if ( ! str_to_real( &d , str ) )
	{
		return( false );
	}

	(*dir) = Degree( d );

	return( true );
}


S_Side  SServer_Message_Translator::lr_to_side( char  l_or_r ,
						S_Side_LR  our_side )
{
	return( lr_to_side( string( 1 , l_or_r ) , our_side ) );
}


S_Side  SServer_Message_Translator::lr_to_side( const string &  l_or_r ,
						S_Side_LR  our_side )
{
	if ( l_or_r == "l"  &&  our_side == S_Side_LR::Left_Side
	  || l_or_r == "r"  &&  our_side == S_Side_LR::Right_Side )
	{
		return( S_Side::Our_Side );
	}
	else if ( l_or_r == "l"  &&  our_side == S_Side_LR::Right_Side
	       || l_or_r == "r"  &&  our_side == S_Side_LR::Left_Side )
	{
		return( S_Side::Opponent_Side );
	}
	else
	{
		return( S_Side::Unknown );
	}
}

int    SServer_Message_Translator::double_to_nearest_int( double  x )
{
	if ( x < 0.0 )
	{
		x -= 0.5;
	}
	else if ( x > 0.0 )
	{
		x += 0.5;
	}

	return( static_cast<int>( x ) );
}
