#include  "active_repository.h"
#include  "monitor_supplyer.h"
#include  "log_supplyer.h"
#include  "client_debug_view.h"
#include  <utility>
#include  <string>
#include  <cmath>
#include  <csignal>

const double	Active_Repository::PENALTY_AREA_LENGTH = 16.5 /* meter */;
const double	Active_Repository::PENALTY_AREA_HALF_WIDTH = 20.16 /* meter */;
const double	Active_Repository::FIELD_HALF_LENGTH = 52.5 /* meter */;

const int	Active_Repository::SIMULATOR_STEP_MSEC = 100 /* msec */;
const long	Active_Repository::AUTO_DROP_WAIT_STEP = 150 /* step */;
const double	Active_Repository::AUTO_DROP_CIRCLE_R = 2.0 /* meter */;
const int	Active_Repository::AUTO_DROP_COUNT = 5 /* sec */;
const long	Active_Repository::AUTO_KICK_OFF_WAIT_STEP = 300 /* step */;
const long	Active_Repository::AUTO_QUIT_WAIT_STEP = 30 /* step */;

Active_Repository::Active_Repository
		   ( const ref_count_ptr<Abstruct_Updator> &  updator ,
		     const std::string &  host_name ,
		     ip_port_number_t  server_port_number ,
		     ip_port_number_t  debug_port_number ,
		     int  server_protocol_version ,
		     bool  log_file_source ,
		     const std::string &  log_file_name ,
		     bool  receive_debug ,
		     bool  auto_drop_flag ,
		     bool  auto_kick_off_flag ,
		     bool  auto_quit_flag ,
		     pid_t  server_process_id )
	: updator( updator ) ,
	  callback( new Active_Repository::Callback( *this ) ) ,
	  param() ,
	  changed_receiver() ,
	  view_supplyer() , server_controll_message_handler() ,
	  debug_connection( debug_port_number ) ,
	  receive_debug_flag( receive_debug ) ,
	  game_over_flag( false ),
	  monitor_responsive( false ) ,
	  server_time( 0 ) ,
	  game_log() ,
	  old_play_mode( Monitor_View_Data::Monitor_Play_Mode::Null_Mode ) ,
	  ball_track( AUTO_DROP_WAIT_STEP ) ,
	  auto_drop_flag( auto_drop_flag && ! log_file_source ) ,
	  auto_drop_count( 0 ) ,
	  auto_kick_off_flag( auto_kick_off_flag && ! log_file_source ) ,
	  auto_kick_off_count( 0 ) ,
	  auto_quit_flag( auto_quit_flag ) , auto_quit_count( 0 ) ,
	  server_process_id( server_process_id ) ,
	  goal_time()
{
	if ( log_file_source )
	{
		this -> server_controll_message_handler
			= static_cast<Interactive_Monitor_View_Supplyer *>(0);

		this -> view_supplyer
			= new Log_Supplyer( updator , callback ,
					    log_file_name );
	}
	else
	{
		this -> server_controll_message_handler
			= new Monitor_Supplyer
				( updator , callback ,
				  host_name , server_port_number ,
				  server_protocol_version );

		this -> view_supplyer = server_controll_message_handler;
	}
}

Active_Repository::~Active_Repository()
{
}

void   Active_Repository::add_changed_event_receiver
		( const ref_count_ptr<Changed_Event_Receiver> &  receiver )
{
	this -> changed_receiver.push_back( receiver );
}

void   Active_Repository::changed_notify() const
{
	for ( size_t  i = 0  ;  i < this -> changed_receiver.size()  ;  i ++ )
	{
		this -> changed_receiver[i] -> notify();
	}
}

const std::map< long , ref_count_ptr<View_Snapshot> > &
					Active_Repository::get_game_log() const
{
	return( this -> game_log );
}

long   Active_Repository::get_server_time() const
{
	return( this -> server_time );
}

const std::vector< std::pair<long,S_Side_LR> > &
	Active_Repository::get_goal_time() const
{
	return( this -> goal_time );
}

bool   Active_Repository::game_over() const
{
	return( this -> game_over_flag );
}

bool   Active_Repository::monitor_connection_responsive() const
{
	return( this -> view_supplyer -> responsive() );
}

void   Active_Repository::update()
{
	//
	// receive monitor message
	//
	for(;;)
	{
		bool	have_s_param;
		SServer_Monitor_Log_Format::server_param_t	s_param;

		ref_count_ptr<const Monitor_View_Data>
			m = view_supplyer -> get( updator ,
						  &have_s_param , &s_param );
		if ( have_s_param )
		{
			param.overwrite_server_param
			  ( s_param.kick_power_rate
			      / SServer_Monitor_Log_Format::SHOWINFO_SCALE_2 ,
			    s_param.stamina_max
			      / SServer_Monitor_Log_Format::SHOWINFO_SCALE_2 ,
			    s_param.stamina_inc_max
			      / SServer_Monitor_Log_Format::SHOWINFO_SCALE_2 );
		}


		if ( ! m )
		{
			break;
		}

		long	t = m -> time_count;

		if ( t < 0 )
		{
			t = 0;
		}

		server_time = t;

		if ( ! game_log[ t ] )
		{
			game_log[ t ] = new View_Snapshot();
		}

		game_log[ t ] -> set_monitor_view_data( m );

		if ( old_play_mode != m -> play_mode
		  && m -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode
						    ::Before_Kick_Off )
		{
			auto_kick_off_count = 0;
		}

		if ( auto_kick_off_flag
		  && m -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode
						    ::Before_Kick_Off
		  && auto_kick_off_count >= 0 )
		{
			if ( (++ auto_kick_off_count)
			     >= AUTO_KICK_OFF_WAIT_STEP )
			{
				this -> send_start();
			}
		}

		if ( old_play_mode != m -> play_mode
		  || m -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode
						    ::Time_Over )
		{
			ball_track.clear();
		}

		if ( auto_drop_flag
		  && m -> play_mode
		     != Monitor_View_Data::Monitor_Play_Mode
						    ::Before_Kick_Off )
		{
			ball_track.push_front( m -> ball );

			while( ball_track.size()
			       > static_cast<size_t>(AUTO_DROP_WAIT_STEP) )
			{
				std::list<D2_Vector>::iterator
					it = ball_track.end();
				it --;
				ball_track.erase( it );
			}

			auto_drop_count = 0;
			std::list<D2_Vector>::iterator	it;
			for ( it = ball_track.begin()  ;
			      it != ball_track.end()  ;  it ++ )
			{
				if ( (*it - (m -> ball)).r()
				     >= AUTO_DROP_CIRCLE_R )
				{
					break;
				}

				auto_drop_count ++;
			}

			if ( auto_drop_count == AUTO_DROP_WAIT_STEP )
			{
				this -> auto_drop_ball
					  ( m -> ball ,
					    m -> play_mode );

			}
		}

		//
		// set goal time
		//
		if ( m -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode::Goal_Left
		  || m -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode::Goal_Right )
		{
			S_Side_LR	side;

			if ( m -> play_mode
			     == Monitor_View_Data
				     ::Monitor_Play_Mode::Goal_Left )
			{
				side = S_Side_LR::Left_Side;
			}
			else
			{
				side = S_Side_LR::Right_Side;
			}

			if ( find( goal_time.begin() ,
				   goal_time.end() ,
				   std::pair<long,S_Side_LR>
				       ( m -> time_count , side ) )
			     == goal_time.end() )
			{
				goal_time
				.push_back( std::pair<long,S_Side_LR>
						( m -> time_count , side ) );
			}
		}

		old_play_mode = m -> play_mode;


		//
		// auto quit
		//
		if ( auto_quit_flag
		  && m -> play_mode
		     == Monitor_View_Data::Monitor_Play_Mode
						    ::Time_Over )
		{
			game_over_flag = true;

			if ( (++ auto_quit_count)
			     >= AUTO_QUIT_WAIT_STEP )
			{
				exit( 0 );
			}
		}

		this -> changed_notify();
	}

	bool	resp = this -> monitor_connection_responsive();

	if ( monitor_responsive != resp )
	{
		this -> changed_notify();
	}

	monitor_responsive = resp;


	//
	// receive debug message
	//
	std::string	debug_message;

	while( this -> receive_debug_flag
	    && this -> debug_connection.recv( &debug_message , false ) )
	{
		ref_count_ptr<Client_Debug_View>
			c = new Client_Debug_View( &param );

		if ( ! (c -> decode( debug_message )) )
		{
			std::cerr << "Bad user debug message format["
				  << debug_message << "]" << std::endl;

			continue;
		}

		if ( c -> field_snapshot() )
		{
			long	t = c -> field_snapshot()
					-> game_state.current_time.main_step();

			if ( ! game_log[ t ] )
			{
				game_log[ t ] = new View_Snapshot();
			}

			game_log[ t ] -> set_client_view
				 ( new Debug_Client_Field_Recog( *c ) );
		}
	}
}

bool   Active_Repository::have_server_process_id() const
{
	return( this -> server_process_id != -1 );
}

void   Active_Repository::kill_server() const
{
	if ( this -> have_server_process_id() )
	{
		std::kill( server_process_id , SIGTERM );
	}
}

const SServer_Param &  Active_Repository::sserver_param() const
{
	return( param );
}

int    Active_Repository::send_start()
{
	this -> auto_kick_off_count = -1;

	if ( this -> server_controll_message_handler )
	{
		return( this -> server_controll_message_handler
							 -> send_start() );
	}
	else
	{
		return( -1 );
	}
}

int    Active_Repository::send_foul( double  x ,  double  y ,
				     S_Side_LR  side )
{
	this -> auto_drop_count = 0;

	if ( this -> server_controll_message_handler )
	{
		return( this -> server_controll_message_handler
						-> send_foul( x , y , side ) );
	}
	else
	{
		return( -1 );
	}
}


//
// Auto Drop Flag & Count
//
void   Active_Repository::set_auto_drop( bool  f )
{
	if ( this -> auto_drop_flag != f )
	{
		ball_track.clear();
	}

	this -> auto_drop_flag = f;
}

bool   Active_Repository::auto_drop_mode() const
{
	return( this -> auto_drop_flag );
}

long   Active_Repository::auto_drop_rest_step() const
{
	return( Active_Repository::AUTO_DROP_WAIT_STEP
			- this -> auto_drop_count );
}

long   Active_Repository::auto_drop_rest_second() const
{
	if ( this -> auto_drop_rest_step() == static_cast<long>(0) )
	{
		return( static_cast<long>(0) );
	}
	else
	{
		return( static_cast<long>
			( (this -> auto_drop_rest_step() - 1)
			   * SIMULATOR_STEP_MSEC / 1000
			  + 1 ) );
	}
}


//
// Auto Kick Off Flag & Count
//
void   Active_Repository::set_auto_kick_off( bool  f )
{
	this -> auto_kick_off_flag = f;
}

bool   Active_Repository::auto_kick_off_mode() const
{
	return( this -> auto_kick_off_flag );
}

long   Active_Repository::auto_kick_off_rest_step() const
{
	if ( this -> auto_kick_off_count < static_cast<long>(0) )
	{
		return( static_cast<long>(0) );
	}
	else
	{
		return( Active_Repository::AUTO_KICK_OFF_WAIT_STEP
			- this -> auto_kick_off_count );
	}
}

long   Active_Repository::auto_kick_off_rest_second() const
{
	if ( this -> auto_kick_off_rest_step() == static_cast<long>(0) )
	{
		return( static_cast<long>(0) );
	}
	else
	{
		return( static_cast<long>
			( (this -> auto_kick_off_rest_step() - 1)
			   * Active_Repository::SIMULATOR_STEP_MSEC / 1000
			  + 1 ) );
	}
}


//
// Auto Quit Flag & Count
//
void   Active_Repository::set_auto_quit( bool  f )
{
	this -> auto_quit_flag = f;
}

bool   Active_Repository::auto_quit_mode() const
{
	return( this -> auto_quit_flag );
}

long   Active_Repository::auto_quit_rest_step() const
{
	if ( this -> auto_quit_count < static_cast<long>(0) )
	{
		return( static_cast<long>(0) );
	}
	else
	{
		return( Active_Repository::AUTO_QUIT_WAIT_STEP
			- this -> auto_quit_count );
	}
}

long   Active_Repository::auto_quit_rest_second() const
{
	if ( this -> auto_quit_rest_step() == static_cast<long>(0) )
	{
		return( static_cast<long>(0) );
	}
	else
	{
		return( static_cast<long>
			( (this -> auto_quit_rest_step() - 1)
			   * Active_Repository::SIMULATOR_STEP_MSEC / 1000
			  + 1 ) );
	}
}




int   Active_Repository::auto_drop_ball
		( const D2_Vector &  ball ,
		  const Monitor_View_Data::Monitor_Play_Mode &  play_mode )
{
	ball_track.clear();

	D2_Vector	drop_point = ball;

	if ( fabs( ball.y() ) <= PENALTY_AREA_HALF_WIDTH
	  && fabs( ball.x() ) >= FIELD_HALF_LENGTH
					     - PENALTY_AREA_LENGTH )
	{
		double	drop_x;
		double	drop_y;

		drop_x = ((ball.x() >= 0) ?
			    + (FIELD_HALF_LENGTH - PENALTY_AREA_LENGTH)
			  : - (FIELD_HALF_LENGTH - PENALTY_AREA_LENGTH));

		drop_y = ((ball.y() >= 0) ?
			    + PENALTY_AREA_HALF_WIDTH
			  : - PENALTY_AREA_HALF_WIDTH );

		drop_point.set( drop_x , drop_y );
	}

	S_Side_LR	which_ball;
	switch( play_mode.advantage() )
	{
	case S_Side_LR::Left_Side:
		which_ball = S_Side_LR::Right_Side;
		break;

	case S_Side_LR::Right_Side:
		which_ball = S_Side_LR::Left_Side;
		break;

	case S_Side_LR::Unknown:
	default:
		which_ball = S_Side_LR::Unknown;
		break;
	}

	return( this -> send_foul( drop_point.x() , drop_point.y() ,
				   which_ball ) );
}
